<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.parameters
 */
/**
 * @file ParameterBuilder.php
 * @brief Parameter builder for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: ParameterBuilder.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * build Parameter from string definition.
 */

require_once("ficus/exception/ClassNotFoundException.php");
require_once("ficus/exception/IllegalTypeException.php");
require_once("ficus/parameters/ValidatableSimpleParameter.php");
require_once("ficus/parameters/ValidatableComplexParameter.php");
require_once("ficus/parameters/ArrayParameter.php");
require_once("ficus/lang/ClassLoader.php");

/**
 * @class Ficus_ParameterBuilder
 */
class Ficus_ParameterBuilder{

	//separator token
	const T_SEPARATOR = "|";
	//comma token
	const T_COMMA = ",";
	//level token
	const T_LEVEL = ":";
	//comment token
	const T_COMMENT = "#";
    //End of Definition token
	const T_EOD = "EOD";
	//array token
	const T_ARRAY = '/([a-zA-Z0-9_:\.]+)\[\]/';

	//name index
	const NAME = 0;
	//typename index
	const TYPENAME = 1;
	//validator index
	const VALIDATOR = 2;

	//validator class path
	const VALIDATOR_PATH = "ficus.validators.Ficus_%sValidator";
	//parameter class name
	const DATATYPE_CLASSNAME = "Ficus_%sParameter";
	//parameter class path
	const DATATYPE_PATH = "ficus.parameters.datatypes.Ficus_%sParameter";

    /**
     * @var validator path.
     */
    private static $validatorPath = '';

    /**
     * Set validator path.
     *
     * @param $validatorPath string validator path.
     */
    public static function setValidatorPath($validatorPath) {
        self::$validatorPath = $validatorPath;
    }

	/**
	 * create root parameter
	 * @param $name string name
	 * @param $validator Ficus_Validator validator for root param
	 * @return Ficus_ValidatableComplexParameter root parameter
	 */
	public static function buildRootParameter($name, $validator = null){
		return new Ficus_ValidatableComplexParameter($name, "", null,
													$validator);
	}

	/**
	 * create Simple Parameter with definition line
	 * @param $definition string definition of simple parameter
	 * @param $value mixed value of this param
	 * @return Ficus_ValidatableSimpleParameter simple parameter
	 */
	public static function buildSimpleParameter($definition, $value = null){
        $array = null;
		$set = self::parseOneLine($definition);

		if(preg_match(self::T_ARRAY, $set[self::TYPENAME], $matches)){
			$array = new Ficus_ArrayParameter($set[self::NAME],
											 $set[self::TYPENAME],
											 $value,
											 $set[self::VALIDATOR]);
			$set[self::TYPENAME] = $matches[1];
		}

		$typename = ucfirst($set[self::TYPENAME]);
		$className = sprintf(self::DATATYPE_CLASSNAME, $typename);
		$classPath = sprintf(self::DATATYPE_PATH, $typename);
		try{
			Ficus_ClassLoader::import($classPath);
			$class = new $className($set[self::NAME],
									$value,
									$set[self::VALIDATOR]);
		}catch(Ficus_ClassNotFoundException $e){
			$class = new Ficus_ValidatableSimpleParameter(
				$set[self::NAME], $set[self::TYPENAME],
				$value, $set[self::VALIDATOR]);
		}
		
		if($value){
			$class->validate();
		}
		if($array){
			$array->append($class);
			return $array;
		}
		return $class;
	}

	/**
	 * create Complex Parameter with definition line
	 * @param $definition string definition of complex parameter
	 * @param $value mixed parameter value
	 * @return Ficus_ValidatableComplexParameter complex parameter
	 */
	public static function buildComplexParameter($definition, $value = null){
		$set = self::parseOneLine($definition);
		if(preg_match(self::T_ARRAY, $set[self::TYPENAME], $matches)){
			$set[self::TYPENAME] = $matches[1];
		}
		return new Ficus_ValidatableComplexParameter($set[self::NAME],
													$set[self::TYPENAME],
													$value,
													$set[self::VALIDATOR]);
	}

	/**
	 * create Array Parameter with definition line
	 * @param $definition string definition of array parameter
	 * @param $value mixed parameter value
	 * @return Ficus_ValidatableComplexParameter array parameter
	 */
	public static function buildArrayParameter($definition, $value = null){
		$set = self::parseOneLine($definition);
		if(preg_match(self::T_ARRAY, $set[self::TYPENAME])){
			return new Ficus_ArrayParameter($set[self::NAME],
										   $set[self::TYPENAME],
										   $value,
										   $set[self::VALIDATOR]);
		}else{
			return null;
		}
	}

	/**
	 * create Typed Parameter
	 * @param $name string name of this parameter
	 * @param $typename string type
	 * @param $value string parameter value
	 * @param $validators array array of Ficus_Validator
	 * @return Ficus_ValidatableSimpleParameter requested Simple Parameter
	 */
	public static function buildSimpleParameterFromTypename(
		          $name, $typename, $value = null, $validators = null){
		$typename = ucfirst($typename);
		$className = sprintf(self::DATATYPE_CLASSNAME, $typename);
		$classPath = sprintf(self::DATATYPE_PATH, $typename);
		try{
			Ficus_ClassLoader::import($classPath);
			$class = new $className($name, $value, $validators);
		}catch(Ficus_ClassNotFoundException $e){
			$class = new Ficus_ValidatableSimpleParameter(
				$name, $typename, $value, $validators);
		}
		if($value){
			$class->validate();
		}
		return $class;
	}

	/**
	 * build parameter from PHP types
	 * @param $name string name of this parameter
	 * @param $value string value
     * @throw Ficus_IllegalTypeException illegal type.
	 */
	public static function buildSimpleParameterFromVariable($name, $value){
		$parameter = null;
		
		if(is_string($value) && class_exists($value)){
			return $this->buildParameterFromClassName($name, $value);
			
		}else if(is_array($value)){
			return $this->buildParameterFromArray($name, $value);
			
		}else if(is_bool($value)){
		    return new Ficus_BooleanParameter($name, $value);
			
		}else if(is_double($value)){
			//In php, there are no difference between double and float.
			//and is_double is aliase of is_float.
			//so we will never create FloatParameter.
			return new Ficus_DoubleParameter($name, $value);
			
		}else if(is_int($value)){
			//same reason as double, we will not create short, byte or any
			//other subclass of Integer.
			return new Ficus_IntegerParameter($name, $value);
			
		}else if(is_object($value)){
			return $this->buildParameterFromObject($name, $value);
			
		}else if(is_string($value)){
			return new Ficus_StringParameter($name, $value);
			
		}else{
			throw new Ficus_IllegalTypeException("this function may not accept functions, null or resource variables");
		}
	}
	
	/**
	 * build Parameter tree from object
	 * @param $name string name of this parameter
	 * @param $object mixed target object
     * @throw Ficus_IllegalArgumentException no object.
	 */
	public static function buildParameterFromObject($name, $object){
		if(is_object($object) === false){
			throw new Ficus_IllegalArgumentException("input must be an object");
		}

		ereg('_(.+)$', get_class($object), $matches);
		$root = new Ficus_ValidatableComplexParameter($name, $matches[1]);
		
		$class = new ReflectionClass($object);
		$props = $class->getProperties();

		foreach($props as $prop){
			$propName = $prop->getName();
			$getMethod = "get" . ucfirst($propName);

			if(method_exists($object, $getMethod)){
				$method = $class->getMethod($getMethod);
			}else if(method_exists($object, $propName)){
				$method = $class->getMethod($propName);
			}else{
				continue;
			}

			$value = $object->{$method->getName()}();
			// we havent ReflectionClass->hasMethod() yet, it will be
			// introduce in php 5.1.0.
			/*
			if($class->hasMethod($getMethod)){
				$value = $object->{$getMethod}();
			}else if($class->hasMethod($propName)){
				$value = $object->{$propName}();
			}else{
				continue;
			}
			*/
			$root->append(self::buildParameterFromVariable($propName, $value));
		}
		return $root;
	}

	/**
     * build Parameter tree from object
     * !!!! this method can not understand class properties when pear doc style
     * !!!! comment is not specified.
     * !!!! this problem blongs to PHP language nature. there is no type!
     * @param $name string name of this parameter
     * @param $className string class name
     * @throw Ficus_IllegalArgumentException no object.
     */
	public static function buildParameterFromClassName($name, $className){
		if(is_string($className) === false){
			throw new Ficus_IllegalArgumentException("input must be an object");
		}

		ereg('([^.]+_[^.]+)$', $className, $matches);
		$root = new Ficus_ValidatableComplexParameter($name, $matches[1]);

		$class = new ReflectionClass($className);
		$props = $class->getProperties();

		foreach($props as $prop){
			$propName = $prop->getName();
			$getMethod = "get" . ucfirst($propName);

			try{
				$method = $class->getMethod($getMethod);
			}catch(ReflectionException $e){
				try{
					$method = $class->getMethod($propName);
				}catch(ReflectionException $e){
					continue;
				}
			}

			//get property type infomation from doc comment
			//I want ReflectionMethod->hasDocComment() ...
			try{
				$comment = $method->getDocComment();
			}catch(ReflectionException $e){
				continue;
			}
            $validators = null;
			if(preg_match("/@xmltype\s+([^\s\|]+)\|?([^\s]+)?/",
								$comment, $matches)){
				$propClassName = $matches[1];
				if(isset($matches[2])){
					$validators = self::buildValidators($matches[2]);
				}
			}else if(preg_match("/@return\s+([^\s]+)/", $comment, $matches)){
				$propClassName = $matches[1];
			}
			if(class_exists($propClassName)){
				$root->append(self::buildParameterFromClassName(
								  $propName, $propClassName));
			}else{
				$root->append(self::buildSimpleParameterFromTypename(
							  $propName, $propClassName, null, $validators));
			}
		}
		return $root;
	}
	

	/**
	 * build Parameter from array
	 * @param $name string name of this parameter
	 * @param $array array array of parameters. must be an named
     * @throw Ficus_IllegalArgumentException no object.
	 */
	public static function buildParameterFromArray($name, $array){
		if(is_array($array) === false){
			throw new Ficus_IllegalArgumentException("this function accepts only an array");
		}

		if(is_array($name)){
			$set = each($name);
			$root = new Ficus_ValidatableComplexParameter(
				$set["key"], $set["value"]);
		}else{
			$root = self::buildRootParameter($name);
		}
		
		foreach($array as $key => $value){
			$root->append(self::buildParameterFromVariable($key, $value));
		}
		return $root;
	}
	
	/**
	 * create Parameter Tree from multi line definition
	 * @param $definition string definiton
	 * @return Ficus_ValidatableComplexParameter requested complex parameter
	 */
	public static function buildParameterFromDefinition($definition){
		$lines = self::explodeNewline($definition);
		return self::parseMultiLine(null, $lines, 0);
	}
	
	/**
	 * one line parser of parameter definition
	 *
	 * line format is like this
	 * &lt;ValiableName> &lt;SEPARATOR> &lt;TypeName> &lt;SEPARATOR>
     *              (&lt;ValidatorName> (&lt;COMMA> &lt;ValidatorName>)?)?
	 * ValiableName, TypeName and ValidatorName must be legal name
	 * @param $line string definition line
	 * @return array elements
     * @throw Ficus_IllegalArgumentException no object.
	 */
	private function parseOneLine($line){
		if(is_string($line) == false){
			throw new Ficus_IllegalArgumentException("input must be formated string");
		}

		$line = trim(str_replace(self::T_LEVEL, "", $line));
		preg_match('/^([^|]+)\|?((?<=\|)[^\|]+)?\|?((?<=\|).+)?$/',
				   $line, $elements);
		array_shift($elements);

        $name = "";
        $typeName = "";
		if(isset($elements[self::NAME])){
           $name = trim($elements[self::NAME]);
        }
        if(isset($elements[self::TYPENAME])){
            $typeName = trim($elements[self::TYPENAME]);
        }

        $validators = null;
		if(isset($elements[self::VALIDATOR])){
			$validators = self::buildValidators($elements[self::VALIDATOR]);
		}
		return array(self::NAME      => $name,
					 self::TYPENAME  => $typeName,
					 self::VALIDATOR => $validators);
	}

	/**
	 * load validators
	 * @param $names string validator names
	 * @return array array of Validator classes
     * @throw Ficus_IllegalArgumentException illegal validator name.
	 */
	private function buildValidators($names){
		if(is_string($names) === false){
			throw new Ficus_IllegalArgumentException("validator names must be string");
		}
		preg_match_all('/[a-zA-Z_]\w*\(.*\)|[a-zA-Z_][^,]*/',
					   $names, $matches);
		$validatorNames = $matches[0];
		$validators = array();
        $parameters = array();
		foreach($validatorNames as $name){
			if(empty($name)){
				continue;
			}
			ereg("([^\(]+)(.*)", trim($name), $regs);
			$className = ucfirst($regs[1]);
            // TODO use seasar.
            switch ($className) {
                case 'PerlRegex':
                case 'PostfixRegex':
                case 'RDF':
                case 'Regex':
                case 'Required':
                case 'SimpleType':
                case 'URI':
                $className = sprintf(self::VALIDATOR_PATH, $className);
                break;
                default:
                $className = sprintf(self::$validatorPath, $className);
                break;
            }

			//create parameter
			if(preg_match_all(
				   '/(?:[^\\\\]["\'])(.+[^\\\\])(?:["\'])|[^,\)\(]+/',
				   $regs[2], $regs)){
				foreach($regs[0] as $key => $reg){
					if(preg_match('/(?:[^\\\\])["\']/', $reg)){
						$parameters[$key] = trim($regs[1][$key]);
					}else{
						$parameters[$key] = trim($reg);
					}
				}
			}

			//load class
			try{
				array_push($validators,
						   Ficus_ClassLoader::load($className, $parameters));
			}catch(Ficus_ClassNotFoundException $e){
                throw new Ficus_IllegalArgumentException(
					"validator $className not found at $name line", $e);
			}
		}
		return $validators;
	}
	
	/**
	 * multi line parser of parameter definition
	 * this is a callback function.
	 *
	 * multi line format is below
	 * whitespace must be ignored
	 * first '\#' is comment
	 * $definition = <<< DEF
	 * \#first line will be used as wsdl message name
	 * StoreResourceRequest
	 * :id      |anyURI      |required,soyaURN
	 * :target  |anyURI      |required
	 * :creator |anyURI      |required
	 * :metadata|base64Binary|required,RDF
	 * :data    |base64Binary|required
	 * DEF;
	 *
	 * nested
	 * $definition = <<< DEF
	 * StoreResourceResponse
	 * :return|StoreResult|
	 * \#nested definition of StoreResult
	 * ::id      |anyURI      |soyaURN
	 * ::target  |anyURI      |
	 * ::creator |anyURI      |
	 * ::metadata|base64Binary|
	 * ::data    |base64Binary|
	 * DEF;
	 *
	 * array indicator is '[]'
	 * array simple type
	 * $definition = <<< DEF
	 * \#first line will be used as wsdl message name
	 * StoreResourceResponse
	 * :id      |anyURI[]     |required,SoyaID
	 * DEF;
	 *
	 * complex type array
	 * $definition = <<< DEF
	 * StoreResourceResponse
	 * :return|StoreResult[]|
	 * \#nested definition of StoreResult
	 * ::id      |anyURI      |soyaURN
	 * ::target  |anyURI      |
	 * ::creator |anyURI      |
	 * ::metadata|base64Binary|
	 * ::data    |base64Binary|
	 * DEF;
	 * and use like this.
	 * $response = $builder->createWithDefString($definition);
	 *
	 * create parameter tree.
	 * @param $root Ficus_ComplexParameter root parameter
	 * @param $lines array array of lines
	 * @param $baseLevel integer base depth level
	 * @return Ficus_ValidatableComplexParameter result
     * @throw Ficus_IllegalArgumentException illegal next level.
	 */
    private function parseMultiLine($root, $lines, $baseLevel){
		for($i = 0; $i < count($lines); $i++){
			if(empty($lines[$i])){
				continue;
			}

			$currentLine  = trim($lines[$i]);
			$currentLevel = self::getLevel($currentLine);
            $nextLevel = 0;
			//ignore comment
			if($currentLevel === self::T_COMMENT){
				continue;
			}
			//check for next line
			for($k = $i + 1; $k <= count($lines); $k++){
                if(isset($lines[$k]) == false){
                    $nextLevel = self::getLevel(null);
                }else{
                    $nextLevel = self::getLevel($lines[$k]);
                }
                
				if($nextLevel === self::T_COMMENT){
					continue;
				}else if($nextLevel === self::T_EOD){
					$nextLevel = $currentLevel;
					break;
				}else if($nextLevel - $baseLevel >= 2){
					throw new
						Ficus_IllegalArgumentException("nest level is wrong");
				}else{
					break;
				}
			}

			if($nextLevel > $baseLevel || $root == null){
				if($currentLevel == 0 && $root == null){
					$set = self::parseOneLine($currentLine);
					$root = self::buildRootParameter($set[self::NAME],
													  $set[self::VALIDATOR]);
					$root = self::parseMultiLine(
						$root,array_slice($lines, $i + 1), $nextLevel);
				}else{
					$array = self::buildArrayParameter($currentLine);
					if($array){
						$branch = self::buildComplexParameter($currentLine);
						$branch = self::parseMultiLine(
							$branch, array_slice($lines, $i + 1), $nextLevel);
						$array->append($branch);
						$root->append($array);
					}else{
						$branch = self::buildComplexParameter($currentLine);
						$root->append(self::parseMultiLine(
										  $branch, array_slice($lines, $i + 1),
										  $nextLevel));
					}
				}
				//jump next target
				for($i = $i + 1; $i < count($lines); $i++){
					$currentLevel = self::getLevel($lines[$i]);
					if($currentLevel == self::T_COMMENT){
						continue;
					}else if($currentLevel == self::T_EOD ||
							 $currentLevel < $nextLevel){
						break;
					}
				}
			}else if($nextLevel == $baseLevel){
				$root->append(self::buildSimpleParameter($currentLine));
			}else{
				return $root;
			}
		}
		return $root;
	}

	/**
	 * get Level of line
	 * @param $string string parameter definition
	 * @return integer level
	 */
	private function getLevel($string){
		if(empty($string)){
			return self::T_EOD;
		}
		if(ereg("^" . self::T_COMMENT, $string)){
			return self::T_COMMENT;
		}
		$level = ereg("^" . self::T_LEVEL . "+", $string, $regs);
		return $level ? strlen($regs[0]) : 0;
	}

	/**
	 * explode lines with new line spesific charactor.
	 * @param $lines string strings
	 * @return array array of string
     * @throw Ficus_IllegalTypeException no string.
	 */
	private function explodeNewline($lines){
		if(is_string($lines) == false){
			throw new Ficus_IllegalTypeException("input must be string");
		}

		if(strpos($lines, "\r\n")){
			$newline = "\r\n";
		}else if(strpos($lines, "\r")){
			$newline = "\r";
		}else if(strpos($lines, "\n")){
			$newline = "\n";
		}else{
			return;
		}
		return explode($newline, $lines);
	}
}
?>
