<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages
 */
/**
 * @file Cookie.php
 * @brief cookie object
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: Cookie.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * cookie object for page
 */

/**
 * @class Ficus_Cookie
 */
class Ficus_Cookie
{
    const VALUE = "value";
    const EXPIRE = "expire";
    const SECURE = "secure";
    const PATH = "path";
    const DOMAIN = "domain";
    const HTTPONLY = "httponly";
    
    /**
     * @var $name string name of cookie
     */
    protected $name = null;

    /**
     * @var $data string data sended
     */
    protected $values = array();

    /**
     * construct
     */
    public function __construct(){
    }
    
    /**
     * setcookie
     * @param $name string name of cookie
     * @param $value string value of cookie
     * @param $path string path
     * @param $domain string domain
     * @param $expire integer expire time, minuites
     * @param $secure boolean secure
     * @param $httponly boolean httponly
     */
    public function set($name, $value, $path = null, $domain = null,
                        $expire = 0, $secure = false, $httponly = false){
        $expire = ($expire === 0) ? 0 : time() - $expire * 60;
        if(is_null($path)){
            $path = new Ficus_String($_SERVER["REQUEST_URI"]);
            $path = ($path->endsWith("/")) ? $path : dirname($path) . "/";
        }
        $domain = (is_null($domain)) ? $_SERVER["SERVER_HOST"] : $domain;
        $secure = ($secure) ? 1 : 0;
        $httponly = ($httponly) ? 1 : 0;
        $this->data[$name] = array(self::VALUE => $value,
                                   self::EXPIRE => $expire,
                                   self::PATH => $path,
                                   self::DOMAIN => $domain,
                                   self::SECURE => $secure,
                                   self::HTTPONLY => $httponly);
        setcookie($name, $value, $expire, $path, $domain, $secure, $httponly);
    }

    /**
     * update cookie value
     * @param $name string name of cookie
     * @param $value string value of cookie
     * @param $expire integer expire time, minuites
     * @param $secure boolean secure
     * @param $httponly boolean httponly
     */
    public function update($name, $value, $path = null, $domain = null,
                           $expire = 0, $secure = false, $httponly = false){
        $this->remove($name);
        $this->set($name, $value, $path, $domain, $expire, $secure, $httponly);
    }

    /**
     * change cookie expires
     * @param $name string name of cookie to delete
     * @param $expire integer expire time, minuites
     */
    public function touch($name, $expire = null){
        $value = $this->data[$name][self::VALUE];
        $path = $this->data[$name][self::PATH];
        $domain = $this->data[$name][self::DOMAIN];
        $secure = $this->data[$name][self::SECURE];
        $httponly = $this->data[$name][self::HTTPONLY];
        if(is_null($expire)){
            $expire = $this->data[$name][self::EXPIRE];
        }
        $this->remove($name);
        $this->set($name, $value, $path, $domain, $expire, $secure, $httponly);
    }

    /**
     * change cookie expires
     * @param $expire integer expire time, minuites
     */
    public function touchAll($expire = null){
        foreach($this->data as $key => $data){
            $this->touch($key, $expire);
        }
    }

    /**
     * get cookie value
     * @param $name string name of cookie
     * @return string value of cookie
     */
    public function get($name){
        if($this->isExists($name) == false){
            throw new Ficus_IllegalArgumentException("cookie $name is not found.");
        }
        $value =  $_COOKIE[$name];
        $this->data[$name] = array(self::VALUE => $value);
        return $value;
    }
    
    /**
     * remove cookie
     * @param $name string name of cookie to delete
     */
    public function remove($name){
        setcookie($name, 0, time() - 3000);
        unset($this->data[$name]);
    }

    /**
     * remove all
     */
    public function removeAll(){
        foreach($this->data as $key => $data){
            $this->remove($key);
        }
    }
    
    /**
     * check is exists
     * @param $name string name of cookie to check
     */
    public function isExists($name){
        if(isset($_COOKIE[$name]) == false){
            return false;
        }
        return true;
    }
}
?>
