<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file AutoClassLoader.php
 * @brief search path for class
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: AutoClassLoader.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * load class from search path.
 */

require_once("ficus/lang/ClassLoader.php");
require_once("ficus/lang/ClassPathElement.php");

/**
 * @class Ficus_AutoClassLoader
 */
class Ficus_AutoClassLoader extends Ficus_ClassLoader
{
    /**
     * @var $paths array search paths
     */
    private static $paths = array();

    /**
     * @return array paths
     */
    public static function paths(){
        return self::$paths;
    }
    
    /**
     * add search path
     * @param $path Ficus_ClassPathElement search path
     * @param $prefix String prefix of class
     */
    public static function add($path, $prefix =""){
        Ficus_Assert::isType($path, "Ficus_ClassPathElement");
        self::$paths[$prefix][] = $path;
    }

    /**
     * import class
     * @param $classPath string class name to import
     * @param $classname string class if classqname and classname are not same
     * @throw Ficus_NotReadyException no initialized.
     * @throw Ficus_ClassNotFoundException class not found.
     * @throw Ficus_MultipleClassDefinitionFoundException multiple definition.
     */
    public static function import($classPath, $classname=""){
        try{
            $result = self::normalizeClassPath($classPath, "");
            $classPath = $result[0];
            $classname = $result[1];
            $prefix = self::getPrefix($classname);
            $index = strrpos($classPath, DIRECTORY_SEPARATOR);
            if($index !== false){
                $filename = substr($classPath, $index + 1);
            }else{
                $filename = $classPath;
            }
            
            $paths = array();
            if(empty($prefix) || isset(self::$paths[$prefix]) == false){
                foreach(self::$paths as $path){
                    $paths = array_merge($paths, $path);
                }
            }else{
                $paths = self::$paths[$prefix];
            }
            
            $classes = array();
            foreach($paths as $path){
                $found = $path->search($filename, true);
                $classes = array_merge($found, $classes);
            }
            if(empty($classes)){
                throw new Ficus_ClassNotFoundException("class is not found");
            }
            $class = $classes[0];
            require_once($class);
            if(!class_exists($classname)){
                return false;
            }
        }catch(Ficus_IllegalArgumentException $e){
            return false;
        }catch(Ficus_ClassNotFoundException $e){
            return false;
        }
        return true;
    }

    /**
     * clear
     */
    public static function clear(){
        self::$paths = array();
    }

    /**
     * get prefix from classname
     * @param $classname string classname
     * @return string prefix
     */
    protected static function getPrefix($classname){
        $index = strpos($classname, "_");
        if($index === false){
            return "";
        }
        $prefix = substr($classname, 0, $index);
        return $prefix;
    }
}
?>
