<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.db.s2dao.models
 */
/**
 * @file Dao.php
 * @brief Dao.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: S2DaoModelFactory.php 2 2007-07-11 10:37:48Z ishitoya $
 *
 */

require_once("ficus/db/s2dao/models/S2DaoModelConstants.php");
require_once("ficus/db/s2dao/models/S2DaoDataAccessObject.php");

/**
 * @class Ficus_S2DaoModelFactory
 */
class Ficus_S2DaoModelFactory implements Ficus_S2DaoModelConstants{
    /**
     * string prefix
     */
    protected static $prefix = "";

    /**
     * string modelPackage
     */
    protected static $modelPackage = "";
    
    /**
     * string autoPackage
     */
    protected static $autoPackage = "";
    

    /**
     * constructor
     */
    public function __construct(){
        self::$prefix = Ficus_Registry::search(self::REGISTRY_PREFIX);
        self::$modelPackage =
            Ficus_Registry::search(self::REGISTRY_PACKAGE_MODEL);
        self::$autoPackage =
            Ficus_Registry::search(self::REGISTRY_PACKAGE_AUTO);
        if(is_null(self::$prefix) ||
           is_null(self::$modelPackage) ||
           is_null(self::$autoPackage)){
            throw new Ficus_NotReadyException("s2dao.prefix, s2dao.package.model, s2dao.package.auto must be registered before runch");
        }
    }
    
    /**
     * get dao instance from argument
     * @param $subject string subject name
     * @param $auto boolean true if auto dto
     * @return DAO class
     */
    public function getDao($subject, $auto = false){
        $dao = $this->getClassName($subject, $auto,
                                   self::DAO_PACKAGE,
                                   self::DAO_SUFFIX);
        $dao = preg_replace("/.*?" . self::$prefix . "/", "", $dao);
        try{
            $dao = Ficus_S2DaoComponentFactory::getComponent($dao);
        }catch(S2Container_ComponentNotFoundRuntimeException $e){
            $dao = self::AUTO_PREFIX . $dao;
            $dao = Ficus_S2DaoComponentFactory::getComponent($dao);
        }
        return new Ficus_S2DaoDataAccessObject($dao);
    }

    /**
     * get dto instance from argument
     * @param $subject string subject name
     * @param $auto boolean true if auto dto
     * @return DTO class
     */
    public function getDto($subject, $auto = null){
        if(is_null($auto)){
            try{
                return $this->getDto($subject, false);
            }catch(Ficus_ClassNotFoundException $e){
                return $this->getDto($subject, true);
            }
        }
        $dto = $this->getClassName($subject, $auto,
                                  self::DTO_PACKAGE,
                                  self::DTO_SUFFIX);
        return Ficus_ClassLoader::load($dto);
    }

    /**
     * get entity instance from argument
     * @param $subject string subject name
     * @param $auto boolean true if auto dto
     * @return Entity class
     */
    public function getEntity($subject, $auto = null){
        if(is_null($auto)){
            try{
                return $this->getEntity($subject, false);
            }catch(Ficus_ClassNotFoundException $e){
                return $this->getEntity($subject, true);
            }
        }
        $entity = $this->getClassName($subject, $auto,
                                     self::ENTITY_PACKAGE,
                                     self::ENTITY_SUFFIX);
        return Ficus_ClassLoader::load($entity);
    }

    /**
     * get class name from argument
     * @param $subject string subject name
     * @param $auto boolean true if auto dto
     * @param $suffix string suffix of class
     * @return string classname
     */
    public static function getClassName($subject, $auto, $package, $suffix){
        $subject = ucfirst($subject);
        if($auto){
            $subject = self::AUTO_PREFIX . $subject;
            $pkg = self::$autoPackage . ".";
        }else{
            $pkg = self::$modelPackage . ".";
        }

        $package = $pkg . $package . ".";
        return $package . self::$prefix . $subject . $suffix;
    }
}
?>
