<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.cli.subversion
 */
/**
 * @file SVNStatusLine.php
 * @brief svn status line
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: SVNStatusLine.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * svn stasus line like
 * U  hogehoge.php
 * @see http://subversion.bluegate.org/doc/re30.html
 */

require_once("ficus/io/File.php");

/**
 * @class Ficus_SVNStatusLine
 */
class Ficus_SVNStatusLine
{
    const STATUS_ADDED = "A";
    const STATUS_DELETED = "D";
    const STATUS_UPDATED = "U";
    const STATUS_MERGE = "G";
    const STATUS_CONFLICT = "C";
    const STATUS_NONE = "_";
    const PATTERN = "/(A|D|U|G|C|_)(A|D|U|G|C| )\s+(.*)/";

    /**
     * @var $fileStatus string status of file
     */
    private $fileStatus;

    /**
     * @var $propertyStatus string status of property
     */
    private $propertyStatus;

    /**
     * @var $file Ficus_File file
     */
    private $file;

    /**
     * constructor
     * @param $line string status line
     */
    public function __construct($line){
        $this->parseLine($line);
    }

    /**
     * parse one line
     * @param $line string status line
     */
    protected function parseLine($line){
        if(preg_match(self::PATTERN, $line, $regs)){
            $this->fileStatus = $regs[1];
            $this->propertyStatus = $regs[2];
            $this->file = new Ficus_File($regs[3]);
        }else{
            throw new Ficus_IllegalArgumentException("this is not a subversion status line. $line");
        }
    }

    /**
     * get status
     * @return string status part eg:UU _U CM etc
     */
    public function getStatus(){
        return $this->fileStatus . $this->propertyStatus;
    }

    /**
     * get file status
     * @return string status of file
     */
    public function getFileStatus(){
        return $this->fileStatus();
    }

    /**
     * get property status
     * @param string status of property
     */
    public function getPropertyStatus(){
        return $this->propertyStatus;
    }

    /**
     * get file object
     * @return Ficus_File file
     */
    public function getFile(){
        return $this->file;
    }

    /**
     * is either file of property Added
     * @return boolean true if either file or property added
     */
    public function isAdded(){
        return $this->isFileAdded() || $this->isPropertyAdded();
    }

    /**
     * is file Added
     * @return boolean true if file added
     */
    public function isFileAdded(){
        return $this->fileStatus == self::STATUS_ADDED;
    }
    
    /**
     * is property Added
     * @return boolean true if property added
     */
    public function isPropertyAdded(){
        return $this->propertyStatus == self::STATUS_ADDED;
    }

    /**
     * is deleted either file or property
     * @return boolean true if file or property deleted
     */
    public function isDeleted (){
        return $this->isFileDeleted() || $this->propertyDeleted();
    }

    /**
     * is file deleted
     * @return boolean true if file deleted
     */
    public function isFileDeleted (){
        return $this->fileStatus == self::STATUS_DELETED;
    }

    /**
     * is property deleted
     * @return boolean true if property deleted
     */
    public function isPropertyDeleted (){
        return $this->propertyStatus == self::STATUS_DELETED;
    }

    /**
     * is updated
     * @return boolean true if anything updated
     */
    public function isUpdated(){
        return $this->isFileUpdated() || $this->isPropertyUpdated();
    }

    /**
     * is file updated
     * @return boolean true when file updated
     */
    public function isFileUpdated(){
        return $this->fileStatus == self::STATUS_UPDATED;
    }

    /**
     * is property updated
     * @return boolean true when property updated
     */
    public function isPropertyUpdated(){
        return $this->propertyStatus == self::STATUS_UPDATED;
    }

    /**
     * is property or file merged
     * @return boolean true if property or file merged
     */
    public function isMerged(){
        return $this->isFileMerged() || $this->isPropertyMerged();
    }

    /**
     * is file merged
     * @return boolean
     */
    public function isFileMerged(){
        return $this->fileStatus == self::STATUS_MERGE;
    }

    /**
     * is property merged
     * @return boolean true if property merged
     */
    public function isPropertyMerged(){
        return $this->propertyStatus == self::STATUS_MERGE;
    }

    /**
     * is something conflicted
     * @return boolean
     */
    public function isConflicted(){
        return $this->isFileConflicted() || $this->isPropertyConflicted();
    }

    /**
     * is file conflicted
     * @return boolean
     */
    public function isFileConflicted(){
        return $this->fileStatus == self::STATUS_CONFLICT;
    }

    /**
     * is property conflicted
     * @return boolean true if property conflicted
     */
    public function isPropertyConflicted(){
        return $this->propertyStatus == self::STATUS_CONFLICT;
    }
}
?>
