<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.cli
 */
/**
 * @file CommandLineManager.php
 * @brief command line command manager class
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: CommandLineManager.php 2 2007-07-11 10:37:48Z ishitoya $
 *
 * Command line command manager
 */

require_once("ficus/lang/reflect/ReflectionClass.php");
require_once("ficus/lang/reflect/annotation/ReflectionAnnotationProperty.php");
require_once("ficus/lang/reflect/annotation/ReflectionAnnotationClass.php");

/**
 * @class Soya_CommandLineManager
 */
class Soya_CommandLineManager{
    private $application = null;
    private $commands = array();
    private $defaultCommand = "";

    public function __construct($application){
        $this->application = $application;
        Ficus_ReflectionAnnotation::setPrefixes(array('Soya_', 'Ficus_'));
        $refClass = new Ficus_ReflectionClass($application);
        $class = new Ficus_ReflectionAnnotationClass($refClass);
        $methods = $class->getMethods();
        foreach($methods as $method){
            if($method->hasAnnotation("CommandLineCommand")){
                $annotation = $method->getAnnotation("CommandLineCommand");
                $command = $this->createExecutableCommand($annotation,
                                                          $application,
                                                          $method->getName());
                $this->add($command);
            }
        }

        $properties = $class->getProperties();
        foreach($properties as $property){
            if($property->hasAnnotation("CommandLineCommand")){
                $annotation = $property->getAnnotation("CommandLineCommand");
                $command = $this->createParameterCommand($annotation,
                                                         $property->getName());
                $this->add($command);
            }
        }
    }
        

    public function add($command){
        $this->commands[$command->name()] = $command;
    }

    private function createParameterCommand($annotation, $name){
        $argument =
          new Soya_CommandLineArgument($annotation->getShort(),
                                       $annotation->getLong());
        $command =
          new Soya_CommandLineCommand($name,
                                      Soya_CommandLineCommand::PARAMETER,
                                      null,
                                      $argument);
        return $command;
    }

    private function createExecutableCommand($annotation, $object, $name){
        $argument =
          new Soya_CommandLineArgument($annotation->getShort(),
                                       $annotation->getLong());
        $command =
          new Soya_CommandLineCommand($name,
                                      Soya_CommandLineCommand::EXECUTABLE,
                                      array($object, $name),
                                      $argument);
        if($annotation->isDefault()){
            $this->defaultCommand = $name;
        }
            
        return $command;
    }

    public function setDefaultCommand($command){
        $this->defaultCommand = $command;
    }

    public function execute($args){
        $prepare = array();
        foreach($this->commands as $command){
            if($command->check($args)){
                $prepare[$command->name()] = $command;
                if($command->type() == Soya_CommandLineCommand::PARAMETER){
                    $setter = "set" . $command->name();
                    $this->application->{$setter}($command->argument());
                }else if($command->type() == Soya_CommandLineCommand::SIMPLE){
                    $setter = "set" . $command->name();
                    $this->application->{$setter}(true);
                }
            }
        }

        $executed = false;
        foreach($prepare as $command){
            if($command->type() == Soya_CommandLineCommand::EXECUTABLE){
                $command->execute();
                $executed = true;
            }
        }

        if($executed == false){
            $this->commands[$this->defaultCommand]->execute();
        }
    }
}
?>
