<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.analyzer
 */
/**
 * @file SourceAnalyzer.php
 * @brief SourceAnalyzer of PHP.
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: SourceAnalyzer.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * Source analyzer of PHP.
 */
require_once('ficus/io/Dir.php');
require_once('ficus/io/PathPattern.php');
require_once('ficus/analyzer/AnalyzeData.php');
require_once('ficus/analyzer/AnnotationAnalyzeData.php');
require_once('ficus/analyzer/ClassAnalyzeData.php');
require_once('ficus/analyzer/FunctionAnalyzeData.php');
require_once('ficus/analyzer/ParameterAnalyzeData.php');
require_once('ficus/analyzer/ThrowAnalyzeData.php');
/**
 * @class Ficus_SourceAnalyzer
 */
class Ficus_SourceAnalyzer {
    /**
     * @var sources.
     */
    private $sources = array();

    /**
     * traverse files in path.
     *
     * @param $path traverse path.
     * @return array sources.
     */
    public function traverse($path) {
        $pattern = new Ficus_PathPattern();
        $pattern->addInclude("**.php");
        $dir = new Ficus_Dir($path, $pattern);
        $dir->each(array($this, 'analyze'));
        return $this->sources;
    }

    /**
     * analyze source.
     *
     * @param $fullpath fullpath.
     */
    public function analyze($fullpath) {
        $this->analyzeClass($fullpath);
        $this->callSource($this->sources[sizeof($this->sources)-1]);
    }

    /**
     * get sources.
     *
     * @return array sources.
     */
    public function getSources() { return $this->sources; }

    /**
     * to string.
     *
     * @return string preview.
     */
    public function __toString() {
        $buf = '';
        foreach ($this->sources as $source) {
            foreach ($source as $class) {
                $buf .= $class->__toString() . "\n";
            }
        }
        return $buf;
    }

    /**
     * call each source.
     *
     * @param $classes array of Ficus_ClassAnalyzeData in source.
     */
    public function callSource($classes) {}

    /**
     * analyze class.
     *
     * @param $path path.
     */
    public function analyzeClass($path) {
        $classes = array();

        $comment = '';
        $annotations = array();

        $tokens= token_get_all(file_get_contents($path));
        foreach ($tokens as $index => $token) {
            if ($token[0] == T_CLASS || $token[0] == T_INTERFACE) {

                $nextToken = self::nextTypeOf(T_STRING, $index, $tokens);
                $class = new Ficus_ClassAnalyzeData($path, $nextToken);
                $class->setAnnotations($annotations);
                $class->setComment($comment);
                $classes[] = $class;

                $comment = '';

            } else if ($token[0] == T_FUNCTION) {

                if ($class == null) {
                    $class = new Ficus_ClassAnalyzeData($path, null);
                    $classes[] = $class;
                }

                $nextToken = self::nextTypeOf(T_STRING, $index, $tokens);
                $function = new Ficus_FunctionAnalyzeData($nextToken);
                $function->setComment($comment);
                $function->setAnnotations($annotations);
                $function->setParameters(self::analyzeParameters($index, $tokens));
                $class->addFunction($function);

                $comment = '';

            } else if ($token[0] == T_DOC_COMMENT) {

                $comment = $token[1];
                $annotations = self::analyzeComment($token[1]);

            } else if ($token[0] == T_THROW) {

                if ($tokens[$index+2][0] != T_NEW) {
                    // T_THROW T_VARIABLE
                    // search backward.
                    $i = $index;
                    do {
                        $i--;
                    } while ($tokens[$i][0] != T_STRING);
                    $nextToken = $tokens[$i][1];
                } else {
                    // T_THROW T_NEW T_STRING
                    // search forward.
                    $nextToken = self::nextTypeOf(T_STRING, $index, $tokens);
                }
                $throw = new Ficus_ThrowAnalyzeData($nextToken);
                $function->addThrow($throw);

            }
        }
        $this->sources[] = $classes;
    }

    /**
     * analyze parameters.
     *
     * @param $index index of tokens.
     * @param $tokens PHP tokens.
     * @return array parameters.
     */
    static function analyzeParameters($index, $tokens) {
        $args = array();
        $arg = null;

        // T_FUNCTION T_WHITESPACE T_STRING (
        $index += 4;
        while (($type = $tokens[$index][0]) != '{' && $type != ';') {

            switch ($type) {

                case T_WHITESPACE:
                $index++;
                continue 2;
                break;

                case ',':
                case ')':
                case ':':
                case ';':
                case '(':
                case '=':
                break;

                case T_VARIABLE:
                $args[] = array();
                $arg = new Ficus_ParameterAnalyzeData($tokens[$index][1]);
                $args[sizeof($args)-1][] = $arg;
                break;

                default:
                if ($arg != null) {
                    $arg->setDefaultValue($tokens[$index][1]);
                }
                break;

            }

            /*
            if (is_string($type)) {
                echo $type . "\n";
            } else if ($type != T_WHITESPACE) {
                // T_VARIABLE
                // T_VARIABLE = T_STRING
                // T_VARIABLE = T_CONSTANT_ENCAPSED_STRING
                // T_VARIABLE = T_ARRAY()
                echo token_name($type) . "\n";
            }
            */

            $index++;
        }
        return $args;
    }

    /**
     * analyze comment.
     *
     * @param $comment
     * @return array annotations.
     */
    static function analyzeComment($comment) {
        $annotations = array();
        foreach (preg_split('/\r|\n/', $comment) as $line) {
            if (preg_match('/(@\S+)\s+(.*)$/', $line, $matches) > 0) {
                $annotation = new Ficus_AnnotationAnalyzeData($matches[1]);
                $annotation->setData($matches[2]);
                $annotations[] = $annotation;
            }
        }
        return $annotations;
    }

    /**
     * search type.
     *
     * @param $type find type.
     * @param $index index of tokens.
     * @param $tokens PHP tokens.
     * @return string value.
     */
    static function nextTypeOf($type, $index, $tokens) {
        $i = $index;
        do {
            $i++;
        } while ($tokens[$i][0] != $type);
        return $tokens[$i][1];
    }
}

?>
