#! /usr/local/bin/ruby -Ku
# encoding: utf-8

##
## feml2.rb: Bayesian Spam Filter
## Copyright (C) 2009, 2015 KOYAMA Hiro <tac@amris.co.jp>
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
##

require 'optparse';
require 'kconv';
require 'fileutils';
require 'pstore';
require 'date';
require "fiddle/import";

require 'bigdecimal';
require 'bigdecimal/util';

AmConf = Hash.new();

# __BEGIN_AM_CONF__
# =====================================================================
#
AmConf['User'] = 'tac';
AmConf['Base'] = '/usr/home/%user%';
AmConf['MailHome'] = AmConf['Base'] + '/Feml2';
AmConf['ConfHome'] = AmConf['MailHome'] + '/conf';

# =====================================================================
#
AmConf['OccurenceStore'] = AmConf['ConfHome'] + '/feml_store.pstore';

AmConf['UfromWhiteList'] = AmConf['ConfHome'] + '/ufrom_white_list.txt';
AmConf['FromWhiteList'] = AmConf['ConfHome'] + '/from_white_list.txt';
AmConf['SubjectWhiteList'] = AmConf['ConfHome'] + '/subject_white_list.txt';
AmConf['ReceivedWhiteList'] = AmConf['ConfHome'] + '/received_white_list.txt';

AmConf['UfromBlackList'] = AmConf['ConfHome'] + '/ufrom_black_list.txt';
AmConf['FromBlackList'] = AmConf['ConfHome'] + '/from_black_list.txt';
AmConf['SubjectBlackList'] = AmConf['ConfHome'] + '/subject_black_list.txt';
AmConf['ReceivedBlackList'] = AmConf['ConfHome'] + '/received_black_list.txt';

# =====================================================================
#
AmConf['LogFile'] = AmConf['MailHome'] + '/feml_log.txt';
AmConf['BackupMailDir'] = AmConf['MailHome'] + '/backup';
AmConf['CleanMailDir'] = AmConf['MailHome'] + '/clean';
AmConf['WhiteMailDir'] = AmConf['MailHome'] + '/white';
AmConf['DirtyMailDir'] = AmConf['MailHome'] + '/dirty';
AmConf['BlackMailDir'] = AmConf['MailHome'] + '/black';

# =====================================================================
#
AmConf['SpamIfEmptyBody'] = 'on';
		# 本文に空行しかない場合、スパムと看做す。
AmConf['SpamIfIllegalDate'] = 'on';
		# 日付 (Date:) が正しくない場合、スパムと看做す。
		# -- 日付文字列として解析不可
		# -- 現在日時との差が1.5日以上
AmConf['SpamIfShiftJisSubject'] = 'on';
		# 「Subject:」行に生のShift-JIS文字列が埋め込まれている場合、
		# スパムと看做す。
		#	EUC-JPなどもそのまま埋め込むのは規格違反だが、
		#	実際に目につくのはShift-JISばかり。
		# ## ISO-8859-1などの文字列がShift-JISと誤認される可能性は
		# ## あるかも知れない。

AmConf['SpamCharsets'] = 'utf-7,unicode-1-1-utf-7,gb2312,gb18030,koi8,koi8-r,koi8-u';
		# この符号化方式が用いられているものはスパムと看做す。

# =====================================================================
#
AmConf['MailBoxFile'] = '/var/mail/%user%';
		# スパムと判定しなかったとき、このファイルに追記する。
		# 但しこれが空であれば、$stdoutに書き出す。

AmConf['LockFile'] = '/tmp/FEML_LOCK_FILE.%user%.txt';
AmConf['ExceptionInfoFile'] = AmConf['MailHome'] + '/exception_info.txt';


# =====================================================================
#
AmConf['SpamThreshold'] = 0.95;

# =====================================================================
#
AmConf['MecabDicDir'] = '/usr/local/lib/mecab/dic/ipadic_neologd';

# __END_AM_CONF__

# ---------------------------------------------------------------------
#
FemlVersion = 'feml-2.0.0';

# =====================================================================
# 設定ファイルがあれば読み込む。
#
begin
	load "#{File::dirname(__FILE__)}/feml_config.rb";
rescue LoadError
end

# ======================================================================
# (参照) mecab.h
#
module Mecab
	extend	Fiddle::Importer;
	dlload	"libmecab.so";

	# ==============================================================
	#
	typealias "mecab_t", "void";
	typealias "mecab_node_t", "void";
	typealias "mecab_path_t", "void";
	typealias "mecab_dictionary_info_t", "void";

	extern	"mecab_t * mecab_new(int, char **)";
	extern	"mecab_t * mecab_new2(const char *)";
	extern	"const char * mecab_version()";
	extern	"const char * mecab_sparse_tostr(mecab_t *, const char *)";
	extern	"const mecab_node_t * mecab_sparse_tonode(mecab_t *, const char *)";
	extern	"const mecab_dictionary_info_t * mecab_dictionary_info(mecab_t *)";
	extern	"void mecab_destroy(mecab_t *)";

	# ==============================================================
	#
	MecabPathT = struct([
		"mecab_node_t * rnode",
		"mecab_path_t * rnext",
		"mecab_node_t * lnode",
		"mecab_path_t * lnext",
		"int cost",
		"float prob",
	]);

	MecabNodeT = struct([
		"mecab_node_t * prev",
		"mecab_node_t * next",
		"mecab_node_t * enext",
		"mecab_node_t * bnext",
		"mecab_path_t * rpath",
		"mecab_path_t * lpath",
		"const char * surface",
		"const char * feature",
		"unsigned int id",
		"unsigned short length",
		"unsigned short rlength",
		"unsigned short rcAttr",
		"unsigned short lcAttr",
		"unsigned short posid",
		"unsigned char char_type",
		"unsigned char stat",
		"unsigned char isbest",
		"float alpha",
		"float beta",
		"float prob",
		"float wcost",
		"long cost",
	]);

	MecabDictionaryInfoT = struct([
		"const char * filename",
		"const char * charset",
		"unsigned int size",
		"int type",
		"unsigned int lsize",
		"unsigned int rsize",
		"unsigned short version",
		"mecab_dictionary_info_t *next",
	]);
end

# ======================================================================
# 字句への分割
#
class Tokenizer
	# ==============================================================
	#
	def initialize(log_fp: nil, mecab_dic_dir: '')
		@log_fp = log_fp || open('/dev/null', 'w');
		@mecab_dic_dir = mecab_dic_dir;
	end

	# ==============================================================
	#
	def get_tokens(source)
		token_array = [];

##		@log_fp.print "Tokenizer#get_tokens: #{source}\n";
		opt = '-Ochasen';
		if @mecab_dic_dir != nil && ! @mecab_dic_dir.empty?
			opt += " -d#{@mecab_dic_dir}";
		end
		tagger = Mecab::mecab_new2(opt);

		dict_info_p = Mecab::mecab_dictionary_info(tagger);
		dict_info = Mecab::MecabDictionaryInfoT.new(dict_info_p);
		dict_filename = dict_info.filename.to_s;
		dict_charset = dict_info.charset.to_s;
		case dict_charset.downcase
		when 'utf8', 'utf-8'
			is_utf8_dict = true;
		else
			is_utf8_dict = false;
		end

		@log_fp.print "   dict_filename = #{dict_filename}\n";
		@log_fp.print "   dict_charset = #{dict_charset}\n";

		source_to_mecab = source;
		if ! is_utf8_dict
			source_to_mecab = source_to_mecab.toeuc;
		end

		node_p = Mecab::mecab_sparse_tonode(tagger, source_to_mecab);
		while ! node_p.null?
			node = Mecab::MecabNodeT.new(node_p);
			node_p = node.next;
			surface = node.surface.to_s(node.length).force_encoding('UTF-8');
			feature = node.feature.to_s.force_encoding('UTF-8');
			if ! is_utf8_dict
				surface = surface.toutf8;
				feature = feature.toutf8;
			end

			word_class = feature.split(/,/)[0];
						# 0: 品詞
						# 1: 品詞再分類1
						# 2: 品詞再分類2
						# 3; 品詞再分類3
						# 4: 活用形
						# 5: 活用型
						# 6: 原形
						# 7: 読み
						# 8: 発音
			next if surface.empty?;
			next if /\A\s*\z/ =~ surface;

			mark = '  ';
			if is_eligible_token(surface, word_class)
				token_array << surface;
				mark = '○';
			end
			@log_fp.print "#{mark} #{surface}: #{feature}\n";
		end
		Mecab::mecab_destroy(tagger);

		return token_array;
	end

	# --------------------------------------------------------------
	#
	def is_eligible_token(s, word_class)
		is_eligible = false

		case word_class
		when '名詞', '動詞', '形容詞', '副詞', '連体詞'
			is_eligible = true if 2 <= s.length;
			is_eligible = true if /\A[\p{Han}\p{Katakana}ー]+\z/ =~ s;
					#
					# - 英単語は「名詞」扱い。
					# - 漢字や片仮名は1文字であっても可。
					#
		when '助詞', '助動詞', '接続詞', '接頭詞', '感動詞'
			is_eligible = true if 2 <= s.gsub(/[^\p{Han}\p{Hiragana}\p{Katakana}ー]/, '').length;

		when 'BOS/EOS'
			is_eligible = false;		# just in case...

		else	# '記号', 'フィラー', 'その他',
			is_eligible = false;		# just in case...
		end

		# 品詞によらず、約物 (句読記号)・記号 (シンボル)・
		# 数字・区切り文字 (空白文字を含む) のみから成るものは不可。
		is_eligible = false if /\A[\p{P}\p{S}\p{N}\p{Z}]+\z/ =~ s;

		return is_eligible;
	end

end

# ======================================================================
# 字句データベース
#
class TokenDB

	# ==============================================================
	#
	def initialize(pstore_file_path, log_fp: nil)
		@pstore_file_path = pstore_file_path;
		@log_fp = log_fp || open('/dev/null', 'w');
	end

	# =============================================================
	# token_arrayの各字句について、修正スパム確率 (fw値) を求める:
	#	c: CLEAN出現数 (過去のCLEANメールに当該字句が出現した回数)
	#	s: SPAM出現数 (過去のSPAMメールに当該字句が出現した回数)
	#	pw = s / (c + s);   # 単純SPAM確率
	#	robx: 0.5; 字句が初めて現れたとき、メールがSPAMである予測確率
	#	robs: 1.0; robxの予測に与える強さ
	#	n: 当該字句の出現数 (= c + s)
	#	fw = (robs * robx + n * pw) / (robs + n)
	# 戻り値: [ [ <字句>, <修正スパム確率>, <CLEAN数>, <SPAM数> ], ...]
	#	robs = 0 とすれば単純スパム確率になる。
	#
	def get_biased_spam_probabilities(token_array, robx = 0.5, robs = 1.0)
		fw_array = [];
		db = PStore.new(@pstore_file_path);
		db.transaction do
			token_array.each do |token|
				occurences = db[token] || [0, 0, 0];
				c = occurences[0];
				s = occurences[1];
				n = c + s;
				pw = (n != 0) ? (s.to_f / n) : 0.0;
				fw = (robs * robx + n * pw) / (robs + n);
				fw_array << [token, fw, c, s];
			end
		end
		return fw_array;
	end

	# =============================================================
	# 字句の出現数を増減する。
	# mode := :add_clean | :add_spam | :sub_clean | :sub_spam |
	#         :add_spam_sub_clean | :add_clean_sub_spam
	#
	def update_occurences(token_array, mode = :add_clean)
		db = PStore.new(@pstore_file_path);
		today_mjd = Date.today.mjd;
		db.transaction do
			token_array.each do |token|
				occurences = db[token] || [0, 0, 0];
					# [ <clean出現数>, <spam出現数> ]
				case mode.to_s
				when 'add_clean'
					occurences[0] += 1;
				when 'add_spam'
					occurences[1] += 1;
				when 'sub_clean'
					occurences[0] =
					    [occurences[0] - 1, 0].max;
				when 'sub_spam'
					occurences[1] =
					    [occurences[1] - 1, 0].max;
				when 'add_spam_sub_clean'
					occurences[0] =
					    [occurences[0] - 1, 0].max;
					occurences[1] += 1;
				when 'add_clean_sub_spam'
					occurences[0] += 1;
					occurences[1] =
					    [occurences[1] - 1, 0].max;
				end
				occurences[2] = today_mjd;
				db[token] = occurences;
			end
		end
	end

	# ==============================================================
	# 最終更新から相当の日数が経過しても出現数が増えない語句*を消去する。
	#	* (出現数合計) * 100 < 経過日数 である語句
	#
	def reduce(factor = 100)
		db = PStore.new(@pstore_file_path);
		today_mjd = Date.today.mjd;
		db.transaction do |pstore|
			db.roots.each do |token|
				next if db[token] == nil;
				c = db[token][0];
				s = db[token][1];
				mjd = db[token][2];
				if (c + s) * factor < today_mjd - mjd
					pstore.delete(token);
				end
			end
		end
	end

	# ==============================================================
	#
	def clear()
		db = PStore.new(@pstore_file_path);
		db.transaction do |pstore|
			db.roots.each do |token|
				pstore.delete(token);
			end
		end
	end

	# ==============================================================
	# インポート
	#
	def import_from_fp(ifp = $stdin)
		import(ifp.readlines);
	end

	# ==============================================================
	#
	def import(str_array)
		FileUtils::mkdir_p(File::dirname(@pstore_file_path));
		db = PStore.new(@pstore_file_path);
		db.transaction do
			str_array.each do |buf|
				buf = buf.toutf8;
				next if /^#/ =~ buf;
				next if /^$/ =~ buf;
				if /(\d+)[\s:]+(\d+)[\s:]+(\d+)[\s:]+(.+)/ =~ buf
					c = $1.to_i;
					s = $2.to_i;
					mjd = $3.to_i;
					token = $4;
					db[token] = [c, s, mjd];
				end
			end
		end
	end

	# ==============================================================
	# エクスポート:
	# <clean出現数> : <spam出現数> : <更新(修正ユリウス)日> : <字句>
	#
	def export_to_fp(ofp = $stdout)
		str_array = export();
		str_array.each do |buf|
			ofp.print buf, "\n";
		end
	end

	# ==============================================================
	#
	def export()
		str_array = [];
		db = PStore.new(@pstore_file_path);
		db.transaction do
			db.roots.sort.each do |token|
				next if db[token] == nil;
				c = db[token][0] || 0;
				s = db[token][1] || 0;
				mjd = db[token][2] || 0;
				str_array << sprintf("%7d : %7d : %7d : %s",
					c, s, mjd, token);
			end
		end
		return str_array;
	end
end

# =====================================================================
# ホワイト・リスト/ブラック・リスト。
#
class BWList
	# =============================================================
	#
	def initialize(list_file_name)
		@list_file_name = list_file_name;
	end

	# =============================================================
	# ホワイト・リスト/ブラック・リストに、該当する項目が
	# 載っているか否か調べる。
	# target_text: 検査対象文字列 (一般にあるヘッダー文字列)。
	#
	def is_in_pattern(target_text)
		File::foreach(@list_file_name) do |buf|
			buf = buf.chomp.sub(/#.*/, '').strip;
			next if /^\*/ =~ buf;
			next if /^$/ =~ buf;

			if %r|^/(.+)/$| =~ buf
					# %r 記法までは組み込まない。
				pattern_string = $1;
				begin
					pattern = Regexp::new(pattern_string);
				rescue RegexpError
					## 正規表現として不正
					pattern = pattern_string;
				end
			else
				pattern = buf;
			end
			if target_text.index(pattern) != nil
				return true;
			end
		end
		return false;
	end
end

# =====================================================================
#
module Kconv
	# ==============================================================
	# MIME解読なしで文字コードを判定する。
	#
	def guess_m0(str)
		return guess(str.gsub(/=\?/, ''))
	end
	module_function :guess_m0;
end

# =====================================================================
# スパム確率の判定
#
class SpamProb

	# =============================================================
	#
	def initialize(pstore_file_path = './feml2.pstore',
				log_fp: nil, mecab_dic_dir: '')
		@pstore_file_path = pstore_file_path;
		@log_fp = log_fp || open('/dev/null', 'w');

		@tokenizer = Tokenizer.new(log_fp: @log_fp,
				mecab_dic_dir: mecab_dic_dir);
		@token_db = TokenDB.new(@pstore_file_path, log_fp: @log_fp);

		@token_array = [];
	end

	# =============================================================
	# SPAM/非SPAM確率を計算し、指標 [0, 1.0] を求める。
	#
	def probability(source)
		# =====================================================
		#
		@token_array = @tokenizer.get_tokens(source);
		return 0 if @token_array.size == 0;

		# =====================================================
		#
		s = probability_index(@token_array, @token_db);

		return s;
	end

	# =============================================================
	# 判定結果に従って字句データベースを更新する。
	# mode: :add_clean | :add_spam
	#
	def update_token_db(mode)
		@token_db.update_occurences(@token_array, mode);
	end

	# -------------------------------------------------------------
	# SPAM/非SPAM確率を計算し、指標を求める。
	#
	def probability_index(token_array, token_db)

		biased_prob_array =
			token_db.get_biased_spam_probabilities(token_array);
			# [ [ <字句>, <修正SPAM確率>, <CLEAN数>, <SPAM数> ], ... ]

		@log_fp.print "** 修正SPAM確率:DB出現数:字句\n";
		biased_prob_array.each do |token, biased_prob, n_clean, n_spam|
			@log_fp.printf("%9.5f%% : %7d : %s\n",
				biased_prob * 100, (n_clean + n_spam), token);
		end

		# -----------------------------------------------------
		# p = ((1 - f(w1)) * ... * (1 - f(wm)))
		# q = (f(w1) * ... * f(wn))
		# lp = log(p)
		# lq = log(q)
		#
		# 直接計算するとFloat::EPSILONよりも小さい値が出てしまうので、
		# 次のように変形して計算する。
		# q = Πf(w) = exp(Σlog(f(w)))
		# lq = log(q) = Σlog(f(w))
		# (同様に)
		# lp = Σlog(1 - f(w))
		#
		lp = 0;
		lq = 0;
		biased_prob_array.each do |token, biased_prob, n_clean, n_spam|
			lp += Math::log(1 - biased_prob);
			lq += Math::log(biased_prob);
		end
		n = biased_prob_array.size;
		@log_fp.print "[A] lp = Σlog(1 - fw) = #{lp}, lq = Σlog(fw) = #{lq}\n";
		@log_fp.print "[A] p = Π(1 - fw) = #{Math::exp(lp)}, q = Πfw = #{Math::exp(lq)}\n";
		@log_fp.print "[A] n = #{n} : 字句の個数\n";

		# -----------------------------------------------------
		# pp = 1 - chi-square(-2 * log(p), 2 * n) ; 非SPAM確率
		# qq = 1 - chi-square(-2 * log(q), 2 * n) ; SPAM確率
		# s = (1 + pp - qq) / 2                   ; 指標
		#
		pp = 1 - chi2q(-2.0 * lp, 2 * n);
		qq = 1 - chi2q(-2.0 * lq, 2 * n);
		@log_fp.print "[A] pp = 1 - χ2(-2 * lp, 2 * n) = #{pp}\n";
		@log_fp.print "[A] qq = 1 - χ2(-2 * lq, 2 * n) = #{qq}\n";
		s = (1.0 + pp - qq) / 2.0;
		@log_fp.print "[A] s = (1.0 + pp - qq) / 2.0 = #{s}\n";

		return s;
	end

	# --------------------------------------------------------------
	# χ^2
	#	m = x2 / 2 として、
	#	s = exp(-m) * Σ(m^i / i!)
	#		ただしΣは、i = 0 .. n / 2 - 1
	#	Floatで表せる範囲を超える場合があるので BigDecimal で計算
	#
	def chi2q(x2, n)
		m = x2 / 2.0;
		sig = 63;				# 精度
		t = BigDecimal::new("1.0");
		s = BigDecimal::new("1.0");
		(1 .. (n / 2) - 1).each do |i|
			## t *= (m / i);
			## s += t;
			t = t.mult((m / i).to_d, sig);
			s = s.add(t, sig);
##			@log_fp.print "[#{i}] t = #{t}, s = #{s}\n";
		end
##		@log_fp.print "[chi2q] Σ(m^i / i!) = #{s}\n";
##		s *= Math::exp(-m);
			# Math::exp(-m) は m が大きい (m > 745) とき
			# アンダーフローする (0 になる) ので、
			# e で m_int_part 回割り算し、その後
			# exp(m_frac_part) で割って小数部分を調整する。
			#
			# BigMath::exp(-m.to_d, sig) は、
			# m > 50 になると正しい結果が得られない (Bug?)。

		m_int_part = m.truncate;		# 整数部分
		m_frac_part = m - m_int_part;		# 小数部分
		e = Math::E.to_d;
		m_int_part.times do |j|
			s = s.div(e, sig);
		end
		s = s.div(Math::exp(m_frac_part).to_d, sig);
##		@log_fp.print "[chi2q] chi2q(x2 = #{x2}, ν = #{n}) => #{s}\n";

		if s.nan?
##			@log_fp.print "[chi2q] s is NaN\n";
			return 1.0
		else
			# s = s.to_f;
				# 単に to_f でFloatに変換しようとすると、
				# s が正の微小値 (Floatでアンダーフロー) のとき
				# Float::MIN (負) や Infinity になってしまう
				# 場合がある。
				# (例) chi2q(3900, 1070)
				#	3900を固定してnを変化させてみると、
				#	n=1070付近で結果が正しくなくなる。
				#
				#	s = 0.611494332424872765043423726400302103456441237131513406914493223E-316
				#	s = s.to_f;	# => Infinity

			f, x, y, z = s.split;
			s = f * (("0." + x).to_f) * (y ** z);

			return [0, [s, 1.0].min].max;
		end
	end

end

# ======================================================================
# Unix Mbox 形式で複数の電子メール・テキストをまとめたファイル
#
class Mbox

	# =============================================================
	#
	def initialize()
		@mbuf_array = [];
		@unix_from_indexes = [];
	end

	# =============================================================
	# 電子メールのテキストをファイルから読み込む。
	#
	def load(file_name)
		open(file_name, "r") do |fp|
			@mbuf_array = fp.readlines();
		end
		split_by_ufrom();
	end

	# =============================================================
	# 電子メールのテキストを、文字列の配列として読み込む。
	#
	def load_array(buf_array)
		@mbuf_array = buf_array;
		split_by_ufrom();
	end

	# -------------------------------------------------------------
	# Unix Mbox形式と想定して区切り行を見つけておく。
	# - 先頭または空行直後の「From 」が基準 (mail.localに合わせた仕様)。
	#
	def split_by_ufrom()

		@mbuf_array.each_index do |n|
			@mbuf_array[n].force_encoding('ASCII-8BIT');
					# この段階ではエンコーディングが不明。
			@mbuf_array[n] = @mbuf_array[n].chomp + "\n";
		end				# 改行コードを標準化する。

		@unix_from_indexes = [ 0 ];	# 「From 」行がない場合の用心
		prev_is_empty_line = false;
		@mbuf_array.each.with_index do |buf, n|
			if prev_is_empty_line && /^From / =~ buf
				@unix_from_indexes << n;
			end
			prev_is_empty_line = (buf == "\n");
		end
		@unix_from_indexes << @mbuf_array.size;	# 末尾
	end
	private :split_by_ufrom;

	# =============================================================
	#
	attr_reader	:unix_from_indexes;

	# =============================================================
	# 「From 」行で区切られた各メッセージについてのイテレーター。
	#
	def each_mbox()
		(0 ... @unix_from_indexes.size - 1).each do |i|
			from_index = @unix_from_indexes[i];
			to_index = @unix_from_indexes[i + 1];
			next if to_index <= from_index;

			yield @mbuf_array[from_index, to_index - from_index];
		end
	end
end

# =====================================================================
# 電子メール・テキストのヘッダー解析、デコード。
#
class ExDecoder

	# =============================================================
	#
	def initialize()
		@mbuf_array = [];
		@mbuf_header_array = [];
		@mbuf_body = '';
	end

	# =============================================================
	# 電子メールのテキストをファイルから読み込む。
	#
	def load(file_name, with_no_parse: false)
		fp = open(file_name, "r");
		load_fp(fp, with_no_parse: with_no_parse);
		fp.close;
	end

	# =============================================================
	# 電子メールのテキストをファイルから読み込む。
	#
	def load_fp(fp, with_no_parse: false)
		fp.set_encoding('ASCII-8BIT');
					# この段階ではエンコーディングが不明。
		@mbuf_array = fp.readlines();
		if ! with_no_parse
			parse_header();
			parse_body();
		end
	end

	# =============================================================
	# 電子メールのテキストを、文字列の配列として読み込む。
	#
	def load_array(buf_array, with_no_parse: false)
		@mbuf_array = Array.new();
		buf_array.each do |buf|
			buf.force_encoding('ASCII-8BIT');
					# この段階ではエンコーディングが不明。
			@mbuf_array << buf;
		end
		if ! with_no_parse
			parse_header();
			parse_body();
		end
	end

	# =============================================================
	#
	def parse()
		parse_header();
		parse_body();
	end

	# =============================================================
	# 電子メールのテキストをそのままファイルに書き出す。
	#
	def store(file_name)
		fp = open(file_name, "w");
		store_fp(fp);
		fp.close;
	end

	# =============================================================
	# 電子メールのテキストをそのままファイルに書き出す。
	#
	def store_fp(fp)
		@mbuf_array.each do |buf|
			fp.print buf;
		end
	end

	# -------------------------------------------------------------
	# mbufのエンコーディングを推測する。
	# (1) charsetの指定があれば一般にはこれがエンコーディングになるが、
	#     読み替えが必要な場合がある。
	#     (1-a) String#encode() で例外が発生するとき。
	#     (1-b) charsetの宣言とmbufのエンコーディングが矛盾している
	#           例が多いとき。
	# (2) charsetの指定がなければ、mbufから推測する。
	#
	def guess_from_encoding(mbuf, charset)

		# -----------------------------------------------------
		# charsetをもとにfrom_encを判断する。
		#
		charset_to_encoding = {
			'iso-2022-jp-2' => 'ISO-2022-JP',
				# Encoding::ISO_2022_JP_2はあるが、
				# Encoding::ConverterNotFoundErrorが起こる。
			'shift-jis' => 'Shift_JIS',
			'sjis' => 'Shift_JIS',
			'x-sjis' => 'Shift_JIS',
			'windows-932' => 'Shift_JIS',
			'gb2312' => 'GB18030',
			'gb18030' => 'GB18030',
			'windows-936' => 'GB18030',
			'euc-cn' => 'GB18030',
			'x-euc' => 'GB18030',
			'iso-8859-1' => 'Windows-1252',		# 西欧
			'us-ascii' => 'Windows-1252',
				# 宣言はISO-8859-1でも実際にはWindows-1252の
				# 文字列になっている例が多い。
			'iso-8859-2' => 'Windows-1250',		# 中欧
			'iso-8859-5' => 'Windows-1251',		# Cyrillic
			'koi8' => 'Windows-1251',		# Cyrillic
			'koi8-r' => 'Windows-1251',	# ロシア語/ブルガリア語
			'koi8-u' => 'Windows-1251',	# ウクライナ語
			'koi8-ru' => 'Windows-1251',	#
			'koi8-t' => 'Windows-1251',	# タジク語
			'koi8-cs' => 'Windows-1251',	# チェコ/スロバキア語
		};
			# Encoding::ConverterNotFoundError など、
			# 不都合が生じうるコードの読み替え仕様。

		from_enc = charset_to_encoding[(charset || '').downcase] ||
			charset || '';

		# -----------------------------------------------------
		# from_encが決まらない場合は元の文字列から推測する。
		#
		guess_charset_to_encoding = {
			Kconv::JIS => 'ISO-2022-JP',
			Kconv::SJIS => 'Shift_JIS',
			Kconv::EUC => 'EUC-JP',
			Kconv::ASCII => 'ASCII',
			Kconv::UTF8 => 'UTF-8',
			Kconv::UTF16 => 'UTF-16',
			Kconv::UNKNOWN => nil,
			Kconv::BINARY => nil,
		};
		if from_enc.empty?
			guess_enc = Kconv::guess_m0(mbuf);
			from_enc = guess_charset_to_encoding[guess_enc];
		end

		return from_enc;
	end

	# =============================================================
	# 文字コードをできるだけ変換し、UTF-8の文字列にする。
	# numchar-input、MIME-[BQ] も考慮する。
	#
	def buf_decode(mbuf, charset = nil)

		mbuf.force_encoding('ASCII-8BIT');

		# -----------------------------------------------------
		# 変換元文字列 (mbuf) のエンコーディング。
		#
		from_enc = guess_from_encoding(mbuf, charset);

		# -----------------------------------------------------
		# 「Content-Transfer-Encoding: 7bit」であっても、
		# 空白を &nbsp; (0xA0) で表して偽装する例があるので、
		# ここで変換する。
		#	「Content-Transfer-Encoding: 7bit」という記述自体も
		#	信頼できないので、これには依存せず文字列そのものを
		#	調べて判断する。
		#	Single-byte系エンコーディングすべてに対処するべきかも
		#	知れないが、経験則として、とりあえずwindows-1252のみ
		#	対象とする。
		#
		if from_enc.downcase == 'windows-1252' ||
		   /\A[\x00-\x7F\xA0]+\z/n =~ mbuf
			mbuf = mbuf.gsub(/\xA0/n, ' ');
		end

		# =====================================================
		#
		mbuf = mbuf.encode('UTF-8', from_enc, {
			:invalid => :replace,
			:undef => :replace,
			:replace => '<?>',
		});

		# -----------------------------------------------------
		# (不正なバイト列)
		# charsetにかかわらず、ISO/IEC 2022のエスケープ・シーケンスが
		# 混入している場合、便宜上 ISO-2022-JP 扱いとして、
		# 重ねて変換する。
		#
		if Kconv::isjis(mbuf)
			mbuf = mbuf.encode('UTF-8', 'ISO-2022-JP', {
				:invalid => :replace,
				:undef => :replace,
				:replace => '<?>',
			});
		end

		# -----------------------------------------------------
		# numchar-inputの変換
		# Unicodeにおける波ダッシュの問題に対処する。
		#
		mbuf = mbuf.gsub(/&#(\d+);/) do
			[ $1.to_i ].pack("U");
		end
		mbuf = mbuf.gsub('〜', '～');

		# -----------------------------------------------------
		# [MIME-B] (base64)、[MIME-Q] (quoted-printable) のデコード。
		#	=?<charset>?[BQ]?<encoded_string>?=
		#		その後にさらにMIMEエンコード文字列があれば、
		#		直後の空白を除去する。そのため、
		#			( +(?==\?))?
		#		により、先読み部分に「=?」がある場合のみ、
		#		空白の並びを含める形で照合する。
		#
		begin
			result = mbuf.gsub(/=\?(.+?)\?B\?([!->@-~]+?)\?=( +(?==\?))?/i) do
				m_charset = $1 || '';
				text = $2;
				s = buf_decode(text.unpack("m")[0], m_charset);
				s;
			end
		rescue ArgumentError
		else
			mbuf = result;
		end

		begin
			result = mbuf.gsub(/=\?(.+?)\?Q\?(.+?)\?=( +(?==\?))?/im) do
				m_charset = $1 || '';
				text = $2;
				s = buf_decode(text.unpack("M")[0], m_charset);
				s;
			end
		rescue ArgumentError
		else
			mbuf = result;
		end

		return mbuf;

	end

	# -------------------------------------------------------------
	# ヘッダー部分を解析する。
	#
	def parse_header()
		# -----------------------------------------------------
		# 空行(ヘッダーと本文の区切り)までを抽出する。
		# - 継続行は前行に連結しておく。
		# - デコードし、UTF-8に標準化する。
		#
		header_body_delim_i = nil;
		@mbuf_array.each.with_index do |mbuf, i|
			if mbuf == nil || mbuf == "\n"		## 空行
				header_body_delim_i = i;
				break;
			end
			if /^\s/ !~ mbuf || @mbuf_header_array.size == 0
					## 先頭が空白類でない場合のほか、
					## 最初の行が継続行の形をしている
					## (おそらく不正な状況) 場合も、
					## 非継続行として扱う。
				@mbuf_header_array << mbuf;
			else		## 継続行
				@mbuf_header_array[-1] += mbuf;
			end
		end

		@from_raw_enc = Kconv::UNKNOWN;
		@subject_raw_enc = Kconv::UNKNOWN;
		@mbuf_header_array.each.with_index do |mbuf, i|
			m_charset = 'iso-8859-1';
			if /^From:/i =~ mbuf
				m_charset = nil;
				@from_raw_enc = Kconv::guess_m0(mbuf);
			end
			if /^Subject:/i =~ mbuf
				m_charset = nil;
				@subject_raw_enc = Kconv::guess_m0(mbuf);
			end
			@mbuf_header_array[i] = buf_decode(mbuf, m_charset);
		end
				# MIME-Qエンコードが継続行にまで続いている
				# 場合があるので、連結した後でデコードする。
				# ヘッダーのコードはiso-8859-1が原則だが、
				# Shift-JISがそのまま埋め込まれている例が
				# 多数あるので、Subject/From行については
				# 例外として扱う。
				#

		if header_body_delim_i == nil		# 空行がみつからない
			@mbuf_body_array = [];
		else
			@mbuf_body_array = @mbuf_array[header_body_delim_i + 1 .. -1];
		end

		# -----------------------------------------------------
		# ヘッダー行から必要な情報を抽出しておく。
		# - エンコーディング。
		# - Content-Type: multipartメッセージの区切り行。
		#
		@content_type_major = 'text';
		@content_type_minor = 'plain';
		@charset = '';
		@boundary = '';
		@mbuf_header_array.each do |mbuf|
			case mbuf
			when /^Content-Transfer-Encoding:\s*(.+)$/i
				@encoding = $1;
			when /^Content-Type:\s*(.+?)\/(.+?);(.*)?/im
				@content_type_major = $1.downcase;
				@content_type_minor = $2.downcase;
				rest = $3;
				if /charset\s*=\s*([\'\"]*)([^\s\1\;]+)\1/i =~ rest
					@charset = $2;
				end
				if /boundary\s*=\s*"(.+?)"/i =~ rest
					@boundary = $1;
				elsif /boundary\s*=\s*([^\s;]+)/i =~ rest
					@boundary = $1;
				end
			when /^Content-Type:\s*(\w+?)\/(\w+)/im
				@content_type_major = $1.downcase;
				@content_type_minor = $2.downcase;
			when /^Content-Type:/im
				@content_type_major = 'unknown';
				@content_type_minor = 'unknown';
			end
		end

		# -----------------------------------------------------
		# multipartの境界行を調べる。
		#
		@boundary_lines = Array.new();
		if @boundary != nil && @boundary != ''
			b_regex = Regexp.new("--#{Regexp.escape(@boundary)}(--)?$");
			end_boundary_appeared = false;
			@mbuf_body_array.each_index do |i|
				mbuf = @mbuf_body_array[i];
				if b_regex =~ mbuf
					end_boundary_appeared = true if $1 == '--';
					@boundary_lines << i;
				end
			end
			if ! end_boundary_appeared
				@boundary_lines << @mbuf_body_array.size;
					# 末尾を示す境界行が欠落している
					# 場合は補う。
			end
		end

##		print "[from_raw_enc] #{@from_raw_enc.inspect}\n";
##		print "[subject_raw_enc] #{@subject_raw_enc.inspect}\n";
##		print "[encoding] #{@encoding}\n";
##		print "[content_type_major] #{@content_type_major}\n";
##		print "[content_type_minor] #{@content_type_minor}\n";
##		print "[boundary] #{@boundary}\n";
##		print "[charset] #{@charset}\n";
##		print "[boundary_lines] #{@boundary_lines.join(', ')}\n";

	end
	private :parse_header;

	# =============================================================
	# 「Subject:」行のエンコーディング (MIMEデコードする前) を返す。
	#	Shift-JISの文字列がそのまま埋め込まれている例が多いので、
	#	これも判定材料として利用する。
	#
	def raw_header_encoding()
		kconv_name_tbl = {
			Kconv::ASCII => 'ascii',
			Kconv::EUC => 'euc-jp',
			Kconv::JIS => 'iso-2022-jp',
			Kconv::SJIS => 'shift_jis',
			Kconv::UTF8 => 'utf-8',
			Kconv::UTF16 => 'utf-16',
			Kconv::UTF32 => 'utf-32',
			Kconv::BINARY => 'binary',
			Kconv::UNKNOWN => 'unknown',
		};
		s = kconv_name_tbl[@subject_raw_enc] || 'unknown';

		return s;
	end

	# =============================================================
	# ヘッダーの内容の要約; 主としてmultipartの場合を想定。
	#
	def header_summary_line()
		return <<-"__SUMMARY__";
## <feml content-type="#{@content_type_major}/#{@content_type_minor}" charset="#{@charset}" encoding="#{@encoding}" />
		__SUMMARY__
	end

	# -------------------------------------------------------------
	# ヘッダーの字句解析について:
	#
	# multipart-boundaryの構文: rfc2046.txt
	#
	#	boundary := 0*69<bchars> bcharsnospace
	#
	#	bchars := bcharsnospace / " "
	#
	#	bcharsnospace := DIGIT / ALPHA / "'" / "(" / ")" /
	#			"+" / "_" / "," / "-" / "." /
	#			"/" / ":" / "=" / "?"
	#
	# rfc2045.txt
	#
	#	parameter := attribute "=" value
	#
	#	attribute := token
	#			; Matching of attributes
	#			; is ALWAYS case-insensitive.
	#
	#	value := token / quoted-string
	#
	#	token := 1*<any (US-ASCII) CHAR except SPACE, CTLs, or tspecials>
	#
	#	tspecials :=  "(" / ")" / "<" / ">" / "@" /
	#			"," / ";" / ":" / "\" / <">
	#			"/" / "[" / "]" / "?" / "="
	#			; Must be in quoted-string,
	#			; to use within parameter values
	#
	# したがって、引用符で囲まずに記述できるboundaryを構成する文字は:
	#		DIGIT / ALPHA / "'" / "+" / "_" / "-" / "."
	#

	# -------------------------------------------------------------
	#
	def each_multipart()
		(0 .. @boundary_lines.size - 2).each do |i|
			beg_index = @boundary_lines[i] + 1;
			end_index = @boundary_lines[i + 1] - 1;
			yield(@mbuf_body_array[beg_index .. end_index]);
		end
	end
	protected :each_multipart;

	# -------------------------------------------------------------
	# Content-Typeを基準として、判断材料として使うか否かを調べる。
	#
	def content_type_is_acceptable()
		is_acceptable = false;
		case @content_type_major
		when 'text'
			case @content_type_minor
			when 'plain'
				is_acceptable = true;
			when 'html'
				is_acceptable = true;
			end
		when 'message'
			is_acceptable = true;
		end
		return is_acceptable;
	end
	private :content_type_is_acceptable;

	# -------------------------------------------------------------
	# 本文を解析する。
	# - multipartの場合、Content-Typeがtextでないチャンクは削除。
	# - base64、quoted-printableでエンコードされている部分はデコード。
	# - 元の文字エンコーディングにかかわらず、UTF-8に変換。
	#
	def parse_body()
		@mbuf_body = '';

		top_line = (@mbuf_body_array[0] || '');
		if /^(MAIL FROM:|QUIT)/ =~ top_line
			sub_ex_d = ExDecoder.new();
			sub_ex_d.load_array(@mbuf_body_array);
			@mbuf_body += 'X-From: ';
			@mbuf_body += sub_ex_d.get_header('From:') || "\n";
			@mbuf_body += 'X-Subject: ';
			@mbuf_body += sub_ex_d.get_header('Subject:') || "\n";
			@mbuf_body += "\n";
			@mbuf_body += sub_ex_d.get_decoded_body;
				#
				# (不正な形式)
				# 本文中に「MAIL FROM:」や「QUIT」から始まる
				# SMTPとのやりとりが入っている。
				# 不正な形式ではあるが、このようなスパムが
				# 多数現れるので、特別扱いして解析する。
				#

		elsif @boundary_lines.size != 0
			self.each_multipart do |mbuf_array|
				sub_ex_d = ExDecoder.new();
				sub_ex_d.load_array(mbuf_array);
				@mbuf_body += %Q|## <feml boundary="#{@boundary}" />\n|;
				@mbuf_body += sub_ex_d.header_summary_line;
				@mbuf_body += sub_ex_d.get_decoded_body;
			end
			@mbuf_body += %Q|## <feml end-boundary="#{@boundary}" />\n|;

		elsif @content_type_major == "message" &&
		      @content_type_minor == "rfc822"
			sub_ex_d = ExDecoder.new();
			sub_ex_d.load_array(@mbuf_body_array);
			@mbuf_body += "## ====================\n";
			@mbuf_body += sub_ex_d.get_decoded_body;
			@mbuf_body += "// ====================\n";

		elsif content_type_is_acceptable()
			tbuf = '';
			case @encoding
			when "base64"
				@mbuf_body_array.each do |buf|
					tbuf += buf.unpack('m*')[0];
				end
			when "quoted-printable"
				@mbuf_body_array.each do |buf|
					tbuf += buf.unpack('M*')[0];
				end
			else
				@mbuf_body_array.each do |buf|
					tbuf += buf;
				end
			end
			tbuf += "\n" if tbuf[-1, 1] != "\n";
			@mbuf_body += buf_decode(tbuf, @charset).
					gsub(/\r\n|\r/m, "\n");
							# 改行の標準化
		else
			### 何も出力しない
		end

		return @mbuf_body;
	end
	protected :parse_body;

	# =============================================================
	# ヘッダーの値(デコード済み)を取得する。
	# 例:
	#	s = dec.get_header('Subject:')	# => (「Subject:」行の内容)
	# 該当するヘッダーがない場合はnil。
	#
	def get_header(key)
		@mbuf_header_array.each do |mbuf|
			if /^#{key}\s*(.+)$/im =~ mbuf
				value = $1;
				begin
					value = value.gsub(/\n\s/, '').strip + "\n";
				rescue
					value = "\n";
				end
				return value;
			end
		end
		return nil;
	end

	# =============================================================
	#
	def get_unix_from()
		unix_from_line = get_header('From ') || '';
		unix_from = '';
		if /^(\S+) / =~ unix_from_line
			unix_from = $1;
		end
		return unix_from;
	end

	# =============================================================
	#
	def get_mail_from()
		from_header = get_header('From:') || '';

		if /<(.+)>/ =~ from_header
			mail_addr = $1;
		elsif /([^ ]+)\s*\(.*\)/ =~ from_header
			mail_addr = $1;
		else
			mail_addr = from_header.strip;
		end

		local_part_chars = '[0-9A-Za-z_!#\$%&\'*+\-\/=\?^_{|}\~\.]+';
		domain_part_chars = '[0-9A-Za-z_\-\.]+';
		valid_pattern = Regexp::new("^(#{local_part_chars})@(#{domain_part_chars})$".force_encoding('ASCII-8BIT'));
		if valid_pattern !~ mail_addr
			mail_addr = '';
		end
		return mail_addr;
	end

	# =============================================================
	# 本文が空であるか否かを判定する。
	#
	def is_empty_body()
		self.get_decoded_body().each_line do |buf|
			if /^\s*$/ !~ buf && %r|<feml.+ />| !~ buf
				return false;
			end
		end
		return true;
	end

	# =============================================================
	# 使われているcharsetを取得する。
	#
	def get_charset_in_use()

		charset_in_use = [];
		if %r|<feml.+charset="([^"]+)".+/>| =~ header_summary_line()
			charset_in_use << $1.downcase;
		end
		self.get_decoded_body().each_line do |buf|
			if %r|<feml.+charset="([^"]+)".+/>| =~ buf
				charset = $1.downcase;
				charset_in_use << charset if ! charset_in_use.include?(charset);
			end
		end
		return charset_in_use;
	end

	# =============================================================
	# 電子メールの本文(デコード済み)を返す。
	#
	def get_decoded_body()
		return @mbuf_body;
	end

	# =============================================================
	# 電子メールのテキスト(デコード済み)を返す。
	#
	def decoded_message()
		result = '';
		@mbuf_header_array.each do |buf|
			result += buf;
		end
		result += "\n";
		result += @mbuf_body;

		return result;
	end

	# =============================================================
	#
	def decode_out(ofp)
		ofp.print decoded_message();
	end

end

# ======================================================================
#
class FemlController
	# ==============================================================
	#
	def initialize(conf = {})
		@conf = conf;
	end

	# ==============================================================
	# 字句データベースの更新
	# mbox_file_name: Unix Mbox形式の電子メール・ファイル
	# mode: :add_spam_sub_clean | :add_clean_sub_spam |
	#	:add_spam | :add_clean
	#
	def update_spam_db(mbox_file_name, mode,
			token_db_store = @conf['OccurenceStore'])
		mb = Mbox.new();
		mb.load(mbox_file_name);
		mb.each_mbox do |mbuf_array|
			exd = ExDecoder.new();
			exd.load_array(mbuf_array);

			subject = (exd.get_header('Subject:') || '') + "\n";
			mail_body = exd.get_decoded_body() || '';

			tk = Tokenizer.new(
				mecab_dic_dir: @conf['MecabDicDir']);
			token_array = tk.get_tokens(subject + remove_tags(mail_body));

			token_db = TokenDB.new(token_db_store);
			token_db.update_occurences(token_array, mode);
		end
	end

	# ==============================================================
	# スパム確率の判定と振り分け
	#
	def do_spam_filter(fp_in, user)
		# ======================================================
		# 準備。ロック、ディレクトリーの準備、メール保存ファイル名。
		#
		lock_fp = do_lock();
		make_directories();
		timestamp = Time::now.strftime("%Y%m%d_%H%M%S") + sprintf("_%05d", $$);
		mail_file_name = "mail_#{timestamp}.txt";

		# ======================================================
		# stdinからのメール・メッセージを、
		# まずはバックアップ・ファイルに書き出す。
		# // 以降の処理で何か問題が生じても、バックアップが残っている
		# // 可能性が高いよう、分析は後回しにする。
		#
		exd = ExDecoder.new();
		exd.load_fp(fp_in, with_no_parse: true);
		exd.store(@conf['BackupMailDir'] + '/' + mail_file_name);

		# ======================================================
		#
		lfp = open(@conf['LogFile'], "a");

		begin
			judge = do_spam_filter_sub(exd, lfp, timestamp);
		rescue => e
			exception_info_file = AmConf['ExceptionInfoFile'];
			if exception_info_file != nil && ! exception_info_file.empty?
				efp = open(exception_info_file, 'a');
				efp.print "## ==== Feml Exception ====\n";
				efp.print "## Backup: #{@conf['BackupMailDir']}/#{mail_file_name}\n";
				efp.print "#{e.message}\n";
				e.backtrace.each do |buf|
					efp.print "#{buf}\n";
				end
				efp.close;
			end

			lfp.print "** ==== Exception ====\n";
			lfp.print "#{e.message}\n";
			e.backtrace.each do |buf|
				lfp.print "#{buf}\n";
			end
		end

		# ======================================================
		# ロック解除、後始末。
		#
		lfp.close();
		do_unlock(lock_fp);

		return judge;
	end

	# --------------------------------------------------------------
	#
	def do_spam_filter_sub(exd, lfp, timestamp)

		# ======================================================
		# 保留にしていたメールの分析をここで実行する。
		#
		exd.parse();

		# ======================================================
		# ログに状況を記録する (前半)。
		#
		lfp.print <<-"__LOG__";
** #{timestamp} ========================================
#{exd.decoded_message}
** -----------------------------------------------------
		__LOG__

		# ======================================================
		# (判定1) ホワイト・リスト/ブラック・リスト。
		#
		judge, reason = lookup_bw_list(exd);

		# ======================================================
		# (判定2) その他の諸条件にもとづく判定。
		#
		if judge == :unknown
			judge, reason = misc_judges(exd);
		end

		# ======================================================
		# (判定3) 出現字句にもとづく判定。
		#	判定1/2で既に結果が出ていてもスパム確率を求め、
		#	ログに出力して参考にする。
		#
		sp = SpamProb.new(@conf['OccurenceStore'],
				log_fp: lfp,
				mecab_dic_dir: @conf['MecabDicDir']);

		subject = (exd.get_header('Subject:') || '') + "\n";
		mail_body = exd.get_decoded_body() || '';
		spam_prob = sp.probability(remove_tags(subject + mail_body));

		if judge == :unknown
			threshold = @conf['SpamThreshold'].to_f;
			if spam_prob < threshold
				judge = :is_clean;
				reason = "< #{threshold * 100}%";
			else
				judge = :is_dirty;
				reason = ">= #{threshold * 100}%";
			end
		end

		# ======================================================
		# 判定結果ごとの処理内容。
		#
		action_rules = {
				# 判定文字列、メールボックスに追記
			:is_white => [ 'White', true ],
			:is_dirty => [ 'Dirty', false ],
			:is_clean => [ 'Clean', true ],
			:is_black => [ 'Black', false ],
		};
		judge_str = "Feml#{action_rules[judge][0]}";
		save_dir_key = "#{action_rules[judge][0]}MailDir";
		append_to_mailbox = action_rules[judge][1];
		update_mode = append_to_mailbox ? :add_clean : :add_spam;

		# ======================================================
		# ログに状況を記録する。
		#
		lfp.print <<-"__LOG__";
** #{timestamp} ----------------------------------------
** #{judge_str} (#{sprintf('%.2f', spam_prob * 100)}%) (#{reason})
** FromLine: #{exd.get_mail_from()}
** UnixFrom: #{exd.get_unix_from()}
** charset: #{exd.get_charset_in_use().join(',')}
** SubjectLine: #{(exd.get_header('Subject:') || '').chomp}

		__LOG__

		# ======================================================
		# 字句データベース更新。
		# 判定結果に応じたディレクトリーに保存。
		#
		sp.update_token_db(update_mode);
		store_path = "#{@conf[save_dir_key]}/mail_#{timestamp}.txt";
		exd.store(store_path);

		# ======================================================
		# メールボックス・ファイルに追記する。
		#
		if append_to_mailbox
			mailbox_file = @conf['MailBoxFile'];
			if mailbox_file != nil && ! mailbox_file.empty?
				mfp = open(mailbox_file, "a");
				exd.store_fp(mfp);
				mfp.print "\n";
				mfp.close();
			else
				exd.store_fp($stdout);
			end
		end
			# -----------------------------------------------
			# /usr/libexec/mail.local を使う場合:
			# - 「From 」行を除いて、一時ファイルに書き出しておく。
			# - 「mail.local -f <ufrom> tac < mail.tmp」の
			#   形で呼び出す。
			#
			# mail.localでは書き出し権限の検査などを厳密に
			# やっているが、実質的な処理はexd.store_fp()と同等。
			#

		return judge;
	end

	# --------------------------------------------------------------
	# 同時に複数のプロセスがmailbox_fileに書き込んだりすることのないよう、
	# 安全のためロックをかけるが、失敗した場合でもとにかく先に進む。
	#
	def do_lock()
		lock_fp = nil;
		rescue_point = '';
		begin
			lock_fp = open(@conf['LockFile'], "a");
		rescue
			# 例外が発生した場合はロックをかけずに先に進む。
			lock_fp = nil;
			rescue_point = 'open';
		else
			begin
				lock_fp.flock(File::LOCK_EX);
			rescue
				lock_fp.close();
				lock_fp = nil;
				rescue_point = 'flock';
			end
		end
		return lock_fp;
	end

	# --------------------------------------------------------------
	# ロック解除、後始末。
	#
	def do_unlock(lock_fp)
		begin
			if lock_fp != nil
				lock_fp.flock(File::LOCK_UN);
				lock_fp.close();
				File::unlink(@conf['LockFile']);
			end
		rescue
			# 例外が発生しても無視して先に進む。
		end
	end

	# --------------------------------------------------------------
	# バックアップ保存その他に必要なディレクトリーを用意する。
	#
	def make_directories()
		[ 'LogFile', 'OccurenceStore' ].each do |f|
			file = @conf[f];
			next if file == nil || file.empty?;
			dir = File::dirname(file);
			if ! File::directory?(dir)
				FileUtils::mkdir_p(dir);
			end
		end

		[ 'BackupMailDir', 'CleanMailDir', 'WhiteMailDir', 'DirtyMailDir', 'BlackMailDir' ].each do |d|
			dir = @conf[d];
			next if dir == nil || dir.empty?;
			if ! File::directory?(dir)
				FileUtils::mkdir_p(dir);
			end
		end
	end

	# --------------------------------------------------------------
	# ブラック・リスト/ホワイト・リストにもとづく判定。
	#
	def lookup_bw_list(exd)
		lookup_rules = [
			# リスト・ファイルのキー、基準文字列、判定
			[ 'UfromWhiteList', exd.get_unix_from(), :is_white ],
			[ 'FromWhiteList', exd.get_mail_from(), :is_white ],
			[ 'ReceivedWhiteList', exd.get_header('Received:'), :is_white ],
			[ 'SubjectWhiteList', exd.get_header('Subject:'), :is_white ],
			[ 'UfromBlackList', exd.get_unix_from(), :is_black ],
			[ 'FromBlackList', exd.get_mail_from(), :is_black ],
			[ 'ReceivedBlackList', exd.get_header('Received:'), :is_black ],
			[ 'SubjectBlackList', exd.get_header('Subject:'), :is_black ],
		];

		lookup_rules.each do |key, comp_str, judge|
			next if @conf[key] == nil || @conf[key].empty?;
			next if ! File.file?(@conf[key]);
			bw_list = BWList.new(@conf[key]);
			if bw_list.is_in_pattern(comp_str || '')
				return judge, key;
			end
		end
		return :unknown, '';
	end

	# --------------------------------------------------------------
	# 諸条件にもとづく判定。
	#
	def misc_judges(exd)
		if is_conf_on('SpamIfEmptyBody')
			if exd.is_empty_body()
				return :is_dirty, 'Empty Body';
			end
		end
		if is_conf_on('SpamIfIllegalDate')
			diag = verify_date(exd.get_header('Date:') || '');
			if ! diag.empty?
				return :is_dirty, diag;
			end
		end
		if is_conf_on('SpamIfShiftJisSubject')
			if exd.raw_header_encoding == 'shift_jis'
				return :is_dirty, 'Shift_Jis Subject';
			end
		end

		s_ch = '';
		charset_in_use = exd.get_charset_in_use();
		if (spam_charsets = @conf['SpamCharsets']) != nil
			spam_charsets_array = spam_charsets.split(/,\s*/);
			s_ch = (spam_charsets_array & charset_in_use).join(',');
					# スパムと看做すcharset
		end
		if ! s_ch.empty?
			return :is_dirty, "Spam Charsets (#{s_ch})";
		end

		return :unknown, '';
	end

	# --------------------------------------------------------------
	# HTMLタグおよびこれに準じるものを削除する。
	#	(スパム確率を求める際には不要)
	#
	def remove_tags(src)
		g_tags = "style";
		tags = "feml";
		tags += "|html|head|body";
		tags += "|frameset|frame|noframes";
		tags += "|title|script|meta|link";
		tags += "|br|p|div|span|hr";
		tags += "|pre|blockquote|q|address";
		tags += "|h1|h2|h3|h4|h5|h6";
		tags += "|strong|em|dfn|code|samp|kbd|var|cite|abbr|acronym";
		tags += "|sup|sub";
		tags += "|a|img";
		tags += "|table|caption|thead|tfoot|tbody|tr|th|td";
		tags += "|ul|ol|li|dl|dt|dd";
		tags += "|applet|object";
		tags += "|form|input|textarea|select|option|label|button";
		tags += "|tt|b|u|i|s|strike|big|small|blink|marquee";
		tags += "|font|basefont|center|map|area|iframe";

		patterns = [
			%r|<!--.+-->|m,
			%r|<(#{g_tags})[^<>]*/?>.+?</\1>|im,
			%r|</?(#{tags})[^<>]*/?>|im,
			%r|<!DOCTYPE\s.+?>|im,
			%r|<\?>|im,
		];
		patterns.each do |pattern|
			src = src.gsub(pattern, ' ');
		end
		return src;
	end

	# --------------------------------------------------------------
	# ブール型の設定項目について、オンになっているかどうかを調べる。
	#	値として空でない文字列が設定されていればオン。
	#
	def is_conf_on(key)
		return (@conf[key] != nil && @conf[key] != '');
	end

	# --------------------------------------------------------------
	# 日付の書式と、妥当な範囲内に入っているかどうかを診断する。
	#
	def verify_date(date_line)
		diag = '';				# 日付の検査結果
		date_std = '';				# 標準書式で表した日付
		date_diff = 0;
		begin
			date_val = DateTime.parse(date_line);
			date_now = DateTime::now;
			date_diff = date_val - date_now;
			date_std = date_val.new_offset(date_now.offset).strftime("%Y-%m-%d %H:%M:%S %z");
		rescue ArgumentError
			date_val = nil;
			diag = "Illegal Date [書式不正] ORIG: #{date_line}";
		else
			if date_diff <= -1.5
				diag = "Illegal Date [過去] (#{date_std})";
			elsif 1.5 < date_diff
				diag = "Illegal Date [未来] (#{date_std})";
			else
				diag = "";
			end
		end
		return diag;
	end
end

# ======================================================================
#
if __FILE__ == $0
	mode = :spam_filter;
	user = AmConf['User'] || 'user';
	import_token_file = nil;
	export_token_file = nil;

	# --------------------------------------------------------------
	#
	opts = OptionParser.new();

	opts.on("-f", "--spam-filter") do |opt|
		mode = :spam_filter;
	end
	opts.on("-s", "--add-spam-sub-clean") do |opt|
		mode = :add_spam_sub_clean;
	end
	opts.on("-c", "--add-clean-sub-spam") do |opt|
		mode = :add_clean_sub_spam;
	end
	opts.on("--add-spam") do |opt|
		mode = :add_spam;
	end
	opts.on("--add-clean") do |opt|
		mode = :add_claen;
	end
	opts.on("-e FILE_PATH", "--export-token-db FILE_PATH") do |opt|
		mode = :export_token_db;
		export_token_file = opt;
	end
	opts.on("-i FILE_PATH", "--import-token-db FILE_PATH") do |opt|
		mode = :import_token_db;
		import_token_file = opt;
	end
	opts.on("--reduce-token-db") do |opt|
		mode = :reduce_token_db;
	end
	opts.on("-v", "--version") do |opt|
		mode = :version;
	end

	opts.on("-u [USER]", "--user [USER]") do |opt|
		user = opt;
	end
	opts.on("-k [Key=Value]", "--key [Key=Value]") do |opt|
		if /(\w+?)\s*=\s*(\S*)/ =~ opt
			key = $1;
			value = $2 || '';
			AmConf[key] = value;
		end
	end

	opts.parse!(ARGV);

	# --------------------------------------------------------------
	#
	AmConf.each_pair do |key, value|
		if value.instance_of?(String)
			value = value.gsub(/%user%/, user);
		end
		AmConf[key] = value;
	end

	# --------------------------------------------------------------
	#
	case mode
	when :spam_filter
		fc = FemlController.new(AmConf);
		fc.do_spam_filter($stdin, user);
	when :add_spam_sub_clean, :add_clean_sub_spam, :add_spam, :add_clean
		fc = FemlController.new(AmConf);
		ARGV.each do |file_name|
			fc.update_spam_db(file_name, mode);
		end
	when :export_token_db
		token_db = TokenDB.new(AmConf['OccurenceStore']);
		open(export_token_file, 'w') do |ofp|
			token_db.export_to_fp(ofp);
		end
	when :import_token_db
		token_db = TokenDB.new(AmConf['OccurenceStore']);
		token_db.clear();
		open(import_token_file, 'r') do |fp|
			token_db.import_from_fp(fp);
		end
	when :reduce_token_db
		token_db = TokenDB.new(AmConf['OccurenceStore']);
		token_db.reduce();
	when :version
		$stderr.print FemlVersion, "\n";
	end

end

