// $Id: exerb.cpp,v 1.147 2003/11/14 14:07:32 yuya Exp $

#include <windows.h>
#include <ruby.h>
#include <mod_zlib.h>
#include <mod_bruby.h>
#include "exerb.h"
#include "module.h"
#include "utility.h"
#include "plugin.h"

////////////////////////////////////////////////////////////////////////////////

typedef struct {
	char *filepath;
	HMODULE handle;
} LOADED_LIBRARY_ENTRY;

typedef struct {
	DWORD base_of_file;
	IMAGE_SECTION_HEADER *secion;
	IMAGE_DOS_HEADER *dos_header;
	IMAGE_NT_HEADERS *nt_headers;
	DWORD base_of_import_table;
	DWORD delta_of_import_table;
	DWORD base_of_name_pool;
	DWORD size_of_name_pool;
} IMPORT_TABLE_INFO;

////////////////////////////////////////////////////////////////////////////////

VALUE rb_mExerb             = 0;
VALUE rb_eExerbRuntimeError = 0;

EXERB_HEADER        g_exerb_header         = {0};
ARCHIVE_HEADER      g_archive_header       = {0};
NAME_TABLE_HEADER   *g_name_table_header   = NULL;
FILE_TABLE_HEADER   *g_file_table_header   = NULL;
PLUGIN_TABLE_HEADER *g_plugin_table_header = NULL;

int g_loaded_library_count = 0;
LOADED_LIBRARY_ENTRY g_loaded_library_table[32] = {0};
LOADED_LIBRARY_ENTRY g_pre_loaded_library_table[8] = {0};

int g_loaded_resource_count = 0;
HMODULE g_loaded_resource_table[4] = {0};

extern "C" VALUE rb_load_path;

////////////////////////////////////////////////////////////////////////////////

int exerb_main(int argc, char** argv, void (*on_fail)(VALUE));
static VALUE exerb_main_in_protect(VALUE data);
static void exerb_mapping();
static void exerb_setup_kcode();
static void exerb_setup_plugin();
static void exerb_setup_file_table();
static void exerb_setup_resource_library();
static void exerb_execute();
static void exerb_cleanup();
static void exerb_unmapping();
extern "C" VALUE exerb_require(VALUE fname);
static bool exerb_find_file_pre_loaded(VALUE filename, VALUE *feature, LOADED_LIBRARY_ENTRY **loaded_library_entry);
static bool exerb_find_file_inside(VALUE filename, WORD *id, VALUE *feature, VALUE *realname);
static bool exerb_find_file_outside(VALUE filename, VALUE *feature, VALUE *realname);
static VALUE exerb_load_ruby_script(FILE_ENTRY_HEADER *file_entry);
static VALUE exerb_load_ruby_script(char *filepath);
static VALUE exerb_load_bruby_binary(FILE_ENTRY_HEADER *file_entry);
static VALUE exerb_load_bruby_binary(char *filepath);
static void exerb_load_extension_library(FILE_ENTRY_HEADER *file_entry);
static void exerb_load_extension_library(char *filepath);
static HMODULE exerb_load_library(FILE_ENTRY_HEADER *file_entry);
static HMODULE exerb_load_library(PLUGIN_ENTRY_HEADER *plugin_entry);
static HMODULE exerb_load_library(char *base_of_file, int size_of_file, char* filepath);
static bool exerb_replace_import_table(char *base_of_file);
static void exerb_replace_import_dll_name(IMPORT_TABLE_INFO *info, char *src_name, char* new_name);
static bool exerb_get_import_table_info(char *base_of_file, IMPORT_TABLE_INFO *info);
static IMAGE_SECTION_HEADER* exerb_get_enclosing_section_header(const IMAGE_NT_HEADERS *nt_headers, const DWORD rva);
static void exerb_call_initialize_function(HMODULE handle, char *filepath);

////////////////////////////////////////////////////////////////////////////////

int
exerb_main(int argc, char** argv, void (*on_fail)(VALUE))
{
	::NtInitialize(&argc, &argv);
	::ruby_init();
	::ruby_set_argv(argc - 1, argv + 1);
	::rb_ary_push(rb_load_path, ::rb_str_new2("."));

	int state = 0, result_code = 0;
	::rb_protect(exerb_main_in_protect, 0, &state);

	if ( state ) {
		if ( ::rb_obj_is_kind_of(ruby_errinfo, rb_eSystemExit) ) {
			result_code = FIX2INT(::rb_iv_get(ruby_errinfo, "status"));
		} else {
			on_fail(ruby_errinfo);
			result_code = 1;
		}
	}

	::ruby_finalize();
	::exerb_cleanup();
	::exerb_unmapping();

	return result_code;
}

static VALUE
exerb_main_in_protect(VALUE data)
{
	::Init_Exerb();

	::exerb_mapping();
	::exerb_setup_kcode();
	::exerb_setup_plugin();
	::exerb_setup_file_table();
	::exerb_setup_resource_library();
	::exerb_execute();

	return Qnil;
}

static void
exerb_mapping()
{
	char self_filepath[MAX_PATH] = "";
	::exerb_get_self_filepath(self_filepath, sizeof(self_filepath));
	
	HANDLE file = ::exerb_fopen_for_read(self_filepath);
	::exerb_fseek(file, -((int)sizeof(EXERB_HEADER)), FILE_END);
	::exerb_fread(file, (void*)&g_exerb_header, sizeof(EXERB_HEADER));

	if ( g_exerb_header.signature1 != EXERB_HEADER_SIGNATURE1 || g_exerb_header.signature2 != EXREB_HEADER_SIGNATURE2 ) {
		::exerb_fclose(file);
		::rb_raise(rb_eExerbRuntimeError, "The executable has invalid signature of exerb header.\nThe core that has not been joined with archive was likely to be invoked.");
	}

	if ( g_exerb_header.offset_of_archive == 0 ) {
		::exerb_fclose(file);
		::rb_raise(rb_eExerbRuntimeError, "The executable has invalid exerb header.");
	}
	
	::exerb_fseek(file, g_exerb_header.offset_of_archive, FILE_BEGIN);
	::exerb_fread(file, &g_archive_header, sizeof(ARCHIVE_HEADER));

	if ( g_archive_header.signature != ARCHIVE_HEADER_SIGNATURE ) {
		::exerb_fclose(file);
		::rb_raise(rb_eExerbRuntimeError, "The executable has invalid signature of archive header.");
	}
	
	if ( g_archive_header.offset_of_name_table   == 0 || g_archive_header.size_of_name_table   == 0 ||
		 g_archive_header.offset_of_file_table   == 0 || g_archive_header.size_of_file_table   == 0 ||
		 g_archive_header.offset_of_plugin_table == 0 || g_archive_header.size_of_plugin_table == 0 ) {
		::exerb_fclose(file);
		::rb_raise(rb_eExerbRuntimeError, "The executable has invalid archive header.");
	}

	g_name_table_header   = (NAME_TABLE_HEADER*)new char[g_archive_header.size_of_name_table];
	g_file_table_header   = (FILE_TABLE_HEADER*)new char[g_archive_header.size_of_file_table];
	g_plugin_table_header = (PLUGIN_TABLE_HEADER*)new char[g_archive_header.size_of_plugin_table];

	::exerb_fseek(file, g_exerb_header.offset_of_archive + g_archive_header.offset_of_name_table, FILE_BEGIN);
	::exerb_fread(file, g_name_table_header, g_archive_header.size_of_name_table);
	::exerb_fseek(file, g_exerb_header.offset_of_archive + g_archive_header.offset_of_file_table, FILE_BEGIN);
	::exerb_fread(file, g_file_table_header, g_archive_header.size_of_file_table);
	::exerb_fseek(file, g_exerb_header.offset_of_archive + g_archive_header.offset_of_plugin_table, FILE_BEGIN);
	::exerb_fread(file, g_plugin_table_header, g_archive_header.size_of_plugin_table);

	::exerb_fclose(file);

	if ( g_name_table_header->signature != NAME_TABLE_HEADER_SIGNATURE ) {
		::rb_raise(rb_eExerbRuntimeError, "The executable has invalid signature of the name table header.");
	}

	if ( g_plugin_table_header->signature != PLUGIN_TABLE_HEADER_SIGNATURE ) {
		::rb_raise(rb_eExerbRuntimeError, "The executable has invalid signature of the plug-in table header.");
	}
}

static void
exerb_setup_kcode()
{
	switch ( g_exerb_header.options.kcode ) {
	case EXERB_HEADER_OPTIONS_KCODE_NONE: ::rb_set_kcode("n"); break;
	case EXERB_HEADER_OPTIONS_KCODE_EUC:  ::rb_set_kcode("e"); break;
	case EXERB_HEADER_OPTIONS_KCODE_SJIS: ::rb_set_kcode("s"); break;
	case EXERB_HEADER_OPTIONS_KCODE_UTF8: ::rb_set_kcode("u"); break;
	}
}

static void
exerb_setup_plugin()
{
#ifdef EXERB_RUNTIMELIB
	mod_zlib_init();
	mod_bruby_init();
#endif

	PLUGIN_ENTRY_HEADER *plugin_entry = GET_FIRST_PLUGIN_ENTRY(g_plugin_table_header);

	for ( int i = 0; i < g_plugin_table_header->number_of_headers; i++, plugin_entry++ ) {
		::exerb_load_library(plugin_entry);
	}
}

static void
exerb_setup_file_table()
{
	if ( g_archive_header.options.compressed ) {
		MOD_ZLIB_DECOMPRESS_PROC mod_zlib_decompress = (MOD_ZLIB_DECOMPRESS_PROC)::exerb_find_plugin_function(MOD_ZLIB_DECOMPRESS_NAME);
		DWORD size_of_decompressed = *((DWORD*)g_file_table_header);
		DWORD size_of_compressed   = *((DWORD*)g_file_table_header + 1);
		char *decompressed = new char[size_of_decompressed];
		char *compressed   = (char*)((DWORD*)g_file_table_header + 2);
		mod_zlib_decompress(decompressed, size_of_decompressed, compressed, size_of_compressed);
		delete[] (char*)g_file_table_header;
		g_file_table_header = (FILE_TABLE_HEADER*)decompressed;
	}

	if ( g_file_table_header->signature != FILE_TABLE_HEADER_SIGNATURE ) {
		::rb_raise(rb_eExerbRuntimeError, "The executable has invalid signature of the file table header.");
	}
}

static void
exerb_setup_resource_library()
{
	FILE_ENTRY_HEADER *file_entry = GET_FIRST_FILE_ENTRY(g_file_table_header);

	for ( int i = 0; i < g_file_table_header->number_of_headers; i++, file_entry++ ) {
		if ( file_entry->type_of_file == FILE_ENTRY_HEADER_TYPE_RESOURCE_LIBRARY ) {
			HMODULE handle = ::exerb_load_library(file_entry);

			if ( g_loaded_resource_count > sizeof(g_loaded_resource_table) / sizeof(HMODULE) ) {
				::rb_raise(rb_eExerbRuntimeError, "the loaded recourse table is too big.");
			}

			g_loaded_resource_table[g_loaded_resource_count] = handle;
			g_loaded_resource_count++;
		}
	}
}

static void
exerb_execute()
{
	FILE_ENTRY_HEADER *file_entry = GET_FIRST_FILE_ENTRY(g_file_table_header);

	bool bruby_is_enable = ::exerb_has_plugin_function(MOD_BRUBY_LOAD_CODE_NAME);

	for ( int i = 0; i < g_file_table_header->number_of_headers; i++, file_entry++ ) {
		if ( file_entry->type_of_file == FILE_ENTRY_HEADER_TYPE_RUBY_SCRIPT ) {
			::ruby_script(GET_NAME_FROM_ENTRY_HEADER(g_name_table_header, ::exerb_find_name_entry(file_entry->id)));
			::exerb_load_ruby_script(file_entry);
			return;
		} else if ( file_entry->type_of_file == FILE_ENTRY_HEADER_TYPE_BRUBY_BINARY && bruby_is_enable ) {
			::ruby_script(GET_NAME_FROM_ENTRY_HEADER(g_name_table_header, ::exerb_find_name_entry(file_entry->id)));
			::exerb_load_bruby_binary(file_entry);
			return;
		}
	}

	::rb_raise(rb_eExerbRuntimeError, "The startup script was not found.");
}

static void
exerb_cleanup()
{
	for ( int i = g_loaded_library_count; i > 0; i-- ) {
		LOADED_LIBRARY_ENTRY *entry = &g_loaded_library_table[i - 1];
		char filepath[MAX_PATH] = "";
		::GetModuleFileName(entry->handle, filepath, sizeof(filepath));
		::FreeLibrary(entry->handle);
		::DeleteFile(filepath);
		delete[] entry->filepath;
	}
}

static void
exerb_unmapping()
{
	delete[] (char*)g_name_table_header;
	delete[] (char*)g_file_table_header;
	delete[] (char*)g_plugin_table_header;
}

extern "C" VALUE
exerb_require(VALUE fname)
{
	::Check_SafeStr(fname);

	LOADED_LIBRARY_ENTRY *loaded_library_entry = NULL;
	WORD id = 0;
	VALUE feature = Qnil, realname = Qnil;

	if ( ::exerb_find_file_pre_loaded(fname, &feature, &loaded_library_entry) ) {
		::rb_provide(RSTRING(feature)->ptr);
		::exerb_call_initialize_function(loaded_library_entry->handle, loaded_library_entry->filepath);
		delete[] loaded_library_entry->filepath;
		loaded_library_entry->filepath = NULL;
		loaded_library_entry->handle   = NULL;
		return Qtrue;
	} else if ( ::exerb_find_file_inside(fname, &id, &feature, &realname) ) {
		if ( ::rb_provided(RSTRING(feature)->ptr) ) {
			return Qfalse;
		}

		FILE_ENTRY_HEADER *file_entry = ::exerb_find_file_entry(id);
		if ( !file_entry ) return Qfalse;

		switch ( file_entry->type_of_file ) {
		case FILE_ENTRY_HEADER_TYPE_RUBY_SCRIPT:
			::rb_provide(RSTRING(feature)->ptr);
			::exerb_load_ruby_script(file_entry);
			return Qtrue;
		case FILE_ENTRY_HEADER_TYPE_BRUBY_BINARY:
			::rb_provide(RSTRING(feature)->ptr);
			::exerb_load_bruby_binary(file_entry);
			return Qtrue;
		case FILE_ENTRY_HEADER_TYPE_EXTENSION_LIBRARY:
			::rb_provide(RSTRING(feature)->ptr);
			::exerb_load_extension_library(file_entry);
			return Qtrue;
		}
	} else if ( ::exerb_find_file_outside(fname, &feature, &realname) ) {
		if ( ::rb_provided(RSTRING(feature)->ptr) ) {
			return Qfalse;
		}

		const char *ext = ::strrchr(RSTRING(feature)->ptr, '.');

		if ( ::stricmp(ext, ".rb") == 0 ) {
			::rb_provide(RSTRING(feature)->ptr);
			::exerb_load_ruby_script(RSTRING(realname)->ptr);
			return Qtrue;
		} else if ( ::stricmp(ext, ".brb") == 0 ) {
			::rb_provide(RSTRING(feature)->ptr);
			::exerb_load_bruby_binary(RSTRING(realname)->ptr);
			return Qtrue;
		} else if ( ::stricmp(ext, ".so") == 0 ) {
			::rb_provide(RSTRING(feature)->ptr);
			::exerb_load_extension_library(RSTRING(realname)->ptr);
			return Qtrue;
		}
	}

	::rb_raise(rb_eLoadError, "No such file to load -- %s", RSTRING(fname)->ptr);

	return Qfalse;
}

static bool
exerb_find_file_pre_loaded(VALUE filename, VALUE *feature, LOADED_LIBRARY_ENTRY **loaded_library_entry)
{
	const char *fname    = RSTRING(filename)->ptr;
	const int  fname_len = ::strlen(fname);

	for ( int i = 0; i < sizeof(g_pre_loaded_library_table) / sizeof(LOADED_LIBRARY_ENTRY); i++ ) {
		const char *name = g_pre_loaded_library_table[i].filepath;
		if ( !name ) continue;

		if ( ::stricmp(name, fname) == 0 ) {
			*feature              = ::rb_str_new2(name);
			*loaded_library_entry = &g_pre_loaded_library_table[i];
			return true;
		} else if ( ::strncmp(name, fname, fname_len) == 0 && ::strcmp(name + fname_len, ".so") == 0 ) {
			*feature              = ::rb_str_new2(name);
			*loaded_library_entry = &g_pre_loaded_library_table[i];
			return true;
		} else if ( ::strncmp(name, fname, fname_len) == 0 && ::strcmp(name + fname_len, ".dll") == 0 ) {
			*feature              = ::rb_str_concat(::rb_str_new2(fname), ::rb_str_new2(".so"));
			*loaded_library_entry = &g_pre_loaded_library_table[i];
			return true;
		}
	}

	return false;
}

static bool
exerb_find_file_inside(VALUE filename, WORD *id, VALUE *feature, VALUE *realname)
{
	NAME_ENTRY_HEADER *name_entry = GET_FIRST_NAME_ENTRY(g_name_table_header);

	const char *fname    = STR2CSTR(filename);
	const int  fname_len = ::strlen(fname);

	for ( int i = 0; i< g_name_table_header->number_of_headers; i++, name_entry++ ) {
		const char *name = GET_NAME_FROM_ENTRY_HEADER(g_name_table_header, name_entry);

		if ( ::strcmp(name, fname) == 0 ) {
			*id       = name_entry->id;
			*feature  = ::rb_str_new2(name);
			*realname = ::rb_str_new2(name);
			return true;
		} else if ( ::strncmp(name, fname, fname_len) == 0 && ::strcmp(name + fname_len, ".rb") == 0 ) {
			*id       = name_entry->id;
			*feature  = ::rb_str_new2(name);
			*realname = ::rb_str_new2(name);
			return true;
		} else if ( ::strncmp(name, fname, fname_len) == 0 && ::strcmp(name + fname_len, ".so") == 0 ) {
			*id       = name_entry->id;
			*feature  = ::rb_str_new2(name);
			*realname = ::rb_str_new2(name);
			return true;
		} else if ( ::strncmp(name, fname, fname_len) == 0 && ::strcmp(name + fname_len, ".dll") == 0 ) {
			*id       = name_entry->id;
			*feature  = ::rb_str_concat(::rb_str_new2(fname), ::rb_str_new2(".so"));
			*realname = ::rb_str_new2(name);
			return true;
		}
	}

	*id       = 0;
	*feature  = Qnil;
	*realname = Qnil;

	return false;
}

static bool
exerb_find_file_outside(VALUE filename, VALUE *feature, VALUE *realname)
{
	const VALUE filename_rb  = ::rb_str_concat(::rb_str_dup(filename), ::rb_str_new2(".rb"));
	const VALUE filename_brb = ::rb_str_concat(::rb_str_dup(filename), ::rb_str_new2(".brb"));
	const VALUE filename_so  = ::rb_str_concat(::rb_str_dup(filename), ::rb_str_new2(".so"));
	const VALUE filename_dll = ::rb_str_concat(::rb_str_dup(filename), ::rb_str_new2(".dll"));

	if ( *realname = ::rb_find_file(*feature = filename)     ) return true;
	if ( *realname = ::rb_find_file(*feature = filename_rb)  ) return true;
	if ( *realname = ::rb_find_file(*feature = filename_brb) ) return true;
	if ( *realname = ::rb_find_file(*feature = filename_so)  ) return true;
	if ( *realname = ::rb_find_file(filename_dll) ) { *feature = filename_so; return true; }
	
	*feature  = Qnil;
	*realname = Qnil;

	return false;
}

static VALUE
exerb_load_ruby_script(FILE_ENTRY_HEADER *file_entry)
{
	static const ID    id_eval = ::rb_intern("eval");
	static const VALUE binding = ::rb_const_get(rb_mKernel, ::rb_intern("TOPLEVEL_BINDING"));
	static const VALUE lineno  = INT2FIX(1);
	
	const VALUE code = ::rb_str_new(GET_FILE_FROM_ENTRY_HEADER(g_file_table_header, file_entry), file_entry->size_of_file);
	const VALUE name = ::rb_str_new2(GET_NAME_FROM_ENTRY_HEADER(g_name_table_header, ::exerb_find_name_entry(file_entry->id)));

	return ::rb_funcall(rb_mKernel, id_eval, 4, code, binding, name, lineno);
}

static VALUE
exerb_load_ruby_script(char *filepath)
{
	::rb_load(::rb_str_new2(filepath), 0);
	return Qnil;
}

static VALUE
exerb_load_bruby_binary(FILE_ENTRY_HEADER *file_entry)
{
	MOD_BRUBY_LOAD_CODE_PROC mod_bruby_load_code_proc = (MOD_BRUBY_LOAD_CODE_PROC)::exerb_find_plugin_function(MOD_BRUBY_LOAD_CODE_NAME);

	char  *code = GET_FILE_FROM_ENTRY_HEADER(g_file_table_header, file_entry);
	DWORD size  = file_entry->size_of_file;
	char  *name = GET_NAME_FROM_ENTRY_HEADER(g_name_table_header, ::exerb_find_name_entry(file_entry->id));

	return mod_bruby_load_code_proc(code, size, name);
}

static VALUE
exerb_load_bruby_binary(char *filepath)
{
	MOD_BRUBY_LOAD_FILE_PROC mod_bruby_load_file_proc = (MOD_BRUBY_LOAD_FILE_PROC)::exerb_find_plugin_function(MOD_BRUBY_LOAD_FILE_NAME);
	
	return mod_bruby_load_file_proc(filepath);
}

static void
exerb_load_extension_library(FILE_ENTRY_HEADER *file_entry)
{
	NAME_ENTRY_HEADER *name_entry = ::exerb_find_name_entry(file_entry->id);
	char *filepath = GET_NAME_FROM_ENTRY_HEADER(g_name_table_header, name_entry);

	HMODULE handle = ::exerb_load_library(file_entry);

	::exerb_call_initialize_function(handle, filepath);
}

static void
exerb_load_extension_library(char *filepath)
{
	HANDLE file    = ::exerb_fopen_for_read(filepath);
	DWORD  size    = ::exerb_fsize(file);
	char   *buffer = new char[size];
	::exerb_fread(file, buffer, size);
	::exerb_fclose(file);

	HMODULE handle = ::exerb_load_library(buffer, size, filepath);
	::exerb_call_initialize_function(handle, filepath);

	delete[] buffer;
}

static HMODULE
exerb_load_library(FILE_ENTRY_HEADER *file_entry)
{
	char *base_of_file = GET_FILE_FROM_ENTRY_HEADER(g_file_table_header, file_entry);
	char *filepath     = GET_NAME_FROM_ENTRY_HEADER(g_name_table_header, ::exerb_find_name_entry(file_entry->id));
	return ::exerb_load_library(base_of_file, file_entry->size_of_file, filepath);
}

static HMODULE
exerb_load_library(PLUGIN_ENTRY_HEADER *plugin_entry)
{
	char *base_of_file = GET_PLUGIN_FROM_ENTRY_HEADER(g_plugin_table_header, plugin_entry);
	return ::exerb_load_library(base_of_file, plugin_entry->size_of_plugin, NULL);
}

static HMODULE
exerb_load_library(char *base_of_file, int size_of_file, char* filepath)
{
	if ( g_loaded_library_count > sizeof(g_loaded_library_table) / sizeof(LOADED_LIBRARY_ENTRY) ) {
		::rb_raise(rb_eExerbRuntimeError, "the loaded library table is too big.");
	}

	::exerb_replace_import_table(base_of_file);

	char tmp_dirpath[MAX_PATH]  = "";
	char tmp_filepath[MAX_PATH] = "";
	::GetTempPath(sizeof(tmp_dirpath), tmp_dirpath);
	::GetTempFileName(tmp_dirpath, "exerb", 0, tmp_filepath);

	HANDLE file = ::exerb_fopen_for_write(tmp_filepath);
	::exerb_fwrite(file, base_of_file, size_of_file);
	::exerb_fclose(file);

	HMODULE handle = ::LoadLibraryEx(tmp_filepath, NULL, LOAD_WITH_ALTERED_SEARCH_PATH);
	if ( !handle ) {
		DWORD error = ::GetLastError();
		::DeleteFile(tmp_filepath);
		::exerb_raise_runtime_error(error);
	}

	g_loaded_library_table[g_loaded_library_count].filepath = ::exerb_strdup(filepath);
	g_loaded_library_table[g_loaded_library_count].handle   = handle;
	g_loaded_library_count++;

	return handle;
}

static bool
exerb_replace_import_table(char *base_of_file)
{
	IMPORT_TABLE_INFO info = {0};
	if ( !::exerb_get_import_table_info(base_of_file, &info) ) return false;

	char self_filepath[MAX_PATH] = "";
	char *self_filename = ::exerb_get_self_filepath(self_filepath, sizeof(self_filepath));

	::exerb_replace_import_dll_name(&info, "exerb_dummy_module.dll", self_filename); // for an exerb plug-in
	::exerb_replace_import_dll_name(&info, "ruby.exe",               self_filename); // for an extension library on static linked ruby
	::exerb_replace_import_dll_name(&info, "msvcrt-ruby18.dll",      self_filename); // for an extension library on dynamic linked ruby
	::exerb_replace_import_dll_name(&info, "cygwin-ruby18.dll",      self_filename); // for experiment
	::exerb_replace_import_dll_name(&info, "cygruby18.dll",          self_filename); // for experiment

	return true;
}

static void
exerb_replace_import_dll_name(IMPORT_TABLE_INFO *info, char *src_name, char* new_name)
{
	DWORD base_of_name = info->base_of_file - info->delta_of_import_table;
	DWORD size_of_new_name = ::strlen(new_name);

	IMAGE_IMPORT_DESCRIPTOR *first_descriptor = (IMAGE_IMPORT_DESCRIPTOR*)(info->base_of_file + info->base_of_import_table);

	for ( IMAGE_IMPORT_DESCRIPTOR *descriptor = first_descriptor; descriptor->Name; descriptor++ ) {
		char *name = (char*)(base_of_name + descriptor->Name);

		if ( ::stricmp(name, src_name) == 0 ) {
			bool found = false;
			for ( IMAGE_IMPORT_DESCRIPTOR *desc = first_descriptor; desc->Name; desc++ ) {
				if ( ::strcmp((char*)(base_of_name + desc->Name), new_name) == 0 ) {
					descriptor->Name = desc->Name;
					found = true;
					break;
				}
			}
			if ( found ) continue;

			if ( ::strlen(name) >= size_of_new_name ) {
				::strcpy(name, new_name);
			} else if ( size_of_new_name + 1 <= info->size_of_name_pool ) {
				DWORD address_of_new_name = info->base_of_name_pool - (size_of_new_name + 1);

				::strcpy((char*)(info->base_of_file + address_of_new_name), new_name);
				descriptor->Name = address_of_new_name + info->delta_of_import_table;

				info->base_of_name_pool -= size_of_new_name + 1;
				info->size_of_name_pool -= size_of_new_name + 1;
			} else {
				::rb_raise(rb_eLoadError, "Couldn't modify DLL's name in the import table. The name of the executable file is too long.");
			}
		} else if ( NAME_ENTRY_HEADER *name_entry = ::exerb_find_name_entry(name) ) {
			FILE_ENTRY_HEADER *file_entry = ::exerb_find_file_entry(name_entry->id);
			BYTE type = file_entry->type_of_file;

			if ( type == FILE_ENTRY_HEADER_TYPE_EXTENSION_LIBRARY || type == FILE_ENTRY_HEADER_TYPE_DYNAMIC_LIBRARY ) {
				HMODULE handle = NULL;
				for ( int i = 0; i < g_loaded_library_count; i++ ) {
					if ( g_loaded_library_table[i].filepath && ::stricmp(g_loaded_library_table[i].filepath, name) == 0 ) {
						handle = g_loaded_library_table[i].handle;
						break;
					}
				}

				if ( !handle ) {
					handle = ::exerb_load_library(file_entry);

					if ( type == FILE_ENTRY_HEADER_TYPE_EXTENSION_LIBRARY ) {
						for ( int i = 0; i < sizeof(g_pre_loaded_library_table) / sizeof(LOADED_LIBRARY_ENTRY); i++ ) {
							if ( !g_pre_loaded_library_table[i].handle ) {
								g_pre_loaded_library_table[i].filepath = ::exerb_strdup(name);
								g_pre_loaded_library_table[i].handle   = handle;
								break;
							}
						}
					}
				}

				char filepath[MAX_PATH] = "";
				char *filename = ::exerb_get_module_filepath(handle, filepath, sizeof(filepath));
				::exerb_replace_import_dll_name(info, name, filename);
			}
		}
	}
}

static bool
exerb_get_import_table_info(char *base_of_file, IMPORT_TABLE_INFO *info)
{
	info->base_of_file = (DWORD)base_of_file;
	info->dos_header   = (IMAGE_DOS_HEADER*)info->base_of_file;
	info->nt_headers   = (IMAGE_NT_HEADERS*)(info->base_of_file + info->dos_header->e_lfanew);
	
	const DWORD import_table_rva = info->nt_headers->OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].VirtualAddress;
	info->secion = ::exerb_get_enclosing_section_header(info->nt_headers, import_table_rva);
	if ( !info->secion ) return false;

	info->delta_of_import_table = info->secion->VirtualAddress - info->secion->PointerToRawData;
	info->base_of_import_table  = import_table_rva - info->delta_of_import_table;

	if ( ::strnicmp((char*)info->secion->Name, ".idata", 8) == 0 ||  // for Boland's Compiler
		 ::strnicmp((char*)info->secion->Name, ".rdata", 8) == 0 ) { // for Microsoft's Compiler
		info->base_of_name_pool = info->secion->PointerToRawData + info->secion->SizeOfRawData;
		info->size_of_name_pool = info->secion->SizeOfRawData    - info->secion->Misc.VirtualSize;
	} else {
		info->base_of_name_pool = 0;
		info->size_of_name_pool = 0;
	}

	return true;
}

static IMAGE_SECTION_HEADER*
exerb_get_enclosing_section_header(const IMAGE_NT_HEADERS *nt_headers, const DWORD rva)
{
	IMAGE_SECTION_HEADER *section = IMAGE_FIRST_SECTION(nt_headers);

	for ( int i = 0; i < nt_headers->FileHeader.NumberOfSections; i++, section++ ) {
		if ( (rva >= section->VirtualAddress) && (rva < (section->VirtualAddress + section->Misc.VirtualSize)) ) {
			return section;
		}
	}

	return NULL;
}

static void
exerb_call_initialize_function(HMODULE handle, char *filepath)
{
	char *filename = ::exerb_get_filename(filepath);
	char funcname[128] = "Init_";
	::strncat(funcname, filename, sizeof(funcname) - ::strlen(funcname) - 1);

	char *ext = ::strrchr(funcname, '.');
	if ( ext && (::stricmp(ext, ".so") == 0 || ::stricmp(ext, ".dll") == 0) ) {
		*ext = '\0';
	}

	void (*init_proc)(void) = (void (*)(void))::GetProcAddress(handle, funcname);
	if ( !init_proc ) ::rb_raise(rb_eExerbRuntimeError, "Couldn't call the initialize function in the extension library. --- %s(%s)", filepath, funcname);

	(*init_proc)();
}

////////////////////////////////////////////////////////////////////////////////
