#! /usr/bin/ruby

#==============================================================================#
# $Id: recipe_yaml.rb,v 1.10 2004/02/14 10:44:04 yuya Exp $
#==============================================================================#

require 'yaml'
require 'exerb/error'
require 'exerb/utility'
require 'exerb/archive'

#==============================================================================#

module Exerb

  class RecipeYaml

    def initialize(filepath)
      @basedir  = File.dirname(File.expand_path(filepath))
      @filename = File.basename(filepath)

      begin
        @root = YAML.load(File.read(filepath))
      rescue Errno::ENOENT => e
        raise(Exerb::ExerbError, "no such file -- #{filepath}")
      rescue ArgumentError => e
        msg = e.message.gsub(/\n/, '\n')
        msg.gsub!(/\r/, '\r')
        msg.gsub!(/\t/, '\t')
        raise(Exerb::ExerbError, format('%s: %s', @filename, msg))
      end

      @general = GeneralSection.new(@root.delete('general'), @filename)
      @path    = PathSection.new(@root.delete('path'), @filename)
      @file    = FileSection.new(@root.delete('file'), @filename)
      raise(Exerb::ExerbError, "#{@filename}: [#{@root.keys.join(',')}] is unknown section") unless @root.empty?
    end

    def make_filename(filename, src, dst)
      return filename.sub(/(\.#{src})?$/i, '.' + dst)
    end
    protected :make_filename

    def archive_filepath(filename = nil)
      return self.make_filename(self.output_filepath(filename), 'exe', 'exa')
    end

    def output_filepath(filename = nil)
      filename ||= @general.output
      filename ||= self.make_filename(@filename, 'exy', 'exe')

      return File.expand_path(filename, @basedir)
    end

    def core_filepath(filepath = nil)
      unless @general.core.nil?
        core   = Exerb::Utility.find_core_by_filepath(File.expand_path(@general.core, @basedir))
        core ||= Exerb::Utility.find_core_by_filename(@general.core)
        core ||= Exerb::Utility.find_core_by_name(@general.core)
        raise(Exerb::ExerbError, "#{@filename}: specified core isn't found -- #{@general.core}") if core.nil?
      end

      filepath ||= core
      filepath ||= Exerb::Utility.find_core_by_name('cui', true)

      return filepath
    end

    def create_archive(kcode = nil)
      archive = Exerb::Archive.new(kcode || @general.kcode)

      search_path  = [@basedir]
      search_path += @path.path.collect { |dir| File.expand_path(dir, @basedir) }
      search_path += $:

      startup_entry = @file.delete(@general.startup)
      raise(Exerb::ExerbError, "#{@filename}: specified startup script isn't found -- #{@general.startup}") if startup_entry.nil?
      add_file_entry(archive, search_path, @general.startup, startup_entry)

      @file.keys.sort.each { |internal_name|
        add_file_entry(archive, search_path, internal_name, @file[internal_name])
      }

      return archive
    end

    def add_file_entry(archive, search_path, internal_name, entry)
      if entry.file
        filepath = File.expand_path(entry.file, @basedir)
        raise("#{@filename}: no such file -- #{entry.file}") unless File.exist?(filepath)
      else
        filepath = search_path.collect { |dir|
          File.join(dir, internal_name)
        }.find { |path|
          File.exist?(path)
        }
        raise("#{@filename}: no such file -- #{internal_name}") unless filepath
      end

      flag = ([entry.type] + entry.flag).inject(0) { |a, b| a | b }

      archive.add_from_file(internal_name, filepath, flag)
    end
    protected :add_file_entry

    class GeneralSection

      def initialize(section, filename)
        @startup = nil
        @output  = nil
        @core    = nil
        @kcode   = nil
        self.analyze(section, filename)
      end

      attr_reader :startup, :output, :core, :kcode

      def analyze(section, filename)
        raise(Exerb::ExerbError, "#{filename}: general section isn't found") if section.nil?
        raise(Exerb::ExerbError, "#{filename}: general section must be Hash object") unless section.kind_of?(Hash)

        @startup = section.delete('startup')
        @output  = section.delete('output')
        @core    = section.delete('core')
        @kcode   = (section.delete('kcode') || 'none').downcase

        known_kcode = ['none', 'euc', 'sjis', 'utf8']

        raise(Exerb::ExerbError, "#{filename}: [#{section.keys.join(',')}] is unknown field at general section") unless section.empty?
        raise(Exerb::ExerbError, "#{filename}: startup field isn't specified at general section") if @startup.nil?
        raise(Exerb::ExerbError, "#{filename}: [#{@kcode}] is unknown kcode at general section") unless known_kcode.include?(@kcode)
      end
      protected :analyze

    end # GeneralSection

    class PathSection

      def initialize(section, filename)
        @path = []
        self.analyze(section, filename)
      end

      attr_reader :path

      def analyze(section, filename)
        return if section.nil?
        raise(Exerb::ExerbError, "#{filename}: path section must be Array object") unless section.kind_of?(Array)

        @path = section
      end
      protected :analyze

    end # PathSection

    class FileSection

      include Enumerable

      def initialize(section, filename)
        @file = {}
        self.analyze(section, filename)
      end

      def analyze(section, filename)
        raise(Exerb::ExerbError, "#{filename}: file section isn't found") if section.nil?
        raise(Exerb::ExerbError, "#{filename}: file section must be Hash object") unless section.kind_of?(Hash)

        section.each { |internal_name, options|
          @file[internal_name] = FileEntry.new(options, filename)
        }
      end
      protected :analyze

      def [](key)
        return @file[key]
      end

      def each(&block)
        return @file.each(&block)
      end

      def keys
        return @file.keys
      end

      def delete(key)
        return @file.delete(key)
      end

      class FileEntry

        def initialize(options, filename)
          @file = nil
          @type = Exerb::FileTable::Entry::FLAG_TYPE_RUBY_SCRIPT
          @flag = []
          self.analyze_options(options, filename)
        end

        attr_reader :file, :type, :flag

        def analyze_options(options, filename)
          return if options.nil?
          raise(Exerb::ExerbError, "#{filename}: file entry must be Hash object") unless options.kind_of?(Hash)
          @file = self.analyze_option_file(options, filename)
          @type = self.analyze_option_type(options, filename)
          @flag = self.analyze_option_flag(options, filename, @type)
        end
        protected :analyze_options

        def analyze_option_file(options, filename)
          return options['file']
        end
        protected :analyze_option_file

        def analyze_option_type(options, filename)
          type = options['type'].to_s.downcase

          case type
          when ''                  then return Exerb::FileTable::Entry::FLAG_TYPE_RUBY_SCRIPT
          when 'script'            then return Exerb::FileTable::Entry::FLAG_TYPE_RUBY_SCRIPT
          when 'bruby_binary'      then return Exerb::FileTable::Entry::FLAG_TYPE_BRUBY_BINARY
          when 'extension-library' then return Exerb::FileTable::Entry::FLAG_TYPE_EXTENSION_LIBRARY
          when 'dynamic-library'   then return Exerb::FileTable::Entry::FLAG_TYPE_DYNAMIC_LIBRARY
          when 'resource-library'  then return Exerb::FileTable::Entry::FLAG_TYPE_RESOURCE_LIBRARY
          when 'data'              then return Exerb::FileTable::Entry::FLAG_TYPE_DATA_BINARY
          else raise(Exerb::ExerbError, "#{filename}: [#{type}] is unknown type at file entry")
          end
        end
        protected :analyze_option_type

        def analyze_option_flag(options, filename, type)
          case options['flag']
          when nil    then flag = []
          when String then flag = [options['flag']]
          when Array  then flag = options['flag']
          else raise(Exerb::ExerbError, "#{filename}: flag field of file entry must be String or Array object")
          end

          case type
          when Exerb::FileTable::Entry::FLAG_TYPE_RUBY_SCRIPT       then known_flags = {} # {'compile_by_bruby' => ...}
          when Exerb::FileTable::Entry::FLAG_TYPE_BRUBY_BINARY      then known_flags = {}
          when Exerb::FileTable::Entry::FLAG_TYPE_EXTENSION_LIBRARY then known_flags = {'no_replace_function' => Exerb::FileTable::Entry::FLAG_NO_REPLACE_FUNCTION}
          when Exerb::FileTable::Entry::FLAG_TYPE_DYNAMIC_LIBRARY   then known_flags = {}
          when Exerb::FileTable::Entry::FLAG_TYPE_RESOURCE_LIBRARY  then known_flags = {}
          when Exerb::FileTable::Entry::FLAG_TYPE_DATA_BINARY       then known_flags = {}
          end

          selected_flags = known_flags.keys.select { |name|
            flag.delete(name)
          }.collect { |name|
            known_flags[name]
          }

          unless flag.empty?
            raise(Exerb::ExerbError, "#{filename}: [#{flag.join(',')}] is unknown flag at file entry")
          end

          return selected_flags
        end
        protected :analyze_option_flag

      end # FileEntry

    end # FileSection

  end # RecipeYaml

end # Exerb

#==============================================================================#

if $0 == __FILE__
  recipe = Exerb::RecipeYaml.new(ARGV.shift)
  p(recipe)
  p(recipe.archive_filepath)
  p(recipe.output_filepath)
  p(recipe.core_filepath)
  recipe.create_archive.write_to_file('tmp.exa')
end

#==============================================================================#
#==============================================================================#
