/* eptex-fp.c: interface between MPFR library and 
   WEB-Pascal source of e-pTeX. 
                by H. Kitagawa (h_kitagawa2001 at yahoo.co.jp)*/


#define	EXTERN 
#include "texd.h"

/* We use 95 bits to pack a float into glue specifications.
   Layout is indicated as follows:

------------------------------------------------------
   STRETCH_ORD (2 bit)        -----                sign bit  S  
                              \----        -----1
   SHRINK_ORD  (2 bit)                       |
   sign bit    (1 bit)  \                    |     exponent part E
   (unused bit)         | Normal part        |         (16 bit)
   int. part   (14 bit) |   of glue spec.  -----12
                        |                  -----2
   frac. part  (16 bit) /                    |
   sign bit    (1 bit)  \                    |     fraction part F
   (unused bit)         | Stretch part       |     ( 2+16+60 = 78 bit)
   int. part   (14 bit) |   of glue spec.    |
   frac. part  (16 bit) /     -----        -----15
                              \----        (unused) 1    
   sign bit    (1 bit)  \                  -----  
   (unused bit)         | Shrink part        |      
   int. part   (14 bit) |   of glue spec.    |
   frac. part  (16 bit) /     -----        -----15
                              \----        (unused) 1

   * Actual value = (1-2S)*2^(E-32767)*(1.F)_2, if E is not 65535.
   * Allowed exponent is -32767 -- 32767. Biased by 32767.
   * Zero is signed. 
     <sgn>/11111111111111111/0000...
   * NaN is not signed.
         0/11111111111111111/0100...
   * Infinity is signed.
     <sgn>/11111111111111111/1000...
   * Machine epsilon is about 3.31e-24
*/

/* from tex.web (my memo)
    width(#) = mem[#+1].cint 
    stretch(#) = mem[#+2].cint 
    shrink(#) = mem[#+3].cint 
    stretch_order = type ( = mem[#].hh.b0 )
    shrink_order = subtype ( = mem[#].hh.b1 )
 */

/*  
   * following functions are supported by MPFR library:
     cosec, sec, cot and these hyperbolic version [120--125],
     zeta (Riemann-zeta) [126], Gamma [127], erf [128], 
     hypot [129], and Bessel functions [130, 131].
 */

#include <mpfr.h>

/* These variables are used in this C source only. */
mpfr_t fbuf[10]; 
char sbuf[83];


void fpizero(integer w) {mpfr_set_ui(fbuf[(w)],0,GMP_RNDN);}
void fpirshiftadd(integer w,integer p) {
  mpfr_t fptemp;
  mpfr_init_set_si(fptemp,p,GMP_RNDN);
  mpfr_mul_ui(fptemp,fbuf[w],10,GMP_RNDN);
  mpfr_add_ui(fbuf[w],fptemp,p,GMP_RNDN);
  mpfr_clear(fptemp);
}
void fpiexpten(integer w,integer p) {
  mpfr_t fptemp, fptemp2;
  mpfr_init_set_si(fptemp,p,GMP_RNDN);
  mpfr_init(fptemp2);
  mpfr_exp10(fptemp2,fptemp,GMP_RNDN);
  mpfr_mul(fptemp,fbuf[w],fptemp2,GMP_RNDN);
  mpfr_swap(fptemp,fbuf[w]);
  mpfr_clear(fptemp);
  mpfr_clear(fptemp2);
}
void fpineg(integer w) {mpfr_neg(fbuf[w],fbuf[w],GMP_RNDN);}
void fpiinttofloat(integer w,integer p, integer d) {
  mpfr_set_si_2exp(fbuf[w],p,d,GMP_RNDN);
}
integer fpifloattoint(integer w) {return mpfr_get_si(fbuf[w],GMP_RNDZ);}
boolean isfloatinrange(integer w) {
  return (mpfr_cmp_si(fbuf[w],0x7FFFFFFF)>0)||(mpfr_cmp_si(fbuf[w],-(0x7FFFFFFF))<0);
}
boolean isfloatnan(integer w) {return mpfr_nan_p(fbuf[w]);}
integer fpifloatsign(integer w) {return mpfr_cmp_si(fbuf[w],0);}
void fpimultwobeki(integer w, integer a){mpfr_mul_2si(fbuf[w],fbuf[w],a,GMP_RNDN);}
boolean isfloatcomparable(integer a,integer b) {
  return !(mpfr_unordered_p(fbuf[a],fbuf[b]));
}
integer fpicompare(integer a,integer b) {return mpfr_cmp(fbuf[a],fbuf[b]);}

/* In next three subroutine/functions, Parameter k (it is value of \\fpoutprec) 
   means ``printed in just k significand digits (in decimal)''. 
   If k<=0 or k>30, we will treat as k=23. */

void fpioutfracinit(integer a,integer k){
  integer g;
  if ((k<=0)||(k>30)) k=23;
  mpfr_get_str(sbuf,&g, 10, k, fbuf[a], GMP_RNDN);
}
integer fpioutfracread(integer x){
  if (sbuf[x]=='-') return 10;
  else if (sbuf[x]=='\0') return 11;
  else return sbuf[x];
}
integer fpioutexpr(integer a,integer k){
  integer g;
  if (mpfr_number_p(fbuf[a])) {
     if ((k<=0)||(k>30)) k=23;
     mpfr_get_str(sbuf,&g, 10, k, fbuf[a], GMP_RNDN);
     return g-1;
  } else return 0;
}


/* Pack fbuf[p] into new glue specification */ 
halfword fpipack (integer p) {
  scanfloat_regmem
  integer g,n; mpfr_t fptemp;
  halfword m;
  m = newspec (membot) ;
  if (mpfr_nan_p(fbuf[p])!=0) {
    mem[m].hh.b0=1; mem[m].hh.b1=3;
    mem[m+1].cint=0xbffd0000; mem[m+2].cint=1; mem[m+3].cint=1; 
    /* 01/11/1@11/1111/1111/1101/0000/0000/0000/0000
        1  3   B     F   F    D    0    0    0    0  */
  } else if (mpfr_inf_p(fbuf[p])!=0) {
    mem[m].hh.b0=1; mem[m].hh.b1=3;
    mem[m+1].cint=0xbffe0000; mem[m+2].cint=1; mem[m+3].cint=1; 
    /* 01/11/1@11/1111/1111/1110/0000/0000/0000/0000
        1  3   B     F   F    E    0    0    0    0  */
    if (mpfr_sgn(fbuf[p])<0) mem[m].hh.b0=3;
  } else if (mpfr_zero_p(fbuf[p])!=0) {
    mem[m].hh.b0=1; mem[m].hh.b1=3;
    mem[m+1].cint=0xbffc0000; mem[m+2].cint=1; mem[m+3].cint=1; 
    /* 01/11/1@11/1111/1111/1100/0000/0000/0000/0000
        1  3   B     F   F    C    0    0    0    0  */
    /* !!! FIXME: if (****) mem[m].hh.b0=3; ( -0 )   */
  } else {
    mpfr_init(fptemp); 
    mpfr_abs(fptemp,fbuf[p],GMP_RNDN);
    mpfr_get_str(sbuf,&g, 2, 79, fptemp, GMP_RNDN);
    mem[m+1].cint=0; mem[m+2].cint=1; mem[m+3].cint=1; 
    for (n=1; n<19; n++) mem[m+1].cint+=(sbuf[n]-'0')<<(18-n);
    for (n=20; n<49; n++) mem[m+2].cint+=(sbuf[n]-'0')<<(49-n);
    for (n=50; n<79; n++) mem[m+3].cint+=(sbuf[n]-'0')<<(79-n);
    if (sbuf[19]=='1') mem[m+2].cint+=0x80000000UL;
    if (sbuf[49]=='1') mem[m+3].cint+=0x80000000UL;
    n=32767+g-1; 
    mem[m+1].cint+=(n&0xfff)<<18; n>>=12;
    mem[m+1].cint+=((unsigned int)(n&1))<<31; n>>=1;
    mem[m].hh.b1=n&3; mem[m].hh.b0=n>>2;
    if (mpfr_sgn(fbuf[p])<0) mem[m].hh.b0+=2;
    mpfr_clear(fptemp);
  }
  return m;
}


/* Unpack a glue specification into fbuf[p] */ 
void fpiunpack (integer w, halfword m) {
  scanfloat_regmem
  int i,g,n;
  if ( (mem[m].hh.b0&1) && (mem[m].hh.b1==3) 
       && ((unsigned int)mem[m+1].cint>=0xbffc0000) ) {
    g=1-(mem[m].hh.b0&2); 
    if ((unsigned int)mem[m+1].cint>=0xbffe0000) mpfr_set_inf(fbuf[w],g);
    else if ((unsigned int)mem[m+1].cint>=0xbffd0000) mpfr_set_nan(fbuf[w]);
    else {
      mpfr_set_si(fbuf[w],0,GMP_RNDN); 
      if (g<0) mpfr_neg(fbuf[w],fbuf[w],GMP_RNDN); 
    }
  } else {
    n=0;
    if (mem[m].hh.b0&2) {n++; sbuf[0]='-';}
    sbuf[n]='1'; sbuf[n+1]='.'; n+=19; 
    g=((unsigned int)mem[m+1].cint)&0x3ffff; 
    for (i=0; i<18; i++) {sbuf[n-i]=(g&1)+'0'; g>>=1;}
    n++; sbuf[n]=(((unsigned int)mem[m+2].cint)>>31)+'0';
    n+=29; g=((unsigned int)mem[m+2].cint)&0x3ffffffe; g>>=1;
    for (i=0; i<29; i++) {sbuf[n-i]=(g&1)+'0'; g>>=1;}
    n++; sbuf[n]=(((unsigned int)mem[m+3].cint)>>31)+'0';
    n+=29; g=((unsigned int)mem[m+3].cint)&0x3ffffffe; g>>=1;
    for (i=0; i<29; i++) {sbuf[n-i]=(g&1)+'0'; g>>=1;}
    n++; sbuf[n]='\0';
    mpfr_strtofr(fbuf[w],sbuf,NULL,2,GMP_RNDN);
    g = ( (mem[m].hh.b0&1)<<15 ) + ( mem[m].hh.b1<<13 )
      + ( (((unsigned int)mem[m+1].cint)>>31) <<12 )
      + ( ((unsigned int)mem[m+1].cint&0x3ffc0000)>>18) -32767;
    mpfr_mul_2si(fbuf[w],fbuf[w],g,GMP_RNDN);
  }
}


/* These procedures are automatically executed in starting/terminating
   e-pTeX.*/
void fpinit (void) {
  int i;
  for (i=0; i<10; i++) mpfr_init2(fbuf[i],79);
  mpfr_set_default_prec(79);
  mpfr_set_default_rounding_mode(GMP_RNDN);
  mpfr_set_emin(-32766); mpfr_set_emax(32768);
}
void fpdest (void) {
  int i;
  for (i=0; i<10; i++) mpfr_clear(fbuf[i]);
}


void fpiunaryoperation(integer w, integer p,integer q) {
  switch(w) {
  case 101: /* negative operation */
    mpfr_neg(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 102: /* square root */
    mpfr_sqrt(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 103: /* e^x */
    mpfr_exp(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 104: /* absolute value */
    mpfr_abs(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 105: /* ceiling function */
    mpfr_ceil(fbuf[q],fbuf[p]); break;
  case 106: /* flooring function */
    mpfr_floor(fbuf[q],fbuf[p]); break;
  case 107: mpfr_sin(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 108: mpfr_cos(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 109: mpfr_tan(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 110: mpfr_sinh(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 111: mpfr_cosh(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 112: mpfr_tanh(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 113: /* natural logarithm */
    mpfr_log(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 114: mpfr_asinh(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 115: mpfr_acosh(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 116: mpfr_atanh(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 117: mpfr_asin(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 118: mpfr_acos(fbuf[q],fbuf[p],GMP_RNDN); break;
  case 119: mpfr_atan(fbuf[q],fbuf[p],GMP_RNDN); break;
  }
}

void fpibinaryoperation(integer w, integer p,integer q, integer r){
  integer g;
  switch(w) {
  case 1: mpfr_add(fbuf[r],fbuf[p],fbuf[q],GMP_RNDN); break;
  case 2: mpfr_sub(fbuf[r],fbuf[p],fbuf[q],GMP_RNDN); break;
  case 3: mpfr_mul(fbuf[r],fbuf[p],fbuf[q],GMP_RNDN); break;
  case 4: mpfr_div(fbuf[r],fbuf[p],fbuf[q],GMP_RNDN); break;
  case 5: 
    if (mpfr_zero_p(fbuf[p])&&mpfr_zero_p(fbuf[q])) 
      mpfr_set_ui(fbuf[r],1,GMP_RNDN);
    else mpfr_pow(fbuf[r],fbuf[p],fbuf[q],GMP_RNDN); 
    break;
  }
}

