/*
 *  Copyright (C) 2006  Takashi Kasuya <kasuya@sfc.keio.ac.jp>
 *
 * This library is free software; you can redistribute it and/or
 *@modify it under the terms of the GNU Lesser General Public
 *@License as published by the Free Software Foundation; either
 *@version 2.1 of the License, or (at your option) any later version.
 *@This library is distributed in the hope that it will be useful,
 *@but WITHOUT ANY WARRANTY; without even the implied warranty of
 *@MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *@Lesser General Public License for more details.
 *
 *@You should have received a copy of the GNU Lesser General Public
 *@License along with this library; if not, write to the Free Software
 *@Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

package jp.ac.naka.ec.entity;



import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Ƃ肠gp
 * @author Takashi Kasuya
 *
 */
public class RequestMessage {

	String methodName;
	static String[] supports = {"String", "Integer", "Float", "Long"};
	List paramList = new ArrayList();

	public RequestMessage() {
	}

	public String getMethod() {
		return methodName;
	}

	public void setMethod(String methodName) {
		this.methodName = methodName;
	}

	public Object[] getParamList() {
		Object[] params = (paramList.size() != 0) ? paramList.toArray() : null;
		return params;
	}

	public void addParam(Object param) {
		this.paramList.add(param);
	}

	public String toXMLForm() {
		StringBuilder builder = new StringBuilder();
		builder.append("<?xml version='1.0' encoding='UTF-8' ?>\r\n");
		builder.append("<message>");
		builder.append("<method>");
		builder.append(methodName);
		builder.append("</method>");
		getParamXMLForm(builder);
		builder.append("</message>");
		return builder.toString();
	}

	private void getParamXMLForm(StringBuilder builder) {

		if (paramList.size() != 0) {
			builder.append("<paramList>");
			for (Object obj : paramList) {
				builder.append("<param>");
				if (obj instanceof String) {
					builder.append("<String>");
					builder.append((String) obj);
					builder.append("</String>");
				} else if (obj instanceof Integer) {
					builder.append("<Integer>");
					builder.append(((Integer) obj).intValue());
					builder.append("</Integer>");
				} else if (obj instanceof Float) {
					builder.append("<Float>");
					builder.append(((Float) obj).floatValue());
					builder.append("</Float>");
				} else if (obj instanceof Long) {
					builder.append("<Long>");
					builder.append(((Long) obj).longValue());
					builder.append("</Long>");
				}
				
				else {
					System.err.println("Not Support Type: " + obj.getClass().toString());
					throw new UnsupportedOperationException(
							"Not Support Type: " + obj.getClass().toString());
				}
				builder.append("</param>");
			}
			builder.append("</paramList>");
		}
	}

	public static RequestMessage parse(Document doc) throws ParseException {
		RequestMessage message = new RequestMessage();

		Element root = doc.getDocumentElement();

		Node method = root.getFirstChild();
		Node paramList = method.getNextSibling();

		message.methodName = method.getFirstChild().getNodeValue();

		if (paramList == null) {
			return message;
		}
		
		NodeList list = paramList.getChildNodes();
		
		
		for (int i = 0; i < list.getLength(); i++) {
			Node node = list.item(i);
			Node typeNode = node.getFirstChild();
			String type = typeNode.getNodeName();
			String value = typeNode.getFirstChild().getNodeValue();
			if (type.equals("String")) {
				message.addParam(value);
			} else if (type.equals("Integer")) {
				message.addParam(Integer.valueOf(value));
			} else if (type.equals("Float")) {
				message.addParam(Float.valueOf(value));
			} else {
				new UnsupportedOperationException("Invalid Type :" + type);

			}

		}
		return message;
	}

	
	public static RequestMessage parse(String doc) throws ParseException {
		RequestMessage message = new RequestMessage();
		int begin = getPosition(doc, "<method>");
		int end = doc.indexOf("</method>");
		if ( begin < 0 || end < 0) {
			throw new ParseException("Invalid message : " + doc, begin);
		}
		String temp = doc.substring(begin, end);
		message.setMethod(temp);
		begin = getPosition(doc, "<paramList>");
		end = doc.indexOf("</paramList>");
		
		temp = doc.substring(begin, end);
		
		while ((begin = getPosition(temp, "<param>")) != -1 && !temp.equals("")) {
			end = temp.indexOf("</param>");
			String param = temp.substring(begin, end);
			Object obj = findParam(param);
			message.addParam(obj);
			temp = temp.substring(end + "</param>".length());
		}
		
		return message;
	}
	
	private static Object findParam(String param) {
		int begin, end, i = 0;
		Object retval = null;
		for (String support:supports) {
			String start_tag = "<" + support + ">";
			String end_tag  = "</" + support + ">";
			if ((begin = param.indexOf(start_tag)) != -1) {
				end = param.indexOf(end_tag);
				begin += start_tag.length();
				String value = param.substring(begin, end);
				
				switch (i) {
				case 0 : retval = value; break;
				case 1 : retval = Integer.parseInt(value); break;
				case 2 : retval = Float.parseFloat(value); break;
				case 3 : retval = Long.parseLong(value); break;
				}
				return retval;
			}
			i++;
		}
		throw new UnsupportedOperationException("Invalid Value : " + param);
	}

	private static int getPosition(String doc, String str) {
		return doc.indexOf(str) + str.length();
	}
	public String toString() {
		String temp = "Method :" + methodName + ", Param :";
		for (Object obj : paramList) {
			temp += obj.toString() + ", ";
		}
		return temp;
	}
}
