/**
 * @file common.c
 * @brief Linuxコンソール版共通関数
 * @author BananaJinn
 * @version $Id: common.c,v 1.2 2007/09/26 15:35:44 bananajinn Exp $
 * 円盤複写屋
 * Copyright (c) 2004-2007 Kagetani Hideto(BananaJinn)
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "typedefs.h"

/**
 * @brief 標準入力からの入力
 * @param[out] buf 入力した文字列を格納するバッファ
 * @param[in] size バッファサイズ
 */
void InputFromStdin(char *buf, int size)
{
  char *tmpbuf;

  if(buf == NULL){
    return;
  }
  if(size <= 1){
    return;
  }
  tmpbuf = malloc(size+1);
  tmpbuf[0] = '\0';
  fgets(tmpbuf, size+1, stdin);
  if(strlen(tmpbuf) > 0){
    if(tmpbuf[strlen(tmpbuf)-1] == '\n'){
      tmpbuf[strlen(tmpbuf)-1] = '\0';
    }
    else{
      while(fgetc(stdin) != '\n');
    }
  }
  strncpy(buf, tmpbuf, size);
  free(tmpbuf);
  return;
}

/**
 * @brief メッセージを表示して Yes か No を選択入力
 * @param[in] name 入力項目名
 * @param[in,out] value 入力結果YesならTRUE、NoならFALSE(初期値設定有効)
 */
void InputYesNo(const char *name, BOOL *value)
{
  char buf[2];

  printf("%s(%s/%s):", name,
	 *value ? "[y]":"y",
	 !(*value) ? "[n]":"n");
  InputFromStdin(buf, sizeof(buf));
  if(strlen(buf) > 0){
    if(*buf=='Y' || *buf=='y'){
      *value = TRUE;
    }
    else{
      *value = FALSE;
    }
  }
  return;
}

/**
 * @brief メッセージを表示して文字列を入力
 * @param[in] name 入力項目名
 * @param[out] value 入力した文字列(初期値設定有効)
 * @param[in] size 入力した文字列を格納可能なサイズ
 */
void InputString(const char *name, char *value, int size)
{
  char *defstr;

  if(value == NULL){
    return;
  }
  defstr = strdup(value);

  printf("%s", name);
  if(strlen(value) > 0){
    printf("([%s])", defstr);
  }
  printf(":");
  InputFromStdin(value, size);
  if(strlen(value) == 0){
    strncpy(value, defstr, size);
  }
  free(defstr);
}

/**
 * @brief メッセージを表示して数値を入力
 * @param[in] name 入力項目名
 * @param[out] value 入力した数値
 */
void InputNumber(const char *name, int *value)
{
  char buf[11+1]="";
  char *p;

  while(1){
    printf("%s([%d]):", name, *value);
    InputFromStdin(buf, sizeof(buf));
    for(p=buf; *p != '\0'; p++){
      if(!((*p >= '0' && *p <= '9') || *p=='-')){
	break;
      }
    }
    if(*p == '\0'){
      break;
    }
  }
  if(strlen(buf) != 0){
    *value = atoi(buf);
  }
}



