/*
 * @file drive.c
 * @brief hCuANZX
 * @author BananaJinn
 * @version $Id: drive.c,v 1.13 2007/10/02 15:10:52 bananajinn Exp $
 * ~Օʉ
 * Copyright (C) 2004-2006 BananaJinn<banana@mxh.mesh.ne.jp>.
 */

#include <stdio.h>
#if !defined(WIN32)
# include <stdlib.h>
# include <string.h>
#endif

#include "mem.h"
#include "drive.h"
#include "cmdlog.h"
#include "cmd.h"
#include "ui.h"
#include "text.h"
#include "log.h"

int SendCmd(CMDDRIVE *drive, BYTE *cdb, DWORD buflen, BYTE reqflag)
{
  int ret;

  if(drive->cmd_type == CMDDRVCTYPE_UNKNOWN){
	drive->cmd_type = CMDDRVCTYPE_CHECKING;
	CheckAtapi(drive);
  }

  drive->cmdcode = cdb[0];
  if(drive->type == CMDDRVTYPE_DRIVE){
	ret = SendAspiCmd(&drive->u.drive, cdb, drive->data_buf, buflen,
					  drive->sense_data, sizeof(drive->sense_data), reqflag);
  }
  else if(drive->type == CMDDRVTYPE_NET){
	ret = SendNetCmd(&drive->u.net, cdb, drive->data_buf, buflen,
					 drive->sense_data, sizeof(drive->sense_data), reqflag);
  }
  else{
	return RET_NG;
  }
  CmdLog(drive, cdb, buflen, reqflag, ret);
#ifdef DEBUGCMD
  DebugLog("SendCmd: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X len=0x%08lX io=%d\n",
	   cdb[0], cdb[1], cdb[2], cdb[3], cdb[4], cdb[5],
	   cdb[6], cdb[7], cdb[8], cdb[9], cdb[10], cdb[11],
	   buflen, reqflag);
#endif

  if(ret != RET_OK){
    DebugLog("SendCmd: Command error. cmd=%02Xh %X-%02X-%02X\n",
	     cdb[0], SD_SENSEKEY(drive), SD_ASC(drive), SD_ASCQ(drive));
  }

  return ret;
}

int SendCmdLongRead(CMDDRIVE *drive, BYTE *cdb, DWORD buflen, BYTE reqflag,
					DWORD total_blocks)
{
  int ret;

  if(drive->cmd_type == CMDDRVCTYPE_UNKNOWN){
	drive->cmd_type = CMDDRVCTYPE_CHECKING;
	CheckAtapi(drive);
  }

  if(drive->type == CMDDRVTYPE_NET){
	ret = SendNetCmdLongRead(&drive->u.net, cdb, drive->data_buf, buflen,
							 drive->sense_data, sizeof(drive->sense_data),
							 reqflag, total_blocks);
  }
  else{
	return RET_NG;
  }

  CmdLog(drive, cdb, buflen, reqflag, ret);
	
  return ret;
}

int InitializeCmdDrive(CMDDRIVE *drive, int bufsize, int cmdlog_size, int type)
{
  int ret;

  if(type != CMDDRVTYPE_DRIVE && type != CMDDRVTYPE_NET &&
	 type != CMDDRVTYPE_IMAGE && type != CMDDRVTYPE_ISO){
	return RET_NG;
  }

  FreeCmdDrive(drive);
  drive->cmd_type = CMDDRVCTYPE_UNKNOWN;

  if(CmdLogInit(drive, cmdlog_size)!=RET_OK){
	return RET_NG;
  }

  if(bufsize>0){
	drive->data_buf = (BYTE *)MemNew(bufsize);
	if(drive->data_buf==NULL){
	  return RET_NG;
	}
	drive->bufsize = bufsize;
	drive->own_alloc = TRUE;
  }

  drive->type = type;
  if(type == CMDDRVTYPE_DRIVE){
	ret = InitializeDrive(&drive->u.drive);
  }
  else if(type == CMDDRVTYPE_NET){
	ret = InitializeSOCKCB(&drive->u.net);
  }
  else if(type == CMDDRVTYPE_IMAGE){
	ret = InitializeImage(&drive->u.image);
  }
  else /*if(type == CMDDRVTYPE_ISO)*/{
	ret = InitializeImage(&drive->u.image);
  }
  if(ret != RET_OK){
	if(drive->own_alloc){
	  MemFree(drive->data_buf);
	  drive->data_buf=NULL;
	}
	MemFree(drive->cmdlog);
	drive->cmdlog=NULL;
	return ret;
  }

  return RET_OK;
}

void FreeCmdDrive(CMDDRIVE *drive)
{
  CmdLogFree(drive);
  if(drive->data_buf!=NULL){
	if(drive->own_alloc){
	  MemFree(drive->data_buf);
	}
	drive->data_buf=NULL;
  }

  if(drive->type == CMDDRVTYPE_DRIVE){
	FreeDrive(&drive->u.drive);
  }
  else if(drive->type == CMDDRVTYPE_NET){
	FreeSOCKCB(&drive->u.net);
  }
  else if(drive->type == CMDDRVTYPE_IMAGE){
	FreeImage(&drive->u.image);
  }
  else if(drive->type == CMDDRVTYPE_ISO){
	FreeImage(&drive->u.image);
  }
}

void CheckAtapi(CMDDRIVE *drive)
{
  BYTE cdb[12];
  BYTE buf[12];
  BYTE *keep;
  int ret;

  keep = drive->data_buf;
  drive->data_buf = buf;

  memset(cdb, 0, sizeof(cdb));
  cdb[0] = CMD_MODE_SENSE10;
  cdb[2] = 0x2a;
  cdb[8] = 12;
  ret = SendCmd(drive, cdb, 12, REQ_DATAIN);
  if(ret!=RET_OK){
	drive->cmd_type = CMDDRVCTYPE_SCSI;
  }
  else{
	drive->cmd_type = (drive->data_buf[8]==0x2a ? CMDDRVCTYPE_ATAPI:CMDDRVCTYPE_SCSI);
  }

  drive->data_buf = keep;
}

int SetDriveAspiSetting(CMDDRIVE *drive, int *ha, int *tg, DWORD *timeout)
{
  drive->cmd_type = CMDDRVCTYPE_UNKNOWN;
  return SetAspiSetting(&drive->u.drive, ha, tg, timeout);
}

void SetDriveBuffer(CMDDRIVE *drive, BYTE *data_buf, int bufsize)
{
  drive->data_buf = data_buf;
  drive->bufsize = bufsize;
}

/**
 * uI[v
 * @param[out]	drive	u\
 * @param[in]	id	uID
 * @param[in]	reader	ǎ摕uǂ(ǎ摕u=TRUE)
 * @param[in]	bWaitDisc  fBXN҂L(L=TRUE)
 * @param[in]	data_buf   f[^obt@
 * @param[in]	bufsize    f[^obt@TCY
 * @retval	RET_OK	I
 * @retval	RET_NG	G[
 */
int OpenDevice(CMDDRIVE *drive, int hid, int tid,
	       BOOL reader, BOOL bWaitDisc,
	       BYTE *data_buf, int bufsize)
{
  int ret;
  DWORD timeout=5*60;

  ret = InitializeCmdDrive(drive, 0, 50, CMDDRVTYPE_DRIVE);
  if(ret!=RET_OK){
	return ret;
  }
  SetDriveBuffer(drive, data_buf, bufsize);
  ret = SetDriveAspiSetting(drive, &hid, &tid, &timeout);
  if(ret!=RET_OK){
	return ret;
  }

  ret = GetDiscType(drive, &drive->disc_type, bWaitDisc);
  if(ret!=RET_OK){
	if(ret!=RET_ABORT && bWaitDisc){
	  UIDispMessage(MSG_CANT_GET_DISC_TYPE, UIDMT_ERROR);
	}
	FreeCmdDrive(drive);
	return ret;
  }
  ret = SendPreventAllow(drive, 1);
  if(ret!=RET_OK){
	DispCommandError(drive);
	FreeCmdDrive(drive);
	return ret;
  }

  UIDispInfo("");
  return RET_OK;
}

/**
 * I[vꂽuN[Y
 * @param[in]	drive	N[Y鑕u\
 */
void CloseDevice(CMDDRIVE *drive)
{
  if(drive->type == CMDDRVTYPE_DRIVE){
	SendPreventAllow(drive, 0);
  }

  FreeCmdDrive(drive);
}

/**
 * ꎞt@CuI[v
 * @param[in]	drive	u\
 * @param[in]	reader	ǎ摕uǂ(ǎ摕u=TRUE)
 * @param[in]	data_buf   f[^obt@
 * @param[in]	bufsize    f[^obt@TCY
 * @param[in]	tmpfile    ꎞt@CpX
 * @retval	RET_OK	I
 * @retval	RET_NG	G[
 */
int OpenTempImageDevice(CMDDRIVE *drive, BOOL reader,
			BYTE *data_buf, int bufsize, const char *tmpfile)
{
  int ret;

  ret = InitializeCmdDrive(drive, 0, 50, CMDDRVTYPE_IMAGE);
  if(ret!=RET_OK){
    return ret;
  }
  SetDriveBuffer(drive, data_buf, bufsize);
  ret = OpenImage(&drive->u.image, tmpfile, reader);
  if(ret!=RET_OK){
    return ret;
  }
  
  return RET_OK;
}

/**
 * ISOC[Wt@CuI[v
 * @param[in]	drive	u\
 * @param[in]	reader	ǎ摕uǂ(ǎ摕u=TRUE)
 * @param[in]	data_buf   f[^obt@
 * @param[in]	bufsize    f[^obt@TCY
 * @retval	RET_OK	I
 * @retval	RET_NG	G[
 */
int OpenISOImageDevice(CMDDRIVE *drive, BOOL reader,
		       BYTE *data_buf, int bufsize)
{
  char tmpfile[_MAX_PATH];
  int ret;

  if(reader){
    tmpfile[0] = '\0';
  }
  else{
    strcpy(tmpfile, "image.iso");
  }
  
  ret = UIFileDialog(reader, tmpfile, sizeof(tmpfile), "iso");
  if(ret!=UIDMRET_OK){
    return RET_ABORT;
  }
  ret = InitializeCmdDrive(drive, 0, 50, CMDDRVTYPE_ISO);
  if(ret!=RET_OK){
    return ret;
  }
  SetDriveBuffer(drive, data_buf, bufsize);
  ret = OpenImage(&drive->u.image, tmpfile, reader);
  if(ret!=RET_OK){
    return ret;
  }

  return RET_OK;
}



/**
 * IWiC[Wt@CuI[v
 * @param[in]	drive	u\
 * @param[in]	reader	ǎ摕uǂ(ǎ摕u=TRUE)
 * @param[in]	data_buf   f[^obt@
 * @param[in]	bufsize    f[^obt@TCY
 * @retval	RET_OK	I
 * @retval	RET_NG	G[
 */
int OpenEnbanImageDevice(CMDDRIVE *drive, BOOL reader,
			 BYTE *data_buf, int bufsize)
{
  char tmpfile[_MAX_PATH];
  int ret;

  if(reader){
    tmpfile[0] = '\0';
  }
  else{
    strcpy(tmpfile, "image.emg");
  }
  
  ret = UIFileDialog(reader, tmpfile, sizeof(tmpfile), "emg");
  if(ret!=UIDMRET_OK){
    return RET_ABORT;
  }
  ret = InitializeCmdDrive(drive, 0, 50, CMDDRVTYPE_IMAGE);
  if(ret!=RET_OK){
    return ret;
  }
  SetDriveBuffer(drive, data_buf, bufsize);
  ret = OpenImage(&drive->u.image, tmpfile, reader);
  if(ret!=RET_OK){
    return ret;
  }

  return RET_OK;
}


/**
 * lbg[NuI[v
 * @param[in]	drive	u\
 * @param[in]	server	ǎ摕u(server)ǂ(ǎ摕u=TRUE)
 * @param[in]	data_buf   f[^obt@
 * @param[in]	bufsize    f[^obt@TCY
 * @retval	RET_OK	I
 * @retval	RET_NG	G[
 */
int OpenNetDevice(CMDDRIVE *drive, BOOL server,
		  BYTE *data_buf, int bufsize)
{
  int ret;
  char remote_host[128];
  int port_number;

  ret = UINetDialog(server, remote_host, sizeof(remote_host), &port_number);
  if(ret!=UIDMRET_OK){
    return RET_ABORT;
  }
  ret = InitializeCmdDrive(drive, 0, 50, CMDDRVTYPE_NET);
  if(ret!=RET_OK){
    return ret;
  }
  SetDriveBuffer(drive, data_buf, bufsize);
  if(server){
    /* T[o[h */
    drive->u.net.u.server.port = (WORD)port_number;
    ret = NACreateServer(&drive->u.net);
    if(ret!=RET_OK){
      DispSocketError(&drive->u.net, ret);
      return ret;
    }
    UIClearAbort();
    while(1){
      if(UICheckAbort()){
	ret = RET_ABORT;
	break;
      }
      ret = NAWaitConnect(&drive->u.net, 1000);
      if(ret==RET_OK){
	break;
      }
      if(ret!=RET_TIMEOUT){
	break;
      }
    }
    if(ret!=RET_OK){
      DispSocketError(&drive->u.net, ret);
      FreeCmdDrive(drive);
      return ret;
    }
  }
  else{
    /* NCAg[h */
	  drive->u.net.remote_host = MemNew(strlen(remote_host)+1);
	  strncpy(drive->u.net.remote_host, remote_host, strlen(remote_host)+1);
    drive->u.net.u.client.remote_port = (WORD)port_number;
    ret = NAConnect(&drive->u.net);
    if(ret!=RET_OK){
      DispSocketError(&drive->u.net, ret);
      FreeCmdDrive(drive);
      return ret;
    }

    ret = GetDiscType(drive, &drive->disc_type, TRUE);
    if(ret!=RET_OK){
      if(ret!=RET_ABORT){
	UIDispMessage(MSG_CANT_GET_DISC_TYPE
		      /*"fBXN^Cv擾Ɏs܂B"*/, UIDMT_ERROR);
      }
      FreeCmdDrive(drive);
      return ret;
    }
    ret = SendPreventAllow(drive, 1);
    if(ret!=RET_OK){
      DispCommandError(drive);
      FreeCmdDrive(drive);
      return ret;
    }
  }
  
  return RET_OK;
}

