using System;
using System.Collections.Generic;
using System.Text;
using MinorShift.Emuera.GameData.Expression;

namespace MinorShift.Emuera.Sub
{

	internal enum TokenType
	{
		Null = 0,
		Numeric = 1,
		Identifer = 2,
		EndOfLine = 3,
		WhiteSpace = 4,
		Operator = 5,
		StartOfExpression = 6,
		EndOfExpression = 7,
		EtcSymbol = 9,
		StringToken = 10,
		StringFormToken = 11,
		EndOfString = 12,
        TernaryOperator = 13,
	}

	internal static class TokenReader
	{
		//1734 速度優先のため頻度の高い関数についてはIList<char>.Containsをやめてswitchで行う
		readonly static IList<char> operators = new char[] { '+', '-', '*', '/', '%', '=', '!', '<', '>', '|', '&', '^', '~', '?', '#'};
		readonly static IList<char> whiteSpaces = new char[] { ' ', '　', '\t' };
		//readonly static IList<char> endOfExpression = new char[] { ')', '}', ']', ',', ':' };
		//readonly static IList<char> startOfExpression = new char[] { '(' };
		//readonly static IList<char> stringToken = new char[] { '\"', };
		//readonly static IList<char> stringFormToken = new char[] { '@', };
		//readonly static IList<char> etcSymbol = new char[] { '[', '{', '$', '\\', };
		readonly static IList<char> decimalDigits = new char[] { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', };
		readonly static IList<char> hexadecimalDigits = new char[] { 'a', 'b', 'c', 'd', 'e', 'f', 'A', 'B', 'C', 'D', 'E', 'F' };

		public static SingleTerm ReadInt64(StringStream st, bool retZero)
		{
			Int64 significand = 0;
			int expBase = 0;
			int exponent = 0;
			int stStartPos = st.CurrentPosition;
			int stEndPos = st.CurrentPosition;
			int fromBase = 10;
			if (st.Current == '0')
			{
				char c = st.Next;
				if ((c == 'x') || (c == 'X'))
				{
					fromBase = 16;
					st.ShiftNext();
					st.ShiftNext();
				}
				else if ((c == 'b') || (c == 'B'))
				{
					fromBase = 2;
					st.ShiftNext();
					st.ShiftNext();
				}
				//8進法は互換性の問題から採用しない。
				//else if (decimalDigits.Contains(c))
				//{
				//    fromBase = 8;
				//    st.ShiftNext();
				//}
			}
            if (!decimalDigits.Contains(st.Current) && retZero && st.Current != '+' && st.Current != '-')
            {
                if (fromBase != 16)
                    return new SingleTerm(0);
                else if (!hexadecimalDigits.Contains(st.Current))
                    return new SingleTerm(0);
            }
			significand = readDigits(st, fromBase);
			if ((st.Current == 'p') || (st.Current == 'P'))
				expBase = 2;
			else if ((st.Current == 'e') || (st.Current == 'E'))
				expBase = 10;
			if (expBase != 0)
			{
				st.ShiftNext();
				unchecked { exponent = (int)readDigits(st, fromBase); }
			}
			stEndPos = st.CurrentPosition;
			if ((expBase != 0) && (exponent != 0))
			{
				double d = significand * Math.Pow(expBase, exponent);
				if ((double.IsNaN(d)) || (double.IsInfinity(d)) || (d > Int64.MaxValue) || (d < Int64.MinValue))
					throw new CodeEE("\"" + st.Substring(stStartPos, stEndPos) + "\"は64ビット符号付整数の範囲を超えています");
				significand = (Int64)d;
			}
			return new SingleTerm(significand);
		}


		private static Int64 readDigits(StringStream st, int fromBase)
		{
			StringBuilder buffer = new StringBuilder();
			char c = st.Current;
			if ((c == '-') || (c == '+'))
			{
				buffer.Append(c);
				st.ShiftNext();
			}
			if (fromBase == 10)
			{
				while (!st.EOS)
				{
					c = st.Current;
					if (decimalDigits.Contains(c))
					{
						buffer.Append(c);
						st.ShiftNext();
						continue;
					}
					break;
				}
			}
			else if (fromBase == 16)
			{
				while (!st.EOS)
				{
					c = st.Current;
					if ( decimalDigits.Contains(c) || hexadecimalDigits.Contains(c) )
					{
						buffer.Append(c);
						st.ShiftNext();
						continue;
					}
					break;
				}
			}
			else if (fromBase == 2)
			{
				while (!st.EOS)
				{
					c = st.Current;
					if ( decimalDigits.Contains(c))
					{
						if ((c != '0') && (c != '1'))
							throw new CodeEE("二進法表記の中で使用できない文字が使われています");
						buffer.Append(c);
						st.ShiftNext();
						continue;
					}
					break;
				}
			}
            try
            {
                return Convert.ToInt64(buffer.ToString(), fromBase);
            }
            catch (FormatException)
            {
                throw new CodeEE("\"" + buffer.ToString() + "\"は整数値に変換できません");
            }
            catch (OverflowException)
            {
                throw new CodeEE("\"" + buffer.ToString() + "\"は64ビット符号付き整数の範囲を超えています");
            }
            catch (ArgumentOutOfRangeException)
            {
                throw new CodeEE("文字列\"" + st.ToString() + "\"は数値として認識できません");
            }
		}


		/// <summary>
		/// 
		/// </summary>
		/// <param name="st"></param>
		/// <param name="d"></param>
		/// <returns></returns>
		public static double ReadDouble(StringStream st)
		{
			//大雑把に読み込んでエラー処理はConvertクラスに任せる。
			//仮数小数部
			StringBuilder buffer = new StringBuilder();

			if (st.Current == '-')
			{
				buffer.Append(st.Current);
				st.ShiftNext();
			}
			while (!st.EOS)
			{//仮数部
				char c = st.Current;
				if ( decimalDigits.Contains(c) || (c == '.'))
				{
					buffer.Append(c);
					st.ShiftNext();
					continue;
				}
				break;
			}
			if ((st.Current == 'e') || (st.Current == 'E'))
			{

				buffer.Append(st.Current);
				st.ShiftNext();
				if (st.Current == '-')
				{
					buffer.Append(st.Current);
					st.ShiftNext();
				}
				while (!st.EOS)
				{//指数部
					char c = st.Current;
					if (decimalDigits.Contains(c) || (c == '.'))
					{
						buffer.Append(c);
						st.ShiftNext();
						continue;
					}
					break;
				}
			}

			//これを使用するのがTIMES第２引数のみなので…
			//try
			//{
				return Convert.ToDouble(buffer.ToString());
			//}
			//catch
			//{
			//    throw new CodeEE("\"" + buffer.ToString() + "\"を実数値に変換できませんでした");
			//}
		}

        public static int SkipAllSpace(StringStream st)
        {
            int count = 0;
            while (whiteSpaces.Contains(st.Current))
            {
                count++;
                st.ShiftNext();
            }
            return count;
        }
        
        public static int SkipWhiteSpace(StringStream st)
		{
			int count = 0;
			while (st.Current == ' ' || st.Current == '\t')
			{
				count++;
				st.ShiftNext();
			}
			return count;
		}

        public static int SkipHalfSpace(StringStream st)
        {
            int count = 0;
            while (st.Current == ' ')
            {
                count++;
                st.ShiftNext();
            }
            return count;
        }

		/// <summary>
		/// 失敗したらEmueraOperatorType.NULL
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static OperatorCode ReadOperator(StringStream st)
		{
			string opStr = ReadOperatorString(st);
			OperatorCode ret = OperatorManager.ToOperatorType(opStr);
			if (ret == OperatorCode.NULL)
				throw new CodeEE("\"" + opStr + "\"は演算子として認識できません");
			return ret;
		}

		//代入演算子のため
		public static string ReadOperatorString(StringStream st)
		{
			StringBuilder buffer = new StringBuilder();
			while (operators.Contains(st.Current))
			{
				buffer.Append(st.Current);
				st.ShiftNext();
			}
			return buffer.ToString();
		}

		/// <summary>
		/// 失敗したらnull
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static string ReadSingleIdentifer(StringStream st)
		{
			int start = st.CurrentPosition;
			while (!st.EOS)
			{
				switch(st.Current)
				{
				    case ' ': case '　': case '\t':
				    case '+': case '-': case '*': case '/': case '%': case '=': case '!': case '<': case '>': case '|': case '&': case '^': case '~': case '?': case '#':
				    case ')': case '}': case ']': case ',': case ':':
					case '(':
				        goto end;
				    case '[': case '{': case '$': case '\\':
				        throw new CodeEE("識別子に不正な文字が使われています");
				}
				st.ShiftNext();
			}
			end:
			return st.Substring(start, st.CurrentPosition - start);
		}

		/// <summary>
		/// 失敗したらnull
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static string ReadIdentiferWithIndex(StringStream st, bool replaced)
		{
			int start = st.CurrentPosition;
			int bracetNest = 0;
			while (!st.EOS)
			{
				switch (st.Current)
				{
				    case '(':
				        st.ShiftNext();
				        bracetNest++;
						while (!st.EOS && bracetNest > 0)
						{//括弧の中なら何でも一連の変数の一部と見なす
							if (st.Current == '(')
							{
								bracetNest++;
							}
							else if (st.Current == ')')
							{
								bracetNest--;
							}
							st.ShiftNext();
						}
				        continue;
				    case ' ': case '　': case '\t':
				    case '+': case '-': case '*': case '/': case '%': case '=': case '!': case '<': case '>': case '|': case '&': case '^': case '~': case '?': case '#':
				    case ')': case '}': case ','://case ':':
				        goto end;
				    case '{': case '$': case '\\':
				        throw new CodeEE("識別子に不正な文字が使われています");
                    case '[':
                        if (replaced)
                            throw new CodeEE("識別子に不正な文字が使われています");
                        break;
                    case ']':
                        if (replaced)
                            goto end;
                        break;
				}
				st.ShiftNext();
			}
		end:
			return st.Substring(start, st.CurrentPosition - start);
		}

        /// <summary>
		/// \"で始まって\"で終わる文字列。本家の文法にはない。
		/// "はエスケープするべし
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		internal static string ReadStringWithDoubleQuotation(StringStream st)
		{
			if (st.Current != '\"')
				throw new ExeEE("\"で始まっていないのにリテラルとして扱われた");
			st.ShiftNext();
			StringBuilder buffer = new StringBuilder();
			while (st.Current != '\"')
			{
				if (st.EOS)
					throw new CodeEE("\"が閉じられていません");
				if (st.Current == '\\')//エスケープ処理
				{
					st.ShiftNext();//\を読み飛ばす
					switch (st.Current)
					{
						case StringStream.EndOfString:
							throw new CodeEE("エスケープ文字\\の後に文字がありません");
						case '\n':
							break;
						case 's':
							buffer.Append(' ');
							break;
						case 'S':
							buffer.Append('　');
							break;
						case 't':
							buffer.Append('\t');
							break;
						case 'n':
							buffer.Append('\n');
							break;
						default:
							buffer.Append(st.Current);
							break;
					}
					st.ShiftNext();//\の次の文字を読み飛ばす
					continue;
				}
				buffer.Append(st.Current);
				st.ShiftNext();
			}
			st.ShiftNext();
			return buffer.ToString();
		}

		/// <summary>
		/// 特定文字で終わる文字列。splitterがnullなら,が区切り文字。printvの'で始まる文字列など
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static string ReadStringEndWith(StringStream st, IList<char> splitter)
		{
			StringBuilder buffer = new StringBuilder();
			if (splitter == null)
				splitter = new char[] { ',' };
			while (!st.EOS)
			{
				char c = st.Current;
				if (splitter.Contains(c))
					break;
				buffer.Append(st.Current);
				st.ShiftNext();
			}
			return buffer.ToString();
		}

		/// <summary>
		/// 特定文字で終わる書式付文字列。splitterがnullなら,が区切り文字。printvの'で始まる文字列など
		/// 
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static string ReadFormStringEndWith(StringStream st, IList<char> splitter)
		{
			StringBuilder buffer = new StringBuilder();
			if (splitter == null)
				splitter = new char[] { ',' };
			int nest = 0;
			bool strIn = false;
            bool inTernary = false;
			while (!st.EOS)
			{
				char c = st.Current;
				if ((nest == 0) && (!strIn) && (!inTernary) && splitter.Contains(c))
					break;
				if (c == '\\')
				{
					buffer.Append(c);
					st.ShiftNext();
                    if (st.Current == '@')
                    {
                        if (inTernary)
                            inTernary = false;
                        else
                            inTernary = true;
                        buffer.Append(st.Current);
                        st.ShiftNext();
                        continue;
                    }
                    if (!st.EOS)
					{
						buffer.Append(st.Current);
						st.ShiftNext();
					}
					continue;
				}
                if ((!strIn)&&(c == '{'))
                    nest++;
				else if ((!strIn)&&(c == '}'))
					nest--;
				else if ((nest == 0) && (c == '%'))
					strIn = !strIn;

				buffer.Append(st.Current);
				st.ShiftNext();
			}
			return buffer.ToString();
		}

        public static bool IsWhiteSpace(char c)
		{
			if (whiteSpaces.Contains(c))
				return true;
			return false;
		}

		public static TokenType GetNextTokenType(StringStream st)
		{
			if (st.EOS)
				return TokenType.EndOfString;
			switch(st.Current)
			{
				case '\n':
					return TokenType.EndOfLine;
			    case ' ': case '\t':
					return TokenType.WhiteSpace;
				case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7': case '8': case '9':
					return TokenType.Numeric;
			    case '+': case '-': case '*': case '/': case '%': case '=': case '!': case '<': case '>': case '|': case '&': case '^': case '~': case '?': case '#':
					return TokenType.Operator;
				case ')': case '}': case ']': case ',': case ':':
					return TokenType.EndOfExpression;
				case '(':
					return TokenType.StartOfExpression;
			    case '[': case '{': case '$':
					return TokenType.EtcSymbol;
				case '\\':
				{
					if (st.Next == '@')
						return TokenType.TernaryOperator;
					else
						return TokenType.EtcSymbol;
				}
				case '\"':
					return TokenType.StringToken;
				case '@':
					return TokenType.StringFormToken;
				default:
					return TokenType.Identifer;
			}
		}

	}
}
