using System;
using System.Collections.Generic;
using System.Text;
using MinorShift.Emuera.Sub;
using System.Text.RegularExpressions;
using MinorShift.Emuera.GameData.Variable;
using MinorShift.Emuera.GameData.Expression;
using MinorShift.Emuera.GameView;
using MinorShift.Emuera.GameData;
using MinorShift.Emuera.GameData.Function;

namespace MinorShift.Emuera.GameProc
{
	internal delegate void PrintWarning(string str, LogicalLine line, int level, bool isError, bool isBackComp);
	internal static class LogicalLineParser
	{
		readonly static Regex regexCom = new Regex("COM[0-9]+");
		readonly static Regex regexComAble = new Regex("COM_ABLE[0-9]+");
		readonly static Regex regexAblup = new Regex("ABLUP[0-9]+");
		private static bool isSystemLabel(string labelName)
		{
			switch (labelName)
			{
				case "EVENTFIRST":
				case "EVENTTRAIN":
				case "EVENTSHOP":
				case "EVENTBUY":
				case "EVENTCOM":
				case "EVENTTURNEND":
				case "EVENTCOMEND":
				case "EVENTEND":
				case "SHOW_STATUS":
				case "SHOW_USERCOM":
				case "USERCOM":
				case "SOURCE_CHECK":
                case "CALLTRAINTEND":
				case "SHOW_JUEL":
				case "SHOW_ABLUP_SELECT":
				case "USERABLUP":
				case "SHOW_SHOP":
				case "SAVEINFO":
				case "USERSHOP":

				case "EVENTLOAD":
				case "TITLE_LOADGAME":
				case "SYSTEM_AUTOSAVE":
				case "SYSTEM_TITLE":
					return true;
			}

			if (labelName.StartsWith("COM"))
			{
				if (regexCom.IsMatch(labelName))
					return true;
				if (regexComAble.IsMatch(labelName))
					return true;
			}
			if (labelName.StartsWith("ABLUP"))
				if (regexAblup.IsMatch(labelName))
					return true;
			return false;
		}

		readonly static char[] badSymbolAsLabelname = new char[]
		{
		    '+', '-', '*', '/', '%', '=', '!', '<', '>', '|', '&', '^', '~',
		    ' ', '　', '\t' ,
		    '\"','(', ')', '{', '[', '}', ']', ',', '.', ':',
		    '\\', '@', '$', '#', '?', ';',
		    //'_'はOK
		};

		public static LogicalLine ParseLine(string str, EmueraConsole console)
		{
			ScriptPosition position = new ScriptPosition(str);
			StringStream stream = new StringStream(str);
			return ParseLine(stream, position, console, null);
		}

        public static LogicalLine ParseLine(StringStream stream, ScriptPosition position, EmueraConsole console, Dictionary<string, string> renameDic)
        {
            int lineNo = position.LineNo;
            string s1 = null;
            string s2 = null;
            string s3 = null;
            string errMes = "";
            TokenReader.SkipWhiteSpace(stream);//先頭のホワイトスペースを読み飛ばす
            if (stream.EOS)
                return null;
            //コメント行かどうかはここに来る前に判定しておく
            //ラベル文
            try
            {
                if ((stream.Current == '$') || (stream.Current == '@'))
                {
                    bool isFunction = (stream.Current == '@');
                    stream.ShiftNext();//@か$を除去

                    string labelName = TokenReader.ReadStringEndWith(stream, new char[] { ',', '(' });
                    List<VariableToken> args = new List<VariableToken>();
                    List<IOperandTerm> defs = new List<IOperandTerm>();
                    labelName = labelName.Trim();
                    if (Config.Instance.IgnoreCase)
                        labelName = labelName.ToUpper();
                    if (labelName.Length == 0)
                    {
                        errMes = "ラベル名がありません";
                        goto err;
                    }
                    //1.721 記号をサポートしない方向に変更。
                    if (labelName.IndexOfAny(badSymbolAsLabelname) >= 0)
                        console.PrintWarning("ラベル名" + labelName + "に\"_\"以外の記号が含まれています。", position, 1);
                    //else if (firstTokenType == TokenType.Numeric)
                    else if (char.IsNumber(labelName[0]))
                        console.PrintWarning("ラベル名" + labelName + "が半角数字から始まっています", position, 0);
                    ////1.734s 変数・命令等の予約語は使えないように変更
					//1.736、$ラベル名についてはチェックしない、@関数名についてはコンフィグ次第に変更
                    else if (isFunction && Config.Instance.WarnFunctionOverloading)
                    {
						if (FunctionMethodCreator.IsSystemFunctionMethod(labelName))
						{
							if (!Config.Instance.AllowFunctionOverloading)
							{
								errMes = "関数名" + labelName + "はEmueraの式中関数として使われています";
								InvalidLabelLine labelLine = new InvalidLabelLine(position, errMes);
								return labelLine;
							}
							else
							{
								console.PrintWarning("関数名" + labelName + "はEmueraの式中関数を上書きします", position, 1);
							}
						}
						else if (BuiltInFunctionManager.ToBuiltInFunction(labelName) != BuiltInFunctionCode.__NULL__ || VariableParser.IsVariable(labelName))
						{
							console.PrintWarning("関数名" + labelName + "はEmueraの変数もしくは命令で使われています", position, 1);
						}
                    }


                    if (!stream.EOS)
                    {
                        if (!isFunction)
                        {
                            console.PrintWarning("$で始まるラベルに引数が設定されています", position, 1);
                        }
                        else
                        {
                            while (!stream.EOS)
                            {
                                if ((stream.Current != ',') && (stream.Current != '('))
                                {
                                    errMes = "引数の書式が間違っています";
                                    goto err;
                                }
                                bool startWithBracket = stream.Current == '(';
                                if (!startWithBracket)
                                    stream.ShiftNext();
                                IOperandTerm[] opArgs = ExpressionParser.ReduceArguments(stream, startWithBracket, true);
                                for (int i = 0; i < (opArgs.Length / 2); i++)
                                {
                                    if (opArgs[i * 2] == null)
                                    {
                                        errMes = "関数定義の引数は省略できません";
                                        InvalidLabelLine labelLine = new InvalidLabelLine(position, errMes);
                                        return labelLine;
                                    }
                                    if (!(opArgs[i * 2] is VariableToken))
                                    {
                                        errMes = "関数定義の引数には変数を指定してください";
                                        InvalidLabelLine labelLine = new InvalidLabelLine(position, errMes);
                                        return labelLine;
                                    }
                                    VariableToken vToken = (VariableToken)opArgs[i * 2];
                                    IOperandTerm def = opArgs[i * 2 + 1];
                                    if (vToken.IsConstant)
                                    {
                                        errMes = "関数定義の引数には代入可能な変数を指定してください";
                                        InvalidLabelLine labelLine = new InvalidLabelLine(position, errMes);
                                        return labelLine;
                                    }
                                    if (!(def is NullTerm))
                                    {
                                        if (!(def is SingleTerm))
                                        {
                                            errMes = "引数の初期値には定数のみを指定できます";
                                            InvalidLabelLine labelLine = new InvalidLabelLine(position, errMes);
                                            return labelLine;
                                        }
                                        if (vToken.Identifier.Code != VariableCode.ARG && vToken.Identifier.Code != VariableCode.ARGS)
                                        {
                                            errMes = "引数の初期値を定義できるのは\"ARG\"と\"ARGS\"のみです";
                                            InvalidLabelLine labelLine = new InvalidLabelLine(position, errMes);
                                            return labelLine;
                                        }
                                        if (vToken.GetOperandType() != def.GetOperandType())
                                        {
                                            errMes = "引数の型と初期値の型が一致していません";
                                            InvalidLabelLine labelLine = new InvalidLabelLine(position, errMes);
                                            return labelLine;
                                        }
                                    }
                                    else
                                    {
                                        if (vToken.GetOperandType() == typeof(Int64))
                                            def = new SingleTerm(0);
                                        else
                                            def = new SingleTerm("");
                                    }
                                    args.Add(vToken);
                                    defs.Add(def);
                                }

                                //stream.ShiftNext();
                                //TokenReader.SkipWhiteSpace(stream);
                                //VariableToken vars = VariableParser.ReduceVariable(stream);
                                //args.Add(vars);
                                //TokenReader.SkipWhiteSpace(stream);
                            }
                        }
                    }
                    if (isFunction)
                    {
                        if (Program.AnalysisMode)
                            console.PrintC("@" + labelName, false);
                        FunctionLabelLine labelLine = new FunctionLabelLine(position, labelName, args, defs);
                        if (isSystemLabel(labelName))
                            labelLine.Depth = 0;
                        return labelLine;
                    }
                    else
                    {
                        return new GotoLabelLine(position, labelName);
                    }
                }
                //前置インクリメント、デクリメント行
                if (stream.Current == '+' || stream.Current == '-')
                {
                    char op = stream.Current;
                    string opType;
                    stream.ShiftNext();
                    if (op == '+')
                        opType = "インクリメント";
                    else
                        opType = "デクリメント";
                    if (stream.Current != op)
                    {
                        errMes = "行が\'" + op + "\'から始まっていますが、" + opType + "ではありません";
                        goto err;
                    }
                    stream.ShiftNext();
                    TokenReader.SkipWhiteSpace(stream);
                    if (Config.Instance.UseRenameFile && renameDic != null && ((stream.ToString().IndexOf("[[") >= 0) && (stream.ToString().IndexOf("]]") >= 0)))
                    {
                        string line = stream.Substring();
                        foreach (KeyValuePair<string, string> pair in renameDic)
                            line = line.Replace(pair.Key, pair.Value);
                        stream = new StringStream(line);
                    }
                    s1 = TokenReader.ReadIdentiferWithIndex(stream, true);
                    if (!VariableParser.IsVariable(s1))
                    {
                        errMes = opType + "の対象が変数ではありません";
                        goto err;
                    }
                    if (!VariableParser.IsInteger(s1))
                    {
                        errMes = "数値変数以外の変数を" + opType + "しようとしました";
                        goto err;
                    }
                    if (VariableIdentifier.GetVariableId(s1.Split(':')[0]).Readonly)
                    {
                        if (console != null)
                            console.PrintWarning("この行では前置" + opType + "以外の処理は無視されます", position, 1);
                    }
                    TokenReader.SkipWhiteSpace(stream);
                    if (!stream.EOS)
                    {
                        errMes = opType + "行で" + opType + "以外の処理が定義されています";
                        goto err;
                    }
                    BuiltInFunctionCode f = BuiltInFunctionCode.SET;
                    string r = s1 + " " + op + " 1";
                    return new InstructionLine(position, f, s1, r);
                }
                s1 = TokenReader.ReadIdentiferWithIndex(stream, false);
                BuiltInFunctionCode func = BuiltInFunctionManager.ToBuiltInFunction(s1);
                //命令文
                if (func != BuiltInFunctionCode.__NULL__)//関数文
                {
                    if (Config.Instance.UseRenameFile && renameDic != null && ((stream.ToString().IndexOf("[[") >= 0) && (stream.ToString().IndexOf("]]") >= 0)))
                    {
                        string line = stream.Substring();
                        foreach (KeyValuePair<string, string> pair in renameDic)
                            line = line.Replace(pair.Key, pair.Value);
                        stream = new StringStream(line);
                    }

                    if (stream.EOS) //引数の無い関数
                        return new InstructionLine(position, func, null);
                    if ((stream.Current != ' ') && (stream.Current != '\t'))
                    {
                        errMes = "命令で行が始まっていますが、命令の直後に半角スペース・タブ以外の文字が来ています";
                        goto err;
                    }
                    stream.ShiftNext();
					s2 = stream.Substring();
                    return new InstructionLine(position, func, s2);
                }
                if (Config.Instance.UseRenameFile && renameDic != null && ((stream.ToString().IndexOf("[[") >= 0) && (stream.ToString().IndexOf("]]") >= 0)))
                {
                    string line = stream.ToString();
                    foreach (KeyValuePair<string, string> pair in renameDic)
                        line = line.Replace(pair.Key, pair.Value);
                    stream = new StringStream(line);
                    TokenReader.SkipWhiteSpace(stream);
                    s1 = TokenReader.ReadIdentiferWithIndex(stream, true);
                }
                //ラベルでも命令でもないので代入文のはず。変数で始まっていなければおかしい。
                if (!VariableParser.IsVariable(s1))
                {
                    errMes = "ラベル文・命令文・代入文のいずれとも解釈できない行です";
                    goto err;
                }
                if ((stream.Current != ' ') && (stream.Current != '\t') && (TokenReader.GetNextTokenType(stream) != TokenType.Operator))
                {
                    errMes = "変数で行が始まっていますが、変数の直後に半角スペース・タブ以外の文字が来ています";
                    goto err;
                }
                if (VariableParser.IsInteger(s1))
                    func = BuiltInFunctionCode.SET;
                else
                    func = BuiltInFunctionCode.SETS;

                TokenReader.SkipWhiteSpace(stream);//次の単語まで読み飛ばす
                //"変数　代入演算子　式"という形のはず。
                if ((stream.EOS) || (TokenReader.GetNextTokenType(stream) != TokenType.Operator))
                {
                    errMes = "行が変数で始まっていますが代入演算子がありません";
                    goto err;
                }
                s2 = TokenReader.ReadOperatorString(stream);
                if (s2 == "==")
                {
                    if (console != null)
                        console.PrintWarning("代入演算子に\"==\"が使われています", position, 0);
                    s2 = "=";//"=="を代入文に使うのは本当はおかしいが結構使われているので仕様にする。
                }
                if (func == BuiltInFunctionCode.SET)
                    TokenReader.SkipWhiteSpace(stream);//数式の代入なら次の単語まで読み飛ばす
                else if (stream.Current == ' ')
                    TokenReader.SkipHalfSpace(stream);//文字列の代入なら半角スペースだけを読み飛ばす

                if (stream.EOS)
                    s3 = "";
                else
					s3 = stream.Substring();

                string right = null;
                if (func == BuiltInFunctionCode.SETS)
                {
                    //right = s3;
                    right = OperatorManager.ReduceAssignmentOperatorString(s2, s1, s3);
                    if (right == null)
                    {
                        errMes = "代入式に使用できない演算子が使われました";
                        goto err;
                    }
                }
                else
                {
                    right = OperatorManager.ReduceAssignmentOperator(s2, s1, s3);
                    if (right == null)
                    {
                        errMes = "代入式に使用できない演算子が使われました";
                        goto err;
                    }
                }

                return new InstructionLine(position, func, s1, right);
            err:
                return new InvalidLine(position, errMes);
            }
            catch (CodeEE e)
            {
                System.Media.SystemSounds.Hand.Play();
                return new InvalidLine(position, e.Message);
            }
        }

		private static void printWarning(string str, LogicalLine line, int level, bool isError, bool isBackComp)
		{
			if (isError)
			{
				line.IsError = true;
				line.ErrMes = str;
			}
			if (printWarningLoad == null)
				return;
			printWarningLoad(str, line, level, isError, isBackComp);
		}

		public static PrintWarning printWarningLoad = null;
        public static InstructionLine CurrentLine = null;
		public static bool SetArgumentTo(InstructionLine line, PrintWarning printW)
		{
			if (line == null)
				return false;
			if (line.Argument != null)
				return true;
			if (line.IsError)
				return false;
			if (!Program.DebugMode && (line.Function & BuiltInFunctionCode.__DEBUG__) == BuiltInFunctionCode.__DEBUG__)
			{//非DebugモードでのDebug系命令。何もしないので引数解析も不要
				line.Argument = null;
				return true;
			}
			printWarningLoad = printW;
			CurrentLine = line;
			BuiltInFunctionCode argType = BuiltInFunctionCode.__ARG_MASK__ & line.Function;
			string rowStr = line.ArgumentStr;
			StringStream st = new StringStream(rowStr);
			Argument arg = null;
			IOperandTerm term = null;
			VariableToken varToken = null;
			StringForm strForm = null;
			try
			{
				switch (argType)
				{
					case BuiltInFunctionCode.__ARG_METHOD__:
						{
							IOperandTerm[] args = ExpressionParser.ReduceArguments(st, false, false);
							string code = line.Function.ToString();
							IOperandTerm mTerm = FunctionMethodCreator.GetFunctionMethod(null, code, args);
							arg = new FunctionArgument(mTerm);
						}
						break;
					case BuiltInFunctionCode.__ARG_VOID__:
						if ((rowStr != null) && (rowStr.Length != 0) && (rowStr.Trim().Length != 0))
							printWarning("引数が不要な命令\"" + line.Function.ToString() + "\"に引数が指定されています", line, 1, false, false);
						arg = new VoidArgument();
						break;
					case BuiltInFunctionCode.__ARG_INT_EXPRESSION__:
						term = ExpressionParser.ReduceIntegerTerm(st, null);
						if (line.Function == BuiltInFunctionCode.RETURN)
						{
							if (!(term is SingleTerm))
							{
								if (term is VariableToken)
									printWarning("RESULTの引数に変数が渡されています(eramaker：常に0を返します)", line, 0, false, true);
								else if (term is UnaryExpressionTerm || term is BinaryExpressionTerm || term is TernaryExpressionTerm)
									printWarning("RESULTの引数に数式が渡されています(eramaker：Emueraとは異なる値を返します)", line, 0, false, true);
							}
						}
						else if (line.Function == BuiltInFunctionCode.CLEARLINE)
						{
							if ((term is SingleTerm) && (term.GetOperandType() == typeof(Int64)))
							{
								if (((SingleTerm)term).Int <= 0L)
									printWarning(line.Function.ToString() + "の引数に0以下の値が渡されています(この行は何もしません)", line, 1, false, false);
							}
						}
						else if (line.Function == BuiltInFunctionCode.FONTSTYLE)
						{
							if ((term is SingleTerm) && (term.GetOperandType() == typeof(Int64)))
							{
								if (((SingleTerm)term).Int < 0L)
									printWarning(line.Function.ToString() + "の引数に負の値が渡されています(結果は不定です)", line, 1, false, false);
							}
						}
						arg = new ExpressionArgument(term);
						break;
					case BuiltInFunctionCode.__ARG_INT_ANY__:
						{
							List<IOperandTerm> termList = new List<IOperandTerm>();
							TokenReader.SkipHalfSpace(st);
							if (st.EOS)
							{
								if (line.Function == BuiltInFunctionCode.RETURN)
								{
									termList.Add(new SingleTerm(0));
									arg = new ExpressionArrayArgument(termList);
									break;
								}
								printWarning("命令" + line.Function.ToString() + "に引数が設定されていません", line, 2, true, false);
								break;
							}
							while (true)
							{
								term = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
								if (term is SingleTerm)
									if (((SingleTerm)term).Int < 0 && line.Function != BuiltInFunctionCode.RETURN)
										printWarning("命令" + line.Function.ToString() + "の引数に負の値が与えられています", line, 2, true, false);
								termList.Add(term);
								if (st.EOS)
									break;
								st.ShiftNext();
								TokenReader.SkipHalfSpace(st);
								if (st.EOS)
								{
									printWarning("\',\'の後ろに引数がありません。", line, 1, false, false);
									break;
								}
							}
							arg = new ExpressionArrayArgument(termList);
						}
						break;
					case BuiltInFunctionCode.__ARG_STR_EXPRESSION__:
						term = ExpressionParser.ReduceStringTerm(st, null);
						arg = new ExpressionArgument(term);
						break;
					case BuiltInFunctionCode.__ARG_EXPRESSION__:
						term = ExpressionParser.ReduceExpressionTerm(st, null);
						arg = new ExpressionArgument(term);
						break;
					case BuiltInFunctionCode.__ARG_STR__:
						if (rowStr == null)
						{
							if (line.Function == BuiltInFunctionCode.ALIGNMENT || line.Function == BuiltInFunctionCode.CUSTOMDRAWLINE || line.Function == BuiltInFunctionCode.BEGIN || line.Function == BuiltInFunctionCode.GOTO
								|| line.Function == BuiltInFunctionCode.STRLEN || line.Function == BuiltInFunctionCode.STRLENU || line.Function == BuiltInFunctionCode.TRYGOTO || line.Function == BuiltInFunctionCode.TRYCGOTO)
							{
								printWarning("命令" + line.Function.ToString() + "に引数が設定されていません", line, 2, true, false);
								break;
							}
							rowStr = "";
							if (line.Function == BuiltInFunctionCode.PRINTL)
								printWarning("PRINTLの後ろに空白がありません(eramaker：\'PRINTL\'を表示)", line, 0, false, true);
						}
						arg = new ExpressionArgument(new SingleTerm(rowStr));
						break;
					case BuiltInFunctionCode.__ARG_FORM_STR__:
						if (rowStr == null)
						{
							{
								if (line.Function == BuiltInFunctionCode.STRLENFORM || line.Function == BuiltInFunctionCode.STRLENFORMU || line.Function == BuiltInFunctionCode.GOTOFORM
									|| line.Function == BuiltInFunctionCode.TRYGOTOFORM || line.Function == BuiltInFunctionCode.TRYCGOTOFORM)
								{
									printWarning("命令" + line.Function.ToString() + "に引数が設定されていません", line, 2, true, false);
									break;
								}
							}
							rowStr = "";
							if (line.Function == BuiltInFunctionCode.PRINTFORML)
								printWarning("PRINTFORMLの後ろに空白がありません(eramaker：\'PRINTFORML\'を表示)", line, 0, false, true);
						}
						strForm = new StringForm(rowStr);
						if (Config.Instance.ReduceFormattedStringOnLoad)
							strForm.Reduce();
						arg = new ExpressionArgument(new StringFormTerm(strForm));
						break;
					case BuiltInFunctionCode.__ARG_FORM_STR_ANY__:
						{
							List<IOperandTerm> termList = new List<IOperandTerm>();
							TokenReader.SkipHalfSpace(st);
							if (rowStr == null)
							{
								if (line.Function == BuiltInFunctionCode.RETURNFORM)
								{
									termList.Add(new StringFormTerm(new StringForm("0")));
									arg = new ExpressionArrayArgument(termList);
									break;
								}
								printWarning("命令" + line.Function.ToString() + "に引数が設定されていません", line, 2, true, false);
								break;
							}
							while (true)
							{
								string baseStr = TokenReader.ReadStringEndWith(st, new char[] { ',' }).Trim();
								strForm = new StringForm(baseStr);
								if (Config.Instance.ReduceFormattedStringOnLoad)
									strForm.Reduce();
								termList.Add(new StringFormTerm(strForm));
								st.ShiftNext();
								if (st.EOS)
									break;
								TokenReader.SkipHalfSpace(st);
								if (st.EOS)
								{
									printWarning("\',\'の後ろに引数がありません。", line, 1, false, false);
									break;
								}
							}
							arg = new ExpressionArrayArgument(termList);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_PRINTV__:
						{
							List<IOperandTerm> list = new List<IOperandTerm>();
							while (!st.EOS)
							{
								TokenReader.SkipWhiteSpace(st);
								if (st.Current == '\'')
								{
									st.ShiftNext();
									if (!st.EOS && st.Current != ',')
										list.Add(new SingleTerm(TokenReader.ReadStringEndWith(st, null)));
								}
								else
									list.Add(ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' }));
								TokenReader.SkipWhiteSpace(st);
								if (st.Current == ',')
									st.ShiftNext();
								else
									break;
							}
							arg = new SpPrintVArgument(list);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_TIMES__:
						TokenReader.SkipWhiteSpace(st);
						if (!Char.IsLetter(st.Current))
						{
							printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 2, true, false);
							break;
						}
						varToken = VariableParser.ReduceVariable(st);
						if (varToken == null)
							printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
						else if (varToken.IsString)
							printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を文字列変数にすることはできません", line, 2, true, false);
						else if (varToken.Identifier.Readonly)
							printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に変更できない変数を指定することはできません", line, 2, true, false);

						TokenReader.SkipWhiteSpace(st);
						if (st.Current != ',')
						{
							if (st.EOS)
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
							else
								printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 2, true, false);
							break;
						}
						st.ShiftNext();
						TokenReader.SkipWhiteSpace(st);
						double d = 0.0;
						if ((TokenReader.GetNextTokenType(st) != TokenType.Numeric) && (st.Current != '-') && (st.Current != '.'))
							printWarning("命令\"" + line.Function.ToString() + "\"の第２引数が実数値ではありません（常に0と解釈されます）", line, 1, false, false);
						else
						{
							try
							{
								d = ExpressionParser.ReduceDoubleTerm(st);
							}
							catch
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数が実数値ではありません（常に0と解釈されます）", line, 1, false, false);
							}
						}
						TokenReader.SkipWhiteSpace(st);
						if (!st.EOS)
						{
							if (st.Current == ',')
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
							else
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数が実数値ではありません（常に0と解釈されます）", line, 1, false, false);
								d = 0.0;
							}
						}
						arg = new SpTimesArgument(varToken, d);
						break;
					case BuiltInFunctionCode.__ARG_SP_BAR__:
						{
							TokenReader.SkipWhiteSpace(st);
							IOperandTerm value = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							if (line.Function == BuiltInFunctionCode.TINPUT)
							{
								if ((value is SingleTerm) && (value.GetOperandType() == typeof(Int64)))
								{
									if (((SingleTerm)value).Int <= 0)
										printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に0以下の値が渡されました(入力処理は行われません)", line, 2, true, false);
								}
							}
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							IOperandTerm max = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							IOperandTerm length = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							TokenReader.SkipWhiteSpace(st);
							if (st.Current == ',')
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
							arg = new SpBarArgument(value, max, length);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_SET__:
						varToken = VariableParser.ReduceVariable(new StringStream(line.AssignmentDestStr));
						term = ExpressionParser.ReduceIntegerTerm(st, null);
						arg = new SpSetArgument(varToken, term);
						break;

					case BuiltInFunctionCode.__ARG_SP_SETS__:
						varToken = VariableParser.ReduceVariable(new StringStream(line.AssignmentDestStr));
						strForm = new StringForm(rowStr);
						if (Config.Instance.ReduceFormattedStringOnLoad)
							strForm.Reduce();
						arg = new SpSetsArgument(varToken, strForm);
						break;
					case BuiltInFunctionCode.__ARG_SP_SWAP__:
						{
							TokenReader.SkipWhiteSpace(st);
							IOperandTerm x = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							IOperandTerm y = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							TokenReader.SkipWhiteSpace(st);
							if (st.Current == ',')
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
							arg = new SpSwapCharaArgument(x, y);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_VAR__:
						{
							TokenReader.SkipWhiteSpace(st);

							string idStr = TokenReader.ReadSingleIdentifer(st);
							VariableIdentifier id = VariableIdentifier.GetVariableId(idStr);
							if (id == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							else if ((!id.IsArray1D && !id.IsArray2D && !id.IsArray3D) || (id.Code == VariableCode.RAND))
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に配列でない変数を指定することはできません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (!st.EOS)
							{
								if (st.Current == ',')
									printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
								else
									printWarning("命令\"" + line.Function.ToString() + "\"の第１引数が変数ではありません", line, 2, true, false);
							}
							arg = new SpVarsizeArgument(id);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_SAVEDATA__:
						{
							TokenReader.SkipWhiteSpace(st);
							IOperandTerm target = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								if (st.EOS)
									printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								else
									printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							TokenReader.SkipWhiteSpace(st);
							term = ExpressionParser.ReduceStringTerm(st, new char[] { ',' });
							if (term == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (!st.EOS)
							{
								if (st.Current == ',')
									printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, true, false);
								else
									printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 1, true, false);
								break;
							}
							if (line.Function == BuiltInFunctionCode.SAVEDATA)
							{
								if (term is SingleTerm && term.GetOperandType() == typeof(string))
								{
									if (((SingleTerm)term).Str.Contains("\n"))
									{
										printWarning("命令\"" + line.Function.ToString() + "\"の第２引数に改行文字が含まれています（セーブデータが破損します）", line, 2, true, false);
										break;
									}
								}
							}
							arg = new SpSaveDataArgument(target, term);
							break;
						}
					case BuiltInFunctionCode.__ARG_SP_TINPUTS__:
						TokenReader.SkipWhiteSpace(st);
						IOperandTerm time = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
						if ((time is SingleTerm) && (time.GetOperandType() == typeof(Int64)))
						{
							if (((SingleTerm)time).Int <= 0)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に0以下の値が渡されました(入力処理は行われません)", line, 2, true, false);
						}
						TokenReader.SkipWhiteSpace(st);
						if (st.Current != ',')
						{
							if (st.EOS)
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
							else
								printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 2, true, false);
							break;
						}
						st.ShiftNext();
						TokenReader.SkipWhiteSpace(st);
						term = ExpressionParser.ReduceStringTerm(st, new char[] { ',' });
						if (term == null)
							printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
						TokenReader.SkipWhiteSpace(st);
						if (st.Current != ',')
						{
							printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
							break;
						}
						st.ShiftNext();
						IOperandTerm disp = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
						TokenReader.SkipWhiteSpace(st);
						if (st.Current == ',')
							printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
						arg = new SpTInputsArgument(time, term, disp);
						break;
					case BuiltInFunctionCode.__ARG_SP_SORTCHARA__:
						{
							varToken = new VariableToken(VariableIdentifier.GetVariableId("NO"), null, null, null);
							SortOrder order = SortOrder.ASCENDING;
							TokenReader.SkipWhiteSpace(st);
							if (!st.EOS && !st.CurrentEqualTo("FORWARD") && !st.CurrentEqualTo("BACK"))
							{
								varToken = VariableParser.ReduceVariable(st);
								if (varToken == null)
									printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
								else if (!varToken.Identifier.IsCharacterData)
									printWarning("命令\"" + line.Function.ToString() + "\"の第１引数はキャラクタ変数でなければなりません", line, 2, true, false);
								TokenReader.SkipWhiteSpace(st);
								if (st.Current != ',' && !st.EOS)
								{
									printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 2, true, false);
								}
							}
							st.ShiftNext();
							TokenReader.SkipWhiteSpace(st);
							if (!st.EOS)
							{
								if (st.CurrentEqualTo("FORWARD"))
								{
									order = SortOrder.ASCENDING;
									st.Jump(7);
								}
								else if (st.CurrentEqualTo("BACK"))
								{
									order = SortOrder.DESENDING;
									st.Jump(4);
								}
								else
									printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 2, true, false);
							}
							TokenReader.SkipWhiteSpace(st);
							if (!st.EOS)
							{
								if (st.Current == ',')
									printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
								else
								{
									printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
									order = SortOrder.ASCENDING;
								}
							}
							arg = new SpSortcharaArgument(varToken, order);
							break;
						}
					case BuiltInFunctionCode.__ARG_SP_CALL__:
						{
							TokenReader.SkipWhiteSpace(st);
							string str = TokenReader.ReadStringEndWith(st, new char[] { ',', '(' });
							str = str.Trim();
							if (str.Length == 0)
							{
								printWarning(line.Function.ToString() + "命令に関数名が指定されていません", line, 2, true, false);
								break;
							}
							TokenReader.SkipWhiteSpace(st);
							IOperandTerm[] args = null;
							char start = st.Current;
							if (st.EOS)
								args = new IOperandTerm[0];
							else if ((start == '(') || (st.Current == ','))
							{
								bool startWithBracket = st.Current == '(';
								if (!startWithBracket)
									st.ShiftNext();
								args = ExpressionParser.ReduceArguments(st, startWithBracket, false);
								if (!st.EOS)
									printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 1, false, false);
							}
							else
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 2, true, false);
								break;
							}
							arg = new SpCallArgment(str, args);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_CALLFORM__:
						{
							TokenReader.SkipWhiteSpace(st);
							List<IOperandTerm> terms = new List<IOperandTerm>();
							string str = TokenReader.ReadFormStringEndWith(st, new char[] { ',', '(' });
							str = str.Trim();
							if (str.Length == 0)
							{
								printWarning(line.Function.ToString() + "命令に関数名が指定されていません", line, 2, true, false);
								break;
							}
							strForm = new StringForm(str);
							IOperandTerm[] args = null;
							char start = st.Current;
							if (st.EOS)
							{
								args = new IOperandTerm[0];
							}
							else if ((start == '(') || (st.Current == ','))
							{
								bool startWithBracket = st.Current == '(';
								if (!startWithBracket)
									st.ShiftNext();
								args = ExpressionParser.ReduceArguments(st, startWithBracket, false);
								if (!st.EOS)
									printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 1, false, false);
							}
							else
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 2, true, false);
								break;
							}
							arg = new SpCallformArgment(strForm, args);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_FOR_NEXT__:
						{
							TokenReader.SkipWhiteSpace(st);
							varToken = VariableParser.ReduceVariable(st);
							IOperandTerm start = null;
							IOperandTerm end = null;
							IOperandTerm step = null;
							if (varToken == null)
							{ printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false); break; }
							else if (!varToken.Identifier.IsInteger)
							{ printWarning("命令\"" + line.Function.ToString() + "\"の第１引数は数値型変数でなければなりません", line, 2, true, false); break; }
							else if (varToken.Identifier.IsCharacterData)
							{ printWarning("命令\"" + line.Function.ToString() + "\"の第１引数にキャラクタ変数を指定することはできません", line, 2, true, false); break; }
							else if (varToken.Identifier.Readonly)
							{ printWarning("命令\"" + line.Function.ToString() + "\"の第１引数は代入可能な変数でなければなりません", line, 2, true, false); break; }
							TokenReader.SkipWhiteSpace(st);
							if (st.EOS)
							{ printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false); break; }
							else if (st.Current != ',')
							{ printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 2, true, false); break; }
							st.ShiftNext();
							if (st.Current == ',')
								start = new SingleTerm(0);
							else
								start = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							TokenReader.SkipWhiteSpace(st);
							if (st.EOS)
							{ printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false); break; }
							else if (st.Current != ',')
							{ printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 2, true, false); break; }
							st.ShiftNext();
							end = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							TokenReader.SkipWhiteSpace(st);
							if (st.EOS)
								step = new SingleTerm(1);
							else if (st.Current == ',')
							{
								st.ShiftNext();
								step = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							}
							else
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 2, true, false);
								break;
							}
							TokenReader.SkipWhiteSpace(st);
							if (st.Current == ',')
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
							arg = new SpForNextArgment(varToken, start, end, step);
							TokenReader.SkipWhiteSpace(st);
							break;
						}

					case BuiltInFunctionCode.__ARG_SP_POWER__:
						{
							TokenReader.SkipWhiteSpace(st);
							varToken = VariableParser.ReduceVariable(st);
							if (varToken == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							else if (varToken.IsString)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を文字列変数にすることはできません", line, 2, true, false);
							else if (varToken.Identifier.Readonly)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に変更できない変数を指定することはできません", line, 2, true, false);

							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							TokenReader.SkipWhiteSpace(st);
							IOperandTerm x = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							TokenReader.SkipWhiteSpace(st);
							IOperandTerm y = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							TokenReader.SkipWhiteSpace(st);
							if (st.Current == ',')
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);

							arg = new SpPowerArgument(varToken, x, y);
							break;
						}
					case BuiltInFunctionCode.__ARG_SP_SWAPVAR__:
						{
							TokenReader.SkipWhiteSpace(st);
							VariableToken x = VariableParser.ReduceVariable(st);
							if (x == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							else if (x.Identifier.Readonly)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に変更できない変数を指定することはできません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							TokenReader.SkipWhiteSpace(st);
							VariableToken y = VariableParser.ReduceVariable(st);
							if (y == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
							else if (y.Identifier.Readonly)
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数に変更できない変数を指定することはできません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (st.Current == ',')
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);

							if (x.GetOperandType() != y.GetOperandType())
							{
								printWarning("引数の型が異なります", line, 2, true, false);
								break;
							}

							arg = new SpSwapVarArgument(x, y);
							break;
						}
					case BuiltInFunctionCode.__ARG_CASE__:
						{
							CaseExpression[] args = ExpressionParser.ReduceCaseExpressions(st);
							arg = new CaseArgument(args);
							if (!st.EOS)
								printWarning("命令\"" + line.Function.ToString() + "\"の書式が間違っています", line, 1, true, false);
						}
						break;
					case BuiltInFunctionCode.__ARG_VAR_INT__:
						{
							TokenReader.SkipWhiteSpace(st);
							varToken = null;
							if (!st.EOS)
								varToken = VariableParser.ReduceVariable(st);
							if (varToken != null)
							{
								if (varToken.IsString)
									printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を文字列変数にすることはできません", line, 2, true, false);
								else if (varToken.Identifier.Readonly)
									printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に変更できない変数を指定することはできません", line, 2, true, false);
							}
							TokenReader.SkipWhiteSpace(st);
							if (st.Current == ',')
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
							arg = new PrintDataArgument(varToken);
						}
						break;
					case BuiltInFunctionCode.__ARG_BIT_ARG__:
						{
							TokenReader.SkipWhiteSpace(st);
							varToken = null;
							varToken = VariableParser.ReduceVariable(st);
							if (varToken == null)
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
								break;
							}
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							IOperandTerm y = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							if (y == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (st.Current == ',')
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
							if (varToken.IsString)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を文字列変数にすることはできません", line, 2, true, false);
							else if (varToken.Identifier.Readonly)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に変更できない変数を指定することはできません", line, 2, true, false);

							if (y is SingleTerm)
							{
								Int64 bit = ((SingleTerm)y).Int;
								if ((bit < 0) || (bit > 63))
								{
									printWarning("命令\"" + line.Function.ToString() + "\"の第２引数が範囲(０～６３)を超えています", line, 2, true, false);
									break;
								}
							}
							arg = new BitArgument(varToken, y);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_VAR_SET__:
						{
							IOperandTerm term3 = null;
							IOperandTerm term4 = null;
							TokenReader.SkipWhiteSpace(st);
							varToken = VariableParser.ReduceVariable(st);
							if (varToken == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							else if (varToken.Identifier.Readonly)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に変更できない変数を指定することはできません", line, 2, true, false);
							term = null;
							TokenReader.SkipWhiteSpace(st);
							if (!st.EOS)
							{
								if (st.Current == ',')
								{
									st.ShiftNext();
									term = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
									if (term == null)
										printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
									TokenReader.SkipWhiteSpace(st);
									if (!st.EOS)
									{
										if (st.Current == ',' && varToken.Identifier.IsArray1D)
										{
											st.ShiftNext();
											term3 = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
											if (term3 == null)
												printWarning("命令\"" + line.Function.ToString() + "\"の第３引数を認識できません", line, 2, true, false);
											else if (term3.GetOperandType() != typeof(Int64))
												printWarning("命令\"" + line.Function.ToString() + "\"の第３引数が数値ではありません", line, 2, true, false);
											TokenReader.SkipWhiteSpace(st);
											if (!st.EOS)
											{
												TokenReader.SkipWhiteSpace(st);
												if (st.Current == ',')
												{
													st.ShiftNext();
													term4 = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
													if (term4 == null)
														printWarning("命令\"" + line.Function.ToString() + "\"の第４引数を認識できません", line, 2, true, false);
													else if (term4.GetOperandType() != typeof(Int64))
														printWarning("命令\"" + line.Function.ToString() + "\"の第４引数が数値ではありません", line, 2, true, false);
													if (!st.EOS)
													{
														if (st.Current == ',')
															printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
														else
															printWarning("命令\"" + line.Function.ToString() + "\"の第４引数を認識できません", line, 2, true, false);
													}
												}
												else
													printWarning("命令\"" + line.Function.ToString() + "\"の第３引数を認識できません", line, 2, true, false);
											}
										}
										else if (st.Current == ',' && !varToken.Identifier.IsArray1D)
											printWarning("命令\"" + line.Function.ToString() + "\"の第３引数以降は1次元配列以外では無視されます", line, 1, false, false);
										else
											printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
									}
								}
								else
									printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							}
							TokenReader.SkipWhiteSpace(st);
							if (term == null)
							{
								if (varToken.IsString)
									term = new SingleTerm("");
								else
									term = new SingleTerm(0);
							}
							else if (term.GetOperandType() != varToken.GetOperandType())
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の２つの引数の型が一致していません", line, 2, true, false);
								break;
							}
							arg = new SpVarSetArgument(varToken, term, term3, term4);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_CVAR_SET__:
						{
							IOperandTerm term4 = null;
							IOperandTerm term5 = null;
							TokenReader.SkipWhiteSpace(st);
							varToken = VariableParser.ReduceVariable(st);
							if (varToken == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							else if (varToken.Identifier.Readonly)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に変更できない変数を指定することはできません", line, 2, true, false);
							else if (!varToken.Identifier.IsCharacterData)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数にキャラクタ変数以外の変数を指定することはできません", line, 2, true, false);
							IOperandTerm index = null;
							TokenReader.SkipWhiteSpace(st);
							if (!st.EOS)
							{
								if (st.Current == ',')
								{
									st.ShiftNext();
									index = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
									if (index == null)
										printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 1, false, false);
								}
								else
									printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							}
							if (index == null)
								index = new SingleTerm(0);
							else if (index is SingleTerm && index.GetOperandType() == typeof(string) && varToken.Identifier.IsArray1D)
							{
								if (!ConstantData.isDefined(varToken.Identifier.Code, ((SingleTerm)index).Str))
									printWarning("文字列" + ((SingleTerm)index).Str + "は変数" + varToken.Identifier.Code.ToString() + "の要素ではありません", line, 2, true, false);
							}
							term = null;
							TokenReader.SkipWhiteSpace(st);
							if (!st.EOS)
							{
								if (st.Current == ',')
								{
									st.ShiftNext();
									TokenReader.SkipWhiteSpace(st);
									term = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
									if (term == null)
										printWarning("命令\"" + line.Function.ToString() + "\"の第３引数を認識できません", line, 1, false, false);
									TokenReader.SkipWhiteSpace(st);
									if (!st.EOS)
									{
										if (st.Current == ',' && varToken.Identifier.IsArray1D)
										{
											st.ShiftNext();
											TokenReader.SkipWhiteSpace(st);
											term4 = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
											if (term4 == null)
												printWarning("命令\"" + line.Function.ToString() + "\"の第４引数を認識できません", line, 2, true, false);
											else if (term4.GetOperandType() != typeof(Int64))
												printWarning("命令\"" + line.Function.ToString() + "\"の第４引数が数値ではありません", line, 2, true, false);
											TokenReader.SkipWhiteSpace(st);
											if (!st.EOS)
											{
												TokenReader.SkipWhiteSpace(st);
												if (st.Current == ',')
												{
													st.ShiftNext();
													term5 = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
													if (term5 == null)
														printWarning("命令\"" + line.Function.ToString() + "\"の第５引数を認識できません", line, 2, true, false);
													else if (term5.GetOperandType() != typeof(Int64))
														printWarning("命令\"" + line.Function.ToString() + "\"の第５引数が数値ではありません", line, 2, true, false);
													if (!st.EOS)
													{
														if (st.Current == ',')
															printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
														else
															printWarning("命令\"" + line.Function.ToString() + "\"の第４引数を認識できません", line, 2, true, false);
													}
												}
												else
													printWarning("命令\"" + line.Function.ToString() + "\"の第４引数を認識できません", line, 2, true, false);
											}
										}
										else if (st.Current == ',' && !varToken.Identifier.IsArray1D)
											printWarning("命令\"" + line.Function.ToString() + "\"の第４引数以降は1次元配列以外では無視されます", line, 1, false, false);
										else
											printWarning("命令\"" + line.Function.ToString() + "\"の第３引数を認識できません", line, 2, true, false);
									}
								}
								else
									printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
							}
							if (term == null)
							{
								if (varToken.IsString)
									term = new SingleTerm("");
								else
									term = new SingleTerm(0);
							}
							else if (term.GetOperandType() != varToken.GetOperandType())
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の２つの引数の型が一致していません", line, 2, true, false);
								break;
							}
							arg = new SpCVarSetArgument(varToken, index, term, term4, term5);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_BUTTON__:
						//case BuiltInFunctionCode.__ARG_SP_BUTTONS__:
						{
							IOperandTerm x, y = null;
							TokenReader.SkipWhiteSpace(st);
							x = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
							if (x == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							else if (x.GetOperandType() != typeof(string))
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数は文字列型でなければなりません", line, 2, true, false);

							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							y = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
							if (y == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (st.Current == ',')
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
							arg = new SpButtonArgument(x, y);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_SET_ARRAY__:
					case BuiltInFunctionCode.__ARG_SP_SETS_ARRAY__:
						{
							Type type = typeof(long);
							if (argType == BuiltInFunctionCode.__ARG_SP_SETS_ARRAY__)
								type = typeof(string);
							varToken = VariableParser.ReduceVariable(new StringStream(line.AssignmentDestStr));
							IOperandTerm[] termList = ExpressionParser.ReduceArguments(st, false, false);
							foreach (IOperandTerm t in termList)
							{
								if ((t == null) || (t.GetOperandType() != type))
								{
									printWarning("代入先と代入する値の型が違います", line, 2, true, false);
									break;
								}
							}
							arg = new SpSetArrayArgument(varToken, termList);
							break;
						}
					case BuiltInFunctionCode.__ARG_SP_COLOR__:
						{
							TokenReader.SkipWhiteSpace(st);
							IOperandTerm r = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							if (r == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								arg = new SpColorArgument(r);
								break;
							}
							st.ShiftNext();
							IOperandTerm g = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							if (g == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							IOperandTerm b = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							if (b == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第３引数を認識できません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (st.Current == ',')
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
							arg = new SpColorArgument(r, g, b);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_SPLIT__:
						{
							TokenReader.SkipWhiteSpace(st);
							IOperandTerm targetString = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
							if (targetString == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							if (targetString.GetOperandType() != typeof(string))
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数は文字列型でなければなりません", line, 2, true, false);

							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							IOperandTerm splitChar = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
							if (splitChar == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
							if (splitChar.GetOperandType() != typeof(string))
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数は文字列型でなければなりません", line, 2, true, false);

							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の第３引数を認識できません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							TokenReader.SkipWhiteSpace(st);
							varToken = VariableParser.ReduceVariable(st);
							if (varToken == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第３引数を認識できません", line, 2, true, false);
							if (varToken.Identifier.Readonly)
								printWarning("命令\"" + line.Function.ToString() + "\"の第３引数に変更できない変数を指定することはできません", line, 2, true, false);
							if (!varToken.Identifier.IsString)
								printWarning("命令\"" + line.Function.ToString() + "\"の第３引数は文字列変数でなければなりません", line, 2, true, false);
							if (!varToken.Identifier.IsArray1D && !varToken.Identifier.IsArray2D && !varToken.Identifier.IsArray3D)
								printWarning("命令\"" + line.Function.ToString() + "\"の第３引数は配列変数でなければなりません", line, 2, true, false);

							TokenReader.SkipWhiteSpace(st);
							if (!st.EOS)
							{
								if (st.Current == ',')
									printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
								else
									printWarning("命令\"" + line.Function.ToString() + "\"の引数が解釈不能です", line, 2, true, false);
							}
							arg = new SpSplitArgument(targetString, splitChar, varToken.Identifier);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_GETINT__:
						{
							TokenReader.SkipWhiteSpace(st);
							varToken = VariableParser.ReduceVariable(st);
							if (varToken == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							else if (varToken.IsString)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を文字列変数にすることはできません", line, 2, true, false);
							else if (varToken.Identifier.Readonly)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に変更できない変数を指定することはできません", line, 2, true, false);

							TokenReader.SkipWhiteSpace(st);
							if (!st.EOS)
							{
								if (st.Current == ',')
									printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
								else
									printWarning("命令\"" + line.Function.ToString() + "\"の引数が解釈不能です", line, 2, true, false);
							}
							arg = new SpGetIntArgument(varToken);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_CONTROL_ARRAY__:
						{
							TokenReader.SkipWhiteSpace(st);
							VariableToken x = VariableParser.ReduceVariable(st);
							if (x == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							else if (x.Identifier.Readonly)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に変更できない変数を指定することはできません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							TokenReader.SkipWhiteSpace(st);
							IOperandTerm y = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							if (y == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							TokenReader.SkipWhiteSpace(st);
							IOperandTerm z = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							if (z == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第３引数を認識できません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (st.Current == ',')
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);

							arg = new SpArrayControlArgument(x, y, z);
						}
						break;
					case BuiltInFunctionCode.__ARG_SP_SHIFT_ARRAY__:
						{
							IOperandTerm term4 = null;
							IOperandTerm term5 = null;
							TokenReader.SkipWhiteSpace(st);
							VariableToken x = VariableParser.ReduceVariable(st);
							if (x == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数を認識できません", line, 2, true, false);
							else if (x.Identifier.Readonly)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に変更できない変数を指定することはできません", line, 2, true, false);
							else if (!x.Identifier.IsArray1D)
								printWarning("命令\"" + line.Function.ToString() + "\"の第１引数に１次元配列もしくは配列型キャラクタ変数以外を指定することはできません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							TokenReader.SkipWhiteSpace(st);
							IOperandTerm y = ExpressionParser.ReduceIntegerTerm(st, new char[] { ',' });
							if (y == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第２引数を認識できません", line, 2, true, false);
							TokenReader.SkipWhiteSpace(st);
							if (st.Current != ',')
							{
								printWarning("命令\"" + line.Function.ToString() + "\"の引数が足りません", line, 2, true, false);
								break;
							}
							st.ShiftNext();
							TokenReader.SkipWhiteSpace(st);
							term = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
							if (term == null)
								printWarning("命令\"" + line.Function.ToString() + "\"の第３引数を認識できません", line, 2, true, false);
							if (line.Function == BuiltInFunctionCode.ARRAYSHIFT)
							{
								if (term.GetOperandType() != x.GetOperandType())
									printWarning("命令\"" + line.Function.ToString() + "\"の第１引数と第３引数の型が違います", line, 2, true, false);
							}
							TokenReader.SkipWhiteSpace(st);
							if (!st.EOS)
							{
								if (st.Current == ',')
								{
									st.ShiftNext();
									TokenReader.SkipWhiteSpace(st);
									term4 = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
									if (term4 == null)
										printWarning("命令\"" + line.Function.ToString() + "\"の第４引数を認識できません", line, 2, true, false);
									else if (term4.GetOperandType() != typeof(Int64))
										printWarning("命令\"" + line.Function.ToString() + "\"の第４引数が数値ではありません", line, 2, true, false);
									TokenReader.SkipWhiteSpace(st);
									if (!st.EOS)
									{
										TokenReader.SkipWhiteSpace(st);
										if (st.Current == ',')
										{
											st.ShiftNext();
											TokenReader.SkipWhiteSpace(st);
											term5 = ExpressionParser.ReduceExpressionTerm(st, new char[] { ',' });
											if (term5 == null)
												printWarning("命令\"" + line.Function.ToString() + "\"の第５引数を認識できません", line, 2, true, false);
											else if (term5.GetOperandType() != typeof(Int64))
												printWarning("命令\"" + line.Function.ToString() + "\"の第５引数が数値ではありません", line, 2, true, false);
											if (!st.EOS)
											{
												if (st.Current == ',')
													printWarning("命令\"" + line.Function.ToString() + "\"の引数が多すぎます", line, 1, false, false);
												else
													printWarning("命令\"" + line.Function.ToString() + "\"の第４引数を認識できません", line, 2, true, false);
											}
										}
										else
											printWarning("命令\"" + line.Function.ToString() + "\"の第４引数を認識できません", line, 2, true, false);
									}
								}
							}
							if (term4 == null)
								term4 = new SingleTerm(0);
							arg = new SpArrayShiftArgument(x, y, term, term4, term5);
						}
						break;
					default:
						printWarning("設定されていない引数型", line, 2, true, false);
						return false;
				}
			}
			catch (CodeEE e)
			{
				System.Media.SystemSounds.Hand.Play();
				printWarning(e.Message, line, 2, true, false);
				return false;
			}
			if (arg == null)
			{
				if (!line.IsError)
				{
					System.Media.SystemSounds.Hand.Play();
					printWarning("命令の引数解析中に特定できないエラーが発生", line, 2, true, false);
				}
				return false;
			}
			line.Argument = arg;
			return true;
		}
	}
}
