
#include "defs.h"

#include "eb123.h"
#include "ebgstsrc.h"
#include "history.h"
#include "mainwnd.h"
#include "render.h"
#include "video.h"

G_DEFINE_TYPE(Video, video, GTK_TYPE_FRAME);

#define video_DECODER    "decoder"
#define video_COLOR	    "colorspace"
#define video_SCALE	    "videoscale"
#define video_SINK	    "videosink"

gboolean video_save_mpeg(RESULT *res, gchar *file)
{
    char binary_data[EB_SIZE_PAGE];
    guint argv[4];
    EB_Error_Code error_code;
    ssize_t read_length;
    FILE *fp;

    BOOK_INFO *binfo = res->binfo;
    if(!binfo)
	return FALSE;

    error_code = eb_set_subbook(binfo->book, binfo->subbook_no);
    if(error_code != EB_SUCCESS)
    {
	LOG(LOG_CRITICAL, "Failed to set subbook %s, %d : %s", binfo->path, binfo->subbook_no, eb_error_message(error_code));
	return FALSE;
    }

    if((error_code = eb_decompose_movie_file_name(argv, res->filename)) != EB_SUCCESS)
	return FALSE;

    error_code = eb_set_binary_mpeg(binfo->book, argv);
    if(error_code != EB_SUCCESS)
    {
	LOG(LOG_CRITICAL, "Failed to set binary mpeg: %s", eb_error_message(error_code));
	return FALSE;
    }

    fp = fopen(file, "wb");
    if(fp == NULL)
    {
	LOG(LOG_CRITICAL, "Failed to open file: %s", file);
	return FALSE;
    }

    for(;;)
    {
	error_code = eb_read_binary(binfo->book, EB_SIZE_PAGE, binary_data, &read_length);
	if(error_code != EB_SUCCESS || read_length == 0)
	{
	    fclose(fp);
	    return FALSE;
	}
	fwrite(binary_data, read_length, 1, fp);
    }

    return TRUE;
}

void video_save_as(GtkWidget *w, gpointer data)
{       
    Video *v = VIDEO(data);
    static gchar *path = NULL;
    path = app_browse_disk(_("Save Video As"), mainwnd_get_wnd(), GTK_FILE_CHOOSER_ACTION_SAVE, GTK_STOCK_SAVE_AS, path);
    if(path)
    {
        RESULT *res = v->res;
        if(video_save_mpeg(res, path) != EB_SUCCESS)
            LOG(LOG_WARNING, _("Failed to save video"));
    }

}

#ifdef ENABLE_GSTREAMER

static gpointer video_parent_class_ptr = NULL;

static void video_finalize(GObject *obj)
{
    g_return_if_fail(obj != NULL);
    g_return_if_fail(IS_VIDEO(obj));

    Video *v = VIDEO(obj);
    g_idle_remove_by_data(v->pipeline);
    gst_element_set_state(v->pipeline, GST_STATE_NULL);
    gst_object_unref(GST_OBJECT(v->pipeline));
    G_OBJECT_CLASS(video_parent_class_ptr)->finalize(obj);
}

static void video_play_pause_set_icon(Video *self, gboolean play)
{
    GtkWidget *image = gtk_bin_get_child(GTK_BIN(self->play_btn));
    gtk_widget_destroy(image);
    image = gtk_image_new_from_stock(play ? GTK_STOCK_MEDIA_PAUSE : GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_button_set_image(GTK_BUTTON(self->play_btn), image);
}

static void video_play_pause(Video *self, gboolean play)
{
    GdkWindow *window = gtk_widget_get_window(self->output);
    if(window && self->pipeline)
    {
	gst_x_overlay_set_xwindow_id(GST_X_OVERLAY(self->sink), GDK_WINDOW_XID(window));
	gst_element_set_state(self->pipeline, play ? GST_STATE_PLAYING : GST_STATE_PAUSED);
	video_play_pause_set_icon(self, play);
    }
}

static gboolean video_pause(gpointer data)
{
    Video *v = VIDEO(data);
    video_play_pause(v, FALSE);
    return FALSE;
}

static gboolean video_reset(gpointer data)
{
    Video *v = VIDEO(data);
    video_play_pause(v, TRUE);
    g_idle_add(video_pause, data);
    return FALSE;
}

static void video_resize(Video *self, gint w, gint h)
{
    GdkScreen *screen = gdk_screen_get_default();
    gint sw = gdk_screen_get_width(screen), sh = gdk_screen_get_height(screen);
    if((w > 0) && (h > 0) && (w <= sw) && (h <= sh))
    {
	gtk_widget_set_size_request(self->output, w, h);
#if 0
	GtkWidget *wnd = gtk_widget_get_toplevel(self->output);
	gtk_widget_set_size_request(wnd, -1, h + 15*GTK_ICON_SIZE_SMALL_TOOLBAR);
#endif
	self->video_resized = TRUE;
    }
}

gboolean video_bus_cb(GstBus *bus, GstMessage *msg, gpointer data)
{
    Video *v = VIDEO(data);
    switch (GST_MESSAGE_TYPE(msg))
    {
	case GST_MESSAGE_STATE_CHANGED:
	{
	    if(v->video_resized)
		break;
	    GstState oldstate, newstate;
	    gst_message_parse_state_changed(msg, &oldstate, &newstate, NULL);
	    if(oldstate == GST_STATE_READY && newstate == GST_STATE_PAUSED)
	    {
		gchar *name = gst_element_get_name(msg->src);
		if(!g_strcmp0(name, video_DECODER))
		{
		    GstPad *pad = gst_element_get_static_pad(GST_ELEMENT(msg->src), "src");
		    if(pad)
		    {
			gint w, h;
			gst_video_get_size(pad, &w, &h);
			video_resize(v, w, h);
		    }
		}
		g_free(name);
	    }
	    break;
	}
	case GST_MESSAGE_EOS:
	{
	    gst_element_set_state(v->pipeline, GST_STATE_NULL);
	    g_idle_add(video_reset, (gpointer)v);
	    break;
	}

	case GST_MESSAGE_ERROR:
	{
	    gchar  *debug;
	    GError *error;

	    gst_message_parse_error(msg, &error, &debug);
	    g_free(debug);

	    LOG(LOG_INFO, "%s\n", error->message);
	    g_error_free(error);

	    break;
	}
	 default:
	    break;
    }

    return TRUE;
}

static gboolean video_timeout(gpointer data)
{
    if(!IS_VIDEO(data)) return FALSE;

    Video *v = VIDEO(data);

    GstFormat format = GST_FORMAT_TIME;
    gchar time[32];
    gint64 pos, len;
    gint hh, mm, ss;

    if(!GST_IS_ELEMENT(v->pipeline))
	return FALSE;

    gst_element_query_position(v->pipeline, &format, &pos);
    gst_element_query_duration (v->pipeline, &format, &len);
    g_print ("Time: %" GST_TIME_FORMAT " / %" GST_TIME_FORMAT "      \r" ,
                     GST_TIME_ARGS (pos), GST_TIME_ARGS (len));

    if(format != GST_FORMAT_TIME)
	return FALSE;

    pos /= 1000000000;
    ss = pos % 60;
    pos /= 60;
    mm = pos % 60;
    hh = pos / 60;

    if(!hh)
	sprintf(time, "%.2d:%.2d", mm, ss);
    else
	sprintf(time, "%.2d:%.2d:%.2d", hh, mm, ss);

    gtk_label_set_text(GTK_LABEL(v->time), time);

    gst_x_overlay_expose(GST_X_OVERLAY(v->sink));

    return TRUE;
}

static void video_output_prepare(Video *self)
{
    GstElement *src, *dec, *color, *scale;
    GstBus *bus;
    RESULT *res = self->res;

    self->pipeline = gst_pipeline_new("pipeline");
    src = g_object_new(EB_GST_TYPE_SRC, NULL);
    EB_GST_SRC(src)->res = res;
    dec = gst_element_factory_make("mpeg2dec", video_DECODER);
    color = gst_element_factory_make("ffmpegcolorspace", video_COLOR);
    scale = gst_element_factory_make("videoscale", video_SCALE);
    self->sink = gst_element_factory_make("ximagesink", video_SINK); 

    gst_bin_add_many(GST_BIN(self->pipeline), src, dec, color, scale, self->sink, NULL);
    gst_element_link_many(src, dec, color, scale, self->sink, NULL);

    bus = gst_pipeline_get_bus(GST_PIPELINE(self->pipeline));
    gst_bus_add_watch(bus, video_bus_cb, (gpointer)self);
    gst_object_unref(bus);

    g_timeout_add(500, video_timeout, (gpointer)self);

    video_reset((gpointer)self);
}

void video_play_pause_cb(GtkWidget *w, gpointer data)
{
    Video *v = VIDEO(data);
    GstState state, pending;
    gst_element_get_state(GST_ELEMENT(v->pipeline), &state, &pending, GST_CLOCK_TIME_NONE);
    gboolean play = (state != GST_STATE_PLAYING);
    video_play_pause(v, play);
}

#endif

static void video_set_property(GObject *object, guint param_id, const GValue *value, GParamSpec *pspec)
{
    Video *self = VIDEO(object);

    switch(param_id)
    {
        case 1:
            self->res = g_value_get_pointer(value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID(object, param_id, pspec);
            break;
    }
}

static void video_class_init(VideoClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
#ifdef ENABLE_GSTREAMER
    gobject_class->finalize = video_finalize;
    video_parent_class_ptr = g_type_class_peek_parent(klass);
#endif
    gobject_class->set_property = video_set_property;
    g_object_class_install_property(gobject_class, 1, g_param_spec_pointer("link", _("Link"), _("Link"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}

static void video_init(Video *self)
{
    GtkWidget *hbox = NULL;
#ifdef ENABLE_GSTREAMER
    self->play_btn = NULL;
    GtkWidget *vbox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 0);
    gtk_container_add(GTK_CONTAINER(self), vbox);

    self->output = gtk_drawing_area_new();
    gtk_box_pack_start(GTK_BOX(vbox), self->output, TRUE, TRUE, 0);
    gtk_widget_set_size_request(self->output, 320, 240);
    self->video_resized = FALSE;

    hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);

    self->play_btn = gtk_button_new();
    GtkWidget *image = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_button_set_image(GTK_BUTTON(self->play_btn), image);
    g_signal_connect(G_OBJECT(self->play_btn), "clicked", G_CALLBACK(video_play_pause_cb), self);
    gtk_box_pack_start(GTK_BOX(hbox), self->play_btn, FALSE, TRUE, 0);

    self->time = gtk_label_new("");
    gtk_misc_set_alignment(GTK_MISC(self->time), 1, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), self->time, TRUE, TRUE, 0);
#else
    hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 0);
    gtk_container_add(GTK_CONTAINER(self), hbox);

    GtkWidget *label = gtk_label_new(_("Video"));
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
#endif
    self->save_btn = gtk_button_new();
    GtkWidget *img = gtk_image_new_from_stock(GTK_STOCK_SAVE_AS, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_container_add(GTK_CONTAINER(self->save_btn), img);
    gtk_box_pack_start(GTK_BOX(hbox), self->save_btn, FALSE, TRUE, 0);
    g_signal_connect(G_OBJECT(self->save_btn), "clicked", G_CALLBACK(video_save_as), self);
}

void video_render(Video *self, RenderTextCtx *ctx)
{
    GtkTextIter iter1, iter2;
    if(render_get_last_mark(ctx, "mpeg", &iter1, &iter2))
    {
        gtk_text_buffer_delete(ctx->buf, &iter1, &iter2);
        gtk_text_buffer_get_end_iter(ctx->buf, &iter1);
        gtk_text_buffer_insert(ctx->buf, &iter1, "\n", -1);
        gtk_text_buffer_get_end_iter(ctx->buf, &iter1);
        GtkTextChildAnchor *anchor = gtk_text_buffer_create_child_anchor(ctx->buf, &iter1);
        gtk_text_view_add_child_at_anchor(GTK_TEXT_VIEW(ctx->view), GTK_WIDGET(self), anchor);
        gtk_text_buffer_insert(ctx->buf, &iter1, "\n", -1);
        g_object_set_data_full(G_OBJECT(self), "video", ctx->link, result_free);
	gtk_widget_show_all(GTK_WIDGET(self));
	gtk_widget_realize(GTK_WIDGET(self));
	while(gtk_events_pending())
	    gtk_main_iteration();
#ifdef ENABLE_GSTREAMER
	video_output_prepare(self);
#endif
    }
}

