
#include "defs.h"

#include "dump.h"
#include "ebook.h"
#include "history.h"
#include "mainwnd.h"
#include "render.h"
#include "textview.h"

G_DEFINE_TYPE(TextView, textview, GTK_TYPE_TEXT_VIEW);

static void textview_search_selection_cb(GtkWidget *w, gpointer data)
{
    gchar *str;
    GtkTextIter start, end;
    GtkTextView *text = GTK_TEXT_VIEW(data);
    GtkTextBuffer *buf = gtk_text_view_get_buffer(text);
    Mainwnd *mw = MAINWND(gtk_widget_get_toplevel(GTK_WIDGET(text)));

    gtk_text_buffer_get_selection_bounds(buf, &start, &end);
    str = gtk_text_buffer_get_text(buf, &start, &end, FALSE);
    mainwnd_search(mw, str, -1);
}

static GtkTextTag* textview_get_link_at_iter(TextView *self, GtkTextIter *iter)
{
    GtkTextTag  *res = NULL;
    GSList      *tag_list = NULL;
    if((tag_list = gtk_text_iter_get_tags(iter)))
    {
        GSList *tag_l = tag_list;
        while(tag_l)
        {
            GtkTextTag *tag = GTK_TEXT_TAG(tag_l->data);
            gpointer p1 = g_object_get_data(G_OBJECT(tag), "link");
            gpointer p2 = g_object_get_data(G_OBJECT(tag), "audio");
            gpointer p3 = g_object_get_data(G_OBJECT(tag), "video");
            if(p1 || p2 || p3)
            {
                res = tag;
                break;
            }
            tag_l = g_slist_next(tag_l);
        }
        g_slist_free(tag_list);
    }
    return res;
}

static GtkTextTag *textview_get_link_under_cursor(TextView *self, gint ex, gint ey)
{
    gint        x, y;
    GtkTextIter iter;

    GtkTextView *view = GTK_TEXT_VIEW(self);
    gtk_text_view_window_to_buffer_coords(view, GTK_TEXT_WINDOW_TEXT, ex, ey, &x, &y);
    gtk_text_view_get_iter_at_location(view, &iter, x, y);
    return textview_get_link_at_iter(self, &iter);
}

void textview_underline_link(TextView *self, GdkEventMotion *event)
{
    GdkCursor   *cursor;
    if(self->tag)
        g_object_set(G_OBJECT(self->tag), "underline", PANGO_UNDERLINE_NONE, NULL);

    if(!event)
        return;

    self->tag = textview_get_link_under_cursor(self, (gint)(event->x), (gint)(event->y));
    cursor = gdk_cursor_new(self->tag ? GDK_HAND2 : GDK_LEFT_PTR);
    gdk_window_set_cursor(gtk_text_view_get_window(GTK_TEXT_VIEW(self), GTK_TEXT_WINDOW_TEXT), cursor);
    g_object_unref(cursor);
    if(self->tag)
        g_object_set(G_OBJECT(self->tag), "underline", PANGO_UNDERLINE_SINGLE, NULL);
}

gint textview_motion_notify_cb(GtkWidget *widget, GdkEventMotion *event)
{
    textview_underline_link(TEXTVIEW(widget), event);
    return !(event->state & GDK_BUTTON1_MASK);
}

gint textview_leave_notify_cb(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
    GdkCursor *cursor = gdk_cursor_new(GDK_LEFT_PTR);
    TextView *view = TEXTVIEW(widget);
    gdk_window_set_cursor(gtk_text_view_get_window(GTK_TEXT_VIEW(widget), GTK_TEXT_WINDOW_TEXT), cursor);
    g_object_unref(cursor);
    textview_underline_link(view, NULL);
    view->tag = NULL;
    return FALSE;
}

static void textview_populate_popup_cb(GtkTextView *text, GtkMenu *menu, gpointer data)
{
    GtkTextBuffer *buf = gtk_text_view_get_buffer(text);
    GtkWidget *item = NULL;
    TextView *t = TEXTVIEW(text);

    if(gtk_text_buffer_get_has_selection(buf))
    {
	item = gtk_separator_menu_item_new();
	gtk_widget_show(item);
	gtk_menu_shell_prepend(GTK_MENU_SHELL(menu), item);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_FIND, NULL);
	gtk_widget_show(item);
	gtk_menu_shell_prepend(GTK_MENU_SHELL(menu), item);
	g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(textview_search_selection_cb), (gpointer)text);
    }
    if(t->dump)
    {
	item = gtk_separator_menu_item_new();
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	item = gtk_check_menu_item_new_with_label(_("Hex dump"));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), dump_dlg_visible(t->dump, DUMP_HEX));
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(dump_hex_cb), t->dump);

	item = gtk_check_menu_item_new_with_label(_("Text dump"));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), dump_dlg_visible(t->dump, DUMP_TEXT));
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(dump_text_cb), t->dump);
    }
    if(t->popupwnd)
	gtk_widget_hide(t->popupwnd);
}

gint textview_button_press_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    TextView *text = TEXTVIEW(widget);
    if((event->type == GDK_BUTTON_PRESS) && (event->button == 1))
    {
        GtkTextTag *tag = textview_get_link_under_cursor(text, (gint)(event->x), (gint)(event->y));
        if(tag)
        {
            RESULT *res = (RESULT*)g_object_get_data(G_OBJECT(tag), "link");
	    if(res)
		textview_open(text, res, TRUE, NULL);
	    return TRUE;
        }
    }
    return FALSE;
}

void textview_open(TextView *self, RESULT *res, gboolean clear, const gchar *word)
{
    if(!res)
        return;
    if(clear)
	textview_clear_textbuf(GTK_TEXT_VIEW(self));

    gchar *text = ebook_get_text(res);
    render_text(res->binfo, self, text, word, self->prefs);
    g_free(text);

    if(self->history)
	history_insert_res(self->history, res);

    if(word && prefs_get_int(self->prefs, "highlight_all_keywords"))
	textview_highlight_word(self, word);
}

static void textview_set_property(GObject *object, guint param_id, const GValue *value, GParamSpec *pspec)
{
    TextView *text = TEXTVIEW(object);

    switch(param_id)
    {
	case 1:
	    text->prefs = PREFS(g_value_get_pointer(value));
	    break;
	case 2:
	    text->history = HISTORY(g_value_get_pointer(value));
	    break;
	case 3:
	    text->dump = DUMP(g_value_get_pointer(value));
	    break;
	case 4:
	    text->popupwnd = GTK_WIDGET(g_value_get_pointer(value));
	    break;
	default:
	    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, param_id, pspec);
	    break;
    }
}

static void textview_class_init(TextViewClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    gobject_class->set_property = textview_set_property;
    g_object_class_install_property(gobject_class, 1, g_param_spec_pointer("prefs", _("Prefs"), _("Prefs"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
    g_object_class_install_property(gobject_class, 2, g_param_spec_pointer("history", _("History"), _("History"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
    g_object_class_install_property(gobject_class, 3, g_param_spec_pointer("dump", _("Dump"), _("Dump"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
    g_object_class_install_property(gobject_class, 4, g_param_spec_pointer("popupwnd", _("Popupwnd"), _("Popupwnd"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}

static void textview_init(TextView *self)
{
    g_signal_connect(G_OBJECT(self), "motion_notify_event", G_CALLBACK(textview_motion_notify_cb), NULL);
    g_signal_connect(G_OBJECT(self), "button_press_event", G_CALLBACK(textview_button_press_cb), NULL);
    g_signal_connect(G_OBJECT(self), "leave_notify_event", G_CALLBACK(textview_leave_notify_cb), NULL);
    g_signal_connect(G_OBJECT(self), "populate_popup", G_CALLBACK(textview_populate_popup_cb), self);
    gtk_widget_add_events(GTK_WIDGET(self), GDK_LEAVE_NOTIFY_MASK);
    
    gtk_text_view_set_editable(GTK_TEXT_VIEW(self), FALSE);
    gtk_text_view_set_left_margin(GTK_TEXT_VIEW(self), 10);
    gtk_text_view_set_right_margin(GTK_TEXT_VIEW(self), 10);
    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(self), GTK_WRAP_WORD);
    gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(self), FALSE);
    self->history = NULL;
    self->popupwnd = NULL;
}

void textview_clear_textbuf(GtkTextView *view)
{
    GtkTextIter start, end;
    GtkTextBuffer *buf = gtk_text_view_get_buffer(view);
    gtk_text_buffer_get_bounds(buf, &start, &end);
    gtk_text_buffer_delete(buf, &start, &end);
    gtk_text_view_scroll_to_iter(view, &start, 0, 1, 0, 0);
}

void textview_insert_message(TextView *self, gchar *msg, gboolean clear)
{
    GtkTextIter iter;
    if(clear)
	textview_clear_textbuf(GTK_TEXT_VIEW(self));
    GtkTextBuffer *buf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(self));
    gtk_text_buffer_get_end_iter(buf, &iter);
    gtk_text_buffer_insert(buf, &iter, msg, -1);
}

void textview_highlight_word(TextView *self, const gchar *word)
{
    GtkTextBuffer *buf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(self));
    GtkTextIter iter, start, end;
    gtk_text_buffer_get_start_iter(buf, &iter);
    //gtk_text_buffer_get_start_iter(buf, &start);
    //gtk_text_buffer_get_end_iter(buf, &end);

    const gchar *color = prefs_get_str(self->prefs, "color.keyword");
    GtkTextTag *tag = gtk_text_buffer_create_tag(buf, NULL, "foreground", color, NULL);
    for(;;)
    {
	gboolean b = gtk_text_iter_forward_search(&iter, word, GTK_TEXT_SEARCH_VISIBLE_ONLY, &start, &end, NULL);
	if(!b)
	    break;
	if(!textview_get_link_at_iter(self, &start))
	    gtk_text_buffer_apply_tag(buf, tag, &start, &end);
	gtk_text_iter_set_offset(&iter, gtk_text_iter_get_offset(&end));
    }
}

void textview_find_next(TextView *self, const gchar *str)
{
    GtkTextBuffer *buf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(self));
    GtkTextIter iter, start, end;
    gboolean b1 = gtk_text_buffer_get_selection_bounds(buf, &start, &end);
    if(b1)
	iter = end;
    else
	gtk_text_buffer_get_start_iter(buf, &iter);
    gboolean b2 = gtk_text_iter_forward_search(&iter, str, GTK_TEXT_SEARCH_VISIBLE_ONLY, &start, &end, NULL);
    if(!b2)
    {
	gtk_text_buffer_get_start_iter(buf, &iter);
	b2 = gtk_text_iter_forward_search(&iter, str, GTK_TEXT_SEARCH_VISIBLE_ONLY, &start, &end, NULL);
    }
    if(b2)
    {
	gtk_text_buffer_select_range(buf, &start, &end);
	gtk_text_view_scroll_to_iter(GTK_TEXT_VIEW(self), &start, False, False, 0, 0);
    }
}

