
#include "defs.h"
#include "eb123.h"
#include "ebook.h"
#include "renderheadword.h"
#include "render.h"

G_DEFINE_TYPE(RenderHeadword, render_headword, GTK_TYPE_CELL_RENDERER);

static void render_headword_get_size(GtkCellRenderer *cell,
	GtkWidget *widget,
	const GdkRectangle *area,
	gint *x_offset,
	gint *y_offset,
	gint *w,
	gint *h);

static void render_headword(GtkCellRenderer      *cell,
	cairo_t              *cr,
	GtkWidget            *widget,
	const GdkRectangle   *background_area,
	const GdkRectangle   *cell_area,
	GtkCellRendererState  flags);

static gpointer render_headword_parent_class_ptr = NULL;

enum
{
    HWR_PROP_0,
    HWR_PROP_TEXT,
    HWR_PROP_RESULT,
};

static void render_headword_set_property(GObject *object, guint param_id, const GValue *value, GParamSpec *pspec)
{
    RenderHeadword *r = RENDER_HEADWORD(object);

    switch(param_id)
    {
	case HWR_PROP_TEXT:
	    if(r->text)
		g_free(r->text);
	    r->text = g_strdup(g_value_get_string(value));
	    break;
	case HWR_PROP_RESULT:
	    r->res = ((RESULT*)g_value_get_pointer(value));
	    break;
	default:
	    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, param_id, pspec);
	    break;
    }
}

static void render_headword_get_property(GObject *object, guint param_id, GValue *value, GParamSpec *pspec)
{
    RenderHeadword *r = RENDER_HEADWORD(object);

    switch (param_id)
    {
	case HWR_PROP_TEXT:
	    g_value_set_string(value, r->text);
	    break;
	case HWR_PROP_RESULT:
	    g_value_set_pointer(value, r->res);
	    break;
	default:
	    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
	    break;
    }
}

typedef struct _render_headword_ctx
{
    guint x, y;
    gboolean sub, sup;
    gboolean pretend;
    cairo_t *cr;
    GtkWidget *widget;
    GtkCellRendererState flags;
    RenderHeadword *r;
} render_headword_ctx;

static void render_headword_characters(void *ctx, const xmlChar *ch, int len)
{
    render_headword_ctx *my = (render_headword_ctx*)ctx;
    gchar *str, *text;
    PangoAttrList *attrs;
    PangoLayout *layout;
    PangoRectangle rect;
    GtkStateType state = (my->flags & GTK_CELL_RENDERER_SELECTED) ? GTK_STATE_FLAG_SELECTED : GTK_STATE_FLAG_NORMAL;
    GtkStyleContext *context = gtk_widget_get_style_context(my->widget);
    GdkRGBA color;
    gtk_style_context_get_color(context, state, &color);
    if(my->sub)
	str = g_strdup_printf("<sub><small>%s</small></sub>", (gchar*)ch);
    else if(my->sup)
	str = g_strdup_printf("<sup><small>%s</small></sup>", (gchar*)ch);
    else
	str = g_strdup((gchar*)ch);
    pango_parse_markup(str, -1, 0, &attrs, &text, NULL, NULL);
    if(text)
    {
        layout = gtk_widget_create_pango_layout(my->widget, text);
        pango_layout_set_attributes(layout, attrs);
    }
    else
        layout = gtk_widget_create_pango_layout(my->widget, str);
    g_free(str);
    pango_layout_get_pixel_extents (layout, NULL, &rect);
    if(!my->pretend)
    {
	gint w, h;
	pango_layout_get_size(layout, &w, &h);
	h /= PANGO_SCALE;
	cairo_set_source_rgb(my->cr, color.red, color.green, color.blue);
	cairo_move_to(my->cr, my->x, my->y - h/2);
	pango_cairo_show_layout(my->cr, layout);
	cairo_fill(my->cr);
    }
    my->x += rect.width;
}

static void render_headword_startElement(void *ctx, const xmlChar *name, const xmlChar **atts)
{
    render_headword_ctx *my = (render_headword_ctx*)ctx;
    if(!g_strcmp0((gchar*)name, "gaiji"))
    {
	gint w, h, x_offset, y_offset;;
	GtkStateType state = (my->flags & GTK_CELL_RENDERER_SELECTED) ? GTK_STATE_FLAG_SELECTED : GTK_STATE_FLAG_NORMAL;
	GtkStyleContext *context = gtk_widget_get_style_context(my->widget);
	GdkRGBA color;
	gtk_style_context_get_color(context, state, &color);
	gchar *spec = gdk_rgba_to_string(&color);
	GdkPixbuf *pixbuf;
	if(spec) g_free(spec);
	render_headword_get_size(NULL, my->widget, NULL, &x_offset, &y_offset, NULL, NULL);
	if((pixbuf = ebook_read_gaiji(my->r->res->binfo, (gchar*)atts[1], &w, &h)))
	{
	    if(!my->pretend)
	    {
		gdk_cairo_set_source_pixbuf(my->cr, pixbuf, my->x, my->y - h/2);
		cairo_paint(my->cr);
	    }
	    my->x += w + x_offset;
	}
	g_object_unref(pixbuf);
    }
    if(!g_strcmp0((gchar*)name, "sub"))
	my->sub = TRUE;
    if(!g_strcmp0((gchar*)name, "sup"))
	my->sup = TRUE;
}

static void render_headword_endElement(void *ctx, const xmlChar *name)
{
    render_headword_ctx *my = (render_headword_ctx*)ctx;
    if(!g_strcmp0((gchar*)name, "sub"))
	my->sub = FALSE;
    if(!g_strcmp0((gchar*)name, "sup"))
	my->sup = FALSE;
}

static void render_headword_get_size(GtkCellRenderer *cell,
	GtkWidget *widget,
	const GdkRectangle *area,
	gint *x_offset,
	gint *y_offset,
	gint *w,
	gint *h)
{
    RenderHeadword *r = (RenderHeadword*)cell;
    if(w)
	*w = 200;
    if(h)
    {
	if(!GTK_IS_WIDGET(widget))
	    return;
	*h = 2*render_get_font_size(widget);
    }
    if(x_offset)
	*x_offset = 2;
    if(y_offset)
	*y_offset = 2;
    if(r)
    {
	render_headword(cell, NULL, widget, NULL, area, 0);
	*w = r->width;
    }
}

/*
 *  Workaround: slightly modify brackets
 *  in heading text for proper parsing by libxml, e.g.
 *  日本語 <eng> -> 日本語 [eng]
 */

static gchar* headword_find_br2ptr(gchar *br)
{
    gint c = 0;
    while(br[0])
    {
	if((br[0] == '>') && !c)
	    return br;
	if(br[0] == '<') c++;
	if(br[0] == '>') c--;
	br++;
    }
    g_assert(0);
    return NULL;
}

static void headword_fix_text(gchar *txt)
{
    gint len = strlen(txt);
    gchar *br = txt;
    while((br = g_strstr_len(br, len - (br - txt), "<")))
    {
	if((br[1] == '/') || \
	    !g_ascii_strncasecmp(br + 1, "sub", 3) || \
	    !g_ascii_strncasecmp(br + 1, "sup", 3) || \
	    !g_ascii_strncasecmp(br + 1, "gaiji", 5))
	{
	    br++;
	    continue;
	}
	br[0] = '[';
	gchar *br2 = headword_find_br2ptr(br);
	if(br2) br2[0] = ']';
    }
}

static void render_headword(GtkCellRenderer      *cell,
	cairo_t              *cr,
	GtkWidget            *widget,
	const GdkRectangle   *background_area,
	const GdkRectangle   *cell_area,
	GtkCellRendererState  flags)
{
    RenderHeadword *r = RENDER_HEADWORD(cell);
    render_headword_ctx ctx;
    ctx.pretend = !(cr || background_area || flags);
    ctx.x = ctx.pretend ? 0 : cell_area->x;
    ctx.y = ctx.pretend ? 0 : (cell_area->y + cell_area->height/2);
    ctx.sub = ctx.sup = FALSE;
    ctx.cr = cr;
    ctx.widget = widget;
    ctx.flags = flags;
    ctx.r = RENDER_HEADWORD(cell);
    htmlSAXHandler cb;
    memset(&cb, 0, sizeof(htmlSAXHandler));
    cb.startElement = &render_headword_startElement;
    cb.endElement = &render_headword_endElement;
    cb.characters = &render_headword_characters;
    headword_fix_text(r->text);
    htmlDocPtr doc = htmlSAXParseDoc((xmlChar*)r->text, ENC_UTF8, &cb, &ctx);
    r->width = ctx.x;
    xmlFreeDoc(doc);
}

static void render_headword_finalize(GObject *object)
{
    RenderHeadword *r = RENDER_HEADWORD(object);
    if(r->text)
	g_free(r->text);
    (*G_OBJECT_CLASS(render_headword_parent_class_ptr)->finalize)(object);
}

static void render_headword_class_init(RenderHeadwordClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    GtkCellRendererClass *eklass = GTK_CELL_RENDERER_CLASS(klass);
    render_headword_parent_class_ptr = g_type_class_peek_parent(klass);
    gobject_class->get_property = render_headword_get_property;
    gobject_class->set_property = render_headword_set_property;
    eklass->get_size = render_headword_get_size;
    eklass->render = render_headword;
    gobject_class->finalize = render_headword_finalize;
    g_object_class_install_property(gobject_class, HWR_PROP_TEXT, g_param_spec_string("text", _("Text"), _("Text to render"), NULL, G_PARAM_READWRITE));
    g_object_class_install_property(gobject_class, HWR_PROP_RESULT, g_param_spec_pointer("result", _("Result"), _("Search result"), G_PARAM_READWRITE));
}

static void render_headword_init(RenderHeadword *self)
{
    self->text = NULL;
    self->res = NULL;
    self->width = 100;
    gtk_cell_renderer_set_alignment(GTK_CELL_RENDERER(self), 0.0, 0.5);
    gtk_cell_renderer_set_padding(GTK_CELL_RENDERER(self), 2, 2);
}

