
#include "defs.h"

#include "builder.h"
#include "dicts.h"
#include "history.h"
#include "mainwnd.h"
#include "popupwnd.h"
#include "prefs.h"
#include "hotkeys.h"
#include "textview.h"

G_DEFINE_TYPE(Prefs, prefs, G_TYPE_OBJECT);

typedef struct
{
    int type;
    char *name;
    union
    {
	int i;
	char *s;
    } val;
} pref1;

static Prefs *_prefs = NULL;

static const pref1 prefs_list[] =
{
    {G_TYPE_INT, "headword.maxhits", {50}},
    {G_TYPE_INT, "dictbar.word_hist", {30}},
    {G_TYPE_INT, "selection.maxchar", {32}},
    {G_TYPE_INT, "selection.lookup_started", {0}},
    {G_TYPE_INT, "hotkeys.ignore_locks", {1}},
    {G_TYPE_INT, "popupwnd.w", {350}},
    {G_TYPE_INT, "popupwnd.h", {450}},
    {G_TYPE_INT, "popupwnd.x", {300}},
    {G_TYPE_INT, "popupwnd.y", {300}},
    {G_TYPE_INT, "popupwnd.remember_pos", {0}},
    {G_TYPE_INT, "popupwnd.lock", {0}},
    {G_TYPE_INT, "popupwnd.jap_only", {0}},
    {G_TYPE_INT, "popupwnd.timeout", {2000}},
    {G_TYPE_INT, "popupwnd.maxhits", {5}},
    {G_TYPE_INT, "popupwnd.search_method", {0}},
    {G_TYPE_INT, "mainwnd.x", {300}},
    {G_TYPE_INT, "mainwnd.y", {300}},
    {G_TYPE_INT, "mainwnd.w", {500}},
    {G_TYPE_INT, "mainwnd.h", {400}},
    {G_TYPE_INT, "mainwnd.remember_pos", {1}},
    {G_TYPE_INT, "mainwnd.search", {0}},
    {G_TYPE_INT, "mainwnd.custom_font", {0}},
    {G_TYPE_INT, "paned.tree_width", {150}},
    {G_TYPE_INT, "highlight_all_keywords", {1}},
    {G_TYPE_INT, "prepend_title", {1}},
    {G_TYPE_INT, "prepend_len", {1}},
    {G_TYPE_INT, "headword.use_tree", {0}},
    {G_TYPE_STRING, "mainwnd.font", {.s = ""}},
    {G_TYPE_STRING, "color.link", {.s = "#0000c0"}},
    {G_TYPE_STRING, "color.keyword", {.s = "#c00000"}},
    {G_TYPE_STRING, "color.title", {.s = "#306090"}}
};

static void prefs_set_property(GObject *object, guint param_id, const GValue *value, GParamSpec *pspec)
{
    Prefs *prefs = PREFS(object);

    switch(param_id)
    {
	case 1:
	    prefs->builder = BUILDER(g_value_get_pointer(value));
	    break;
	default:
	    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, param_id, pspec);
	    break;
    }
}


static void prefs_class_init(PrefsClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    gobject_class->set_property = prefs_set_property;
    g_object_class_install_property(gobject_class, 1, g_param_spec_pointer("builder", _("Builder"), _("Builder"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}

static void prefs_init(Prefs *self)
{
    if(!_prefs)
	_prefs = self;
    else
	g_assert(1);

    //GstElementClass *gstelement_class = GST_ELEMENT_CLASS(klass);
    //gst_element_class_add_pad_template(gstelement_class, gst_static_pad_template_get(&prefs_template));

    g_datalist_init(&(self->data));

    gchar *home_dir = getenv("HOME");
    gchar *userdir = g_strdup_printf("%s%s.%s", home_dir, G_DIR_SEPARATOR_S, PACKAGE_NAME);
    prefs_set_str(self, "userdir", userdir);

    GDir *dir;
    if((dir = g_dir_open(userdir, 0, NULL)) == NULL)
    {
        if(g_mkdir_with_parents(userdir, 493))
        {
            LOG(LOG_CRITICAL, "Failed to create directory: %s\n", userdir);
            exit(1);
        }
    }
    else
        g_dir_close(dir);
}

enum
{
    PREF_TITLE,
    PREF_NUMBER,
    PREF_N
};

static void prefs_selection_changed_cb(GtkTreeSelection *selection, gpointer data)
{
    GtkTreeIter iter;
    GtkTreeModel *model;
    if(!gtk_tree_selection_get_selected(selection, &model, &iter)) return;

    GtkNotebook *notebook = GTK_NOTEBOOK(data);
    gint n;
    gtk_tree_model_get(model, &iter, 1, &n, -1);
    if(n >= 0)
        gtk_notebook_set_current_page(notebook, n);
}

static const pref1* prefs_get1(Prefs *p, const gchar *str)
{
    gint i;
    for(i = 0; i < SZ(prefs_list); i++)
    {
	const pref1 *p1 = &(prefs_list[i]);
	if(!g_strcmp0(p1->name, str))
	    return p1;
    }
    return 0;
}

static void prefs_set_widget_data(Prefs *p, Builder *b, const gchar *str)
{
    const pref1 *p1 = prefs_get1(p, str);
    if(p1->type == G_TYPE_INT)
    {
	gint value = prefs_get_int(p, str);
	builder_set_int(b, str, value);
    }
    if(p1->type == G_TYPE_STRING)
    {
	const gchar *value = prefs_get_str(p, str);
	builder_set_str(b, str, value);
    }
}

static void prefs_get_widget_data(Prefs *p, Builder *b, const gchar *str)
{
    const pref1 *p1 = prefs_get1(p, str);
    if(p1->type == G_TYPE_INT)
    {
	gint value = builder_get_int(b, str);
	prefs_set_int(p, str, value);
    }
    if(p1->type == G_TYPE_STRING)
    {
	const gchar *value = builder_get_str(b, str);
	prefs_set_str(p, str, value);
    }
}

void prefs_close_wnd(Prefs *prefs)
{
    GtkWidget *w = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(prefs->builder), "preferences"));
    gtk_widget_hide(w);
}

gboolean preferences_delete_event_cb(GtkWidget *widget, GdkEvent *event, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    gtk_widget_hide(widget);
    popupwnd_lookup_resume(mw->popupwnd);
    return TRUE;
}

void prefs_ok_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    GtkWidget *wnd = gtk_widget_get_toplevel(widget);
    gtk_widget_hide(wnd);
    gint i;
    for(i = 0; i < SZ(prefs_list); i++)
    {
	const pref1 *p1 = &(prefs_list[i]);
	prefs_get_widget_data(mw->prefs, mw->builder, p1->name);
    }
    prefs_save(mw->prefs);
    mainwnd_reset_font(mw);
    mainwnd_reset_headwords(mw);
    GtkBox *vbox = GTK_BOX(gtk_builder_get_object(GTK_BUILDER(mw->builder), "hotkeys_vbox"));
    hotkeys_reload(mw->hotkeys, vbox);
    hotkeys_local_install(mw->hotkeys);
    hotkeys_global_install(mw->hotkeys);
    hotkeys_save(mw->hotkeys);
    popupwnd_lookup_resume(mw->popupwnd);
}

void prefs_cancel_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    prefs_close_wnd(mw->prefs);
    popupwnd_lookup_resume(mw->popupwnd);
}

void prefs_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    Prefs *prefs = mw->prefs;

    GtkWidget *wnd = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(mw->builder), "preferences"));
    gtk_widget_hide_on_delete(wnd);

    GtkTreeView *prefs_tree = GTK_TREE_VIEW(gtk_builder_get_object(GTK_BUILDER(mw->builder), "prefs_tree"));
    GtkTreeStore *prefs_store = GTK_TREE_STORE(gtk_builder_get_object(GTK_BUILDER(mw->builder), "prefs_store"));
    GtkTreeSelection *select = gtk_tree_view_get_selection(prefs_tree);
    GtkTreeIter iter1, iter2;
    static gboolean preparewnd = True;
    if(preparewnd)
    {
	GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_insert_column_with_attributes(prefs_tree, -1, _("Items"), renderer, "text", 0, NULL);

	GtkNotebook *notebook = GTK_NOTEBOOK(gtk_builder_get_object(GTK_BUILDER(mw->builder), "notebook1"));
	gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
	g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(prefs_selection_changed_cb), notebook);

	gint n = gtk_notebook_get_n_pages(notebook);
	gint i;
	for(i = 0; i < n; i++)
	{
	    GtkWidget *p = gtk_notebook_get_nth_page(notebook, i);
	    GtkWidget *l = gtk_notebook_get_tab_label(notebook, p);
	    const gchar *text = gtk_label_get_text(GTK_LABEL(l));
	    if(text[0] != '_')
	    {
		gtk_tree_store_append(prefs_store, &iter1, NULL);
		gtk_tree_store_set(prefs_store, &iter1, 0, _(text), 1, i, -1);
	    }
	    else
	    {
		gtk_tree_store_append(prefs_store, &iter2, &iter1);
		gtk_tree_store_set(prefs_store, &iter2, 0, _(&(text[1])), 1, i, -1);
	    }
	}
	builder_install_text_cellrenderer(mw->builder, "popupwnd.search_method");
	for(i = 0; i < SZ(prefs_list); i++)
	{
	    const pref1 *p1 = &(prefs_list[i]);
	    prefs_set_widget_data(prefs, mw->builder, p1->name);
	}
	GtkWidget *w = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(mw->builder), "mainwnd.font"));
	gtk_widget_set_sensitive(w, prefs_get_int(mw->prefs, "mainwnd.custom_font"));

	Hotkeys *hotkeys = mw->hotkeys;
	GtkBox *vbox = GTK_BOX(gtk_builder_get_object(GTK_BUILDER(mw->builder), "hotkeys_vbox"));
	for(i = 0; (w = hotkeys_edit_nth(hotkeys, i)); i++)
	{
	    gtk_box_pack_start(vbox, w, FALSE, FALSE, 0);
	}
	preparewnd = False;
    }
    if(gtk_tree_model_get_iter_first(GTK_TREE_MODEL(prefs_store), &iter1))
    {
	gtk_tree_selection_select_iter(select, &iter1);
    }
    gtk_tree_view_expand_all(prefs_tree);
    popupwnd_lookup_suspend(mw->popupwnd);
    gtk_window_set_transient_for(GTK_WINDOW(wnd), GTK_WINDOW(mw));
    gtk_widget_show_all(wnd);
}

void prefs_save(Prefs *self)
{
    gint i, j;
    LOG(LOG_INFO, "prefs_save");
    gchar filename[PATH_MAX], buff[128];
    const gchar *str;
    xmlDocPtr doc = xmlNewDoc((xmlChar*)"1.0");
    doc->children = xmlNewDocRawNode(doc, NULL, (xmlChar*)"Prefs", NULL);

    const gchar *userdir = prefs_get_str(self, "userdir");
    sprintf(filename, "%s%s%s", userdir, G_DIR_SEPARATOR_S, FILENAME_PREFS);
    
    for(i = 0 ; i < SZ(prefs_list); i++)
    {
	const pref1 *p = &(prefs_list[i]);
        switch(prefs_list[i].type)
        {
        case G_TYPE_INT:
        case G_TYPE_BOOLEAN:
	    j = prefs_get_int(self, p->name);
            sprintf(buff, "%d", j);
            break;
        case G_TYPE_FLOAT:
            break;
        case G_TYPE_STRING:
	    str = prefs_get_str(self, p->name);
	    if(str)
		sprintf(buff, "%s", str);
            break;
        }
        xmlNodePtr node = xmlAddChild(doc->children, xmlNewNode(NULL, (xmlChar*)prefs_list[i].name));
        xmlNewProp(node, (xmlChar*)"value", (xmlChar*)buff);
    }
    xmlSaveFormatFileEnc(filename, doc, "utf8", 0);
    xmlFreeDoc(doc);
}

void prefs_load_elem(void *ctx, const xmlChar *name, const xmlChar **atts)
{
    gint i;
    xmlParserCtxt *ctxt = (xmlParserCtxt*)ctx;
    xmlSAXHandler *cb = ctxt->sax;
    Prefs *prefs = PREFS(cb->_private);
    for(i = 0; i < SZ(prefs_list); i++)
    {
        if(!g_strcmp0((gchar*)name, prefs_list[i].name))
        {
            switch(prefs_list[i].type){
            case G_TYPE_INT:
            case G_TYPE_BOOLEAN:
		prefs_set_int(prefs, (const gchar*)name, atoi((gchar*)atts[1]));
                break;
            case G_TYPE_STRING:
		prefs_set_str(prefs, (const gchar*)name, strdup((gchar*)atts[1]));
                break;
            case G_TYPE_FLOAT:
                break;
            }
        }
    }
}

void prefs_load(Prefs *self)
{
    gchar filename[PATH_MAX];
    xmlSAXHandler cb;
    const gchar *userdir = prefs_get_str(self, "userdir");
    sprintf(filename, "%s%s%s", userdir, G_DIR_SEPARATOR_S, FILENAME_PREFS);
    if(!g_file_test(filename, G_FILE_TEST_IS_REGULAR))
        return;
    memset(&cb, 0, sizeof(xmlSAXHandler));
    cb.startElement = &prefs_load_elem;
    cb._private = self;
    xmlDocPtr doc = xmlSAXParseFile(&cb, filename, 0);
    xmlFreeDoc(doc);
}

gint prefs_get_int(Prefs *self, const gchar *name)
{
    glong l = (glong)g_datalist_get_data(&self->data, name);
    return (gint)l;
}

void prefs_set_int(Prefs *self, const gchar *name, gint value)
{
    glong l = value;
    g_datalist_set_data(&self->data, name, (gpointer)l);
}

const gchar* prefs_get_str(Prefs *self, const gchar *name)
{
    gpointer ptr = g_datalist_get_data(&self->data, name);
    return (const gchar*)ptr;
}

void prefs_set_str(Prefs *self, const gchar *name, const gchar *value)
{
    g_datalist_set_data(&self->data, name, (gpointer)value);
}

const gchar* prefs_get_userdir()
{
    return prefs_get_str(_prefs, "userdir");
}

void prefs_custom_font_toggled_cb(GtkToggleButton *btn, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    GtkWidget *w = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(mw->builder), "mainwnd.font"));
    gtk_widget_set_sensitive(w, gtk_toggle_button_get_active(btn));
}

