
#include "defs.h"

#include "dicts.h"
#include "ebook.h"
#include "jcode.h"
#include "history.h"
#include "mainwnd.h"
#include "popupwnd.h"
#include "textview.h"

#define SELECTION_TIMEOUT 300
#define POPUPWND_TIMEOUT 1000
#define DISTANCE_OFFSET	20

Popupwnd *_popupwnd = NULL;

G_DEFINE_TYPE(Popupwnd, popupwnd, GTK_TYPE_WINDOW);

static void popupwnd_set_property(GObject *object, guint param_id, const GValue *value, GParamSpec *pspec)
{
    Popupwnd *pw = POPUPWND(object);

    switch(param_id)
    {
	case 1:
	    pw->mainwnd = g_value_get_pointer(value);
	    Mainwnd *mw = MAINWND(pw->mainwnd);
	    pw->prefs = mw->prefs;
	    pw->builder = mw->builder;
	    GtkWidget *vbox = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(pw->builder), "vbox9"));
	    gtk_widget_reparent(vbox, GTK_WIDGET(pw));
	    GtkContainer *scroll = GTK_CONTAINER(gtk_builder_get_object(GTK_BUILDER(pw->builder), "popupwnd_scroll"));
	    pw->text = g_object_new(TYPE_TEXTVIEW, "prefs", pw->prefs, NULL);
	    gtk_container_add(scroll, GTK_WIDGET(pw->text));
	    break;
	default:
	    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, param_id, pspec);
	    break;
    }
}

static gboolean popuwnd_hide_timeout_cb(gpointer data)
{
    Popupwnd *pw = POPUPWND(data);
    GtkWidget *wt = GTK_WIDGET(pw);

    if(popupwnd_locked(pw))
	return True;
    if(!gtk_widget_get_visible(wt))
	return False;

    if(pw->menu)
    {
	if(gtk_widget_get_visible(pw->menu))
	    return True;
    }

    int x, y, w, h, cx, cy;
    gtk_window_get_position(GTK_WINDOW(wt), &x, &y);
    gtk_window_get_size(GTK_WINDOW(wt), &w, &h);
    cx = x + w/2;
    cy = y + h/2;

    GdkScreen *screen = gtk_window_get_screen(GTK_WINDOW(pw));
    GdkDisplay *display = gdk_screen_get_display(screen);
    GdkDeviceManager *devman = gdk_display_get_device_manager(display);
    GdkDevice *dev = gdk_device_manager_get_client_pointer(devman);
    gdk_device_get_position(dev, NULL, &x, &y);
    //gdk_display_get_pointer(display, NULL, &x, &y, NULL);

    if((abs(x - cx) < (w/2 + DISTANCE_OFFSET)) && (abs(y - cy) < (h/2 + DISTANCE_OFFSET)))
	return True;

    gtk_widget_hide(wt);
    return False;
}

static void popupwnd_timer(Popupwnd *self)
{
    static gint timeout_id = 0;
    if(timeout_id != 0)
	g_source_remove(timeout_id);
    timeout_id = 0;
    timeout_id = g_timeout_add(POPUPWND_TIMEOUT, popuwnd_hide_timeout_cb, self);
}

static void popupwnd_class_init(PopupwndClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    gobject_class->set_property = popupwnd_set_property;
    g_object_class_install_property(gobject_class, 1, g_param_spec_pointer("mainwnd", _("Mainwnd"), _("Mainwnd"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}

static void popupwnd_init(Popupwnd *self)
{
    if(!_popupwnd)
        _popupwnd = self;
    else
        g_assert(1);

    self->timeout_id = 0;
    self->lookup_started = False;
    self->lookup_suspended = True;
    self->drag = False;
    self->menu = NULL;
    self->binfo = NULL;
    self->results = g_sequence_new(result_free);
    self->x = -1;
    self->y = -1;
    gtk_window_set_position(GTK_WINDOW(self), GTK_WIN_POS_CENTER);
}

const gchar* popupwnd_selection_copy(Popupwnd *self, gboolean force_copy)
{
    GtkClipboard* clipboard = gtk_clipboard_get(GDK_SELECTION_PRIMARY);
    gchar *str = gtk_clipboard_wait_for_text(clipboard);
    static gchar *sel = NULL, *sel_prev = NULL;
    gint maxchar = prefs_get_int(self->prefs, "selection.maxchar");
    if(!str || (self->lookup_suspended && !force_copy))
    {
        if(sel_prev)
            g_free(sel_prev);
        sel_prev = NULL;
        return NULL;
    }
    g_strstrip(str);
    if(strlen(str) > maxchar)
    {
        if(sel_prev)
            g_free(sel_prev);
        sel_prev = NULL;
        g_free(str);
        return NULL;
    }
    if(force_copy)
        sel = str;
    else
    {
        if(!sel)
        {
            sel = str;
            return NULL;
        }
        if(strncmp(str, sel, maxchar))
        {
            g_free(sel);
            sel = str;
            return NULL;
        }
        g_free(str);
        if((sel_prev) ? !strncmp(sel_prev, sel, maxchar) : FALSE)
            return NULL;
    }
    sel_prev = sel;
    sel = NULL;
    return sel_prev;
}

static void popupwnd_dict_changed_cb(GtkWidget *w, gpointer data)
{
    if((gpointer)_popupwnd->binfo == data)
	return;
    _popupwnd->binfo = (BOOK_INFO*)data;
    popupwnd_search(_popupwnd, NULL);
}

static gboolean popupwnd_dict_menu_append(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
    Popupwnd *pw = POPUPWND(data);
    static GSList *group = NULL;
    static GtkWidget *submenu = NULL;
    GtkWidget *item;
    if(!model)
    {
        if(pw->menu)
            gtk_widget_destroy(pw->menu);
        group = NULL;
	pw->menu = gtk_menu_new();
        item = gtk_radio_menu_item_new_with_label(group, _("Main window selection"));
        g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(popupwnd_dict_changed_cb), NULL);
        group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(item));
        gtk_menu_shell_append(GTK_MENU_SHELL(pw->menu), item);
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE); 
        submenu = NULL;
        return FALSE;
    }
    if(gtk_tree_path_get_depth(path) == 1)
    {
        gchar *name;
        submenu = gtk_menu_new();
        gtk_tree_model_get(model, iter, DICT_ALIAS, &name, -1);
        gtk_menu_shell_append(GTK_MENU_SHELL(pw->menu), item = gtk_menu_item_new_with_label(name));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);
        g_free(name);
    }
    else
    {
        gchar *name;
        BOOK_INFO *binfo;
        gtk_tree_model_get(model, iter, DICT_ALIAS, &name, DICT_BINFO, &binfo, -1);
        item = gtk_radio_menu_item_new_with_label(group, binfo->title);
        group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(item));
        gtk_menu_shell_append(GTK_MENU_SHELL(submenu), item);
	if(binfo == pw->binfo)
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE); 
        g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(popupwnd_dict_changed_cb), binfo);
        g_free(name);
    }
    return FALSE;
}

static void popupwnd_dict_menu_create(Popupwnd *self, gboolean newmenu)
{
    if(newmenu || !self->menu)
    {
        popupwnd_dict_menu_append(NULL, NULL, NULL, self);
	GtkTreeModel *store = GTK_TREE_MODEL(gtk_builder_get_object(GTK_BUILDER(self->builder), "dicts_store"));
        gtk_tree_model_foreach(GTK_TREE_MODEL(store), popupwnd_dict_menu_append, self);
    }
}

void popupwnd_dict_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    Popupwnd *pw = mw->popupwnd;
    popupwnd_dict_menu_create(pw, False);
    gtk_widget_show_all(pw->menu);
    gtk_menu_popup(GTK_MENU(pw->menu), NULL, NULL, NULL, NULL, 0, gtk_get_current_event_time());
}

void popupwnd_render(gpointer data, gpointer user_data)
{
    Popupwnd *pw = POPUPWND(user_data);
    const gchar *word = builder_get_str(pw->builder, "popupwnd_label");
    RESULT *res = (RESULT*)data;
    textview_open(pw->text, res, FALSE, word);
}

static void popupwnd_calculate_coordinates(Popupwnd *self, gint *x, gint *y, gint *w, gint *h)
{
    GdkModifierType mask;
    gint pointer_x, pointer_y, root_x, root_y;

    *w = prefs_get_int(self->prefs, "popupwnd.w");
    *h = prefs_get_int(self->prefs, "popupwnd.h");
    *w = (*w > 100) ? *w : 100;
    *h = (*h > 100) ? *h : 100;
    gboolean remember_pos = prefs_get_int(self->prefs, "popupwnd.remember_pos");
    if(remember_pos && (self->x >= 0) && (self->y >= 0))
    {
	*x = self->x;
	*y = self->y;
	builder_set_int(self->builder, "popupwnd_lock_btn", True);
    }
    else
    {
	GdkWindow *root_win = gdk_x11_window_foreign_new_for_display(gdk_display_get_default(), GDK_ROOT_WINDOW());
	root_x = gdk_window_get_width(root_win);
	root_y = gdk_window_get_height(root_win);
	gdk_window_get_pointer(root_win, &pointer_x, &pointer_y, &mask);
	*x = pointer_x + 10;
	*y = pointer_y + 10;

	if(*x + *w > root_x)
	    *x = root_x - *w;

	if(*y + *h > root_y)
	    *y = root_y - *h;
    }
}

void popupwnd_search(Popupwnd *self, const gchar *str)
{
    GtkLabel *label = GTK_LABEL(gtk_builder_get_object(GTK_BUILDER(self->builder), "popupwnd_label"));
    popupwnd_dict_menu_create(self, False);
    textview_clear_textbuf(GTK_TEXT_VIEW(self->text));
    if(str)
	gtk_label_set_text(label, str);
    if(!str)
    {
	str = gtk_label_get_text(label);
	if(!str)
	    textview_insert_message(self->text, "<No text selected>", TRUE);
    }
    if(!gtk_widget_get_visible(GTK_WIDGET(self)))
    {
	gint x, y, w, h;
	popupwnd_calculate_coordinates(self, &x, &y, &w, &h);
	gtk_window_move(GTK_WINDOW(self), x, y);
	gtk_widget_set_size_request(GTK_WIDGET(self), w, h);
    }

    popupwnd_timer(self);
    gtk_widget_show_all(GTK_WIDGET(self));
    while(gtk_events_pending())
	gtk_main_iteration();

    if(!str)
	return;

    gint search_method = prefs_get_int(self->prefs, "popupwnd.search_method");
    gint maxhits = prefs_get_int(self->prefs, "popupwnd.maxhits");
    if(self->binfo)
        ebook_search_book(str, search_method, self->results, maxhits, self->binfo);
    else
        ebook_search(str, search_method, self->results, maxhits, False);

    if(g_sequence_get_length(self->results))
	g_sequence_foreach(self->results, popupwnd_render, self);
    else
	textview_insert_message(self->text, "<No hits>", TRUE);
}

gboolean popupwnd_search_cb(gpointer data)
{
    Popupwnd *self = POPUPWND(data);
    const gchar *sel = popupwnd_selection_copy(self, False);
    if(sel)
    {
	gint jap_only = prefs_get_int(self->prefs, "popupwnd.jap_only");
	if(jap_only)
	    if(!validate_utf8(sel, True))
		return True;
        popupwnd_search(self, sel);
    }
    return True;
}

void popupwnd_lookup_resume(Popupwnd *self)
{
    if(self->lookup_started)
    {
	if(self->timeout_id != 0)
	    g_source_remove(self->timeout_id);
	self->timeout_id = g_timeout_add(SELECTION_TIMEOUT, popupwnd_search_cb, self);
    }
    self->lookup_suspended = FALSE;
}

void popupwnd_lookup_start(Popupwnd *self)
{
    self->lookup_started = TRUE;
    popupwnd_lookup_resume(self);
}

void popupwnd_lookup_suspend(Popupwnd *self)
{
    if(self->timeout_id != 0)
	g_source_remove(self->timeout_id);
    self->timeout_id = 0;
    self->lookup_suspended = TRUE;
    gtk_widget_hide(GTK_WIDGET(self));
}

void popupwnd_lookup_stop(Popupwnd *self)
{
    self->lookup_started = False;
    popupwnd_lookup_suspend(self);
}

void popupwnd_find_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    const gchar *str = builder_get_str(mw->builder, "popupwnd_label");
    gtk_window_deiconify(GTK_WINDOW(mw));
    mainwnd_search(mw, str, -1);
}

void popupwnd_close_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    Popupwnd *self = mw->popupwnd;
    gtk_widget_hide(GTK_WIDGET(self));
}

gint popupwnd_evtbox_motion_notify_event_cb(GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    Popupwnd *self = mw->popupwnd;
    if(!self->drag)
	return TRUE;
    gint x = self->dx + event->x_root;
    gint y = self->dy + event->y_root;
    if(x < 0) x = 0;
    if(y < 0) y = 0;
    gtk_window_move(GTK_WINDOW(self), x, y);
    self->x = x;
    self->y = y;
    return TRUE;
}

gint popupwnd_evtbox_button_press_event_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    Popupwnd *self = mw->popupwnd;
    if(event->type==GDK_2BUTTON_PRESS)
    {
	self->drag = FALSE;
	popupwnd_lookup_stop(self);
    }
    else
    {
	self->drag = TRUE;
	gint ex, ey, wx, wy;
	ex = event->x_root;
	ey = event->y_root;
	gtk_window_get_position(GTK_WINDOW(self), &wx, &wy);
	self->dx = wx - ex;
	self->dy = wy - ey;
    }
    return FALSE;
}

gint popupwnd_evtbox_button_release_event_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    Popupwnd *self = mw->popupwnd;
    self->drag = FALSE;
    return FALSE;
}

void popupwnd_search_selection(gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    const gchar *word = popupwnd_selection_copy(mw->popupwnd, True);
    popupwnd_search(mw->popupwnd, word);
}

gboolean popupwnd_locked(Popupwnd *self)
{
    if(!gtk_widget_get_visible(GTK_WIDGET(self)))
	return False;
    GtkToggleButton *btn = GTK_TOGGLE_BUTTON(gtk_builder_get_object(GTK_BUILDER(self->builder), "popupwnd_lock_btn"));
    return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(btn));
}

