
#include "defs.h"

#include "dicts.h"
#include "eb123.h"
#include "ebook.h"
#include "icon.xpm"
#include "history.h"
#include "jcode.h"
#include "mainwnd.h"
#include "popupwnd.h"
#include "hotkeys.h"
#include "renderheadword.h"

static Mainwnd *_mainwnd = NULL;

G_DEFINE_TYPE(Mainwnd, mainwnd, GTK_TYPE_WINDOW);

static void mainwnd_class_init(MainwndClass *klass)
{
}

static void mainwnd_init(Mainwnd *self)
{
    if(!_mainwnd)
        _mainwnd = self;
    else
        g_assert(1);

    self->builder = g_object_new(TYPE_BUILDER, NULL);
    self->prefs = g_object_new(TYPE_PREFS, "builder", self->builder, NULL);
    self->dicts = g_object_new(TYPE_DICTS, "builder", self->builder, NULL);
    self->history = g_object_new(TYPE_HISTORY, "builder", self->builder, "prefs", self->prefs, NULL);
    self->hotkeys = g_object_new(TYPE_HOTKEYS, "mainwnd", self, "prefs", self->prefs, NULL);
    self->dump = g_object_new(TYPE_DUMP, "builder", self->builder, "history", self->history, NULL);
    self->results = g_sequence_new(result_free);
    gtk_builder_connect_signals(GTK_BUILDER(self->builder), self);
    GtkWidget *vbox = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(self->builder), "vbox1"));
    gtk_widget_reparent(vbox, GTK_WIDGET(self));
    GtkContainer *scroll = GTK_CONTAINER(gtk_builder_get_object(GTK_BUILDER(self->builder), "scrolledwindow2"));
    self->popupwnd = g_object_new(TYPE_POPUPWND, "mainwnd", self, "type", GTK_WINDOW_POPUP, NULL);
    self->text = g_object_new(TYPE_TEXTVIEW, "prefs", self->prefs, "history", self->history, "dump", self->dump, "popupwnd", self->popupwnd, NULL);
    gtk_container_add(scroll, GTK_WIDGET(self->text));
    self->word = NULL;
}

void mainwnd_headword_append_cb(gpointer data, gpointer user_data)
{
    Mainwnd *mw = MAINWND(user_data);
    GtkTreeView *tree = GTK_TREE_VIEW(gtk_builder_get_object(GTK_BUILDER(mw->builder), "mainwnd_results"));
    GtkTreeStore *store = GTK_TREE_STORE(gtk_tree_view_get_model(tree));
    gint n = prefs_get_int(mw->prefs, "prepend_len");
    RESULT *res = (RESULT*)(data);
    static GtkTreeIter iter1;
    static RESULT *res1 = NULL;
    static gint	rootindent = 0;
    if(res->book_title)
	g_free(res->book_title);
    res->book_title = g_strdup(res->binfo->title);
    if(g_utf8_strlen(res->book_title, 10*n) > n)
    {
	gchar *offset = g_utf8_offset_to_pointer(res->book_title, n);
	gchar *offset2 = g_utf8_offset_to_pointer(res->book_title, n + 1);
	for(; offset != offset2;)
	{
	    *offset++ = 0;
	}
    }
    if(prefs_get_int(mw->prefs, "headword.use_tree"))
    {
	n = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(store), NULL);
	GtkTreeViewColumn *col = gtk_tree_view_get_column(tree, 0);
	GtkTreePath *path = NULL;
	GdkRectangle r;
	if(!n)
	{
	    gtk_tree_store_append(store, &iter1, NULL);
	    gtk_tree_store_set(store, &iter1, 0, res->heading, 1, res, 2, res->book_title, -1);
	    res1 = res;
	    path = gtk_tree_model_get_path(GTK_TREE_MODEL(store), &iter1);
	    gtk_tree_view_get_cell_area(tree, path, col, &r);
	    rootindent = r.x;
	    mw->indent = 0;
	}
	else if(!g_strcmp0(res1->binfo->title, res->binfo->title))
	{
	    GtkTreeIter iter;
	    gtk_tree_store_append(store, &iter, &iter1);
	    gtk_tree_store_set(store, &iter, 0, res->heading, 1, res, 2, res->book_title, -1);
	    path = gtk_tree_model_get_path(GTK_TREE_MODEL(store), &iter);
	    gtk_tree_view_get_cell_area(tree, path, col, &r);
	    if(!mw->indent)
		mw->indent = abs(rootindent - r.x);
	}
	else
	{
	    gtk_tree_store_append(store, &iter1, NULL);
	    gtk_tree_store_set(store, &iter1, 0, res->heading, 1, res, 2, res->book_title, -1);
	    res1 = res;
	}
	gtk_tree_path_free(path);
    }
    else
    {
	gtk_tree_store_append(store, &iter1, NULL);
	gtk_tree_store_set(store, &iter1, 0, res->heading, 1, res, 2, res->book_title, -1);
    }
}

gboolean mainwnd_headword_counters_cb(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
    if(gtk_tree_path_get_depth(path) > 1)
	return False;

    Mainwnd *mw = MAINWND(data);
    RESULT *res;
    gtk_tree_model_get(model, iter, 1, &res, -1);
    if(prefs_get_int(mw->prefs, "headword.use_tree"))
    {
	gint n = gtk_tree_model_iter_n_children(model, iter);
	gchar buf[16];
	g_sprintf(buf, "(%d)", n + 1);
	gchar *ptr = g_strstr_len(res->heading, -1, buf);
	if(ptr)
	    return True;
	gchar *str = g_malloc0(strlen(res->heading) + 16);
	g_sprintf(str, "%s (%d)", res->heading, n + 1);
	g_free(res->heading);
	res->heading = str;
	gtk_tree_store_set(GTK_TREE_STORE(model), iter, 0, res->heading, -1);
    }
    else
    {
	gchar *ptr1 = g_strstr_len(res->heading, -1, "(");
	gchar *ptr2 = g_strstr_len(res->heading, -1, ")");
	if(ptr1 && ptr2)
	{
	    if(ptr1 < ptr2)
		for(; ptr1 != ptr2 ;)
		    *ptr1++ = 0;
	    gtk_tree_store_set(GTK_TREE_STORE(model), iter, 0, res->heading, -1);
	}
    }
    return False;
}

void mainwnd_update_results(Mainwnd *self)
{
    GtkTreeView *tree = GTK_TREE_VIEW(gtk_builder_get_object(GTK_BUILDER(self->builder), "mainwnd_results"));
    GtkTreeModel *model = gtk_tree_view_get_model(tree);
    gtk_tree_store_clear(GTK_TREE_STORE(model));
    if(!g_sequence_get_length(self->results))
    {
	textview_insert_message(self->text, _("<No hits>"), True);
	return;
    }
    g_sequence_foreach(self->results, mainwnd_headword_append_cb, self);
    gtk_tree_view_columns_autosize(tree);
    gtk_tree_view_collapse_all(tree);
    gtk_tree_view_set_level_indentation(tree, -self->indent);
    gtk_tree_model_foreach(model, mainwnd_headword_counters_cb, self);

    GtkTreeIter iter;
    GtkTreeSelection *select = gtk_tree_view_get_selection(tree);
    if(gtk_tree_model_get_iter_first(model, &iter))
    {
        gtk_tree_view_scroll_to_point(tree, 0, 0);
        gtk_tree_selection_select_iter(select, &iter);
	builder_grab_focus(self->builder, "mainwnd_results");
    }
}

static void mainwnd_results_selection_changed_cb(GtkTreeSelection *selection, gpointer data)
{
    Mainwnd *self = MAINWND(data);
    if(!selection)
	return;

    GtkTreeIter iter;
    GtkTreeModel *store;
    if(!gtk_tree_selection_get_selected(selection, &store, &iter))
        return;

    RESULT *res;
    gtk_tree_model_get(store, &iter, 1, &res, -1);
    mainwnd_open(self, res);
    gtk_widget_show_all(GTK_WIDGET(self));
    gtk_window_present(GTK_WINDOW(self));
}

void mainwnd_exit()
{
    Mainwnd *mw = _mainwnd;
    Prefs *prefs = mw->prefs;

    GtkWindow *wnd = GTK_WINDOW(mw);
    int x, y, w, h;
    gtk_window_get_position(wnd, &x, &y);
    gtk_window_get_size(wnd, &w, &h);
    prefs_set_int(prefs, "mainwnd.x", x);
    prefs_set_int(prefs, "mainwnd.y", y);
    prefs_set_int(prefs, "mainwnd.w", w);
    prefs_set_int(prefs, "mainwnd.h", h);

    GtkPaned *p = GTK_PANED(gtk_builder_get_object(GTK_BUILDER(mw->builder), "hpaned1"));
    w = gtk_paned_get_position(p);
    prefs_set_int(prefs, "paned.tree_width", w);

    history_save_words(mw->history);
    prefs_save(prefs);
    dicts_save(mw->dicts);
    hotkeys_save(mw->hotkeys);
    gtk_main_quit();
}

void mainwnd_exit_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    mainwnd_exit();
}

gboolean mainwnd_delete_event_cb(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
    gtk_widget_hide(widget);
    return TRUE;
}

void mainwnd_search_clear_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *wnd = MAINWND(data);
    GtkComboBox *combo = GTK_COMBO_BOX(gtk_builder_get_object(GTK_BUILDER(wnd->builder), "mainwnd_search"));
    GtkEntry *entry = GTK_ENTRY(gtk_bin_get_child(GTK_BIN(combo)));
    gtk_entry_set_text(entry, "");
    gtk_widget_grab_focus(GTK_WIDGET(entry));
}

void mainwnd_search(Mainwnd *self, const gchar *word, gint method)
{
    if(self->word)
    {
	g_free(self->word);
	self->word = NULL;
    }
    if(word)
	self->word = g_strdup(word);
    if(method == -1)
	method = builder_get_int(self->builder, "mainwnd_search_method");

    if(word)
	builder_set_str(self->builder, "mainwnd_search", word);
    history_insert_word(self->history, word);
    gint maxhits = prefs_get_int(self->prefs, "headword.maxhits");
    ebook_search(word, method, self->results, maxhits, prefs_get_int(self->prefs, "headword.use_tree"));
    mainwnd_update_results(self);
    gtk_window_deiconify(GTK_WINDOW(self));
    gtk_widget_show_all(GTK_WIDGET(self));
    gtk_window_present(GTK_WINDOW(self));
}

void mainwnd_search_(gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    const gchar *word = builder_get_str(mw->builder, "mainwnd_search");
    mainwnd_search(mw, word, -1);
}

void mainwnd_about_btn_clicked_cb(GtkWidget *w, gpointer data)
{
    gtk_show_about_dialog(GTK_WINDOW(data),
            "program-name", "eb123",
            "comments", _("A simple EPWING dictionary viewer"),
            "copyright", "Copyright \xc2\xa9 2009-2011 evgeny",
            "version", PACKAGE_VERSION,
            "website", "http://eb123.sourceforge.jp/",
            "website-label", "eb123.sourceforge.jp",
            NULL);
}

void mainwnd_clear_combo(gpointer data)
{
    mainwnd_search_clear_clicked_cb(NULL, data);
}

gint mainwnd_get_font_size()
{
    const gchar *fontstr = prefs_get_str(_mainwnd->prefs, "mainwnd.font");
    if(fontstr)
    {
	PangoFontDescription *desc = pango_font_description_from_string(fontstr);
	gint sz = pango_font_description_get_size(desc);
	return sz/1024;
    }
    return 16;
}

void mainwnd_reset_font(Mainwnd *self)
{
    gint custom_font = prefs_get_int(self->prefs, "mainwnd.custom_font");
    const gchar *fontstr = prefs_get_str(self->prefs, "mainwnd.font");
    const gchar *font = custom_font ? fontstr : "";

    static GtkCssProvider *provider = NULL;
    GdkScreen *screen = gtk_window_get_screen(GTK_WINDOW(self));
    if(provider)
    {
	gtk_style_context_remove_provider_for_screen(screen, GTK_STYLE_PROVIDER(provider));
	g_object_unref(provider);
	provider = NULL;
    }
    if(custom_font)
    {
	provider = gtk_css_provider_new();
	gchar *s = g_strdup_printf("* {font: %s;}", font);
	if(gtk_css_provider_load_from_data(GTK_CSS_PROVIDER (provider), s, -1, NULL))
	    gtk_style_context_add_provider_for_screen(screen, GTK_STYLE_PROVIDER(provider), GTK_STYLE_PROVIDER_PRIORITY_USER);
	g_free(s);
    }
}

void mainwnd_results_row_expanded_collapsed_cb(GtkTreeView *tree, GtkTreeIter *iter, GtkTreePath *path, gpointer data)
{
    GtkTreeSelection *select = gtk_tree_view_get_selection(tree);
    gtk_tree_selection_select_path(select, path);
    gtk_tree_view_columns_autosize(tree);
}

void mainwnd_reset_headwords(Mainwnd *self)
{
    GtkTreeView *tree = GTK_TREE_VIEW(gtk_builder_get_object(GTK_BUILDER(self->builder), "mainwnd_results"));
    GtkTreeViewColumn *column = gtk_tree_view_get_column(tree, 0);
    static int btree_old = 0;
    int btree = prefs_get_int(self->prefs, "headword.use_tree");
    if(btree != btree_old)
    {
	if(prefs_get_int(self->prefs, "headword.use_tree"))
	{
	    gtk_tree_view_column_set_visible(column, True);
	    gtk_tree_view_set_show_expanders(tree, True);
	}
	else
	{
	    gint prepend = prefs_get_int(self->prefs, "prepend_title");
	    gtk_tree_view_column_set_visible(column, prepend);
	    gtk_tree_view_set_show_expanders(tree, False);
	}
	btree_old = btree;
	mainwnd_update_results(self);
    }
    else
    {
	GtkTreeSelection *selection = gtk_tree_view_get_selection(tree);
	mainwnd_results_selection_changed_cb(selection, (gpointer)self);
    }
    const gchar *color = prefs_get_str(self->prefs, "color.title");
    g_object_set(self->dictr, "foreground", color, NULL);
}

static void mainwnd_scan_start_stop(Mainwnd *self, gboolean scan)
{
    builder_set_int(self->builder, "mainwnd_scan", scan);
    prefs_set_int(self->prefs, "selection.lookup_started", scan);
    if(scan)
	popupwnd_lookup_start(self->popupwnd);
    else
	popupwnd_lookup_stop(self->popupwnd);
}

void mainwnd_scan_toggled_cb(GtkToggleButton *btn, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    if(!gtk_widget_get_visible(GTK_WIDGET(mw))) return;
    gint scan = gtk_toggle_button_get_active(btn);
    mainwnd_scan_start_stop(mw, scan);
}

void mainwnd_scan_toggled2_cb(GtkCheckMenuItem *item, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    gint scan = gtk_check_menu_item_get_active(item);
    mainwnd_scan_start_stop(mw, scan);
}

void mainwnd_status_icon_menu_cb(GtkStatusIcon *status_icon, guint button, guint activate_time, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    GtkWidget *menu = gtk_menu_new(), *item;

    gboolean scan = builder_get_int(mw->builder, "mainwnd_scan");
    item = gtk_check_menu_item_new_with_label(_("Scan"));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    g_signal_connect(G_OBJECT(item), "toggled", G_CALLBACK(mainwnd_scan_toggled2_cb), data);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), scan);

    item = gtk_image_menu_item_new_from_stock(GTK_STOCK_QUIT, NULL);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(mainwnd_exit_btn_clicked_cb), NULL);
    gtk_widget_show_all(menu);
    gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, gtk_get_current_event_time());
}

void mainwnd_iconify_restore(gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    if(gtk_widget_get_visible(GTK_WIDGET(mw)))
    {
	prefs_close_wnd(mw->prefs);
	gtk_widget_hide(GTK_WIDGET(mw));
    }
    else
	gtk_window_present(GTK_WINDOW(mw));
}

void mainwnd_iconify_cb(GtkWidget *widget, gpointer data)
{
    mainwnd_iconify_restore(data);
}

void mainwnd_status_icon_add(Mainwnd *self)
{
    GdkPixbuf *pixbuf;
    GtkStatusIcon *icon;
    pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)icon_xpm);
    icon = gtk_status_icon_new_from_pixbuf(pixbuf);
    g_signal_connect(G_OBJECT(icon), "activate", G_CALLBACK(mainwnd_iconify_cb), self);
    g_signal_connect(G_OBJECT(icon), "popup_menu", G_CALLBACK(mainwnd_status_icon_menu_cb), self);
    g_object_unref(pixbuf);
}

static void mainwnd_dict_remove_cb(GtkWidget *w, gpointer data)
{
    GtkContainer *c = GTK_CONTAINER(data);
    if(GTK_IS_TOOL_ITEM(w))
        gtk_container_remove(c, w);
}

void mainwnd_dict_buttons_remove(Mainwnd *self)
{
    GtkWidget *tb = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(self->builder), "mainwnd_dicts"));
    gtk_widget_hide(tb);
    gtk_container_foreach(GTK_CONTAINER(tb), mainwnd_dict_remove_cb, tb);
    gtk_widget_show_all(tb);
}

void mainwnd_dict_toggled_cb(GtkWidget *w, gpointer data)
{
    BOOK_INFO *binfo = (BOOK_INFO*)data;
    binfo->active = gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(w));
}

void mainwnd_dict_buttons_add(Mainwnd *self)
{
    GtkToolbar *tb = GTK_TOOLBAR(gtk_builder_get_object(GTK_BUILDER(self->builder), "mainwnd_dicts"));
    gtk_toolbar_insert(tb, gtk_separator_tool_item_new(), -1);
    gint n = builder_get_int(self->builder, "mainwnd_dict_group");
    if(n < 0)
	return;
    GtkTreeIter iter;
    if(!dicts_get_nth(self->dicts, n, &iter))
	return;
    gchar *title;
    BOOK_INFO *binfo;
    GtkTreeModel *store = GTK_TREE_MODEL(self->dicts->store);
    do
    {
	gtk_tree_model_get(store, &iter, DICT_ALIAS, &title, DICT_BINFO, &binfo, -1);
	GtkToolItem *toggle = gtk_toggle_tool_button_new();
        gtk_tool_button_set_label(GTK_TOOL_BUTTON(toggle), title);
	if(binfo->book)
	    gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(toggle), binfo->active);
	else
            gtk_widget_set_sensitive(GTK_WIDGET(toggle), FALSE);
        gtk_tool_item_set_homogeneous(toggle, FALSE);
        g_signal_connect(G_OBJECT(toggle),"toggled", G_CALLBACK(mainwnd_dict_toggled_cb), (gpointer)binfo);

	gtk_toolbar_insert(tb, toggle, -1);
	gtk_widget_set_tooltip_text(GTK_WIDGET(toggle), title);
	g_free(title);
	if(!gtk_tree_model_iter_next(store, &iter))
            break;
        gtk_toolbar_insert(tb, gtk_separator_tool_item_new(), -1);
    } while(TRUE);
    gtk_widget_show_all(GTK_WIDGET(tb));
}

void mainwnd_dict_buttons_update(Mainwnd *self)
{
    mainwnd_dict_buttons_remove(self);
    mainwnd_dict_buttons_add(self);
}

void mainwnd_dict_group_changed_cb(GtkComboBox *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    mainwnd_dict_buttons_update(mw);
}

void mainwnd_dict_groups_update(Mainwnd *self)
{
    GtkWidget *w = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(self->builder), "mainwnd_dict_group"));
    g_signal_handlers_block_by_func(w, mainwnd_dict_group_changed_cb, self);

    GtkTreeStore *dicts_store = GTK_TREE_STORE(gtk_builder_get_object(GTK_BUILDER(self->builder), "dicts_store"));
    GtkComboBoxText *combo = GTK_COMBO_BOX_TEXT(gtk_builder_get_object(GTK_BUILDER(self->builder), "mainwnd_dict_group"));
    gchar *sgroup = g_strdup(builder_get_str(self->builder, "mainwnd_dict_group"));
    gint ngroup = builder_get_int(self->builder, "mainwnd_dict_group");	
    if(ngroup < 0) ngroup = 0;
    gtk_combo_box_text_remove_all(combo);
    GtkTreeIter iter;
    if(gtk_tree_model_get_iter_first(GTK_TREE_MODEL(dicts_store), &iter))
    {
	gint i = 0;
	do { 
	    gchar *title;
	    gtk_tree_model_get(GTK_TREE_MODEL(dicts_store), &iter, DICT_ALIAS, &title, -1);
	    gtk_combo_box_text_append_text(combo, title);
	    if(ngroup == i++)
	    {
		if(g_strcmp0(sgroup, title))
		    ngroup = 0;
	    }
	    g_free(title);
	} while(gtk_tree_model_iter_next(GTK_TREE_MODEL(dicts_store), &iter));
	if(i <= ngroup)
	    ngroup = 0;
    }
    g_free(sgroup);
    builder_set_int(self->builder, "mainwnd_dict_group", ngroup);

    g_signal_handlers_unblock_by_func(w, mainwnd_dict_group_changed_cb, self);
}

void mainwnd_search_btn_clicked_cb(GtkWidget *w, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    const gchar *word = builder_get_str(mw->builder, "mainwnd_search");
    mainwnd_search(mw, word, -1);
}

gboolean mainwnd_results_key_press_event_cb(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    GtkTreeView *tree = GTK_TREE_VIEW(gtk_builder_get_object(GTK_BUILDER(mw->builder), "mainwnd_results"));
    GtkTreeModel *model = gtk_tree_view_get_model(tree);
    if(prefs_get_int(mw->prefs, "headword.use_tree") && ((event->keyval == GDK_KEY_Left) || (event->keyval == GDK_KEY_Right)))
    {
	GtkTreeSelection *select = gtk_tree_view_get_selection(tree);
	GtkTreeIter iter;
	if(!gtk_tree_selection_get_selected(select, NULL, &iter))
	    return False;
	GtkTreePath *path = gtk_tree_model_get_path(model, &iter);
	if(event->keyval == GDK_KEY_Right)
	    gtk_tree_view_expand_row(tree, path, False);
	if(event->keyval == GDK_KEY_Left)
	{
	    if(gtk_tree_path_get_depth(path) == 2)
	    {
		gtk_tree_path_up(path);
		gtk_tree_model_get_iter(model, &iter, path);
	    }
	    gtk_tree_view_collapse_row(tree, path);
	    gtk_tree_selection_select_iter(select, &iter);
	}
	gtk_tree_path_free(path);
	return True;
    }
    return FALSE;
}

gboolean mainwnd_key_press_event_cb(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    if(event->keyval == GDK_KEY_BackSpace)
    {
	GtkWidget *find = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(mw->builder), "mainwnd_find"));
	if(gtk_widget_has_focus(find))
	    return False;
	GtkWidget *combo = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(mw->builder), "mainwnd_search"));
	GtkWidget *child = gtk_bin_get_child(GTK_BIN(combo));
	if(!gtk_widget_has_focus(child))
	{
	    gtk_widget_grab_focus(combo);
	    gtk_editable_set_position(GTK_EDITABLE(child), -1);
	    return TRUE;
	}
    }
    return FALSE;
}

void mainwnd_prepare(Mainwnd *self)
{
    GdkPixbuf *pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)icon_xpm);
    gtk_window_set_default_icon(pixbuf);
    g_object_unref(pixbuf);
    gtk_widget_hide_on_delete(GTK_WIDGET(self));

    prefs_load(self->prefs);
    history_load_words(self->history);
    hotkeys_load(self->hotkeys);
    hotkeys_local_install(self->hotkeys);
    hotkeys_global_install(self->hotkeys);

    builder_install_text_cellrenderer(self->builder, "mainwnd_search_method");
    builder_install_text_cellrenderer(self->builder, "mainwnd_dict_group");
    builder_install_text_cellrenderer(self->builder, "headword.use_tree");

    GtkComboBox *combo = GTK_COMBO_BOX(gtk_builder_get_object(GTK_BUILDER(self->builder), "mainwnd_search"));
    g_signal_connect(gtk_bin_get_child(GTK_BIN(combo)), "activate", G_CALLBACK(mainwnd_search_btn_clicked_cb), self);

    mainwnd_status_icon_add(self);
    g_signal_connect(G_OBJECT(self), "delete_event", G_CALLBACK(mainwnd_delete_event_cb), self);
    g_signal_connect(G_OBJECT(self), "key_press_event", G_CALLBACK(mainwnd_key_press_event_cb), self);

    gint active = dicts_load(self->dicts);
    mainwnd_dict_groups_update(self);
    builder_set_int(self->builder, "mainwnd_dict_group", active);
    mainwnd_dict_buttons_update(self);

    gint i;
    GtkTreeIter iter;
    GtkListStore *store = GTK_LIST_STORE(gtk_builder_get_object(GTK_BUILDER(self->builder), "mainwnd_search_method_store"));
    for(i = 0; i < 4; i++)
    {
	const gchar *name = ebook_index_to_method_name(i);
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, 0, name, -1);
    }
    builder_set_int(self->builder, "mainwnd_search_method", 0);

    gint w = prefs_get_int(self->prefs, "mainwnd.w");
    gint h = prefs_get_int(self->prefs, "mainwnd.h");
    if((w > 100) && (h > 50))
	gtk_window_resize(GTK_WINDOW(self), w, h);
    w = prefs_get_int(self->prefs, "paned.tree_width");
    GtkPaned *p = GTK_PANED(gtk_builder_get_object(GTK_BUILDER(self->builder), "hpaned1"));
    gtk_paned_set_position(p, w);
    if(prefs_get_int(self->prefs, "mainwnd.remember_pos"))
    {
	gint x = prefs_get_int(self->prefs, "mainwnd.x");
	gint y = prefs_get_int(self->prefs, "mainwnd.y");
        gtk_window_move(GTK_WINDOW(self), x, y);
    }
    GtkTreeView *tree = GTK_TREE_VIEW(gtk_builder_get_object(GTK_BUILDER(self->builder), "mainwnd_results"));
    GtkTreeSelection *select = gtk_tree_view_get_selection(tree);
    gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
    g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(mainwnd_results_selection_changed_cb), self);

    GtkCellRenderer *renderer = g_object_new(TYPE_RENDER_HEADWORD, NULL); //gtk_cell_renderer_text_new();
    gtk_tree_view_insert_column_with_attributes(tree, -1, _("Result"), renderer, "text", 0, "result", 1, NULL);
    const gchar *color = prefs_get_str(self->prefs, "color.title");
    self->dictr = g_object_new(GTK_TYPE_CELL_RENDERER_TEXT, "foreground", color, NULL);
    gtk_tree_view_insert_column_with_attributes(tree, 0, _("Book"), self->dictr, "text", 2, NULL);

    mainwnd_reset_font(self);
    mainwnd_reset_headwords(self);

    gboolean b = prefs_get_int(self->prefs, "selection.lookup_started");
    builder_set_str(self->builder, "mainwnd_search", "");
    mainwnd_scan_start_stop(self, b);
    builder_grab_focus(self->builder, "mainwnd_search");
}

void mainwnd_dicts_update(Mainwnd *self)
{
    mainwnd_dict_groups_update(self);
    mainwnd_dict_buttons_update(self);
}

gboolean mainwnd_get_dicts(GtkTreeModel **store, GtkTreeIter *iter)
{
    Mainwnd *self = _mainwnd;
    int n = builder_get_int(self->builder, "mainwnd_dict_group");
    *store = GTK_TREE_MODEL(gtk_builder_get_object(GTK_BUILDER(self->builder), "dicts_store"));
    GtkTreeIter iter1;
    if(!gtk_tree_model_iter_nth_child(*store, &iter1, NULL, n))
	return FALSE;
    if(!gtk_tree_model_iter_children(*store, iter, &iter1))
        return FALSE;
    return TRUE;
}

GtkWindow *mainwnd_get_wnd()
{
    return GTK_WINDOW(_mainwnd);
}

GSequence *mainwnd_get_results()
{
    return _mainwnd->results;
}

void mainwnd_insert_text(gchar *txt, gboolean clear)
{
    TextView *text = _mainwnd->text;
    textview_insert_message(text, txt, clear);
}

gint mainwnd_get_search_method()
{
    return builder_get_int(_mainwnd->builder, "mainwnd_search_method");
}

void mainwnd_open(Mainwnd *self, RESULT *res)
{
    textview_open(self->text, res, TRUE, self->word);
    gtk_widget_show_all(GTK_WIDGET(self));
    dump_update(self->dump);
}

void mainwnd_menu_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    mainwnd_search(mw, NULL, SEARCH_METHOD_MENU);
}

void mainwnd_copyright_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    mainwnd_search(mw, NULL, SEARCH_METHOD_COPYRIGHT);
}

void mainwnd_prev_clicked_cb(GtkWidget *w, gpointer data)
{
    history_prev(data);
}

void mainwnd_search_selection(gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    const gchar *word = popupwnd_selection_copy(mw->popupwnd, True);
    mainwnd_search(mw, word, -1);
}

void mainwnd_find(gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    builder_set_visible(mw->builder, "mainwnd_find_panel", True);
    builder_set_str(mw->builder, "mainwnd_find", "");
    builder_grab_focus(mw->builder, "mainwnd_find");
}

void mainwnd_find_close_btn_clicked_cb(GtkWidget *w, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    builder_set_visible(mw->builder, "mainwnd_find_panel", False);
}

void mainwnd_find_btn_clicked_cb(GtkWidget *w, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    const gchar *str = builder_get_str(mw->builder, "mainwnd_find");
    textview_find_next(mw->text, str);
}

void mainwnd_find_activate_cb(GtkWidget *w, gpointer data)
{
    mainwnd_find_btn_clicked_cb(w, data);
}

