
#include "defs.h"

#include "eb123.h"
#include "history.h"
#include "hotkeys.h"
#include "mainwnd.h"
#include "prefs.h"

typedef struct
{
    gchar *name;
    void (* func)(gpointer data);
    gboolean global;
} HOTKEY_COMMAND;

typedef struct
{
    guint32 mask, keyval;
    gboolean enabled;
} HOTKEY_EVENT;

const HOTKEY_COMMAND hotkeys_list[] = {
    { _("Search"),			mainwnd_search_, FALSE},
    { _("Go back"),			history_prev, FALSE},
    { _("Clear search field"),		mainwnd_clear_combo, FALSE},
    { _("Find in main window"),		mainwnd_find, FALSE},
//    { _("Next hit"),			headword_next, FALSE},
//    { _("Previous hit"),		headword_prev, FALSE},
    { _("Select all dictionaries"),	dicts_select_all, FALSE},
    { _("Unselect all dictionaries"),	dicts_unselect_all, FALSE},
    { _("Select next group"),		dicts_next_group, FALSE},
    { _("Select previous group"),	dicts_prev_group, FALSE},
    { _("Search in main window"),	mainwnd_search_selection, TRUE},
    { _("Search in popup window"),	popupwnd_search_selection, TRUE},
    { _("Iconify/restore main window"), mainwnd_iconify_restore, TRUE},
    { _("Quit program"),		mainwnd_exit, FALSE}
};

G_DEFINE_TYPE(Hotkeys, hotkeys, G_TYPE_OBJECT);

static Hotkeys *_hotkeys = NULL;

static void hotkeys_set_property(GObject *object, guint param_id, const GValue *value, GParamSpec *pspec)
{   
    Hotkeys *hotkeys = HOTKEYS(object);
    switch(param_id)
    {
        case 1:
            hotkeys->mainwnd = g_value_get_pointer(value);
            break;
        case 2:
            hotkeys->prefs = PREFS(g_value_get_pointer(value));
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID(object, param_id, pspec);
            break;
    }
}
    
static void hotkeys_class_init(HotkeysClass *klass)
{   
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    gobject_class->set_property = hotkeys_set_property;
    g_object_class_install_property(gobject_class, 1, g_param_spec_pointer("mainwnd", _("Mainwnd"), _("Mainwnd"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
    g_object_class_install_property(gobject_class, 2, g_param_spec_pointer("prefs", _("Prefs"), _("Prefs"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}   
    
static void hotkeys_init(Hotkeys *self)
{
    if(!_hotkeys)
	_hotkeys = self;
    else
	g_assert(1);
    gint i;
    HOTKEY_EVENT *events = g_try_new0(HOTKEY_EVENT, sizeof(hotkeys_list));
    if(!events) return;
    self->list = NULL;
    for(i = 0; i < SZ(hotkeys_list); i++)
	self->list = g_list_append(self->list, (gpointer)&(events[i]));
}

static gboolean hotkeys_local_activated_cb(GtkAccelGroup *accelgroup, GObject *arg1, guint arg2, GdkModifierType arg3, gpointer data)
{
    gint i;
    guint cmp;
    HOTKEY_EVENT *evt;
    Hotkeys *self = HOTKEYS(data);

    gulong keyval = (gulong)arg2;
    gulong mask = arg3;
    for(i = 0; (evt = g_list_nth_data(self->list, i)); i++)
    {
	if(!keyval) continue;
	if(evt->keyval != keyval) continue;
	cmp = (evt->mask ^ mask);
	if(prefs_get_int(self->prefs, "hotkeys.ignore_locks"))
	    cmp = cmp & !(GDK_MOD2_MASK | GDK_LOCK_MASK);
	if(!cmp)
	{
	    hotkeys_list[i].func(self->mainwnd);
	    return TRUE;
	}
    }
    return FALSE;
}

void hotkeys_local_install(Hotkeys *self)
{
    static GtkAccelGroup *accel_group = NULL;
    GClosure *closure;
    gint i;
    HOTKEY_EVENT *evt;
    if(accel_group)
    {
        gtk_window_remove_accel_group(mainwnd_get_wnd(), accel_group);
        g_object_unref(accel_group);
    }
    accel_group = gtk_accel_group_new();
    gtk_window_add_accel_group(mainwnd_get_wnd(), accel_group);
    g_signal_connect(G_OBJECT(accel_group), "accel-activate", G_CALLBACK(hotkeys_local_activated_cb), self);

    for(i = 0; (evt = g_list_nth_data(self->list, i)); i++)
    {
	gulong keyval = evt->keyval;
	if(!evt->enabled || hotkeys_list[i].global) continue;
	if((evt->mask == 0) && (evt->keyval == GDK_KEY_Return)) continue;
	closure = g_cclosure_new(G_CALLBACK(hotkeys_local_activated_cb), (gpointer)keyval, NULL);
	gtk_accel_group_connect(accel_group, evt->keyval, evt->mask, GTK_ACCEL_VISIBLE, closure);
    }
}

static GdkFilterReturn hotkeys_global_cb(GdkXEvent *gdk_xevent, GdkEvent *event, gpointer data)
{
    XEvent *xevent = (XEvent *) gdk_xevent;
    Display *d = XOpenDisplay(NULL);
    const guint mods = LockMask | Mod2Mask;
    guint i;
    HOTKEY_EVENT *evt;
    Hotkeys *self = HOTKEYS(data);

    switch(xevent->type)
    {
	case KeyPress:
	{
	    for(i = 0; (evt = g_list_nth_data(self->list, i)); i++)
	    {
		if(!hotkeys_list[i].global) continue;
		gboolean mask_cmp;
		if(prefs_get_int(self->prefs, "hotkeys.ignore_locks"))
		    mask_cmp = ((xevent->xkey.state | mods) == (evt->mask | mods));
		else
		    mask_cmp = (xevent->xkey.state == evt->mask);
		if(mask_cmp && (XKeysymToKeycode(d, evt->keyval) == xevent->xkey.keycode))
		{
		    hotkeys_list[i].func(self->mainwnd);
		    break;
		}
	    }
	    break;
	}
	default:
	    break;
    }

    return GDK_FILTER_CONTINUE;
}

void hotkeys_global_install(Hotkeys *self)
{
    static gboolean cb_added = FALSE;
    GdkWindow *rootwin = gdk_get_default_root_window();
    Display *d = GDK_WINDOW_XDISPLAY(rootwin);
    gint i, j;
    HOTKEY_EVENT *evt;

    if(!cb_added)
    {
	gdk_window_add_filter(rootwin, hotkeys_global_cb, self);
	cb_added = TRUE;
    }
    const guint mod_masks [] = {
	0,
	LockMask, /* Caps Lock */
	Mod2Mask, /* Num Lock */
	LockMask | Mod2Mask
    };
    // remove all hotkeys
    for(i = 0; (evt = g_list_nth_data(self->list, i)); i++)
    {
	if(!hotkeys_list[i].global) continue;
	for(j = 0; j < G_N_ELEMENTS(mod_masks); j++)
	    XUngrabKey(d, XKeysymToKeycode(d, evt->keyval), evt->mask | mod_masks[j], DefaultRootWindow(d));
    }
    // install currently assigned hotkeys
    for(i = 0; (evt = g_list_nth_data(self->list, i)); i++)
    {
	if(!hotkeys_list[i].global) continue;
	if(prefs_get_int(self->prefs, "hotkeys.ignore_locks"))
	{
	    for(j = 0; j < G_N_ELEMENTS(mod_masks); j++)
		XGrabKey(d, XKeysymToKeycode(d, evt->keyval), evt->mask | mod_masks[j], DefaultRootWindow(d), True, GrabModeAsync, GrabModeAsync);
	}
	else
	    XGrabKey(d, XKeysymToKeycode(d, evt->keyval), evt->mask, DefaultRootWindow(d), True, GrabModeAsync, GrabModeAsync);
    }
}

HOTKEY_EVENT* hotkeys_find(Hotkeys *self, const gchar *name)
{
    gint i;
    for(i = 0; i < SZ(hotkeys_list); i++)
    {
	if(!g_strcmp0(hotkeys_list[i].name, name))
	    return (HOTKEY_EVENT*)g_list_nth_data(self->list, i);
    }
    return NULL;
}

void hotkeys_save(Hotkeys *self)
{
    gchar filename[PATH_MAX], buff[16];
    gint i;
    HOTKEY_EVENT *evt;
    xmlDocPtr doc = xmlNewDoc((xmlChar*)"1.0");
    doc->children = xmlNewDocRawNode(doc, NULL, (xmlChar*)"hotkeys", NULL);
    const gchar *userdir = prefs_get_userdir();
    sprintf(filename, "%s%s%s", userdir, G_DIR_SEPARATOR_S, FILENAME_HOTKEYS);
    for(i = 0; (evt = g_list_nth_data(self->list, i)); i++)
    {
	if(!evt->enabled) continue;
	xmlNodePtr node1 = xmlAddChild((xmlNodePtr)doc->children, xmlNewNode(NULL, (xmlChar*)"hotkeys"));
	xmlNewProp(node1, (xmlChar*)"command", (xmlChar*)hotkeys_list[i].name);
	sprintf(buff, "0x%04x", evt->mask);
	xmlNewProp(node1, (xmlChar*)"mask", (xmlChar*)buff);
	sprintf(buff, "0x%04x", evt->keyval);
	xmlNewProp(node1, (xmlChar*)"keyval", (xmlChar*)buff);
    }
    xmlSaveFormatFileEnc(filename, doc, "utf8", 0);
    xmlFreeDoc(doc);
}

void hotkeys_load_item(void *ctx, const xmlChar *name, const xmlChar **atts)
{
    if(!atts) return;

    xmlParserCtxt *ctxt = (xmlParserCtxt*)ctx;
    xmlSAXHandler *cb = ctxt->sax;
    Hotkeys *self = HOTKEYS(cb->_private);
    HOTKEY_EVENT *evt = hotkeys_find(self, (gchar*)atts[1]);
    if(!evt) return;

    evt->mask = strtol((gchar*)atts[3], NULL, 16);
    evt->keyval = strtol((gchar*)atts[5], NULL, 16);
    evt->enabled = (evt->keyval != 0);
}

/* Add local some default hotkeys */
void hotkeys_add_defaults(Hotkeys *self)
{
    gint i;
    for(i = 0; hotkeys_list[i].name; i++)
    {
        if(hotkeys_list[i].func == mainwnd_exit)
        {
	    HOTKEY_EVENT *evt = (HOTKEY_EVENT*)g_list_nth_data(self->list, i);
            evt->mask = 0x4;
            evt->keyval = 0x71;
	    evt->enabled = TRUE;
        }
        else if(hotkeys_list[i].func == dicts_select_all)
        {
	    HOTKEY_EVENT *evt = (HOTKEY_EVENT*)g_list_nth_data(self->list, i);
            evt->mask = 0x4;
            evt->keyval = 0x65;
	    evt->enabled = TRUE;
        }
        else if(hotkeys_list[i].func == dicts_unselect_all)
        {
	    HOTKEY_EVENT *evt = (HOTKEY_EVENT*)g_list_nth_data(self->list, i);
            evt->mask = 0x4;
            evt->keyval = 0x64;
	    evt->enabled = TRUE;
        }
        else if(hotkeys_list[i].func == mainwnd_clear_combo)
        {
	    HOTKEY_EVENT *evt = (HOTKEY_EVENT*)g_list_nth_data(self->list, i);
            evt->mask = 0x4;
            evt->keyval = 0x77;
	    evt->enabled = TRUE;
        }
        else if(hotkeys_list[i].func == mainwnd_iconify_restore)
        {
	    HOTKEY_EVENT *evt = (HOTKEY_EVENT*)g_list_nth_data(self->list, i);
            evt->mask = 0x4;
            evt->keyval = 0x69;
	    evt->enabled = TRUE;
        }
    }
}

void hotkeys_load(Hotkeys *self)
{
    gchar filename[PATH_MAX];
    xmlSAXHandler cb;
    const gchar *userdir = prefs_get_userdir();
    sprintf(filename, "%s%s%s", userdir, G_DIR_SEPARATOR_S, FILENAME_HOTKEYS);
    if(!g_file_test(filename, G_FILE_TEST_IS_REGULAR))
    {
        hotkeys_add_defaults(self);
        return;
    }
    memset(&cb, 0, sizeof(xmlSAXHandler));
    cb.startElement = &hotkeys_load_item;
    cb._private = self;
    xmlDocPtr doc = xmlSAXParseFile(&cb, filename, 0);
    xmlFreeDoc(doc);
}

void hotkey_to_string(guint mask, guint keyval, gchar *key)
{
    key[0] = '\0';

    if(mask & GDK_CONTROL_MASK)
        strcat(key, "<Ctrl>");

    if(mask & GDK_SHIFT_MASK)
        strcat(key, "<Shift>");

    if(mask & GDK_LOCK_MASK)
        strcat(key, "<Lock>");

    if(mask & GDK_MOD1_MASK)
        strcat(key, "<Alt>");

    if(mask & GDK_MOD2_MASK)
            strcat(key, "<NumLock>");

    if(mask & GDK_MOD3_MASK)
        strcat(key, "<Mod3>");

    if(mask & GDK_MOD4_MASK)
        strcat(key, "<Mod4>");

    if(mask & GDK_MOD5_MASK)
        strcat(key, "<ScrollLock>");

    if(mask & GDK_BUTTON1_MASK)
        strcat(key, "<Button1>");

    if(mask & GDK_BUTTON2_MASK)
        strcat(key, "<Button2>");

    if(mask & GDK_BUTTON3_MASK)
        strcat(key, "<Button3>");

    if(mask & GDK_BUTTON4_MASK)
        strcat(key, "<Button4>");

    if(mask & GDK_BUTTON5_MASK)
        strcat(key, "<Button5>");

    if(mask & GDK_RELEASE_MASK)
        strcat(key, "<Release>");

    strcat(key, gdk_keyval_name(keyval));
}

GtkEntry* hotkeys_hbox_get_entry(GtkBox *hbox)
{
    GList *list = gtk_container_get_children(GTK_CONTAINER(hbox));
    return GTK_ENTRY(list->next->data);
}

GtkLabel* hotkeys_hbox_get_label(GtkBox *hbox)
{
    GList *list = gtk_container_get_children(GTK_CONTAINER(hbox));
    return GTK_LABEL(list->data);
}

static gboolean hotkeys_grab_key_cb(GtkDialog *dlg, GdkEventKey *evt, GtkBox *hbox)
{
    gchar key[256];

    switch (evt->keyval){
    case GDK_KEY_Shift_L:
    case GDK_KEY_Shift_R:
    case GDK_KEY_Control_L:
    case GDK_KEY_Control_R:
    case GDK_KEY_Meta_L:
    case GDK_KEY_Meta_R:
    case GDK_KEY_Alt_L:
    case GDK_KEY_Alt_R:
    case GDK_KEY_Caps_Lock:
    case GDK_KEY_Shift_Lock:
    case GDK_KEY_Scroll_Lock:
    case GDK_KEY_Num_Lock:
    case GDK_KEY_Kana_Lock:
        return(FALSE);
        break;
    }

    if(evt->keyval == GDK_KEY_Escape)
    {
        gtk_dialog_response(dlg, GTK_RESPONSE_NO);
        return TRUE;
    }

    if(prefs_get_int(_hotkeys->prefs, "hotkeys.ignore_locks"))
        evt->state = evt->state & (~GDK_LOCK_MASK) & (~GDK_MOD2_MASK);
    hotkey_to_string(evt->state, evt->keyval, key);

    gtk_entry_set_text(hotkeys_hbox_get_entry(hbox), key);
    gtk_dialog_response(dlg, GTK_RESPONSE_OK);

    return TRUE;
}

static void hotkeys_edit_cb(GtkWidget *widget, GtkBox *hbox)
{
    GtkWidget* dlg = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_NONE,
                                            _("Press a key (or a key combination) to setup hotkey,\n"
                                            "\"Clear\" button or \"Esc\" key to remove current hotkey,\n"
                                            "\"Cancel\" to close this message."));
    gtk_dialog_add_buttons(GTK_DIALOG(dlg), GTK_STOCK_CLEAR, GTK_RESPONSE_NO, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL);

    gtk_window_set_title(GTK_WINDOW(dlg), _("Grabbing a key."));
    g_signal_connect(G_OBJECT(dlg), "key-press-event", G_CALLBACK(hotkeys_grab_key_cb), hbox);

    if(gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_NO)
        gtk_entry_set_text(hotkeys_hbox_get_entry(hbox), "");
    gtk_widget_destroy(dlg);
}

GtkWidget* hotkeys_edit_nth(Hotkeys *self, gint n)
{
    if(n >= SZ(hotkeys_list))
	return NULL;

    GtkWidget *hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 5);
    if(hotkeys_list[n].global)
	gtk_widget_set_tooltip_text(hbox, _("Global hotkey"));

    GtkWidget *label = gtk_label_new(hotkeys_list[n].name);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

    GtkWidget *btn = gtk_button_new_with_label(_("Edit..."));
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked", G_CALLBACK(hotkeys_edit_cb), hbox);

    GtkWidget *entry = gtk_entry_new();
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_box_pack_end(GTK_BOX(hbox), entry, FALSE, FALSE, 0);

    HOTKEY_EVENT *evt = (HOTKEY_EVENT*)g_list_nth_data(self->list, n);
    gchar keystr[64];
    if(evt ? evt->enabled : FALSE)
    {
	hotkey_to_string(evt->mask, evt->keyval, keystr);
	gtk_entry_set_text(GTK_ENTRY(entry), keystr);
    }
    return hbox;
}

void hotkeys_reload_keys_cb(GtkWidget *w, gpointer data)
{
    Hotkeys *self = HOTKEYS(data);
    GtkBox *hbox = GTK_BOX(w);
    GtkLabel *label = hotkeys_hbox_get_label(hbox);
    GtkEntry *entry = hotkeys_hbox_get_entry(hbox);
    const gchar *name = gtk_label_get_text(label);
    const gchar *keys = gtk_entry_get_text(entry);
    HOTKEY_EVENT* evt = hotkeys_find(self, name);
    if(!evt) return;
    if(!strlen(keys)) return;
    guint mask, keyval;
    gtk_accelerator_parse(keys, &keyval, &mask);
    evt->mask = mask;
    evt->keyval = keyval;
    evt->enabled = (evt->keyval != 0);
}

void hotkeys_reload(Hotkeys *self, GtkBox *vbox)
{
    gint i;
    for(i = 0; i < SZ(hotkeys_list); i++)
    {
        HOTKEY_EVENT* evt = (HOTKEY_EVENT*)g_list_nth_data(self->list, i);
        evt->mask = evt->keyval = 0;
        evt->enabled = FALSE;
    }
    gtk_container_forall(GTK_CONTAINER(vbox), hotkeys_reload_keys_cb, self);
}

