
#include "defs.h"

#include "hook.h"

static EB_Error_Code hook_text(EB_Book *book, EB_Appendix *appendix, void *container, EB_Hook_Code code, int argc, const unsigned int *argv);
static EB_Error_Code hook_wave(EB_Book *book, EB_Appendix *appendix, void *container, EB_Hook_Code code, int argc, const unsigned int *argv);
static EB_Error_Code hook_mpeg(EB_Book *book, EB_Appendix *appendix, void *container, EB_Hook_Code code, int argc, const unsigned int *argv);

static EB_Hook hooks_text[] = {
    {EB_HOOK_SET_INDENT,            hook_text},
    {EB_HOOK_NEWLINE,               hook_text},
    {EB_HOOK_NARROW_FONT,           hook_text},
    {EB_HOOK_WIDE_FONT,	            hook_text},
    {EB_HOOK_BEGIN_REFERENCE,       hook_text},
    {EB_HOOK_END_REFERENCE,         hook_text},
    {EB_HOOK_BEGIN_CANDIDATE,       hook_text},
    {EB_HOOK_END_CANDIDATE_LEAF,    hook_text},
    {EB_HOOK_END_CANDIDATE_GROUP,   hook_text},
    {EB_HOOK_BEGIN_COLOR_BMP,       hook_text},
    {EB_HOOK_BEGIN_COLOR_JPEG,      hook_text},
    {EB_HOOK_END_COLOR_GRAPHIC,     hook_text},
    {EB_HOOK_BEGIN_MONO_GRAPHIC,    hook_text},
    {EB_HOOK_END_MONO_GRAPHIC,      hook_text},
    {EB_HOOK_BEGIN_GRAY_GRAPHIC,    hook_text},
    {EB_HOOK_END_GRAY_GRAPHIC,      hook_text},
    {EB_HOOK_BEGIN_WAVE,            hook_wave},
    {EB_HOOK_END_WAVE,              hook_wave},
    {EB_HOOK_BEGIN_MPEG,            hook_mpeg},
    {EB_HOOK_END_MPEG,              hook_mpeg},
    {EB_HOOK_BEGIN_SUPERSCRIPT,     hook_text},
    {EB_HOOK_END_SUPERSCRIPT,       hook_text},
    {EB_HOOK_BEGIN_SUBSCRIPT,       hook_text},
    {EB_HOOK_END_SUBSCRIPT,         hook_text},
    {EB_HOOK_BEGIN_EMPHASIS,        hook_text},
    {EB_HOOK_END_EMPHASIS,          hook_text},
    {EB_HOOK_BEGIN_KEYWORD,         hook_text},
    {EB_HOOK_END_KEYWORD,           hook_text},
    {EB_HOOK_BEGIN_DECORATION,      hook_text},
    {EB_HOOK_END_DECORATION,        hook_text},
    {EB_HOOK_NULL, NULL},
};


static EB_Hook hooks_heading[] = {
    {EB_HOOK_NEWLINE,               hook_text},
    {EB_HOOK_NARROW_FONT,           hook_text},
    {EB_HOOK_WIDE_FONT,	            hook_text},
    {EB_HOOK_NARROW_JISX0208,       eb_hook_euc_to_ascii},
    {EB_HOOK_BEGIN_SUPERSCRIPT,     hook_text},
    {EB_HOOK_END_SUPERSCRIPT,       hook_text},
    {EB_HOOK_BEGIN_SUBSCRIPT,       hook_text},
    {EB_HOOK_END_SUBSCRIPT,         hook_text},
    {EB_HOOK_NULL, NULL},
};

static EB_Hook hooks_candidate[] = {
    {EB_HOOK_NARROW_FONT,           hook_text},
    {EB_HOOK_WIDE_FONT,	            hook_text},
    {EB_HOOK_BEGIN_CANDIDATE,       hook_text},
    {EB_HOOK_END_CANDIDATE_LEAF,    hook_text},
    {EB_HOOK_END_CANDIDATE_GROUP,   hook_text},
  {EB_HOOK_NULL, NULL},
};

static EB_Error_Code hook_text(EB_Book *book, EB_Appendix *appendix, void *container, EB_Hook_Code code, int argc, const unsigned int *argv)
{
    char text[128];
    if(code == EB_HOOK_NARROW_FONT)
        sprintf(text, "<gaiji code=h%04x>", argv[0]);
    else if(code == EB_HOOK_WIDE_FONT)
        sprintf(text, "<gaiji code=z%04x>", argv[0]);
    else if(code == EB_HOOK_SET_INDENT)
        sprintf(text, "<indent position=%d>", argv[1]);
    else if(code == EB_HOOK_NEWLINE)
        sprintf(text, "\n");
    else if(code == EB_HOOK_BEGIN_REFERENCE)
        sprintf(text, "<reference>");
    else if(code == EB_HOOK_END_REFERENCE)
        sprintf(text, "</reference page=0x%x offset=0x%x>", argv[1], argv[2]);
    else if(code == EB_HOOK_BEGIN_CANDIDATE)
        sprintf(text, "<candidate>");
    else if(code == EB_HOOK_END_CANDIDATE_LEAF)
        sprintf(text, "</candidate>");
    else if(code == EB_HOOK_END_CANDIDATE_GROUP)
        sprintf(text, "</candidate page=0x%x offset=0x%x>", argv[1], argv[2]);
    else if(code == EB_HOOK_BEGIN_SUPERSCRIPT)
        sprintf (text, "<sup>");
    else if(code == EB_HOOK_END_SUPERSCRIPT)
        sprintf(text, "</sup>");
    else if(code == EB_HOOK_BEGIN_SUBSCRIPT)
        sprintf(text, "<sub>");
    else if(code == EB_HOOK_END_SUBSCRIPT)
        sprintf(text, "</sub>");
    else if(code == EB_HOOK_BEGIN_EMPHASIS)
        sprintf(text, "<emphasis>");
    else if(code == EB_HOOK_END_EMPHASIS)
        sprintf(text, "</emphasis>");
    else if(code == EB_HOOK_BEGIN_KEYWORD)
        sprintf(text, "<keyword argv1=%x>", argv[1]);
    else if(code == EB_HOOK_END_KEYWORD)
        sprintf(text, "</keyword>");
    else if(code == EB_HOOK_BEGIN_DECORATION)
        sprintf(text, "<modification method=%d>", argv[1]);
    else if(code == EB_HOOK_END_DECORATION)
        sprintf(text, "</modification>");
    else if(code == EB_HOOK_BEGIN_COLOR_JPEG)
        sprintf(text, "<jpeg page=0x%x offset=0x%x>", argv[2], argv[3]);
    else if(code == EB_HOOK_BEGIN_COLOR_BMP)
        sprintf(text, "<bmp page=0x%x offset=0x%x>", argv[2], argv[3]);
    else if(code == EB_HOOK_BEGIN_MONO_GRAPHIC)
        sprintf(text, "<mono width=%d height=%d>", argv[3], argv[2]);
    else if(code == EB_HOOK_END_MONO_GRAPHIC)
        sprintf (text, "</mono page=0x%x offset=0x%x>", argv[1], argv[2]);
    else if(code == EB_HOOK_BEGIN_GRAY_GRAPHIC)
        sprintf (text, "<gray page=0x%x offset=0x%x>", argv[2], argv[3]);
    else
        return EB_SUCCESS;
    eb_write_text_string(book, text);
    return EB_SUCCESS;
}

/*
 * Hook for a reference to WAVE sound data.
 */
static EB_Error_Code hook_wave(EB_Book *book, EB_Appendix *appendix, void *container, EB_Hook_Code code, int argc, const unsigned int *argv)
{
    off_t start_location, end_location;
    gint data_size;
    gchar text[64];

    /*
     * Set binary context.
     */
    start_location = (off_t)(argv[2] - 1) * EB_SIZE_PAGE + argv[3];
    end_location   = (off_t)(argv[4] - 1) * EB_SIZE_PAGE + argv[5];
    data_size = end_location - start_location;
    switch (code)
    {
    case EB_HOOK_BEGIN_WAVE:
        eb_write_text_string(book, "<wave>");
        break;
    case EB_HOOK_END_WAVE:
        sprintf (text, "</wave page=0x%x offset=0x%x size=%d>", argv[2], argv[3], data_size);
        eb_write_text_string(book, text);
        break;
    }
    return EB_SUCCESS;
}


/*
 * Hook for a reference to MPEG sound data.
 */
static EB_Error_Code hook_mpeg(EB_Book *book, EB_Appendix *appendix, void *container, EB_Hook_Code code, int argc, const unsigned int *argv)
{
    char file_name[EB_MAX_DIRECTORY_NAME_LENGTH + 1], text[256];

    switch (code)
    {
    case EB_HOOK_BEGIN_MPEG:
        break;
    case EB_HOOK_END_MPEG:
        if (eb_compose_movie_file_name(argv + 2, file_name) != EB_SUCCESS)
            return EB_SUCCESS;
        sprintf(text, "<mpeg filename=%s>", file_name);
        eb_write_text_string(book, text);
        break;
    }
    return EB_SUCCESS;
}

EB_Error_Code hooksets_init()
{
    EB_Error_Code error_code;

    eb_initialize_hookset(&hooksets.text);
    error_code = eb_set_hooks(&hooksets.text, hooks_text);
    if(error_code != EB_SUCCESS)
    {
        LOG(LOG_ERROR, "Failed to set hooksets.text) : %s\n", eb_error_message(error_code));
        return 1;
    }
    eb_initialize_hookset(&hooksets.heading);
    error_code = eb_set_hooks (&hooksets.heading, hooks_heading);
    if(error_code != EB_SUCCESS)
    {
        LOG(LOG_ERROR, "Failed to set hooksets.heading) : %s\n", eb_error_message(error_code));
        return 1;
    }
    eb_initialize_hookset(&hooksets.candidate);
    error_code = eb_set_hooks(&hooksets.candidate, hooks_candidate);
    if(error_code != EB_SUCCESS)
    {
        LOG(LOG_ERROR, "Failed to set hooksets.candidate) : %s\n", eb_error_message(error_code));
        return 1;
    }
    return EB_SUCCESS;
}

void hooksets_finalize()
{
    eb_finalize_hookset(&hooksets.text);
    eb_finalize_hookset(&hooksets.heading);
    eb_finalize_hookset(&hooksets.candidate);
}

