
#include "defs.h"

#include "history.h"
#include "mainwnd.h"
#include "prefs.h"

extern Builder *_builder;
extern Prefs *_prefs;

G_DEFINE_TYPE(History, history, G_TYPE_OBJECT);

static void history_set_property(GObject *object, guint param_id, const GValue *value, GParamSpec *pspec)
{
    History *history = HISTORY(object);
    switch(param_id)
    {
	case 1:
	    history->builder = BUILDER(g_value_get_pointer(value));
	    break;
	case 2:
	    history->prefs = PREFS(g_value_get_pointer(value));
	    break;
	default:
	    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, param_id, pspec);
	    break;
    }
}

static void history_class_init(HistoryClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    gobject_class->set_property = history_set_property;
    g_object_class_install_property(gobject_class, 1, g_param_spec_pointer("builder", _("Builder"), _("Builder"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
    g_object_class_install_property(gobject_class, 2, g_param_spec_pointer("prefs", _("Prefs"), _("Prefs"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}

static void history_init(History *self)
{
}

void history_prev(gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    History *self = mw->history;
    GList *hist, *hist1;
    RESULT *res;

    hist = g_list_last(self->list);
    if(!hist)
        return;

    hist1 = g_list_previous(hist);
    if(!hist1)
        return;

    result_free((RESULT *)hist->data);
    self->list = g_list_delete_link(self->list, hist);
    
    res = (RESULT *)(hist1->data);
    mainwnd_open(MAINWND(mainwnd_get_wnd()), res);
}

void history_insert_res(History *self, RESULT *res)
{
    GList *hist = g_list_last(self->list);
    if(hist)
    {
        RESULT *res1 = (RESULT*)hist->data;
        if((res1->binfo == res->binfo) && (res1->pos.page == res->pos.page) && (res1->pos.offset == res->pos.offset))
            return;
    }
    self->list = g_list_append(self->list, result_duplicate(res));
}

RESULT *history_last_result(History *self)
{
    GList *hist = g_list_last(self->list);
    if(hist)
        return (RESULT*)hist->data;
    return NULL;
}

gboolean history_remove_word(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
    gchar *str, *tmp = (gchar*)data;
    GtkListStore *store = GTK_LIST_STORE(model);
    gtk_tree_model_get(model, iter, 0, &str, -1);
    if(!g_strcmp0(str, tmp))
    {
        gtk_list_store_remove(store, iter);
        return TRUE;
    }
    return FALSE;
}

void history_insert_word(History *self, const gchar *word)
{
    if(!word)
	return;
    if(!strlen(word))
	return;

    GtkComboBox *mainwnd_search = GTK_COMBO_BOX(gtk_builder_get_object(GTK_BUILDER(_builder), "mainwnd_search"));
    GtkTreeModel *model = gtk_combo_box_get_model(mainwnd_search);

    gtk_tree_model_foreach(model, history_remove_word, (gpointer)word);
    gtk_tree_model_foreach(model, history_remove_word, (gpointer)word);
    gint n = prefs_get_int(self->prefs, "dictbar.word_hist");
    if(n < 10) n = 10;

    while(gtk_tree_model_iter_n_children(model, NULL) > n)
	gtk_combo_box_text_remove(GTK_COMBO_BOX_TEXT(mainwnd_search), n - 1);
    gtk_combo_box_text_prepend_text(GTK_COMBO_BOX_TEXT(mainwnd_search), word);
    gtk_combo_box_set_active(mainwnd_search, 0);
}

gboolean history_save_word(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
    guchar *str;
    xmlNodePtr node = xmlAddChild((xmlNodePtr)data, xmlNewNode(NULL, (xmlChar*)"word"));
    gtk_tree_model_get(model, iter, 0, &str, -1);
    xmlNewProp(node, (xmlChar*)"val", str);
    g_free(str);
    return FALSE;
}

void history_save_words(History *self)
{
    gchar filename[PATH_MAX];
    GtkComboBox *mainwnd_search = GTK_COMBO_BOX(gtk_builder_get_object(GTK_BUILDER(self->builder), "mainwnd_search"));
    GtkTreeModel *model = gtk_combo_box_get_model(mainwnd_search);
    xmlDocPtr doc = xmlNewDoc((xmlChar*)"1.0");
    doc->children = xmlNewDocRawNode(doc, NULL, (xmlChar*)"History", NULL);
    const gchar *userdir = prefs_get_userdir();
    sprintf(filename, "%s%s%s", userdir, G_DIR_SEPARATOR_S, FILENAME_HISTORY);
    gtk_tree_model_foreach(model, history_save_word, doc->children);
    xmlSaveFormatFileEnc(filename, doc, "utf8", 0);
    xmlFreeDoc(doc);
}

void history_load_word(void *ctx, const xmlChar *name, const xmlChar **atts)
{
    xmlParserCtxt *ctxt = (xmlParserCtxt*)ctx;
    xmlSAXHandler *cb = ctxt->sax;
    History *self = HISTORY(cb->_private);
    if(!g_strcmp0((gchar*)name, "word"))
    {
	GtkComboBox *mainwnd_search = GTK_COMBO_BOX(gtk_builder_get_object(GTK_BUILDER(self->builder), "mainwnd_search"));
	GtkTreeModel *model = gtk_combo_box_get_model(mainwnd_search);
	GtkTreeIter iter;
	gtk_list_store_append(GTK_LIST_STORE(model), &iter);
	gtk_list_store_set(GTK_LIST_STORE(model), &iter, 0, (gchar*)atts[1], -1);
    }
}

void history_load_words(History *self)
{
    gchar filename[PATH_MAX];
    xmlSAXHandler cb;
    const gchar *userdir = prefs_get_userdir();
    sprintf(filename, "%s%s%s", userdir, G_DIR_SEPARATOR_S, FILENAME_HISTORY);
    if(!g_file_test(filename, G_FILE_TEST_IS_REGULAR))
        return;
    memset(&cb, 0, sizeof(xmlSAXHandler));
    cb.startElement = &history_load_word;
    cb._private = self;
    xmlDocPtr doc = xmlSAXParseFile(&cb, filename, 0);
    xmlFreeDoc(doc);
}

RESULT *result_new(BOOK_INFO *binfo, EB_Position *pos)
{
    RESULT *res = g_new0(RESULT, 1);
    res->binfo = binfo;
    if(pos)
        memcpy(&res->pos, pos, sizeof(EB_Position));
    return res;
}

RESULT *result_duplicate(RESULT *res)
{
    RESULT *result;
    
    g_assert(res != NULL);

    result = g_new0(RESULT, 1);
    result_copy(result, res);
    return result;
}

void result_copy(RESULT *to, RESULT *from)
{
    g_assert(from != NULL);
    g_assert(to != NULL);

    if(from->book_title)
        to->book_title = g_strdup(from->book_title);

    if(from->heading)
        to->heading = g_strdup(from->heading);

    if(from->filename)
        to->filename = g_strdup(from->filename);

    to->binfo = from->binfo;
    to->pos = from->pos;
}

void result_free(gpointer data)
{
    RESULT *res = (RESULT*)data;
    if(res->book_title)
        g_free(res->book_title);
    if(res->heading)
        g_free(res->heading);
    if(res->filename)
	g_free(res->filename);
    g_free(res);
}

gboolean result_compare(RESULT *res1, RESULT *res2)
{
    if(!res1 || !res2)
	return FALSE;
    if(res1->binfo != res2->binfo)
        return FALSE;
    if(res1->pos.page != res2->pos.page)
        return FALSE;
    if(res1->pos.offset != res2->pos.offset)
        return FALSE;
    return TRUE;
}

