
#include "defs.h"

#include "builder.h"
#include "dicts.h"
#include "ebook.h"
#include "history.h"
#include "hook.h"
#include "jcode.h"
#include "mainwnd.h"
#include "textview.h"

#ifndef MAX_BUFSIZE
#define MAX_BUFSIZE 65535
#endif

typedef struct
{
    gchar *name;
    int (* have_method)(EB_Book *book);
    EB_Error_Code (* search)(EB_Book *book, const gchar* input_word);
} SEARCH_METHOD;

struct _ebook
{
    GSequence   *results;
} ebook;

static const SEARCH_METHOD methods[] =
{
    {_("Forward"),	    eb_have_word_search,	eb_search_word},
    {_("Backward"),	    eb_have_endword_search,	eb_search_endword},
    {_("Exactword"),	    eb_have_exactword_search,	eb_search_exactword},
    {_("Keyword"),	    eb_have_keyword_search,	ebook_search_keyword},
    {_("Menu"),		    eb_have_menu,		NULL},
    {_("Copyright"),	    eb_have_copyright,		NULL},
};

const gchar* ebook_index_to_method_name(gint index)
{
    if(index < 0)
	return NULL;
    if(index >= SZ(methods))
	return NULL;
    SEARCH_METHOD m = methods[index];
    return m.name;
}

gboolean ebook_has_method(EB_Book *book, gint index)
{
    if(index < 0)
	return False;
    if(index >= SZ(methods))
	return False;
    return methods[index].have_method(book);
}

gboolean ebook_load(BOOK_INFO *binfo)
{
    gint subcount;
    gchar buff[512];
    EB_Error_Code error_code;
    EB_Subbook_Code sublist[EB_MAX_SUBBOOKS];
    if(!binfo)
	return False;
    if(!binfo->path)
        return False;

    binfo->book = (EB_Book*)g_malloc0(sizeof(EB_Book) + 5);
    if(!binfo->book)
        return False;

    eb_initialize_book(binfo->book);
    error_code = eb_bind(binfo->book, binfo->path);
    if(error_code != EB_SUCCESS)
    {
        g_free(binfo->book);
	binfo->book = NULL;
        LOG(LOG_CRITICAL, "Failed to bind the book %s: %s", binfo->path, eb_error_message(error_code));
        return False;
    }
    error_code = eb_subbook_list(binfo->book, sublist, &subcount);
    if(error_code != EB_SUCCESS)
    {
        g_free(binfo->book);
	binfo->book = NULL;
        LOG(LOG_CRITICAL, "Failed to get a subbook list: %s", eb_error_message(error_code));
        return False;
    }
    if(binfo->subbook_no >= subcount)
    {
        g_free(binfo->book);
	binfo->book = NULL;
        return False;
    }
    error_code = eb_subbook_directory2(binfo->book, sublist[binfo->subbook_no], buff);
    if (error_code != EB_SUCCESS)
    {
        g_free(binfo->book);
	binfo->book = NULL;
        LOG(LOG_CRITICAL, "Failed to get the directory: %s", eb_error_message(error_code));
        return False;;
    }
    binfo->subbook_dir = g_strdup(buff);
    error_code = eb_subbook_title2(binfo->book, sublist[binfo->subbook_no], buff);
    if(error_code != EB_SUCCESS)
    {
        g_free(binfo->book);
	binfo->book = NULL;
        LOG(LOG_CRITICAL, "Failed to get the title: %s", eb_error_message(error_code));
        return False;
    }
    binfo->title = iconv_convert(ENC_EUC_JP, ENC_UTF8, buff);
    if(!binfo->title)
    {
        g_free(binfo->book);
	binfo->book = NULL;
        return False;
    }
    error_code = eb_set_subbook(binfo->book, binfo->subbook_no);
    if(error_code != EB_SUCCESS)
    {
        g_free(binfo->book);
	binfo->book = NULL;
        LOG(LOG_CRITICAL, "Failed to get the title: %s", eb_error_message(error_code));
        return False;
    }
    if(eb_have_font(binfo->book, EB_FONT_16))
        eb_set_font(binfo->book, EB_FONT_16);

    return True;
}

void ebook_free(BOOK_INFO *binfo)
{
    if(binfo->title)
        g_free(binfo->title);
    if(binfo->path)
        g_free(binfo->path);
    if(binfo->appendix_path)
        g_free(binfo->appendix_path);
    if(binfo->subbook_dir)
        g_free(binfo->subbook_dir);
    if(binfo->appendix)
        eb_finalize_appendix(binfo->appendix);
    if(binfo->book)
    {
        eb_finalize_book(binfo->book);
        g_free(binfo->book);
    }
    g_free(binfo);
}

gint ebook_guess_gaiji_size(EB_Book *book)
{
    gint size = EB_FONT_16;
    gint font_h = mainwnd_get_font_size();
    if(font_h < 24)
        size = EB_FONT_16;
    else if(font_h < 30)
        size = EB_FONT_24;
    else if(font_h < 48)
        size = EB_FONT_30;
    else
        size = EB_FONT_48;
    return size;
}

void ebook_clear_results(GSequence *results)
{
    GSequenceIter *begin, *end;
    begin = g_sequence_get_begin_iter(results);
    end = g_sequence_get_end_iter(results);
    g_sequence_remove_range(begin, end);
}

gchar *ebook_get_text(RESULT *res)
{
    gchar *p, text[MAX_BUFSIZE + 1];
    ssize_t len;
    EB_Error_Code error_code;

    error_code = eb_seek_text(res->binfo->book, &res->pos);
    if(error_code != EB_SUCCESS)
    {
        LOG(LOG_CRITICAL, "Failed to seek text: %s", eb_error_message(error_code));
        return NULL;
    }

    error_code = eb_read_text(res->binfo->book, NULL, &hooksets.text, NULL, MAX_BUFSIZE, text, &len);
    if (error_code != EB_SUCCESS)
    {
        LOG(LOG_CRITICAL, "Failed to read text: %s", eb_error_message(error_code));
        return NULL;
    }
    text[len] = '\0';
    p = malloc(len + 1);
    g_strlcpy(p, text, len + 1);
    return p;
}

gchar *ebook_get_raw_text(EB_Book *book, gint page, gint offset)
{
    gchar *data;
    ssize_t len;
    EB_Position pos;
    EB_Error_Code error_code;

    data = g_strnfill(EB_SIZE_PAGE, 0);

    pos.page = page;
    pos.offset = offset;
    eb_seek_text(book, &pos);
    error_code = eb_read_rawtext(book, EB_SIZE_PAGE, data, &len);
    if(error_code != EB_SUCCESS || !len)
        return NULL;

    return data;
}

gchar **ebook_bitmap_to_xpm(const gchar *bitmap, gint width, gint height, gchar *color)
{
    gchar **xpm, *xpm_p;
    const unsigned char *bitmap_p = (const unsigned char *)bitmap;
    gint i, j, gaiji = 2;

    xpm = g_new(gchar *, height + 4 + gaiji);
    xpm[0] = g_strdup_printf("%d %d 2 1", width, height + gaiji);
    xpm[1] = g_strdup_printf(" 	c None");
    if(color == NULL)
        xpm[2] = g_strdup_printf(". 	c Black");
    else
        xpm[2] = g_strdup_printf(". 	c %s", color);

    for(i = 0; i < gaiji; i++)
    {
        xpm[i+3] = g_new(gchar, width + 1);
        memset(xpm[i+3], ' ', width);
        xpm[i+3][width] = '\0';
    }

    for(; i < height + 2; i++)
    {
        xpm[i+3] = g_new(gchar, width + 1);
        xpm_p = xpm[i+3];
        for (j = 0; j + 7 < width; j += 8, bitmap_p++)
        {
            *xpm_p++ = (*bitmap_p & 0x80) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x40) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x20) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x10) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x08) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x04) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x02) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x01) ? '.' : ' ';
        }
        if (j < width)
        {
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x80) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x40) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x20) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x10) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x08) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x04) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x02) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x01) ? '.' : ' ';
            bitmap_p++;
        }
        *xpm_p = '\0';
    }
    xpm[i+3] = '\0';

    return xpm;
}

GdkPixbuf* ebook_read_gaiji(BOOK_INFO *binfo, gchar *code, gint *w, gint *h)
{
    EB_Book *book = binfo->book;
    gint char_no = strtol(&code[1], NULL, 16), size;
    gchar bitmap[EB_SIZE_WIDE_FONT_48], **xpm;
    EB_Error_Code ec = EB_SUCCESS;
    GdkPixbuf *pixbuf;

    if((size = ebook_guess_gaiji_size(book)) < 0)
        return NULL;
    ec = eb_set_font(book, size);

    memset(bitmap, 0, EB_SIZE_WIDE_FONT_48*sizeof(gchar));

    ec = eb_font_height(book, h);
    if(code[0] == 'h')
    {
        ec = eb_narrow_font_width(book, w);
        ec = eb_narrow_font_character_bitmap(book, char_no, bitmap);
    }
    else
    {
        ec = eb_wide_font_width(book, w);
        ec = eb_wide_font_character_bitmap(book, char_no, bitmap);
    }
    if(ec != EB_SUCCESS)
        return NULL;

    xpm = ebook_bitmap_to_xpm(bitmap, *w, *h, NULL);

    pixbuf = gdk_pixbuf_new_from_xpm_data((const char**)xpm);
    return pixbuf;
}

static void ebook_search_results_filter(gpointer data, gpointer user_data)
{
    RESULT **res = (RESULT**)user_data;
    if(result_compare((RESULT*)data, *res))
    {
	result_free(*res);
	*res = NULL;
    }
}

static gint ebook_search_results_save(BOOK_INFO *binfo, const gchar *word, gint maxhits)
{
    gint i, hits_count;
    gchar heading[MAX_BUFSIZE];
    ssize_t len;
    EB_Hit *hits = (EB_Hit*)g_new0(EB_Hit, maxhits);
    EB_Error_Code error_code;
    RESULT *res;

    error_code = eb_hit_list(binfo->book, maxhits, hits, &hits_count);
    if(error_code != EB_SUCCESS)
    {
        g_free(hits);
        return error_code;
    }
    for(i = 0; (i < maxhits) && (i < hits_count); i++)
    {
	res = (RESULT*)g_new0(RESULT, 1);
	if(!res)
	{
	    LOG(LOG_ERROR, "No memory");
	    break;
	}
	error_code = eb_seek_text(binfo->book, &(hits[i].heading));
	if(error_code != EB_SUCCESS)
	{
	    g_free(res);
	    continue;
	}
	error_code = eb_read_heading(binfo->book, binfo->appendix, &hooksets.heading, NULL, MAX_BUFSIZE, heading, &len);
	if(error_code != EB_SUCCESS)
	{
	    g_free(res);
	    break;
	}
	heading[len] = '\0';

	res->heading = iconv_convert(ENC_EUC_JP, ENC_UTF8, heading);
	res->binfo = binfo;
	res->pos = hits[i].text;
	g_sequence_foreach(ebook.results, ebook_search_results_filter, &res);
	if(res)
	    g_sequence_append(ebook.results, res);
    }
    g_free(hits);
    return error_code;
}

EB_Error_Code ebook_search_keyword(EB_Book *book, const char *input_word)
{
    gchar **words = g_strsplit_set(input_word, " \n\r\t", 0);
    EB_Error_Code error_code = eb_search_keyword(book, (const gchar* const*)words);
    g_strfreev(words);
    return error_code;
}

gboolean ebook_search_book(const gchar *word, gint method, GSequence *results, gint maxhits, BOOK_INFO *binfo)
{
    if(!binfo)
	return False;
    if(!binfo->book)
	return False;
    if(!methods[method].have_method(binfo->book))
	return False;
    gchar *euc_str = NULL;
    if(word)
    {
	euc_str = iconv_convert(ENC_UTF8, ENC_EUC_JP, word);
	if(!validate_euc(euc_str, False))
	{
	    g_free(euc_str);
	    return False;
	}
    }
    if(results)
    {
        ebook_clear_results(results);
        ebook.results = results;
    }
    if((method == SEARCH_METHOD_MENU) || (method == SEARCH_METHOD_COPYRIGHT))
    {
	EB_Position pos;
	EB_Error_Code err;
	gboolean menu = (method == SEARCH_METHOD_MENU);
	if(menu)
	    err = eb_menu(binfo->book, &pos);
	else
	    err = eb_copyright(binfo->book, &pos);
	if(err == EB_SUCCESS)
	{
	    RESULT *res = (RESULT *)calloc(sizeof(RESULT), 1);
	    res->heading = g_strdup_printf("%s : %s", menu ? _("menu") : _("copyright"), binfo->title);
	    res->binfo = binfo;
	    res->pos = pos;
	    g_sequence_append(ebook.results, res);
	}
    }
    else
    {
	if(methods[method].search(binfo->book, euc_str) == EB_SUCCESS)
	{
	    ebook_search_results_save(binfo, euc_str, maxhits);
	}
    }
    g_free(euc_str);
    return True;
}

gboolean ebook_search(const gchar *word, gint method, GSequence *results, gint maxhits, gboolean max_per_dict)
{
    GtkTreeIter iter;
    GtkTreeModel *store;
    BOOK_INFO *binfo;

    ebook_clear_results(results);

    if(!mainwnd_get_dicts(&store, &iter))
        return False;

    ebook.results = results;
    gboolean ret = False;
    do {
        gtk_tree_model_get(store, &iter, DICT_BINFO, &binfo, -1);
	if(!binfo)
	    continue;
        if(!binfo->active)
            continue;
        ret = ret | ebook_search_book(word, method, NULL, maxhits, binfo);
	if(!max_per_dict)
	    maxhits -= g_sequence_get_length(ebook.results);
	if(maxhits <= 0)
	    break;
    } while (gtk_tree_model_iter_next(store, &iter));
    ebook.results = NULL;
    return ret;
}

GdkPixbuf *ebook_load_image(BOOK_INFO *binfo, EB_Position *pos)
{
    EB_Book *book = binfo->book;
    gchar *data = NULL, page[EB_SIZE_PAGE]; 
    ssize_t len = 0, len1 = 0;
    GError *err = NULL;
    GdkPixbuf *pixbuf = NULL;
    GInputStream *ism = 0;
    EB_Error_Code error_code = eb_set_binary_color_graphic(book, pos);
    if(error_code != EB_SUCCESS)
    {
        LOG(LOG_CRITICAL, "Failed to set binary color graphic: %s", eb_error_message(error_code));
        return NULL;
    }
    for(;;)
    {
        error_code = eb_read_binary(book, EB_SIZE_PAGE, page, &len1);
        if(error_code != EB_SUCCESS || !len1)
            break;
        data = g_renew(gchar, data, len + len1 + 1);
        g_memmove(&(data[len]), page, len1);
        len += len1;
        data[len] = 0;
    }
    if(!data)
        return NULL;
    ism = g_memory_input_stream_new_from_data(data, len, g_free);
    pixbuf = gdk_pixbuf_new_from_stream(ism, NULL, &err);
    if(g_input_stream_close(ism, NULL, &err))
        g_free(data);
    return pixbuf;
}

