
#include "defs.h"
#include "dump.h"
#include "ebook.h"
#include "history.h"
#include "jcode.h"
#include "mainwnd.h"

G_DEFINE_TYPE(Dump, dump, G_TYPE_OBJECT);

static void dump_set_property(GObject *object, guint param_id, const GValue *value, GParamSpec *pspec)
{
    Dump *dump = DUMP(object);
    switch(param_id)
    {
	case 1:
	    dump->builder = BUILDER(g_value_get_pointer(value));
	    GtkWidget *dumphex = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(dump->builder), "dumphex"));
	    GtkWidget *dumptext = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(dump->builder), "dumptext"));
	    gtk_widget_hide_on_delete(dumphex);
	    gtk_widget_hide_on_delete(dumptext);
	    break;
	case 2:
	    dump->history = HISTORY(g_value_get_pointer(value));
	    break;
	default:
	    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, param_id, pspec);
	    break;
    }
}

static void dump_class_init(DumpClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    gobject_class->set_property = dump_set_property;
    g_object_class_install_property(gobject_class, 1, g_param_spec_pointer("builder", _("Builder"), _("Builder"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
    g_object_class_install_property(gobject_class, 2, g_param_spec_pointer("history", _("history"), _("history"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}

static void dump_init(Dump *self)
{
}

static void dump_hex_page(Dump *self)
{
    gint i, page;
    const gchar *p;
    gchar *text, *tmp, *p_hex = NULL, *p_char = NULL, hex_buff[512], char_buff[512];
    GtkTextIter iter;
    GtkTextTag  *tag;
    RESULT *res = history_last_result(self->history);

    if(!res)
	return;

    p = builder_get_str(self->builder, "dumphex_page");
    if(!strlen(p))
	return;

    page = strtol(p, NULL, 16);

    text = ebook_get_raw_text(res->binfo->book, page, 0);
    if(!text)
	return;

    GtkTextView *view = GTK_TEXT_VIEW(gtk_builder_get_object(GTK_BUILDER(self->builder), "dumphex_text"));
    textview_clear_textbuf(view);
    GtkTextBuffer *buf = gtk_text_view_get_buffer(view);

    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(view), GTK_WRAP_NONE);
    tag = gtk_text_buffer_create_tag(buf, NULL, "family", "Monospace", NULL);
    gtk_text_buffer_get_start_iter(buf, &iter);
    for(i = 0;  i < EB_SIZE_PAGE; i = i + 2)
    {
	if((i % 16) == 0)
	{
	    p_hex = hex_buff;
	    p_char = char_buff;
	    memset(p_hex, 0, 512);
	    memset(p_char, 0, 512);
	    sprintf(p_hex, "0x%02x ",(i / 16));
	    p_hex += 5;
	    sprintf(p_hex, "(0x%08x)  ",(page - 1) * EB_SIZE_PAGE + i);
	    p_hex += 14;
	    sprintf(p_char, " ");
	    p_char += 1;
	}
	sprintf(p_hex, "%02x ",(unsigned char)text[i]);
	p_hex += 3;
	sprintf(p_hex, "%02x ",(unsigned char)text[i+1]);
	p_hex += 3;
	if(is_jis_printable((guchar*)&text[i]))
	{
	    *p_char = text[i] + 0x80;
	    p_char++;
	    *p_char = text[i + 1] + 0x80;
	    p_char++;
	}
	else
	{
	    *p_char = 0xA1;
	    p_char++;
	    *p_char = 0xA5;
	    p_char++;
	}
	if((i % 16) == 14)
	{
	    gtk_text_buffer_insert_with_tags(buf, &iter, hex_buff, -1, tag, NULL);
	    tmp = iconv_convert(ENC_EUC_JP, ENC_UTF8, char_buff);
	    if(tmp)
	    {
		gtk_text_buffer_insert_with_tags(buf, &iter, tmp, -1, tag, NULL);
		g_free(tmp);
	    }
	    gtk_text_buffer_insert_with_tags(buf, &iter, "\r\0", -1, tag, NULL);
	}
    }
    g_free(text);
}

static void dump_text_page(Dump *self)
{
    gchar *text, *utf_text;
    GtkTextIter iter;

    RESULT *res = history_last_result(self->history);
    if(!res)
	return;
    text = ebook_get_text(res);
    if(!text)
	return;

    GtkTextView *view = GTK_TEXT_VIEW(gtk_builder_get_object(GTK_BUILDER(self->builder), "dumptext_text"));
    textview_clear_textbuf(view);
    GtkTextBuffer *buf = gtk_text_view_get_buffer(view);

    gtk_text_buffer_get_start_iter(buf, &iter);
    utf_text = iconv_convert(ENC_EUC_JP, ENC_UTF8, text);
    gtk_text_buffer_insert(buf, &iter, utf_text, -1);
    g_free(text);
    g_free(utf_text);
}

void dump_hex(Dump *self)
{
    RESULT *res = history_last_result(self->history);
    if(!res)
	return;

    gchar buff[32];
    sprintf(buff, "0x%.8X", res->pos.page);
    builder_set_str(self->builder, "dumphex_page", buff);

    dump_hex_page(self);
}

void dump_text(Dump *self)
{
    RESULT *res = history_last_result(self->history);
    if(!res)
	return;

    gchar buff[32];
    sprintf(buff, "0x%.8X", res->pos.page);
    builder_set_str(self->builder, "dumptext_page", buff);
    sprintf(buff, "0x%.8X", res->pos.offset);
    builder_set_str(self->builder, "dumptext_offset", buff);

    dump_text_page(self);
}

void dump_hex_cb(GtkWidget *w, gpointer data)
{
    Dump *d = DUMP(data);
    dump_dlg_show(d, DUMP_HEX, !dump_dlg_visible(d, DUMP_HEX));
}

void dump_text_cb(GtkWidget *w, gpointer data)
{
    Dump *d = DUMP(data);
    dump_dlg_show(d, DUMP_TEXT, !dump_dlg_visible(d, DUMP_TEXT));
}

void dump_update(Dump *self)
{
    if(dump_dlg_visible(self, DUMP_HEX));
	dump_hex(self);
    if(dump_dlg_visible(self, DUMP_TEXT));
	dump_text(self);
}

gboolean dump_dlg_visible(Dump *self, gint n)
{
    if(n == DUMP_HEX)
	return gtk_widget_get_visible(GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(self->builder), "dumphex")));
    if(n == DUMP_TEXT)
	return gtk_widget_get_visible(GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(self->builder), "dumptext")));
    return False;
}

gboolean dump_dlg_show(Dump *self, gint n, gboolean show)
{
    if(n == DUMP_HEX)
    {
	GtkWidget *dlg = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(self->builder), "dumphex"));
	if(show)
	{
	    dump_hex(self);
	    gtk_widget_show_all(dlg);
	}
	else
	    gtk_widget_hide(dlg);
    }
    if(n == DUMP_TEXT)
    {
	GtkWidget *dlg = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(self->builder), "dumptext"));
	if(show)
	{
	    dump_text(self);
	    gtk_widget_show_all(dlg);
	}
	else
	    gtk_widget_hide(dlg);
    }
    return False;
}

gboolean dumphex_delete_event_cb(GtkWidget *w, GdkEvent *event, gpointer user_data)
{
    gtk_widget_hide(w);
    return TRUE;
}

gboolean dumptext_delete_event_cb(GtkWidget *w, GdkEvent *event, gpointer user_data)
{
    gtk_widget_hide(w);
    return TRUE;
}

void dumphex_close_btn_clicked_cb(GtkWidget *w, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    dump_dlg_show(mw->dump, DUMP_HEX, False);
}

void dumptext_close_btn_clicked_cb(GtkWidget *w, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    dump_dlg_show(mw->dump, DUMP_TEXT, False);
}

static void dump_hex_page_change(Dump *self, gboolean next)
{
    gint page;
    const gchar *p;
    gchar buff[32];

    p = builder_get_str(self->builder, "dumphex_page");
    page = strtol(p, NULL, 16);
    if(!page)
	return;
    page += next ? 1 : -1;

    sprintf(buff, "0x%.8X", page);
    builder_set_str(self->builder, "dumphex_page", buff);
    dump_hex_page(self);
}

void dumphex_prev_btn_clicked_cb(GtkWidget *w, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    dump_hex_page_change(mw->dump, False);
}

void dumphex_next_btn_clicked_cb(GtkWidget *w, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    dump_hex_page_change(mw->dump, True);
}

