
#include "defs.h"

#include "dicts.h"
#include "eb123.h"
#include "ebook.h"
#include "jcode.h"
#include "mainwnd.h"
#include "popupwnd.h"
#include "prefs.h"

G_DEFINE_TYPE(Dicts, dicts, G_TYPE_OBJECT);

static gint dicts_selection_depth = -1;

static void dicts_set_property(GObject *object, guint param_id, const GValue *value, GParamSpec *pspec)
{
    Dicts *dicts = DICTS(object);
    switch(param_id)
    {
	case 1:
	    dicts->builder = BUILDER(g_value_get_pointer(value));
	    dicts->store = GTK_TREE_STORE(gtk_builder_get_object(GTK_BUILDER(dicts->builder), "dicts_store"));
	    break;
	default:
	    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, param_id, pspec);
	    break;
    }
}

static void dicts_class_init(DictsClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    gobject_class->set_property = dicts_set_property;
    g_object_class_install_property(gobject_class, 1, g_param_spec_pointer("builder", _("Builder"), _("Builder"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}

static void dicts_init(Dicts *self)
{
}

static void dicts_cell_edited_cb(GtkCellRendererText *cell, const gchar *path_string, const gchar *new_text, gpointer data)
{
    Dicts *dicts = DICTS(data);
    gchar *old_text;
    gint column = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(cell), "column"));
    GtkTreeIter iter;
    GtkTreePath *path = gtk_tree_path_new_from_string(path_string);
    gtk_tree_model_get_iter(GTK_TREE_MODEL(dicts->store), &iter, path);
    gtk_tree_model_get(GTK_TREE_MODEL(dicts->store), &iter, column, &old_text, -1);
    g_free(old_text);
    gtk_tree_store_set(dicts->store, &iter, column, new_text, -1);
    gtk_tree_path_free(path);
}

static gboolean dicts_row_drop_possible(GtkTreeDragDest *drag_dest, GtkTreePath *dest, GtkSelectionData *selection_data)
{
    if(!dest) return FALSE;
    return (gtk_tree_path_get_depth(dest) == dicts_selection_depth);
}

static void dicts_item_move(Dicts *self, gboolean up)
{
    GtkTreeIter iter, iter2;
    GtkTreeSelection *selection;
    GtkTreePath *path, *path_orig;

    GtkTreeView *tree = GTK_TREE_VIEW(gtk_builder_get_object(GTK_BUILDER(self->builder), "dicts_tree"));
    selection = gtk_tree_view_get_selection(tree);
    if(!gtk_tree_selection_get_selected(selection, NULL, &iter))
    {
	LOG(LOG_WARNING, _("No dictionary selected"));
	return;
    }
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(self->store), &iter);
    path_orig = gtk_tree_path_copy(path);

    up ? gtk_tree_path_prev(path) : gtk_tree_path_next(path);
    if(gtk_tree_path_compare(path, path_orig) && gtk_tree_model_get_iter(GTK_TREE_MODEL(self->store), &iter2, path)) 
    {
	gtk_tree_store_swap(self->store, &iter, &iter2);
	gtk_tree_selection_select_iter(selection, &iter);
    }
    else if(gtk_tree_path_get_depth(path) > 1)
    {
	gtk_tree_path_up(path);
	gtk_tree_path_free(path_orig);
	path_orig = gtk_tree_path_copy(path);
	up ? gtk_tree_path_prev(path) : gtk_tree_path_next(path);
	if(gtk_tree_path_compare(path, path_orig))
	{
	    GtkTreeIter parent;
	    if(gtk_tree_model_get_iter(GTK_TREE_MODEL(self->store), &parent, path))
	    {
		if(up)
		    gtk_tree_store_append(self->store, &iter2, &parent);
		else
		    gtk_tree_store_prepend(self->store, &iter2, &parent);

		gchar *title;
		gboolean editable;
		BOOK_INFO *binfo;
		gtk_tree_model_get(GTK_TREE_MODEL(self->store), 
			&iter, 
			DICT_ALIAS, &title,
			DICT_BINFO, &binfo,
			DICT_EDITABLE, &editable,
			-1);
		gtk_tree_store_set(self->store, &iter2,
			DICT_ALIAS, title,
			DICT_BINFO, binfo,
			DICT_EDITABLE, editable,
			-1);
		g_free(title);

		gtk_tree_store_remove(self->store, &iter);
		gtk_tree_view_expand_row(tree, path, TRUE);
		gtk_tree_selection_select_iter(gtk_tree_view_get_selection(tree), &iter2);
	    }
	}
    }
    gtk_tree_path_free(path);
    gtk_tree_path_free(path_orig);
}

void dicts_scroll_down(Dicts *self)
{
    while(gtk_events_pending())
	gtk_main_iteration();
    GtkTreeView *tree = GTK_TREE_VIEW(gtk_builder_get_object(GTK_BUILDER(self->builder), "dicts_tree"));
    GtkTreeSelection *sel = gtk_tree_view_get_selection(tree);
    GtkTreeIter iter;
    gtk_tree_selection_get_selected(sel, NULL, &iter);
    GtkTreePath *path = gtk_tree_model_get_path(GTK_TREE_MODEL(self->store), &iter);
    gtk_tree_view_scroll_to_cell(tree, path, NULL, True, .5, .5);
    gtk_tree_path_free(path);
}

void dicts_add_btn_clicked_cb(GtkWidget *w, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    Dicts *self = mw->dicts;
    gchar name[32];
    gint n = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(self->store), NULL);
    GtkTreeIter iter;

    sprintf(name, "Group %d", n + 1);
    gtk_tree_store_append(GTK_TREE_STORE(self->store), &iter, NULL);
    gtk_tree_store_set(GTK_TREE_STORE(self->store), &iter, DICT_ALIAS, name, DICT_EDITABLE, TRUE, -1);
    GtkTreeView *tree = GTK_TREE_VIEW(gtk_builder_get_object(GTK_BUILDER(self->builder), "dicts_tree"));
    GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));
    gtk_tree_selection_select_iter(selection, &iter);
    gtk_widget_grab_focus(GTK_WIDGET(tree));
    dicts_scroll_down(self);
}

static gboolean dicts_drag_data_received_cb(GtkTreeDragDest *drag_dest, GtkTreePath *dest, GtkSelectionData  *selection_data)
{
    return FALSE;
}

void dicts_info_update_search_cb(GtkWidget *w, gpointer data)
{
    EB_Book *book = (EB_Book*)data;
    gint i;
    const gchar *name = NULL;
    for(i = 0; (name = ebook_index_to_method_name(i)); i++)
    {
	if(!g_strcmp0(gtk_label_get_text(GTK_LABEL(w)), name))
	{
	    (book ? ebook_has_method(book, i) : FALSE) ? gtk_widget_show(w) : gtk_widget_hide(w);
	    break;
	}
    }
}

static BOOK_INFO* dicts_load_binfo(const gchar *path, gint subbook_no)
{
    BOOK_INFO *binfo = (BOOK_INFO*)g_new0(BOOK_INFO, 1);
    if(!binfo)
	return NULL;
    binfo->path = g_strdup(path);
    binfo->subbook_no = subbook_no;
    ebook_load(binfo);
    return binfo;
}

static void dicts_search_recursive(Dicts *self, gchar *path, gint depth)
{
    static gboolean group_exists = FALSE;
    static gchar *dirname = NULL;
    static GtkTreeIter iter;
    const gchar *name;
    gchar fullpath[PATH_MAX];
    GDir *dir;
    GtkTreeIter iter1;
    g_strstrip(path);

    if(!(dir = g_dir_open(path, 0, NULL)))
    {
	LOG(LOG_CRITICAL, "Failed to open directory %s", path);
	return;
    }
    if(!depth)
    {
	// remove trailing separators, if needed
	while(strlen(path) > 1)
	{
	    if(!g_strcmp0(&(path[strlen(path) - 1]), G_DIR_SEPARATOR_S))
		path[strlen(path) - 1] = 0;
	    else
		break;
	}
	dirname = g_strrstr(path, G_DIR_SEPARATOR_S);
	dirname = &(dirname[1]);
	if(strlen(dirname))
	    dirname = g_strdup(dirname);
	else
	    dirname = g_strdup(_("Results"));
	group_exists = FALSE;
    }
    GtkTreeView *dicts_tree = GTK_TREE_VIEW(gtk_builder_get_object(GTK_BUILDER(self->builder), "dicts_tree"));
    GtkTreeStore *dicts_store = GTK_TREE_STORE(gtk_builder_get_object(GTK_BUILDER(self->builder), "dicts_store"));
    while((name = g_dir_read_name(dir)))
    {
	sprintf(fullpath, "%s%s%s", path, G_DIR_SEPARATOR_S, name);
	if(g_file_test(fullpath, G_FILE_TEST_IS_DIR))
	{
	    if(depth < builder_get_int(self->builder, "dicts_search_depth"))
		dicts_search_recursive(self, fullpath, depth + 1);
	}
	if(g_file_test(fullpath, G_FILE_TEST_IS_REGULAR))
	{
	    if(!strcasecmp(name, "catalog") || !strcasecmp(name, "catalogs"))
	    {
		gint i;
		BOOK_INFO *binfo;
		GtkTreePath *tree_path;
		for(i = 0;; i++)
		{
		    binfo = dicts_load_binfo(path, i);
		    if(!binfo->book)
		    {
			ebook_free(binfo);
			break;
		    }
		    if(!group_exists)
		    {
			gtk_tree_store_append(dicts_store, &iter, NULL);
			gtk_tree_store_set(dicts_store, &iter, DICT_ALIAS, dirname, DICT_EDITABLE, TRUE, -1);
			GtkTreeSelection *selection = gtk_tree_view_get_selection(dicts_tree);
			gtk_tree_selection_select_iter(selection, &iter);
			gtk_widget_grab_focus(GTK_WIDGET(dicts_tree));
			group_exists = TRUE;
		    }
		    gtk_tree_store_append(dicts_store, &iter1, &iter);
		    gtk_tree_store_set(dicts_store, &iter1, DICT_ALIAS, binfo->title, DICT_BINFO, binfo, DICT_EDITABLE, TRUE, -1);
		    tree_path = gtk_tree_model_get_path(GTK_TREE_MODEL(dicts_store), &iter);
		    gtk_tree_view_expand_row(dicts_tree, tree_path, TRUE);
		    gtk_tree_path_free(tree_path);
		}
	    }
	}
    }
    if(!depth)
    {
	g_free(dirname);
	dirname = NULL;
	group_exists = FALSE;
    }
    g_dir_close(dir);
}

void dicts_browse_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    static gchar *path = NULL;
    path = app_browse_disk("Select directory", GTK_WINDOW(gtk_widget_get_toplevel(widget)), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER, GTK_STOCK_OPEN, path);
    if(path)
    {
	builder_set_str(mw->builder, "dicts_search_path", path);
	builder_grab_focus(mw->builder, "dicts_search_path");
    }
}

void dicts_search_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    Dicts *self = mw->dicts;
    GtkEntry *path = GTK_ENTRY(gtk_builder_get_object(GTK_BUILDER(mw->builder), "dicts_search_path"));
    if(!gtk_entry_get_text_length(path))
	return;
    gchar *dirname = (gchar*)gtk_entry_get_text(path);
    dirname = g_strdup(dirname);
    dicts_search_recursive(self, dirname, 0);
    g_free(dirname);
    gtk_entry_set_text(path, "");
    dicts_scroll_down(self);
}

void dicts_close_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    GtkWidget *wnd = gtk_widget_get_toplevel(widget);
    gtk_widget_hide(wnd);
    dicts_save(mw->dicts);
    mainwnd_dicts_update(mw);
    popupwnd_lookup_resume(mw->popupwnd);
}

gboolean dictionaries_delete_event_cb(GtkWidget *widget, GdkEvent *event, gpointer data)
{
    dicts_close_btn_clicked_cb(widget, data);
    return TRUE;
}

static void dicts_selection_changed_cb(GtkTreeSelection *selection, gpointer data)
{
    gint n;
    GtkTreeIter iter;
    GtkTreePath *path;
    GtkTreeModel *store;
    BOOK_INFO *binfo = NULL;
    Mainwnd *mw = MAINWND(data);

    if(gtk_tree_selection_get_selected(selection, &store, &iter))
    {
	path = gtk_tree_model_get_path(store, &iter);
	dicts_selection_depth = gtk_tree_path_get_depth(path);
	n = gtk_tree_path_get_depth(path);
	gtk_tree_path_free(path);
	if(n == 2)
	    gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, DICT_BINFO, &binfo, -1);
    }
    else
	dicts_selection_depth = -1;
    builder_set_str(mw->builder, "dicts_book_title", binfo ? (binfo->title ? binfo->title : "N/A") : "");
    builder_set_str(mw->builder, "dicts_book_path", binfo ? binfo->path : "");
    builder_set_int(mw->builder, "dicts_subbook_no", binfo ? binfo->subbook_no : -1);
    builder_set_str(mw->builder, "dicts_appendix_path", binfo ? (binfo->appendix_path ? binfo->appendix_path : "") : "");
    builder_set_int(mw->builder, "dicts_appendix_subbook_no", binfo ? binfo->appendix_subbook_no : -1);
    GtkBox *box = GTK_BOX(gtk_builder_get_object(GTK_BUILDER(mw->builder), "dicts_search_methods"));
    gtk_container_foreach(GTK_CONTAINER(box), dicts_info_update_search_cb, binfo ? (gpointer)binfo->book : NULL);
}

void dicts_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    Dicts *self = mw->dicts;
    GtkWidget *wnd = GTK_WIDGET(gtk_builder_get_object(GTK_BUILDER(mw->builder), "dictionaries"));
    gtk_widget_hide_on_delete(wnd);
    GtkTreeIter iter1;
    GtkTreeView *tree = GTK_TREE_VIEW(gtk_builder_get_object(GTK_BUILDER(self->builder), "dicts_tree"));
    GtkTreeSelection *select = gtk_tree_view_get_selection(tree);
    GtkBox *box = GTK_BOX(gtk_builder_get_object(GTK_BUILDER(self->builder), "dicts_search_methods"));
    static gboolean preparewnd = True;
    if(preparewnd)
    {
	GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_insert_column_with_attributes(tree, -1, _("Name"), renderer, "text", DICT_ALIAS, "editable", DICT_EDITABLE, NULL);
	g_signal_connect(G_OBJECT(renderer), "edited", G_CALLBACK(dicts_cell_edited_cb), self);
	gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
	g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(dicts_selection_changed_cb), data);
	gint i;
	const gchar *name;
	for(i = 0; (name = ebook_index_to_method_name(i)); i++)
	    gtk_box_pack_start(box, gtk_label_new(name), False, False, 2);

	GtkTreeDragDestIface *iface = GTK_TREE_DRAG_DEST_GET_IFACE(self->store);
	iface->row_drop_possible = dicts_row_drop_possible;
	g_signal_connect(G_OBJECT(tree), "drag_data_received", G_CALLBACK(dicts_drag_data_received_cb), self);
	gtk_tree_view_set_reorderable(tree, TRUE);
	preparewnd = False;
    }
    if(gtk_tree_model_get_iter_first(GTK_TREE_MODEL(self->store), &iter1))
    {   
        gtk_tree_selection_select_iter(select, &iter1);
    }
    builder_set_int(self->builder, "dicts_search_depth", 1);
    builder_set_str(self->builder, "dicts_search_path", "");
    builder_grab_focus(self->builder, "dicts_search_path");
    popupwnd_lookup_suspend(mw->popupwnd);
    gtk_window_set_transient_for(GTK_WINDOW(wnd), GTK_WINDOW(mw));
    gtk_widget_show_all(wnd);
    gtk_container_foreach(GTK_CONTAINER(box), dicts_info_update_search_cb, NULL);
}

gboolean dicts_save_item(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
    guchar *title;
    gint n = gtk_tree_path_get_depth(path);
    static xmlNodePtr group;
    if(n == 1)
    {
	xmlNodePtr node = (xmlNodePtr)data;
	gtk_tree_model_get(model, iter, DICT_ALIAS, &title, -1);
	group = xmlAddChild(node, xmlNewNode(NULL, (xmlChar*)"group"));
	xmlNewProp(group, (xmlChar*)"name", title);
	g_free(title);
    }
    else if(n == 2)
    {
	gchar buf[16];
	xmlNodePtr node = xmlAddChild(group, xmlNewNode(NULL, (xmlChar*)"dict"));
	BOOK_INFO *binfo;
	GtkTreeStore *dicts_store = GTK_TREE_STORE(model);
	gtk_tree_model_get(GTK_TREE_MODEL(dicts_store), iter, DICT_ALIAS, &title, DICT_BINFO, &binfo, -1);
	xmlNewProp(node, (xmlChar*)"name", (xmlChar*)title);
	xmlNewProp(node, (xmlChar*)"path", (xmlChar*)binfo->path);
	sprintf(buf, "%d", binfo->subbook_no);
	xmlNewProp(node, (xmlChar*)"subbook", (xmlChar*)buf);
	xmlNewProp(node, (xmlChar*)"appendix_path", (xmlChar*)(binfo->appendix_path ? binfo->appendix_path : " "));
	sprintf(buf, "%d", binfo->appendix_subbook_no);
	xmlNewProp(node, (xmlChar*)"appendix_subbook", (xmlChar*)buf);
	sprintf(buf, "%d", binfo->active ? 1 : 0);
	xmlNewProp(node, (xmlChar*)"active", (xmlChar*)buf);
	g_free(title);
    }
    return FALSE;
}

void dicts_save(Dicts *self)
{
    gchar filename[PATH_MAX], buf[16];
    LOG(LOG_INFO, "dicts_save");
    xmlDocPtr doc = xmlNewDoc((xmlChar*)"1.0");
    doc->children = xmlNewDocRawNode(doc, NULL, (xmlChar*)"Dictionaries", NULL);
    sprintf(buf, "%d", builder_get_int(self->builder, "mainwnd_dict_group"));
    xmlNewProp(doc->children, (xmlChar*)"active", (xmlChar*)buf);
    const gchar *userdir = prefs_get_userdir();
    sprintf(filename, "%s%s%s", userdir, G_DIR_SEPARATOR_S, FILENAME_DICTS);
    GtkTreeStore *dicts_store = GTK_TREE_STORE(gtk_builder_get_object(GTK_BUILDER(self->builder), "dicts_store"));
    gtk_tree_model_foreach(GTK_TREE_MODEL(dicts_store), dicts_save_item, doc->children);
    xmlSaveFormatFileEnc(filename, doc, "utf8", 0);
    xmlFreeDoc(doc);
}

BOOK_INFO* dicts_load_dict(gchar **data)
{
    gboolean active = FALSE;
    gchar *book_path = NULL, *appendix_path = NULL;
    gint subbook_no = 0, appendix_subbook_no = 0, i;
    BOOK_INFO *binfo;
    for(i = 1; i < g_strv_length(data); i++)
    {
	if(!g_strcmp0(data[i - 1], "active"))
	    active = atoi(data[i]);
	else if(!g_strcmp0(data[i - 1], "path") ? strlen(data[i]): FALSE)
	    book_path = data[i];
	else if(!g_strcmp0(data[i - 1], "appendix_path") ? strlen(data[i]) : FALSE)
	    appendix_path = data[i];
	else if(!g_strcmp0(data[i - 1], "subbook"))
	    subbook_no = atoi(data[i]);
	else if(!g_strcmp0(data[i - 1], "appendix_subbook"))
	    appendix_subbook_no = atoi(data[i]);
    }
    if(!book_path)
	return NULL;
    binfo = dicts_load_binfo(book_path, subbook_no);
    if(!binfo)
	return NULL;
    binfo->active = active;
    if(appendix_path)
	binfo->appendix_path = g_strdup(appendix_path);
    binfo->appendix_subbook_no = appendix_subbook_no;
    return binfo;
}

void dicts_startElement(void *ctx, const xmlChar *name, const xmlChar **atts)
{
    static GtkTreeIter iter;
    xmlParserCtxt *ctxt = (xmlParserCtxt*)ctx;
    xmlSAXHandler *cb = ctxt->sax;
    Dicts *dicts = DICTS(cb->_private);
    if(!g_strcmp0((gchar*)name, "Dictionaries"))
    {
	dicts->active_group = atoi((gchar*)atts[1]);
    }
    GtkTreeStore *dicts_store = GTK_TREE_STORE(gtk_builder_get_object(GTK_BUILDER(dicts->builder), "dicts_store"));
    if(!g_strcmp0((gchar*)name, "group"))
    {
	gtk_tree_store_append(dicts_store, &iter, NULL);
	gtk_tree_store_set(dicts_store, &iter, DICT_ALIAS, g_strdup((gchar*)atts[1]), DICT_EDITABLE, TRUE, -1);
    }
    else if(!g_strcmp0((gchar*)name, "dict"))
    {
	BOOK_INFO *binfo = dicts_load_dict((gchar**)atts);
	//if(binfo)
	{
	    GtkTreeIter child;
	    gtk_tree_store_append(dicts_store, &child, &iter);
	    gtk_tree_store_set(dicts_store, &child, DICT_ALIAS, atts[1], DICT_BINFO, binfo, DICT_EDITABLE, TRUE, -1);
	}
    }
}

gint dicts_load(Dicts *self)
{
    gchar filename[PATH_MAX];
    xmlSAXHandler cb;
    const gchar *userdir = prefs_get_userdir();
    sprintf(filename, "%s%s%s", userdir, G_DIR_SEPARATOR_S, FILENAME_DICTS);
    if(!g_file_test(filename, G_FILE_TEST_IS_REGULAR))
	return 0;
    memset(&cb, 0, sizeof(xmlSAXHandler));
    cb.startElement = &dicts_startElement;
    cb._private = self;
    xmlDocPtr doc = xmlSAXParseFile(&cb, filename, 0);
    xmlFreeDoc(doc);
    return self->active_group;
}

gboolean dicts_get_nth(Dicts *self, gint n, GtkTreeIter *iter)
{
    GtkTreeIter iter1;
    if(!gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(self->store), &iter1, NULL, n))
	return FALSE;
    if(!gtk_tree_model_iter_children(GTK_TREE_MODEL(self->store), iter, &iter1))
	return FALSE;
    return TRUE;
}

void dicts_delete_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    gboolean b;
    gchar *title;
    GtkTreeIter iter, iter2;
    BOOK_INFO *binfo;
    Mainwnd *mw = MAINWND(data);
    Dicts *dicts = mw->dicts;

    GtkTreeView *dicts_tree = GTK_TREE_VIEW(gtk_builder_get_object(GTK_BUILDER(dicts->builder), "dicts_tree"));
    GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(dicts_tree));
    if(!gtk_tree_selection_get_selected(selection, NULL, &iter))
	return;

    GtkTreePath *path = gtk_tree_model_get_path(GTK_TREE_MODEL(dicts->store), &iter);
    b = (gtk_tree_path_get_depth(path) == 1);
    while(b ? gtk_tree_model_iter_children(GTK_TREE_MODEL(dicts->store), &iter2, &iter) : TRUE)
    {
	GtkTreeIter *iter1 = b ? &iter2 : &iter;
	gtk_tree_model_get(GTK_TREE_MODEL(dicts->store), iter1, DICT_ALIAS, &title, DICT_BINFO, &binfo, -1);
	gtk_tree_store_remove(dicts->store, iter1);
	ebook_free(binfo);
	g_free(title);
	if(!b) break;
    }
    if(b)
	gtk_tree_store_remove(GTK_TREE_STORE(dicts->store), &iter);
    gtk_tree_path_free(path);
}

gboolean dicts_tree_key_release_event_cb(GtkWidget *w, GdkEventKey *evt, gpointer data)
{
    if(evt->keyval == GDK_DELETE)
    {
	dicts_delete_btn_clicked_cb(w, data);
    }
    return TRUE;
}

void dicts_up_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    dicts_item_move(mw->dicts, True);
}

void dicts_down_btn_clicked_cb(GtkWidget *widget, gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    dicts_item_move(mw->dicts, False);
}

static void dicts_btn_set_active(GtkWidget *widget, gpointer data)
{
    if(GTK_IS_TOGGLE_TOOL_BUTTON(widget))
        gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(widget), data ? True : False);
}

void dicts_select_all(gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    GtkToolbar *tb = GTK_TOOLBAR(gtk_builder_get_object(GTK_BUILDER(mw->builder), "mainwnd_dicts"));
    gtk_container_foreach(GTK_CONTAINER(tb), dicts_btn_set_active, (gpointer)True);
}

void dicts_unselect_all(gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    GtkToolbar *tb = GTK_TOOLBAR(gtk_builder_get_object(GTK_BUILDER(mw->builder), "mainwnd_dicts"));
    gtk_container_foreach(GTK_CONTAINER(tb), dicts_btn_set_active, (gpointer)False);
}

void dicts_next_group(gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    GtkComboBox *combo = GTK_COMBO_BOX(gtk_builder_get_object(GTK_BUILDER(mw->builder), "mainwnd_dict_group"));
    int n = gtk_combo_box_get_active(combo);
    GtkTreeModel *model = gtk_combo_box_get_model(combo);
    int n1 = gtk_tree_model_iter_n_children(model, NULL) - 1;
    if(n < n1) n += 1;
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo), n);
}

void dicts_prev_group(gpointer data)
{
    Mainwnd *mw = MAINWND(data);
    GtkComboBox *combo = GTK_COMBO_BOX(gtk_builder_get_object(GTK_BUILDER(mw->builder), "mainwnd_dict_group"));
    int n = gtk_combo_box_get_active(combo);
    if(n > 0) n -= 1;
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo), n);
}

