
#include "defs.h"

#include "eb123.h"
#include "ebgstsrc.h"
#include "history.h"
#include "mainwnd.h"
#include "render.h"
#include "audio.h"

G_DEFINE_TYPE(Audio, audio, GTK_TYPE_FRAME);

EB_Error_Code audio_save_wave(RESULT *res, gchar *file)
{
    char binary_data[EB_SIZE_PAGE];
    EB_Error_Code error_code;
    EB_Position end_position;
    ssize_t read_len;
    size_t write_len;
    FILE *fp;

    end_position.page = res->pos.page + (res->size / EB_SIZE_PAGE);
    end_position.offset = res->pos.offset + (res->size % EB_SIZE_PAGE);
    if (EB_SIZE_PAGE <= end_position.offset)
    {
        end_position.offset -= EB_SIZE_PAGE;
        end_position.page++;
    }
    error_code = eb_set_binary_wave(res->binfo->book, &res->pos, &end_position);
    if (error_code != EB_SUCCESS)
    {
        LOG(LOG_CRITICAL, "Failed to set binary wave: %s", eb_error_message(error_code));
        return error_code;
    }

    fp = fopen(file, "wb");
    if(!fp)
    {
        LOG(LOG_CRITICAL, "Failed to open file : %s", file);
        return EB_ERR_BAD_FILE_NAME;
    }

    for(;;)
    {
        error_code = eb_read_binary(res->binfo->book, EB_SIZE_PAGE, binary_data, &read_len);
        if(error_code != EB_SUCCESS || read_len == 0)
        {
            fclose(fp);
            return error_code;
        }

        // If there are extra data (32 bytes) before fmt chunk,remove them.
        if((strncmp("fmt ", &binary_data[44], 4) == 0) && (strncmp("fmt ", &binary_data[12], 4) != 0))
        {
            LOG(LOG_CRITICAL, "Warning: extra header found in WAVE data.");
            write_len = fwrite(binary_data, 12, 1, fp);
            write_len = fwrite(&binary_data[44], read_len - 44, 1, fp);
        }
        else
            write_len = fwrite(binary_data, read_len, 1, fp);
    }
    (void)write_len;
    return EB_SUCCESS;
}

void audio_save_as(GtkWidget *w, gpointer data)
{       
    static gchar *path = NULL;
    path = app_browse_disk(_("Save Audio As"), mainwnd_get_wnd(), GTK_FILE_CHOOSER_ACTION_SAVE, GTK_STOCK_SAVE_AS, path);
    if(path)
    {
        RESULT *res = (RESULT*)data;
        if(audio_save_wave(res, path) != EB_SUCCESS)
            LOG(LOG_WARNING, _("Failed to save audio"));
    }

}

gboolean audio_context_menu(GtkWidget *w, GdkEventButton *event, gpointer user_data)
{
    if(event->button != 3) return FALSE;
    GtkWidget *menu = gtk_menu_new(), *item;
    item = gtk_image_menu_item_new_from_stock(GTK_STOCK_SAVE_AS, NULL);
    g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(audio_save_as), user_data);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    gtk_widget_show_all(menu);
    gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, gtk_get_current_event_time());
    return TRUE;
}

#ifdef ENABLE_GSTREAMER

gboolean audio_bus_cb(GstBus *bus, GstMessage *msg, gpointer data)
{
    GstElement *pipeline = GST_ELEMENT(data);
    switch (GST_MESSAGE_TYPE (msg))
    {
	case GST_MESSAGE_EOS:
	{
	    gst_element_set_state(pipeline, GST_STATE_NULL);
	    gst_object_unref(GST_OBJECT (pipeline));
	    break;
	}

	case GST_MESSAGE_ERROR:
	{
	    gchar  *debug;
	    GError *error;

	    gst_message_parse_error(msg, &error, &debug);
	    g_free(debug);

	    LOG(LOG_INFO, "%s\n", error->message);
	    g_error_free(error);

	    break;
	}
	 default:
	    break;
    }

    return TRUE;
}

static void audio_pad_added_cb(GstElement *dec, GstPad *pad, gpointer data)
{
    GstElement *sink = (GstElement*)data;
    GstPad *sinkpad = gst_element_get_pad(sink, "sink");

    gst_pad_link(pad, sinkpad);
    gst_object_unref(sinkpad);
}

void audio_play(GtkWidget *w, gpointer data)
{
    GstElement *src, *dec, *sink, *pipeline = gst_pipeline_new ("pipeline");
    GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE (pipeline));
    RESULT *res = (RESULT*)data;

    gst_bus_add_watch(bus, audio_bus_cb, (gpointer)pipeline);
    gst_object_unref(bus);

    src = g_object_new(EB_GST_TYPE_SRC, NULL);
    EB_GST_SRC(src)->res = res;
    dec = gst_element_factory_make("wavparse", "decoder");
    sink = gst_element_factory_make("alsasink", "sink");

    gst_bin_add_many(GST_BIN(pipeline), src, dec, sink, NULL);
    gst_element_link(src, dec);
    g_signal_connect(dec, "pad-added", G_CALLBACK(audio_pad_added_cb), sink);

    gst_element_set_state(pipeline, GST_STATE_PLAYING);
}

#endif

static void audio_set_property(GObject *object, guint param_id, const GValue *value, GParamSpec *pspec)
{
    Audio *self = AUDIO(object);
    RESULT *res;
    switch(param_id)
    {
        case 1:
            res = g_value_get_pointer(value);
	    g_signal_connect(G_OBJECT(self->save_btn), "clicked", G_CALLBACK(audio_save_as), res);
#ifdef ENABLE_GSTREAMER
	    if(self->play_btn)
		g_signal_connect(G_OBJECT(self->play_btn), "clicked", G_CALLBACK(audio_play), res);
#endif
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID(object, param_id, pspec);
            break;
    }
}

static void audio_class_init(AudioClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    gobject_class->set_property = audio_set_property;
    g_object_class_install_property(gobject_class, 1, g_param_spec_pointer("link", _("Link"), _("Link"), G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}

static void audio_init(Audio *self)
{
    self->play_btn = NULL;
    GtkWidget *img;
    GtkWidget *hbox = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 0);
    gtk_container_add(GTK_CONTAINER(self), hbox);
#ifdef ENABLE_GSTREAMER
    self->play_btn = gtk_button_new();
    img = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_button_set_image(GTK_BUTTON(self->play_btn), img);
    gtk_box_pack_start(GTK_BOX(hbox), self->play_btn, FALSE, TRUE, 0);
#else
    GtkWidget *label = gtk_label_new(_("Audio"));
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
#endif
    self->save_btn = gtk_button_new();
    img = gtk_image_new_from_stock(GTK_STOCK_SAVE_AS, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_container_add(GTK_CONTAINER(self->save_btn), img);
    gtk_box_pack_start(GTK_BOX(hbox), self->save_btn, FALSE, TRUE, 0);
}

void audio_render(Audio *self, RenderTextCtx *ctx)
{
    GtkTextIter iter1, iter2;
    if(render_get_last_mark(ctx, "wave", &iter1, &iter2))
    {
        gtk_text_buffer_delete(ctx->buf, &iter1, &iter2);
        gtk_text_buffer_get_end_iter(ctx->buf, &iter1);
        gtk_text_buffer_insert(ctx->buf, &iter1, "\n", -1);
        gtk_text_buffer_get_end_iter(ctx->buf, &iter1);
        GtkTextChildAnchor *anchor = gtk_text_buffer_create_child_anchor(ctx->buf, &iter1);
        gtk_text_view_add_child_at_anchor(GTK_TEXT_VIEW(ctx->view), GTK_WIDGET(self), anchor);
        gtk_text_buffer_insert(ctx->buf, &iter1, "\n", -1);
        g_object_set_data_full(G_OBJECT(self), "audio", ctx->link, result_free);
	gtk_widget_show_all(GTK_WIDGET(self));
	gtk_widget_realize(GTK_WIDGET(self));
    }
}

