/*
 *  CUnit - A Unit testing framework library for C.
 *  Copyright (C) 2001  Anil Kumar
 *  Copyright (C) 2004  Anil Kumar, Jerry St.Clair
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 *	Contains Interface for console Run tests.
 *
 *	Created By      : Anil Kumar on ...(in month of Aug 2001)
 *	Last Modified   : 09/Aug/2001
 *	Comment         : Single interface to Console_run_tests
 *	EMail           : aksaharan@yahoo.com
 *
 *	Modified        : 20-Jul-2004 (JDS)
 *	Comment         : New interface, doxygen comments
 *	EMail           : jds2@users.sourceforge.net
 */

/** @file
 * Console interface with interactive output (user interface).
 */
/** @addtogroup Console
 * @{
 */

#ifndef _CUNIT_CONSOLE_H
#define _CUNIT_CONSOLE_H


#ifndef _CUNIT_CUNIT_H
#define _CUNIT_CUNIT_H

#include <string.h>
#include <math.h>


#ifndef _CUNIT_CUERROR_H
#define _CUNIT_CUERROR_H

#include <errno.h>

/*------------------------------------------------------------------------*/
/** CUnit error codes.
 * If codes are added or removed, be sure to make a change to the
 * error messages in CUError.c/get_error_desc().
 * @see CU_set_error()
 * @see CU_get_error()
 * @see CU_get_error_msg()
 */
typedef enum {
  /* basic errors */
  CUE_SUCCESS           = 0,  /**< No error condition. */
  CUE_NOMEMORY          = 1,  /**< Memory allocation failed. */

  /* Test Registry Level Errors */
  CUE_NOREGISTRY        = 10,  /**< Test registry not initialized. */
  CUE_REGISTRY_EXISTS   = 11,  /**< Attempt to CU_set_registry() without CU_cleanup_registry(). */

  /* Test Suite Level Errors */
  CUE_NOSUITE           = 20,  /**< A required CU_pSuite pointer was NULL. */
  CUE_NO_SUITENAME      = 21,  /**< Required CU_Suite name not provided. */
  CUE_SINIT_FAILED      = 22,  /**< Suite initialization failed. */
  CUE_SCLEAN_FAILED     = 23,  /**< Suite cleanup failed. */
  CUE_DUP_SUITE         = 24,  /**< Duplicate suite name not allowed. */

  /* Test Case Level Errors */
  CUE_NOTEST            = 30,  /**< A required CU_pTest pointer was NULL. */
  CUE_NO_TESTNAME       = 31,  /**< Required CU_Test name not provided. */
  CUE_DUP_TEST          = 32,  /**< Duplicate test case name not allowed. */
  CUE_TEST_NOT_IN_SUITE = 33,  /**< Test not registered in specified suite. */

  /* File handling errors */
  CUE_FOPEN_FAILED      = 40,  /**< An error occurred opening a file. */
  CUE_FCLOSE_FAILED     = 41,  /**< An error occurred closing a file. */
  CUE_BAD_FILENAME      = 42,  /**< A bad filename was requested (NULL, empty, nonexistent, etc.). */
  CUE_WRITE_ERROR       = 43,  /**< An error occurred during a write to a file. */
} CU_ErrorCode;

/*------------------------------------------------------------------------*/
/** CUnit error action codes.
 * These are used to set the action desired when an error
 * condition is detected in the CUnit framework.
 * @see CU_set_error_action()
 * @see CU_get_error_action()
 */
typedef enum CU_ErrorAction {
  CUEA_IGNORE,    /**< Runs should be continued when an error condition occurs (if possible). */
  CUEA_FAIL,      /**< Runs should be stopped when an error condition occurs. */
  CUEA_ABORT,     /**< The application should exit() when an error conditions occurs. */
} CU_ErrorAction;

/* Error handling & reporting functions. */

#ifdef __cplusplus
extern "C" {
#endif

CU_ErrorCode   CU_get_error(void);
const char*    CU_get_error_msg(void);
void           CU_set_error_action(CU_ErrorAction action);
CU_ErrorAction CU_get_error_action(void);

#ifdef CUNIT_BUILD_TESTS
void test_cunit_CUError(void);
#endif

/* Internal function - users should not generally call this function */
void  CU_set_error(CU_ErrorCode error);

#ifdef __cplusplus
}
#endif

#ifdef USE_DEPRECATED_CUNIT_NAMES
/** Deprecated (version 1). @deprecated Use CU_get_error_msg(). */
#define get_error() CU_get_error_msg()
#endif  /* USE_DEPRECATED_CUNIT_NAMES */

#endif  /*  _CUNIT_CUERROR_H  */


//#include "TestDB.h"   /* not needed here - included for user convenience */

#ifdef __cplusplus
extern "C" {
#endif

/*  Max string lengths for names (includes terminating NULL. */
/** Maximum length of a test name string. */
#define MAX_TEST_NAME_LENGTH	256
/** Maximim length of a suite name string. */
#define MAX_SUITE_NAME_LENGTH	256

/* Global type Definitions to be used for boolean operators. */
#ifndef BOOL
  /** Boolean type for CUnit use. */
	#define BOOL 	int
#endif

#ifndef FALSE
  /** Boolean FALSE for CUnit use. */
  #define FALSE	0
#endif

#ifndef TRUE
  /** Boolean TRUE for CUnit use. */
	#define TRUE	1
#endif

//#include "TestRun.h"  /* not needed here - include (after BOOL define) for user convenience */

/** Record a pass condition without performing a logical test. */
#define CU_PASS(msg) \
  { CU_assertImplementation(TRUE, __LINE__, ("CU_PASS(" #msg ")"), __FILE__, "", FALSE); }

/** Simple assertion.
 * Reports failure with no other action.
 */
#define Assert(value) \
  { CU_assertImplementation((value), __LINE__, #value, __FILE__, "", FALSE); }

/** Simple assertion.
 * Reports failure and causes test to abort.
 */
#define AssertFatal(value) \
  { CU_assertImplementation((value), __LINE__, #value, __FILE__, "", TRUE); }

/** Simple assertion.
 * Reports failure with no other action.
 */
#define CU_TEST(value) \
  { CU_assertImplementation((value), __LINE__, #value, __FILE__, "", FALSE); }

/** Simple assertion.
 * Reports failure and causes test to abort.
 */
#define CU_TEST_FATAL(value) \
  { CU_assertImplementation((value), __LINE__, #value, __FILE__, "", TRUE); }

/** Record a failure without performing a logical test. */
#define CU_FAIL(msg) \
  { CU_assertImplementation(FALSE, __LINE__, ("CU_FAIL(" #msg ")"), __FILE__, "", FALSE); }

/** Record a failure without performing a logical test, and abort test. */
#define CU_FAIL_FATAL(msg) \
  { CU_assertImplementation(FALSE, __LINE__, ("CU_FAIL_FATAL(" #msg ")"), __FILE__, "", TRUE); }

/** Asserts that value is TRUE.
 * Reports failure with no other action.
 */
#define AssertTrue(value) \
  { CU_assertImplementation((value), __LINE__, ("CU_ASSERT_TRUE(" #value ")"), __FILE__, "", FALSE); }

/** Asserts that value is TRUE.
 * Reports failure and causes test to abort.
 */
#define AssertTrueFatal(value) \
  { CU_assertImplementation((value), __LINE__, ("CU_ASSERT_TRUE_FATAL(" #value ")"), __FILE__, "", TRUE); }

/** Asserts that value is FALSE.
 * Reports failure with no other action.
 */
#define AssertFalse(value) \
  { CU_assertImplementation(!(value), __LINE__, ("CU_ASSERT_FALSE(" #value ")"), __FILE__, "", FALSE); }

/** Asserts that value is FALSE.
 * Reports failure and causes test to abort.
 */
#define AssertFalseFatal(value) \
  { CU_assertImplementation(!(value), __LINE__, ("CU_ASSERT_FALSE_FATAL(" #value ")"), __FILE__, "", TRUE); }

/** Asserts that actual == expected.
 * Reports failure with no other action.
 */
#define AssertEqual( actual, expected ) \
  { CU_assertImplementation(((actual) == (expected)), __LINE__, ("CU_ASSERT_EQUAL(" #actual "," #expected ")"), __FILE__, "", FALSE); }

/** Asserts that actual == expected.
 * Reports failure and causes test to abort.
 */
#define AssertEqualFatal(actual, expected) \
  { CU_assertImplementation(((actual) == (expected)), __LINE__, ("CU_ASSERT_EQUAL_FATAL(" #actual "," #expected ")"), __FILE__, "", TRUE); }

/** Asserts that actual != expected.
 * Reports failure with no other action.
 */
#define AssertNotEqual(actual, expected) \
  { CU_assertImplementation(((actual) != (expected)), __LINE__, ("CU_ASSERT_NOT_EQUAL(" #actual "," #expected ")"), __FILE__, "", FALSE); }

/** Asserts that actual != expected.
 * Reports failure and causes test to abort.
 */
#define AssertNotEqualFatal(actual, expected) \
  { CU_assertImplementation(((actual) != (expected)), __LINE__, ("CU_ASSERT_NOT_EQUAL_FATAL(" #actual "," #expected ")"), __FILE__, "", TRUE); }

/** Asserts that pointers actual == expected.
 * Reports failure with no other action.
 */
#define AssertPtrEqual(actual, expected) \
  { CU_assertImplementation(((void*)(actual) == (void*)(expected)), __LINE__, ("CU_ASSERT_PTR_EQUAL(" #actual "," #expected ")"), __FILE__, "", FALSE); }

/** Asserts that pointers actual == expected.
 * Reports failure and causes test to abort.
 */
#define AssertPtrEqualFatal(actual, expected) \
  { CU_assertImplementation(((void*)(actual) == (void*)(expected)), __LINE__, ("CU_ASSERT_PTR_EQUAL_FATAL(" #actual "," #expected ")"), __FILE__, "", TRUE); }

/** Asserts that pointers actual != expected.
 * Reports failure with no other action.
 */
#define AssertPtrNotEqual(actual, expected) \
  { CU_assertImplementation(((void*)(actual) != (void*)(expected)), __LINE__, ("CU_ASSERT_PTR_NOT_EQUAL(" #actual "," #expected ")"), __FILE__, "", FALSE); }

/** Asserts that pointers actual != expected.
 * Reports failure and causes test to abort.
 */
#define AssertPtrNotEqualFatal(actual, expected) \
  { CU_assertImplementation(((void*)(actual) != (void*)(expected)), __LINE__, ("CU_ASSERT_PTR_NOT_EQUAL_FATAL(" #actual "," #expected ")"), __FILE__, "", TRUE); }

/** Asserts that pointer value is NULL.
 * Reports failure with no other action.
 */
#define AssertPtrNull(value) \
  { CU_assertImplementation((NULL == (void*)(value)), __LINE__, ("CU_ASSERT_PTR_NULL(" #value")"), __FILE__, "", FALSE); }

/** Asserts that pointer value is NULL.
 * Reports failure and causes test to abort.
 */
#define AssertPtrNullFatal(value) \
  { CU_assertImplementation((NULL == (void*)(value)), __LINE__, ("CU_ASSERT_PTR_NULL_FATAL(" #value")"), __FILE__, "", TRUE); }

/** Asserts that pointer value is not NULL.
 * Reports failure with no other action.
 */
#define AssertPtrNotNull(value) \
  { CU_assertImplementation((NULL != (void*)(value)), __LINE__, ("CU_ASSERT_PTR_NOT_NULL(" #value")"), __FILE__, "", FALSE); }

/** Asserts that pointer value is not NULL.
 * Reports failure and causes test to abort.
 */
#define AssertPtrNotNullFatal(value) \
  { CU_assertImplementation((NULL != (void*)(value)), __LINE__, ("CU_ASSERT_PTR_NOT_NULL_FATAL(" #value")"), __FILE__, "", TRUE); }

/** Asserts that string actual == expected.
 * Reports failure with no other action.
 */
#define AssertStringEqual(actual, expected) \
  { CU_assertImplementation(!(strcmp((const char*)(actual), (const char*)(expected))), __LINE__, ("CU_ASSERT_STRING_EQUAL(" #actual ","  #expected ")"), __FILE__, "", FALSE); }

/** Asserts that string actual == expected.
 * Reports failure and causes test to abort.
 */
#define AssertStringEqualFatal(actual, expected) \
  { CU_assertImplementation(!(strcmp((const char*)(actual), (const char*)(expected))), __LINE__, ("CU_ASSERT_STRING_EQUAL_FATAL(" #actual ","  #expected ")"), __FILE__, "", TRUE); }

/** Asserts that string actual != expected.
 * Reports failure with no other action.
 */
#define AssertStringNotEqual(actual, expected) \
  { CU_assertImplementation((strcmp((const char*)(actual), (const char*)(expected))), __LINE__, ("CU_ASSERT_STRING_NOT_EQUAL(" #actual ","  #expected ")"), __FILE__, "", FALSE); }

/** Asserts that string actual != expected.
 * Reports failure and causes test to abort.
 */
#define AssertStringNotEqualFatal(actual, expected) \
  { CU_assertImplementation((strcmp((const char*)(actual), (const char*)(expected))), __LINE__, ("CU_ASSERT_STRING_NOT_EQUAL_FATAL(" #actual ","  #expected ")"), __FILE__, "", TRUE); }

/** Asserts that string actual == expected with length specified.
 * The comparison is limited to count characters.
 * Reports failure with no other action.
 */
#define AssertNstringEqual(actual, expected, count) \
  { CU_assertImplementation(!(strncmp((const char*)(actual), (const char*)(expected), (size_t)(count))), __LINE__, ("CU_ASSERT_NSTRING_EQUAL(" #actual ","  #expected "," #count ")"), __FILE__, "", FALSE); }

/** Asserts that string actual == expected with length specified.
 * The comparison is limited to count characters.
 * Reports failure and causes test to abort.
 */
#define AssertNstringEqualFatal(actual, expected, count) \
  { CU_assertImplementation(!(strncmp((const char*)(actual), (const char*)(expected), (size_t)(count))), __LINE__, ("CU_ASSERT_NSTRING_EQUAL_FATAL(" #actual ","  #expected "," #count ")"), __FILE__, "", TRUE); }

/** Asserts that string actual != expected with length specified.
 * The comparison is limited to count characters.
 * Reports failure with no other action.
 */
#define AssertNstringNotEqual(actual, expected, count) \
  { CU_assertImplementation((strncmp((const char*)(actual), (const char*)(expected), (size_t)(count))), __LINE__, ("CU_ASSERT_NSTRING_NOT_EQUAL(" #actual ","  #expected "," #count ")"), __FILE__, "", FALSE); }

/** Asserts that string actual != expected with length specified.
 * The comparison is limited to count characters.
 * Reports failure and causes test to abort.
 */
#define AssertNstringNotEqualFatal(actual, expected, count) \
  { CU_assertImplementation((strncmp((const char*)(actual), (const char*)(expected), (size_t)(count))), __LINE__, ("CU_ASSERT_NSTRING_NOT_EQUAL_FATAL(" #actual ","  #expected "," #count ")"), __FILE__, "", TRUE); }

/** Asserts that double actual == expected within the specified tolerance.
 * If actual is within granularity of expected, the assertion passes.
 * Reports failure with no other action.
 */
#define AssertDoubleEqual(actual, expected, granularity) \
  { CU_assertImplementation(((fabs((double)(actual) - (expected)) <= fabs((double)(granularity)))), __LINE__, ("CU_ASSERT_DOUBLE_EQUAL(" #actual ","  #expected "," #granularity ")"), __FILE__, "", FALSE); }

/** Asserts that double actual == expected within the specified tolerance.
 * If actual is within granularity of expected, the assertion passes.
 * Reports failure and causes test to abort.
 */
#define AssertDoubleEqualFatal(actual, expected, granularity) \
  { CU_assertImplementation(((fabs((double)(actual) - (expected)) <= fabs((double)(granularity)))), __LINE__, ("CU_ASSERT_DOUBLE_EQUAL_FATAL(" #actual ","  #expected "," #granularity ")"), __FILE__, "", TRUE); }

/** Asserts that double actual != expected within the specified tolerance.
 * If actual is within granularity of expected, the assertion fails.
 * Reports failure with no other action.
 */
#define AssertDoubleNotEqual(actual, expected, granularity) \
  { CU_assertImplementation(((fabs((double)(actual) - (expected)) > fabs((double)(granularity)))), __LINE__, ("CU_ASSERT_DOUBLE_NOT_EQUAL(" #actual ","  #expected "," #granularity ")"), __FILE__, "", FALSE); }

/** Asserts that double actual != expected within the specified tolerance.
 * If actual is within granularity of expected, the assertion fails.
 * Reports failure and causes test to abort.
 */
#define AssertDoubleNotEqualFatal(actual, expected, granularity) \
  { CU_assertImplementation(((fabs((double)(actual) - (expected)) > fabs((double)(granularity)))), __LINE__, ("CU_ASSERT_DOUBLE_NOT_EQUAL_FATAL(" #actual ","  #expected "," #granularity ")"), __FILE__, "", TRUE); }

#ifdef __cplusplus
}
#endif

#ifdef USE_DEPRECATED_CUNIT_NAMES
/** Deprecated (version 1). @deprecated Use CU_ASSERT_FATAL. */
#define ASSERT(value) { if (FALSE == (int)(value)) { CU_assertImplementation((BOOL)value, __LINE__, #value, __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_TRUE_FATAL. */
#define ASSERT_TRUE(value) { if (FALSE == (value)) { CU_assertImplementation(FALSE, __LINE__, ("ASSERT_TRUE(" #value ")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_FALSE_FATAL. */
#define ASSERT_FALSE(value) { if (FALSE != (value)) { CU_assertImplementation(FALSE, __LINE__, ("ASSERT_FALSE(" #value ")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_EQUAL_FATAL. */
#define ASSERT_EQUAL(actual, expected) { if ((actual) != (expected)) { CU_assertImplementation(FALSE, __LINE__, ("ASSERT_EQUAL(" #actual "," #expected ")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_NOT_EQUAL_FATAL. */
#define ASSERT_NOT_EQUAL(actual, expected) { if ((void*)(actual) == (void*)(expected)) { CU_assertImplementation(FALSE, __LINE__, ("ASSERT_NOT_EQUAL(" #actual "," #expected ")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_PTR_EQUAL_FATAL. */
#define ASSERT_PTR_EQUAL(actual, expected) { if ((void*)(actual) != (void*)(expected)) { CU_assertImplementation(FALSE, __LINE__, ("ASSERT_PTR_EQUAL(" #actual "," #expected ")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_PTR_NOT_EQUAL_FATAL. */
#define ASSERT_PTR_NOT_EQUAL(actual, expected) { if ((void*)(actual) == (void*)(expected)) { CU_assertImplementation(FALSE, __LINE__, ("ASSERT_PTR_NOT_EQUAL(" #actual "," #expected ")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_PTR_NULL_FATAL. */
#define ASSERT_PTR_NULL(value)  { if (NULL != (void*)(value)) { CU_assertImplementation(FALSE, __LINE__, ("ASSERT_PTR_NULL(" #value")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_PTR_NOT_NULL_FATAL. */
#define ASSERT_PTR_NOT_NULL(value) { if (NULL == (void*)(value)) { CU_assertImplementation(FALSE, __LINE__, ("ASSERT_PTR_NOT_NULL(" #value")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_STRING_EQUAL_FATAL. */
#define ASSERT_STRING_EQUAL(actual, expected) { if (strcmp((const char*)actual, (const char*)expected)) { CU_assertImplementation(FALSE, __LINE__, ("ASSERT_STRING_EQUAL(" #actual ","  #expected ")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_STRING_NOT_EQUAL_FATAL. */
#define ASSERT_STRING_NOT_EQUAL(actual, expected) { if (!strcmp((const char*)actual, (const char*)expected)) { CU_assertImplementation(TRUE, __LINE__, ("ASSERT_STRING_NOT_EQUAL(" #actual ","  #expected ")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_NSTRING_EQUAL_FATAL. */
#define ASSERT_NSTRING_EQUAL(actual, expected, count) { if (strncmp((const char*)actual, (const char*)expected, (size_t)count)) { CU_assertImplementation(FALSE, __LINE__, ("ASSERT_NSTRING_EQUAL(" #actual ","  #expected "," #count ")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_NSTRING_NOT_EQUAL_FATAL. */
#define ASSERT_NSTRING_NOT_EQUAL(actual, expected, count) { if (!strncmp((const char*)actual, (const char*)expected, (size_t)count)) { CU_assertImplementation(TRUE, __LINE__, ("ASSERT_NSTRING_NOT_EQUAL(" #actual ","  #expected "," #count ")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_DOUBLE_EQUAL_FATAL. */
#define ASSERT_DOUBLE_EQUAL(actual, expected, granularity) { if ((fabs((double)actual - expected) > fabs((double)granularity))) { CU_assertImplementation(FALSE, __LINE__, ("ASSERT_DOUBLE_EQUAL(" #actual ","  #expected "," #granularity ")"), __FILE__, "", FALSE); return; }}
/** Deprecated (version 1). @deprecated Use CU_ASSERT_DOUBLE_NOT_EQUAL_FATAL. */
#define ASSERT_DOUBLE_NOT_EQUAL(actual, expected, granularity) { if ((fabs((double)actual - expected) <= fabs((double)granularity))) { CU_assertImplementation(TRUE, __LINE__, ("ASSERT_DOUBLE_NOT_EQUAL(" #actual ","  #expected "," #granularity ")"), __FILE__, "", FALSE); return; }}
#endif  /* USE_DEPRECATED_CUNIT_NAMES */

#endif  /*  _CUNIT_CUNIT_H  */



#ifndef _CUNIT_TESTDB_H
#define _CUNIT_TESTDB_H

#include <setjmp.h>   /* jmp_buf */

//#include "CUnit.h"
//#include "CUError.h"

#ifdef __cplusplus
extern "C" {
#endif

/*  Type definition for Initialization/Cleaup/TestFunction */
typedef int (*CU_InitializeFunc)(void);   /**< Signature for suite initialization function. */
typedef int (*CU_CleanupFunc)(void);      /**< Signature for suite cleanup function. */
typedef void (*CU_TestFunc)(void);        /**< Signature for a testing function in a test case. */

/** CUnit test case data type.
 * CU_Test is a linked list of unit tests.  Each test
 * has a name and a callable test function, as well as
 * links to the next and previous tests in the list.  A
 * test also holds a jmp_buf reference for use in
 * implementing fatal assertions.
 * <P>
 * Generally, the linked list includes tests which are
 * associated with each other in a CU_Suite.  As a result,
 * tests are run in the order in which they are added to a
 * suite (see CU_add_test()).
 * <P>
 * In the current implementation, the name of each CU_Test
 * in a suite must have a unique name.  There is no
 * restriction on the test function.  This means that the
 * same function could, in principle, be called more than
 * once as long as it is registered with different tests
 * having distinct names.
 * @see CU_Suite
 * @see CU_TestRegistry
 */
typedef struct CU_Test
{
  char*           pName;                  /**< Test name. */
  CU_TestFunc     pTestFunc;              /**< Pointer to the test function. */
  jmp_buf*        pJumpBuf;               /**< Jump buffer for setjmp/longjmp test abort mechanism. */

  struct CU_Test* pNext;                  /**< Pointer to the next test in linked list. */
  struct CU_Test* pPrev;                  /**< Pointer to the previous test in linked list. */

} CU_Test;
typedef CU_Test* CU_pTest;                /**< Pointer to a CUnit test case. */

/** CUnit suite data type.
 * CU_Suite is a linked list of CU_Test containers.
 * Each suite has a name and count of associated unit
 * tests.  It also holds a pointer to optional
 * initialization and cleanup functions.  If non-NULL,
 * these are called before and after running the suite's
 * tests, respectively.  In addition, the suite holds a
 * pointer to the head of the linked list of associated
 * CU_Test objects.  Finally, pointers to the next and
 * previous suites in the linked list are maintained.
 * <P>
 * Generally, the linked list includes suites which are
 * associated with each other in a CU_TestRegistry.  As a
 * result, suites are run in the order in which they are
 * registered (see CU_add_suite()).
 * <P>
 * In the current implementation, the name of each CU_Suite
 * in a test registry must have a unique name.  There is no
 * restriction on the contained tests.  This means that the
 * same CU_Test could, in principle, be run more than
 * once as long as it is registered with different suites
 * having distinct names.
 * @see CU_Test
 * @see CU_TestRegistry
 */
typedef struct CU_Suite
{
  char*             pName;                /**< Suite name. */
  CU_pTest          pTest;                /**< Pointer to the 1st test in the suite. */
  CU_InitializeFunc pInitializeFunc;      /**< Pointer to the suite initialization function. */
  CU_CleanupFunc    pCleanupFunc;         /**< Pointer to the suite cleanup function. */

  unsigned int      uiNumberOfTests;      /**< Number of tests in the suite. */
  struct CU_Suite*  pNext;                /**< Pointer to the next suite in linked list. */
  struct CU_Suite*  pPrev;                /**< Pointer to the previous suite in linked list. */

} CU_Suite;
typedef CU_Suite* CU_pSuite;              /**< Pointer to a CUnit suite. */

/** CUnit test registry data type.
 * CU_TestRegisty is the repository for suites containing
 * unit tests.  The test registry maintains a count of the
 * number of CU_Suite objects contained in the registry, as
 * well as a count of the total number of CU_Test objects
 * associated with those suites.  It also holds a pointer
 * to the head of the linked list of CU_Suite objects.
 * <P>
 * With this structure, the user will normally add suites
 * implictly to the internal test registry using CU_add_suite(),
 * and then add tests to each suite using CU_add_test().
 * Test runs are then initiated using one of the appropriate
 * functions in TestRun.c via one of the interfaces.
 * <P>
 * Automatic creation and destruction of the internal registry 
 * and its objects is available using CU_initialize_registry() 
 * and CU_cleanup_registry(), respectively.  For internal and
 * testing purposes, the internal registry can be retrieved and
 * assigned.  Functions are also provided for creating and
 * destroying independent registries.
 * <P>
 * Note that earlier versions of CUnit also contained a
 * pointer to a linked list of CU_FailureRecord objects
 * (termed _TestResults).  This has been removed from the
 * registry and relocated to TestRun.c.
 * @see CU_Test
 * @see CU_Suite
 * @see CU_initialize_registry()
 * @see CU_cleanup_registry()
 * @see CU_get_registry()
 * @see CU_set_registry()
 * @see CU_create_new_registry()
 * @see CU_destroy_existing_registry()
 */
typedef struct CU_TestRegistry
{
#ifdef USE_DEPRECATED_CUNIT_NAMES
  /** Union to support v1.1-1 member name. */
  union {
    unsigned int uiNumberOfSuites;        /**< Number of suites in the test registry. */
    unsigned int uiNumberOfGroups;        /**< Deprecated (version 1). @deprecated Use uiNumberOfSuites. */
  };
  unsigned int uiNumberOfTests;           /**< Number of tests in the test registry. */
  /** Union to support v1.1-1 member name. */
  union {
    CU_pSuite    pSuite;                  /**< Pointer to the 1st suite in the test registry. */
    CU_pSuite    pGroup;                  /**< Deprecated (version 1). @deprecated Use pSuite. */
  };
#else
  unsigned int uiNumberOfSuites;          /**< Number of suites in the test registry. */
  unsigned int uiNumberOfTests;           /**< Number of tests in the test registry. */
  CU_pSuite    pSuite;                    /**< Pointer to the 1st suite in the test registry. */
#endif
} CU_TestRegistry;
typedef CU_TestRegistry* CU_pTestRegistry;  /**< Pointer to a CUnit test registry. */

/* Public interface functions */
CU_ErrorCode CU_initialize_registry(void);
void         CU_cleanup_registry(void);

CU_pSuite CU_add_suite(const char* strName, CU_InitializeFunc pInit, CU_CleanupFunc pClean);
CU_pTest  CU_add_test(CU_pSuite pSuite, const char* strName, CU_TestFunc pTestFunc);

/** Shortcut macro for adding a test to a suite. */
#define CU_ADD_TEST(suite, test) (CU_add_test(suite, ##test, (CU_TestFunc)test))

/*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*/
/*  This section is based conceptually on code
 *  Copyright (C) 2004  Aurema Pty Ltd.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  Derived from code contributed by K. Cheung and Aurema Pty Ltd. (thanks!)
 *    test_case_t, test_group_t, test_suite_t
 */

/** Test case parameters.
 * This data type is provided to assist CUnit users
 * manage collections of test and suites.  It is
 * intended to be used to build arrays of test case
 * parameters that can be then be referred to in
 * a CU_suite_info_t variable.
 */
typedef struct CU_TestInfo {
	char        *pName;     /**< Test name. */
	CU_TestFunc pTestFunc;  /**< Test function. */
} CU_TestInfo;
typedef CU_TestInfo* CU_pTestInfo;  /**< Pointer to CU_TestInfo type. */

/** Suite parameters.
 * This data type is provided to assist CUnit users
 * manage collections of test and suites.  It is
 * intended to be used to build arrays of suite
 * parameters that can be passed to a bulk registration
 * function such as CU_register_suite() or
 * CU_register_suites().
 */
typedef struct CU_SuiteInfo {
	char              *pName;        /**< Suite name. */
	CU_InitializeFunc pInitFunc;     /**< Suite initialization function. */
	CU_CleanupFunc    pCleanupFunc;  /**< Suite cleanup function */
	CU_TestInfo       *pTests;       /**< Test case array - must be NULL terminated. */
} CU_SuiteInfo;
typedef CU_SuiteInfo* CU_pSuiteInfo;  /**< Pointer to CU_SuiteInfo type. */

/** NULL CU_test_info_t to terminate arrays of tests. */
#define CU_TEST_INFO_NULL { NULL, NULL }
/** NULL CU_suite_info_t to terminate arrays of suites. */
#define CU_SUITE_INFO_NULL { NULL, NULL, NULL, NULL }

CU_ErrorCode CU_register_suites(CU_SuiteInfo suite_info[]);
CU_ErrorCode CU_register_nsuites(int suite_count, ...);

#ifdef USE_DEPRECATED_CUNIT_NAMES
typedef CU_TestInfo test_case_t;    /**< Deprecated (version 1). @deprecated Use CU_TestInfo. */
typedef CU_SuiteInfo test_group_t;  /**< Deprecated (version 1). @deprecated Use CU_SuiteInfo. */

/** Deprecated (version 1). @deprecated Use CU_SuiteInfo and CU_TestInfo. */
typedef struct test_suite {
	char *name;            /**< Suite name.  Currently not used. */
	test_group_t *groups;  /**< Test groups.  This must be a NULL terminated array. */
} test_suite_t;

/** Deprecated (version 1). @deprecated Use CU_TEST_INFO_NULL. */
#define TEST_CASE_NULL { NULL, NULL }
/** Deprecated (version 1). @deprecated Use CU_TEST_GROUP_NULL. */
#define TEST_GROUP_NULL { NULL, NULL, NULL, NULL }

/** Deprecated (version 1). @deprecated Use CU_register_suites(). */
#define test_group_register(tg) CU_register_suites(tg)

/** Deprecated (version 1). @deprecated Use CU_SuiteInfo and CU_register_suites(). */
int test_suite_register(test_suite_t *ts)
{
	test_group_t *tg;
	int error;

	for (tg = ts->groups; tg->pName; tg++)
		if ((error = CU_register_suites(tg)) != CUE_SUCCESS)
			return error;

	return CUE_SUCCESS;
}
#endif    /* USE_DEPRECATED_CUNIT_NAMES */
/*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*/

#ifdef USE_DEPRECATED_CUNIT_NAMES
typedef CU_InitializeFunc InitializeFunc; /**< Deprecated (version 1). @deprecated Use CU_InitializeFunc. */
typedef CU_CleanupFunc CleanupFunc;       /**< Deprecated (version 1). @deprecated Use CU_CleanupFunc. */
typedef CU_TestFunc TestFunc;             /**< Deprecated (version 1). @deprecated Use CU_TestFunc. */

typedef CU_Test _TestCase;                /**< Deprecated (version 1). @deprecated Use CU_Test. */
typedef CU_pTest PTestCase;               /**< Deprecated (version 1). @deprecated Use CU_pTest. */

typedef CU_Suite  _TestGroup;             /**< Deprecated (version 1). @deprecated Use CU_Suite. */
typedef CU_pSuite PTestGroup;             /**< Deprecated (version 1). @deprecated Use CU_pSuite. */

typedef CU_TestRegistry  _TestRegistry;   /**< Deprecated (version 1). @deprecated Use CU_TestRegistry. */
typedef CU_pTestRegistry PTestRegistry;   /**< Deprecated (version 1). @deprecated Use CU_pTestRegistry. */

/* Public interface functions */
/** Deprecated (version 1). @deprecated Use CU_initialize_registry(). */
#define initialize_registry() CU_initialize_registry()
/** Deprecated (version 1). @deprecated Use CU_cleanup_registry(). */
#define cleanup_registry() CU_cleanup_registry()
/** Deprecated (version 1). @deprecated Use CU_add_suite(). */
#define add_test_group(name, init, clean) CU_add_suite(name, init, clean)
/** Deprecated (version 1). @deprecated Use CU_add_test(). */
#define add_test_case(group, name, test) CU_add_test(group, name, test)

/* private internal CUnit testing functions */
/** Deprecated (version 1). @deprecated Use CU_get_registry(). */
#define get_registry() CU_get_registry()
/** Deprecated (version 1). @deprecated Use CU_set_registry(). */
#define set_registry(reg) CU_set_registry((reg))

/** Deprecated (version 1). @deprecated Use CU_get_suite_by_name(). */
#define get_group_by_name(group, reg) CU_get_suite_by_name(group, reg)
/** Deprecated (version 1). @deprecated Use CU_get_test_by_name(). */
#define get_test_by_name(test, group) CU_get_test_by_name(test, group)

/** Deprecated (version 1). @deprecated Use ADD_TEST_TO_SUITE. */
#define ADD_TEST_TO_GROUP(group, test) (CU_add_test(group, ##test, (CU_TestFunc)test))
#endif  /* USE_DEPRECATED_CUNIT_NAMES */

/* Internal CUnit system functions.  Should not be routinely called by users. */
CU_pTestRegistry CU_get_registry(void);
CU_pTestRegistry CU_set_registry(CU_pTestRegistry pTestRegistry);
CU_pTestRegistry CU_create_new_registry(void);
void             CU_destroy_existing_registry(CU_pTestRegistry* ppRegistry);
CU_pSuite        CU_get_suite_by_name(const char* szSuiteName, CU_pTestRegistry pRegistry);
CU_pTest         CU_get_test_by_name(const char* szTestName, CU_pSuite pSuite);

#ifdef CUNIT_BUILD_TESTS
void test_cunit_TestDB(void);
#endif

#ifdef __cplusplus
}
#endif

#endif  /*  _CUNIT_TESTDB_H  */

#ifdef __cplusplus
extern "C" {
#endif

void CU_console_run_tests(void);

#ifdef USE_DEPRECATED_CUNIT_NAMES
/** Deprecated (version 1). @deprecated Use CU_console_run_tests(). */
#define console_run_tests() CU_console_run_tests()
#endif  /* USE_DEPRECATED_CUNIT_NAMES */

#ifdef __cplusplus
}
#endif
#endif  /*  _CUNIT_CONSOLE_H_  */
/** @} */



#ifndef _CUNIT_TESTRUN_H
#define _CUNIT_TESTRUN_H

//#include "console.h"
//#include "CUError.h"
//#include "TestDB.h"

#ifdef __cplusplus
extern "C" {
#endif

/* CU_FailureRecord type definition. */
/** Data type for holding assertion failure information (linked list). */
typedef struct CU_FailureRecord
{
  unsigned int  uiLineNumber;     /**< Line number of failure. */
  char*         strFileName;      /**< Name of file where failure occurred. */
  char*         strCondition;     /**< Test condition which failed. */
  CU_pTest      pTest;            /**< Test containing failure. */
  CU_pSuite     pSuite;           /**< Suite containing test having failure. */

  struct CU_FailureRecord* pNext; /**< Pointer to next record in linked list. */
  struct CU_FailureRecord* pPrev; /**< Pointer to previous record in linked list. */

} CU_FailureRecord;
typedef CU_FailureRecord* CU_pFailureRecord;  /**< Pointer to CU_FailureRecord. */

/* CU_RunSummary type definition. */
/** Data type for holding statistics and assertion failures for a test run. */
typedef struct CU_RunSummary
{
  unsigned int nSuitesRun;        /**< Number of suites completed during run. */
  unsigned int nSuitesFailed;     /**< Number of suites for which initialization failed. */
  unsigned int nTestsRun;         /**< Number of tests completed during run. */
  unsigned int nTestsFailed;      /**< Number of tests containing failed assertions. */
  unsigned int nAsserts;          /**< Number of assertions tested during run. */
  unsigned int nAssertsFailed;    /**< Number of failed assertions. */
  unsigned int nFailureRecords;   /**< Number of failure records generated. */
} CU_RunSummary;
typedef CU_RunSummary* CU_pRunSummary;  /**< Pointer to CU_RunSummary. */

/* Type Definitions for Message Handlers. */
/** Message handler called at the start of a test.
 * The parameters are the test and suite being run.
 * The test run is considered in progress when the
 * message handler is called.
 */
typedef void (*CU_TestStartMessageHandler)(const CU_pTest pTest, const CU_pSuite pSuite);
/** Message handler called at the completion of a test.
 * The parameters are the test and suite being run, plus
 * a pointer to the first failure record applicable to
 * this test.  If the test did not have any assertion
 * failures, pFailure will be NULL.
 * The test run is considered in progress when the
 * message handler is called.
 */
typedef void (*CU_TestCompleteMessageHandler)(const CU_pTest pTest, const CU_pSuite pSuite,
                                              const CU_pFailureRecord pFailure);
/** Message handler called at the completion of a test run.
 * The parameter is a pointer to the linked list holding
 * the failure records for the test run.
 * The test run is considered completed when the
 * message handler is called.
 */
typedef void (*CU_AllTestsCompleteMessageHandler)(const CU_pFailureRecord pFailure);
/** Message handler called when a suite initializer fails.
 * The test run is considered in progress when the
 * message handler is called.
 */
typedef void (*CU_SuiteInitFailureMessageHandler)(const CU_pSuite pSuite);

/* Get/Set functions for Message Handlers. */
void CU_set_test_start_handler(CU_TestStartMessageHandler pTestStartMessage);
void CU_set_test_complete_handler(CU_TestCompleteMessageHandler pTestCompleteMessage);
void CU_set_all_test_complete_handler(CU_AllTestsCompleteMessageHandler pAllTestsCompleteMessage);
void CU_set_suite_init_failure_handler(CU_SuiteInitFailureMessageHandler pSuiteInitFailureMessage);

CU_TestStartMessageHandler        CU_get_test_start_handler(void);
CU_TestCompleteMessageHandler     CU_get_test_complete_handler(void);
CU_AllTestsCompleteMessageHandler CU_get_all_test_complete_handler(void);
CU_SuiteInitFailureMessageHandler CU_get_suite_init_failure_handler(void);

/* Functions for running registered tests and suites. */
CU_ErrorCode CU_run_all_tests(void);
CU_ErrorCode CU_run_suite(CU_pSuite pSuite);
CU_ErrorCode CU_run_test(CU_pSuite pSuite, CU_pTest pTest);

/* Functions for getting information about the previous test run. */
unsigned int CU_get_number_of_suites_run(void);
unsigned int CU_get_number_of_suites_failed(void);
unsigned int CU_get_number_of_tests_run(void);
unsigned int CU_get_number_of_tests_failed(void);
unsigned int CU_get_number_of_asserts(void);
unsigned int CU_get_number_of_successes(void);
unsigned int CU_get_number_of_failures(void);
unsigned int CU_get_number_of_failure_records(void);
const CU_pFailureRecord CU_get_failure_list(void);
const CU_pRunSummary CU_get_run_summary(void);

/* Functions for internal & testing use. */
CU_pSuite CU_get_current_suite(void);
CU_pTest  CU_get_current_test(void);
BOOL      CU_is_test_running(void);
void      CU_clear_previous_results(void);

/* Assertion implementation function. */
BOOL CU_assertImplementation(BOOL bValue,
                             unsigned int uiLine,
                             char strCondition[],
                             char strFile[],
                             char strFunction[],
                             BOOL bFatal);

#ifdef USE_DEPRECATED_CUNIT_NAMES
typedef CU_FailureRecord  _TestResult;  /**< @deprecated Use CU_FailureRecord. */
typedef CU_pFailureRecord PTestResult;  /**< @deprecated Use CU_pFailureRecord. */
#endif  /* USE_DEPRECATED_CUNIT_NAMES */

#ifdef CUNIT_BUILD_TESTS
void test_cunit_TestRun(void);
#endif

#ifdef __cplusplus
}
#endif

#endif  /*  _CUNIT_TESTRUN_H  */



#ifndef _CUNIT_MYMEM_H
#define _CUNIT_MYMEM_H

#ifdef __cplusplus
extern "C" {
#endif

#ifdef MEMTRACE
  void* CU_calloc(size_t nmemb, size_t size, unsigned int uiLine, const char* szFileName);
  void* CU_malloc(size_t size, unsigned int uiLine, const char* szFileName);
  void  CU_free(void *ptr, unsigned int uiLine, const char* szFileName);
  void* CU_realloc(void *ptr, size_t size, unsigned int uiLine, const char* szFileName);
  void  CU_dump_memory_usage(const char*);

  /** c-allocate with memory tracking. */
  #define CU_CALLOC(x, y)         CU_calloc((x), (y), __LINE__, __FILE__)
  /** m-allocate with memory tracking. */
  #define CU_MALLOC(x)            CU_malloc((x), __LINE__, __FILE__)
  /** Free with memory tracking. */
  #define CU_FREE(x)              CU_free((x), __LINE__, __FILE__)
  /** Reallocate with memory tracking. */
  #define CU_REALLOC(x, y)        CU_realloc((x), (y), __LINE__, __FILE__)
  /** Generate report on tracked memory. */
  #define CU_DUMP_MEMORY_USAGE(x) CU_dump_memory_usage((x))
#else
  /** Standard calloc() if MEMTRACE not defined. */
  #define CU_CALLOC(x, y)         calloc((x), (y))
  /** Standard malloc() if MEMTRACE not defined. */
  #define CU_MALLOC(x)            malloc((x))
  /** Standard free() if MEMTRACE not defined. */
  #define CU_FREE(x)              free((x))
  /** Standard realloc() if MEMTRACE not defined. */
  #define CU_REALLOC(x, y)        realloc((x), (y))
  /** No-op if MEMTRACE not defined. */ 
  #define CU_DUMP_MEMORY_USAGE(x)
#endif

#ifdef CUNIT_BUILD_TESTS
/** Disable memory allocation for testing purposes. */
void test_cunit_deactivate_malloc(void);
/** Enable memory allocation for testing purposes. */
void test_cunit_activate_malloc(void);
/** Retrieve number of memory events for a given pointer */
unsigned int test_cunit_get_n_memevents(void* pLocation);
/** Retrieve number of allocations for a given pointer */
unsigned int test_cunit_get_n_allocations(void* pLocation);
/** Retrieve number of deallocations for a given pointer */
unsigned int test_cunit_get_n_deallocations(void* pLocation);

void test_cunit_MyMem(void);
#endif


#ifdef __cplusplus
}
#endif

#endif  /*  _CUNIT_MYMEM_H  */
