#ifndef INCLUDE_DOCMI_FILE_WRITER_H
#define INCLUDE_DOCMI_FILE_WRITER_H

#include "DicomData.h"
#include "filesize.h"

namespace docmi {

class DicomEncoder
{
protected:
	DicomEncoder(){}
public:
	virtual ~DicomEncoder(){}
	virtual void writeElement( FILE* f, const Element* e ) = 0;

	void writeSequence( FILE* f, const Sequence* s )
	{
		const Elements* elements = s->getElements();

		for( Elements::const_iterator it=elements->begin(); it != elements->end(); ++it ){
			if( (*it)->elementID() != 0x0000 ){ // O[v͕ς邱Ƃ̂ŏ܂Ȃ
				writeElement( f, *it );
			}
		}
	}

protected:
	void writeTagID( FILE* f, ushort group_id, ushort element_id ){
		writeFile( (uchar*)&group_id, 2, f );
		writeFile( (uchar*)&element_id, 2, f );
	}

	void write4ByteLength( FILE* f, uint len ){
		writeFile( (uchar*)&len, 4, f );
	}

	void writeImplicitTag( FILE* f, ushort group_id, ushort element_id, uint len ){
		writeTagID( f, group_id, element_id );
		write4ByteLength( f, len );
	}

	// t@CɎw肳ꂽ̃f[^
	void writeFile( const uchar* buf, uint length, FILE* f )
	{
		if( length != fwrite( buf, 1, length, f ) ){
			throw Exception( "t@C݂Ɏs" );
		}
	}
};

class ImplicitEncoder : public DicomEncoder
{
public:
	virtual void writeElement( FILE* f, const Element* e )
	{
		writeTagID( f, e->groupID(), e->elementID() );

		const Sequences* sequences = e->getSequences();
		if( e->isSQVR() || sequences->size() > 0 ){
			uint IMPLICIT_LEN = 0xFFFFFFFF;
			write4ByteLength( f, IMPLICIT_LEN );

			for( Sequences::const_iterator s=sequences->begin(); s != sequences->end(); ++s ){

				if( (*s)->length() > 0 || (*s)->getElementCount() == 0 ){
					writeImplicitTag( f, 0xFFFE, 0xE000, (*s)->length() );
					writeFile( (*s)->data(), (*s)->length(), f );
				} else {
					writeImplicitTag( f, 0xFFFE, 0xE000, IMPLICIT_LEN );
					writeSequence( f, *s );
					writeImplicitTag( f, 0xFFFE, 0xE00D, 0 );
				}
			}
			
			writeImplicitTag( f, 0xFFFE, 0xE0DD, 0 );

		} else {
			write4ByteLength( f, e->length() );
			writeFile( e->data(), e->length(), f );
		}
	}

	
};

class ExplicitEncoder : public DicomEncoder
{
public:
	virtual void writeElement( FILE* f, const Element* e )
	{
		writeTagID( f, e->groupID(), e->elementID() );

		if( e->is4ByteLengthVR() ){
			writeFile( (uchar*)e->getVR().c_str(), 2, f );
			uchar nulldata[2] = { 0, 0 };
			writeFile( nulldata, 2, f );

			const Sequences* sequences = e->getSequences();
			if( e->isSQVR() || sequences->size() > 0 ){

				uint IMPLICIT_LEN = 0xFFFFFFFF;
				write4ByteLength( f, IMPLICIT_LEN );
				writeFile( e->data(), e->length(), f );

				for( Sequences::const_iterator s=sequences->begin(); s != sequences->end(); ++s ){

					if( (*s)->length() > 0 || (*s)->getElementCount() == 0 ){
						writeImplicitTag( f, 0xFFFE, 0xE000, (*s)->length() );
						writeFile( (*s)->data(), (*s)->length(), f );
					} else {
						writeImplicitTag( f, 0xFFFE, 0xE000, IMPLICIT_LEN );
						writeSequence( f, *s );
						writeImplicitTag( f, 0xFFFE, 0xE00D, 0 );
					}
				}
				
				writeImplicitTag( f, 0xFFFE, 0xE0DD, 0 );

			} else {
				write4ByteLength( f, e->length() );
				writeFile( e->data(), e->length(), f );
			}

		} else {
			write2ByteLengthElement( f, e );
		}
	}

	void write2ByteLengthElement( FILE* f, const Element* e )
	{
		writeFile( (uchar*)e->getVR().c_str(), 2, f );
		ushort len = e->length();
		writeFile( (uchar*)&len, 2, f );
		writeFile( e->data(), len, f );
	}
};

class FileWriter
{
public:
	enum FILETYPE {
		TYPE_EXPLICIT_LITTLE,
		TYPE_IMPLICIT_LITTLE
	};
private:
	FILETYPE m_file_type;
	InputController* m_input;

public:

	FileWriter(){
		setTransferSyntax( "1.2.840.10008.1.2" );
		setImplementationClassUid( "999.999.999.1234567890" );
		setImplementationVersionName( "DOCMI2005" );
		setFileType( TYPE_IMPLICIT_LITTLE );
	}
	~FileWriter(){}
	void setTransferSyntax( const char* transfer_syntax ){
		this->transfer_syntax = transfer_syntax;
	}
	void setImplementationClassUid( const char* implementation_class_uid ){
		this->implementation_class_uid = implementation_class_uid;
	}
	void setImplementationVersionName( const char* implementation_version_name ){
		this->implementation_version_name = implementation_version_name;
	}
	void setFileType( FILETYPE file_type ){
		this->m_file_type = file_type;
	}

private:
	String transfer_syntax;
	String implementation_class_uid;
	String implementation_version_name;

public:

	/** DICOMf[^t@Cɕۑ
	 *
	 *  throw Exception
	 */
	void save( const Sequence& data, const char* path ){


		FILE* f = fopen( path, "wb" );
		if( f == NULL ){
			throw Exception( "t@CJ܂ł" );
		}

		try{
			writeFilePreamble( f );
			writeFileMetaData( f, data );

			DicomEncoder* encoder;

			if( m_file_type == TYPE_EXPLICIT_LITTLE ){
				encoder = new ExplicitEncoder;
			} else {
				encoder = new ImplicitEncoder;
			}

			encoder->writeSequence( f, &data );
			delete encoder;
			fclose( f );
		} catch( Exception& x ){
			throw x;
		}
	}

private:

	// t@CɎw肳ꂽ̃f[^
	void writeFile( const uchar* buf, uint length, FILE* f )
	{
		if( length != fwrite( buf, 1, length, f ) ){
			throw Exception( "t@C݂Ɏs" );
		}
	}

	// t@CvAuo͂
	void writeFilePreamble( FILE* f )
	{
		uchar preamble[128];
		memset( preamble, 0, 128 );
		writeFile( preamble, 128, f );

		uchar dicm[] = { 'D', 'I', 'C', 'M' };
		writeFile( dicm, 4, f );
	}

	// t@C^f[^o͂
	void writeFileMetaData( FILE* f, const Sequence& data )
	{
		ConcreteSequence temp;

		uchar infover[2] = { 0, 1 };
		temp.modifyElement( ConcreteElement( 0x0002, 0x0001, infover, 2 ) );

		{
			const Element* e = data.search(0x0008,0x0016);
			const uchar* newdata = NULL;
			int data_len = 0;
			if( e ){
				newdata = e->data();
				data_len = e->length();
			}
			temp.modifyElement( ConcreteElement( 0x0002, 0x0002, newdata, data_len ) );
		}
		{
			const Element* e = data.search(0x0008,0x0018);
			const uchar* newdata = NULL;
			int data_len = 0;
			if( e ){
				newdata = e->data();
				data_len = e->length();
			}
			temp.modifyElement( ConcreteElement( 0x0002, 0x0003, newdata, data_len ) );
		}
		{
			ConcreteElement e( 0x0002, 0x0010 );
			e.setNullendString( transfer_syntax );
			temp.modifyElement( e );
		}
		{
			ConcreteElement e( 0x0002, 0x0012 );
			e.setNullendString( implementation_class_uid );
			temp.modifyElement( e );
		}
		{
			ConcreteElement e( 0x0002, 0x0013 );
			e.setSpaceendString( implementation_version_name );
			temp.modifyElement( e );
		}

		int grouplen = calcGroupLen( 0x0002, true, &temp );

		ExplicitEncoder encoder;
		encoder.writeElement( f, &ConcreteElement( 0x0002, 0x0000, (uchar*)&grouplen, 4 ) );
		encoder.writeSequence( f, &temp );
	}

	void writeTagID( FILE* f, ushort group_id, ushort element_id )
	{
		writeFile( (uchar*)&group_id, 2, f );
		writeFile( (uchar*)&element_id, 2, f );
	}

	void write4ByteLength( FILE* f, uint len )
	{
		writeFile( (uchar*)&len, 4, f );
	}

	void writeImplicitTag( FILE* f, ushort group_id, ushort element_id, uint len )
	{
		writeTagID( f, group_id, element_id );
		write4ByteLength( f, len );
	}

public:
	// O[v^OɊ܂߂O[v(O[v^OO[v)vZĕԂ
	// group_id == 0xFFFF ȂASẴ^O̒Ԃ
	int calcGroupLen( ushort group_id, bool isExplicit, const Sequence* dicom_data )
	{
		const Elements* elements = dicom_data->getElements();
		int len = 0;

		for( Elements::const_iterator it=elements->begin(); it != elements->end(); ++it ){
			if( (*it)->groupID() == group_id || group_id == 0xFFFF){
				if( (*it)->groupID() == group_id && (*it)->elementID() == 0x0000 ) continue;

				len += 4;
				if( !isExplicit ){
					len += 4;
				} else {
					if( (*it)->is4ByteLengthVR() ){
						len += 8;
					} else {
						len += 4;
					}
				}
				len += (*it)->length();

				const Sequences* sequences = (*it)->getSequences();
				if( sequences->size() > 0 ){
					for( Sequences::const_iterator s=sequences->begin(); s != sequences->end(); ++s ){
						len += 8; // (FFFE,E000) + 4byte len
						len += (*s)->length();
						len += calcGroupLen( 0xFFFF, isExplicit, *s );
						len += 8; // (FFFE,E00D) + 4byte len
					}
					len += 8; // (FFFE,E0DD) + 4byte len
				}
			}
		}

		return len;
	}
};

} // namespace docmi

#endif
