# vim: fileencoding=utf8
import codecs
import struct
import re
import datetime
import time
from sets import Set
import xml.dom.minidom as minidom
import amf, amf.utils
try:
    from cStringIO import StringIO
except ImportError:
    from StringIO import StringIO


class TypeCodes:
    DOUBLE      = 0
    BOOL        = 1
    UTF8        = 2
    OBJECT      = 3
    MOVIECLIP   = 4
    NULL        = 5
    UNDEFINED   = 6
    REFERENCE   = 7
    MIXEDARRAY  = 8
    ENDOFOBJECT = 9
    ARRAY       = 10
    DATE        = 11
    LONGUTF8    = 12
    UNSUPPORTED = 13
    RECOREDSET  = 14
    XML         = 15
    TYPEDOBJECT = 16


class AMFReader(object):

    def __init__(self, raw_post_data):
       self.data = StringIO(raw_post_data)
       self.content_length = len(raw_post_data)
       self.message = amf.AMFMessage()

    def read(self):
        self.read_headers()
        self.read_bodies()
        return self.message

    def read_headers(self):
        self.read_byte()
        self.read_byte()
        header_count = self.read_int()
        for i in range(header_count):
            name = self.read_utf()
            required = self.read_boolean()
            self.data.seek(4,1)
            type = self.read_byte()
            value = self.read_data(type)
            amfHeader = {'name':name, 'required':required, 'value':value}
            self.message.add_header(amfHeader)

    def read_bodies(self):
        bodyCount = self.read_int()
        for i in range(bodyCount):
            target = self.read_utf(); 
            response = self.read_utf()
            self.data.read(4)
            type = self.read_byte()
            data = self.read_data(type)
            amf_body = amf.AMFMessageBody(target, response, data)
            self.message.add_body(amf_body)
            
    def read_utf(self):
        len = self.read_int()
        if len == 0:
            #return u''
            return ''
        val = self.data.read(len)
        #return unicode(val,'utf_8')
        return val
    
    def read_byte(self):
        return ord(self.data.read(1))
        
    def read_int(self):
        return ((ord(self.data.read(1)) << 8) | ord(self.data.read(1)));

    def read_object(self):
        key = self.read_utf()
        type = self.read_byte()
        ret = {}
        while type != TypeCodes.ENDOFOBJECT:
            val = self.read_data(type)
            ret[key] = val
            key = self.read_utf()
            type = self.read_byte()
        return ret
    
    def read_reference(self):
        reference = self.read_int();
        return "(unresolved object #%s)" % reference

    def read_mixed_array(self):
        self.data.read(4)
        return self.read_mixed_object()

    def read_mixed_object(self):
        key = self.read_utf()
        type = self.read_byte()
        ret = {}
        while type != TypeCodes:
            val = self.read_data(type)
            if isinstance(key, (int, long, float)):
                key = float(key)
            ret[key] = val
            key = self.read_utf()
            type = self.read_byte()
        return ret
    
    def read_long(self):
        return ((ord(self.data.read(1)) << 24) |
                (ord(self.data.read(1)) << 16) |
                (ord(self.data.read(1)) << 8) |
                ord(self.data.read(1)))

    def read_array(self):
        ret = []
        len = self.read_long()
        for i in range(len):
            type = self.read_byte()
            ret.append(self.read_data(type))
        return ret

    def read_date(self):
        ms = self.read_double() # date in milliseconds from 01/01/1970
        offset = self.read_int()
        if offset > 720:
            offset = - (65536 - offset)
        offset *= -60
        h = offset / 3600
        class TZ(datetime.tzinfo):
            def utcoffset(self, dt):
                return datetime.timedelta(hours=h)
            def dst(self, dt):
                return datetime.timedelta(0)
            def tzname(self, dt):
                return "JST" # TODO: How to deal with other timezone?
        tz = TZ()  
        return datetime.datetime.fromtimestamp(ms / 1000.0, tz) 

    def read_long_utf(self):
        len = self.read_long()
        val = self.data.read(len)
        #return unicode(val, 'utf_8')
        return val
    
    def read_xml(self):
        xmlStr = self.read_long_utf()
        return minidom.parseString(xmlStr.encode('utf_8')) # minidom cannot deal with unicode string

    def read_custom_class(self):
        type = self.read_utf().replace('..', '')
        obj = self.read_object()
        amf.logger.debug("read_custom_class() -- type=%s, object=%s", type, str(obj))
        if '_explicitType' not in obj:
            obj['_explicitType'] = type
        return amf.utils.classcast(type, obj)

    def read_double(self):
        bytes = self.data.read(8)
        bytes = bytes[::-1] # big endian
        return struct.unpack('d', bytes)[0]

    def read_boolean(self):
        return self.read_byte() == 1

    func_map = {
        TypeCodes.DOUBLE      : read_double,
        TypeCodes.BOOL        : read_boolean,
        TypeCodes.UTF8        : read_utf,
        TypeCodes.OBJECT      : read_object,
        TypeCodes.MOVIECLIP   : None,
        TypeCodes.NULL        : None,
        TypeCodes.REFERENCE   : read_reference,
        TypeCodes.MIXEDARRAY  : read_mixed_array,
        TypeCodes.ARRAY       : read_array,
        TypeCodes.DATE        : read_date,
        TypeCodes.LONGUTF8    : read_long_utf,
        TypeCodes.UNSUPPORTED : None,
        TypeCodes.XML         : read_xml,
        TypeCodes.TYPEDOBJECT : read_custom_class,
    }

    def read_data(self, type):
        if type not in self.func_map:
            raise RuntimeException("Unsupported type")
        func = self.func_map[type]
        if func is not None:
            return func(self)
        return None


class AMFWriter(object):

    def __init__(self):
        pass

    def write(self, message):
        self.response = StringIO()

        self.write_int(0)
        header_count = len(message.headers)
        self.write_int(header_count)
        for header in message.headers:
            self.write_utf(header['name'])
            self.write_byte(0)
            self.write_long(-1)
            self.write_data(header['value'])

        body_count = len(message.bodies)
        self.write_int(body_count)
        for body in message.bodies:
          self.write_utf(body.target)
          self.write_utf('null')
          self.write_long(-1)
          self.write_data(body.data)

        try:
            return self.response.getvalue()
        finally:
            self.response.close()

    def write_int(self, n):
        d = struct.pack('H', n)
        d = d[::-1]
        self.response.write(d)

    def write_byte(self, b):
        self.response.write(struct.pack('b', b))

    def write_long(self, l):
        d = struct.pack('L', l)
        d = d[::-1]
        self.response.write(d)

    def write_utf(self, s):
        s = self._encode_to_utf8(s)
        self.write_int(len(s))
        self.response.write(s)

    def write_binary(self, s):
        self.write_int(len(s))
        self.rsponse.write(s)

    def write_long_utf(self, s):
        s = self._encode_to_utf8(s)
        self.write_long(len(s))
        self.response.write(s)

    def _encode_to_utf8(self, s):
        if isinstance(s, unicode):
            s = s.encode('utf_8')
        return s 

    def write_string(self, s):
        s = self._encode_to_utf8(s)
        count = len(s) 
        if (count < 65536):
            self.write_byte(TypeCodes.UTF8)
            self.write_utf(s)
        else:
            self.write_byte(TypeCodes.LONGUTF8)
            self.write_long_utf(s)
    
    def write_array(self, a):
        self.write_byte(TypeCodes.ARRAY)
        self.write_long(len(a))
        for e in a:
            self.write_data(e)

    def write_double(self, d):
        b = struct.pack('d', d)
        r = b[::-1] # big endian
        self.response.write(r)    

    def write_object(self, obj):
        self.write_byte(TypeCodes.OBJECT)
        if 'iteritems' in dir(obj):
            items = obj.iteritems()
        else:
            items = obj.__dict__.iteritems()
        for key, value in items:
            self.write_utf(key)
            self.write_data(value)
        self.write_int(0)
        self.write_byte(9)

    def write_number(self, n):
        self.write_byte(TypeCodes.DOUBLE)
        self.write_double(float(n))

    def write_null(self):
        self.write_byte(TypeCodes.NULL)

    def write_boolean(self, b):
        self.write_byte(TypeCodes.BOOL)
        self.write_byte(b)

    def write_datetime(self, d):
        timestamp = time.mktime(d.timetuple()) # TODO cannot deal with 'date' properly
        self.write_byte(TypeCodes.DATE)
        self.write_double(timestamp * 1000)
        self.write_int(0)

    def write_xml(self, document):
        self.write_byte(TypeCodes.XML)
        xmlstr = re.sub(r'\>(\n|\r|\r\n| |\t)*\<', '><', document.toxml().strip())
        self.write_long_utf(xmlstr)
	
    def write_custom_class(self, d):
        self.write_byte(TypeCodes.TYPEDOBJECT)
        type = amf.utils.get_as_type(d)
        self.write_utf(type)
        if 'iteritems' in dir(d): # d is a dict 
            items = d.iteritems()
        else:
            items = d.__dict__.iteritems()
        for key, value in items:
            self.write_utf(key)
            self.write_data(value)
        self.write_int(0)
        self.write_byte(TypeCodes.ENDOFOBJECT)

    def write_data(self, d):
        if isinstance(d, (int, long, float)):
            self.write_number(d)
            return
        elif isinstance(d, (str, unicode)):
            self.write_string(d)
            return
        elif isinstance(d, bool):
            self.write_boolean(d)
            return
        elif isinstance(d, (list, tuple, Set)):
            self.write_array(d)
            return
        elif isinstance(d, dict):
            self.write_object(d)
            return
        elif isinstance(d, (datetime.datetime, datetime.date)):
            self.write_datetime(d)
            return
        elif d is None:
            self.write_null()
            return
        elif isinstance(d, minidom.Document): # XML
            self.write_xml(d)
            return
        else:
            self.write_custom_class(d)
            return

