/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE MP3FRAUNHOFERENC SOURCE CODE.           *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE MP3FRAUNHOFERENC SOURCE CODE IS (C) COPYRIGHT 2008 Cocha     *
 * http://sourceforge.jp/projects/directshow/                       *
 *                                                                  *
 ********************************************************************/

#include "MP3FraunhoferEnc.h"

// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CMP3FraunhoferEnc::GetInFormat(WAVEFORMATEX *pwf)
{  // ̓s̃tH[}bg擾

   CheckPointer(m_pInput, E_POINTER);

   if(m_pInput->IsConnected() == FALSE)
      return VFW_E_NOT_CONNECTED;

   ::CopyMemory(pwf, &m_inFormat, sizeof(m_inFormat));
   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CMP3FraunhoferEnc::CheckBitrate(int nOutBitrate)
{  // nOutBitratẽrbg[gŏo͂ł邩`FbN

   CheckPointer(m_pInput, E_POINTER);

   m_pAcmFraunhofer = new CAcmFraunhofer();

   if(m_pInput->IsConnected() == FALSE)
      return VFW_E_NOT_CONNECTED;

   return m_pAcmFraunhofer->CheckFormat(&m_inFormat, nOutBitrate);
}
// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CMP3FraunhoferEnc::CheckBitrate(WAVEFORMATEX *pInFormat, int nOutBitrate)
{  // pInFormat̓͂̏ꍇɁAnOutBitratẽrbg[gŏo͂ł邩`FbN

   return m_pAcmFraunhofer->CheckFormat(pInFormat, nOutBitrate);
}
// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CMP3FraunhoferEnc::GetBitrate(int *pnOutBitrate)
{  // o͂̃rbg[g擾

   CheckPointer(pnOutBitrate, E_POINTER);

   if(m_pInput->IsConnected() == FALSE)
      return VFW_E_NOT_CONNECTED;

   *pnOutBitrate = m_nBitrate;

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CMP3FraunhoferEnc::SetBitrate(int nOutBitrate)
{  // o͂̃rbg[gݒ肷

   CheckPointer(m_pInput, E_POINTER);

   if(m_pInput->IsConnected() == FALSE)
      return VFW_E_NOT_CONNECTED;

   m_nBitrate = nOutBitrate;

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
CMP3FraunhoferEnc::CMP3FraunhoferEnc(IUnknown *pUnk, HRESULT *phr) :
   CTransformFilter(FILTER_NAME, pUnk, CLSID_MP3FraunhoferEnc)
{  // RXgN^

   // ϐ̏
   m_nBitrate = 0;
   m_nInBufferSize = 0;
   m_nOutBufferSize = 0;
   m_pAcmFraunhofer = new CAcmFraunhofer();

   // R[fbN̊mF
   HRESULT hr;
   WAVEFORMATEX wf;

   wf.wFormatTag = WAVE_FORMAT_PCM;
   wf.cbSize = 0;
   wf.wBitsPerSample = 16;
   wf.nChannels = 2;
   wf.nBlockAlign = wf.nChannels * 2;

   wf.nSamplesPerSec = 44100;
   wf.nAvgBytesPerSec = wf.nBlockAlign * wf.nSamplesPerSec;


   hr = m_pAcmFraunhofer->CheckFormat(&wf, 128000);
   if(hr == S_OK)
   {
      m_bProfessional = true;

      if(phr)
         *phr = S_OK;
   }
   else
   {
      wf.nSamplesPerSec = 22050;
      wf.nAvgBytesPerSec = wf.nBlockAlign * wf.nSamplesPerSec;

      hr = m_pAcmFraunhofer->CheckFormat(&wf, 56000);
      if(hr == S_OK)
      {
         m_bProfessional = false;

         if(phr)
            *phr = S_OK;
      }
      else
      {
         ::MessageBox(NULL, TEXT("FraunhoferR[fbN݂܂B"), TEXT("G["), MB_OK);

         if(phr)
            *phr = E_FAIL;
      }
   }
}
// -----------------------------------------------------------------------------------------------------------------------------------
CMP3FraunhoferEnc::~CMP3FraunhoferEnc()
{  // fXgN^

   if(m_pAcmFraunhofer != NULL)
   {
      delete m_pAcmFraunhofer;
      m_pAcmFraunhofer = NULL;
   }
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CMP3FraunhoferEnc::CheckInputType(const CMediaType *mtIn)
{  // ̓s󂯓邩`FbN

   CAutoLock cObjectLock(m_pLock);

   if(*mtIn->Subtype() == MEDIASUBTYPE_PCM && *mtIn->FormatType() == FORMAT_WaveFormatEx)
   {
      if(mtIn->FormatLength() >= sizeof(WAVEFORMATEX) && mtIn->IsTemporalCompressed() == FALSE)
      {
         WAVEFORMATEX *pwf = (WAVEFORMATEX *)mtIn->pbFormat;

         if(pwf->wFormatTag == WAVE_FORMAT_PCM)
         {
            if(m_bProfessional == true)
            {  // Professionalł̏ꍇ
               if(pwf->nSamplesPerSec <= 48000 && pwf->wBitsPerSample == 16)
               {         
                  ::CopyMemory(&m_inFormat, pwf, sizeof(m_inFormat));
                  return S_OK;
               }
            }
            else
            {  // Advancedł̏ꍇ
               if(pwf->nSamplesPerSec <= 24000 && pwf->wBitsPerSample == 16)
               {
                  ::CopyMemory(&m_inFormat, pwf, sizeof(m_inFormat));
                  return S_OK;
               }
            }
         }
      }
   }

   return S_FALSE;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CMP3FraunhoferEnc::GetMediaType(int iPosition, CMediaType *pOutMediaType)
{  // o̓s̃fBA^Cvݒ肷

   CheckPointer(m_pInput, E_POINTER);

   if(iPosition < 0)
      return E_INVALIDARG;

   if(iPosition > 0)
      return VFW_S_NO_MORE_ITEMS;

   pOutMediaType->SetType(&MEDIATYPE_Audio);
   pOutMediaType->SetSubtype(&WMMEDIASUBTYPE_MP3);

   if (m_pInput->IsConnected() == FALSE)
   {
      pOutMediaType->SetFormatType(&FORMAT_None);
      return S_OK;
   }

   if(m_nBitrate == 0)
   {  // o̓rbg[g܂ݒ肳ĂȂꍇAϊłrbg[gT

      int i;
      HRESULT hr;
      int anBitrate[19] = { 8000, 16000, 18000, 20000, 24000, 32000, 40000, 48000, 56000, 64000, 80000, 96000, 112000, 128000, 160000, 192000, 224000, 256000, 320000};

      for(i=13;i>=0;i--)
      {
         hr = m_pAcmFraunhofer->CheckFormat(&m_inFormat, anBitrate[i]);
         if(hr == S_OK)
            break;
      }

      if(i == -1)
         return E_UNEXPECTED;

      m_nBitrate = anBitrate[i];
   }

   // o̓tH[}bg擾
   m_pAcmFraunhofer->GetOutFormat(&m_inFormat, m_nBitrate, &m_outFormat);

   pOutMediaType->AllocFormatBuffer(sizeof(MPEGLAYER3WAVEFORMAT));
   pOutMediaType->SetFormatType(&FORMAT_WaveFormatEx);
   pOutMediaType->SetFormat((LPBYTE)&m_outFormat, sizeof(MPEGLAYER3WAVEFORMAT));
   pOutMediaType->SetTemporalCompression(FALSE);

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CMP3FraunhoferEnc::CheckTransform(const CMediaType *mtIn, const CMediaType *mtOut)
{  // ̓fBA^Cvo̓fBA^Cvƌ݊邩`FbN

   if(*mtIn->Subtype() == MEDIASUBTYPE_PCM)
   {
      if(*mtOut->Subtype() == WMMEDIASUBTYPE_MP3)
         return S_OK;
   }

   return VFW_E_TYPE_NOT_ACCEPTED;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CMP3FraunhoferEnc::DecideBufferSize(IMemAllocator *pAlloc, ALLOCATOR_PROPERTIES *pProperties)
{  // o̓s̃obt@vݒ肷

   CheckPointer(pAlloc, E_POINTER);
   CheckPointer(pProperties, E_POINTER);
   CheckPointer(m_pInput, E_FAIL);

   if(m_pInput->IsConnected()==FALSE)
      return E_FAIL;

   // ͂̃f[^TCY擾
   HRESULT hr;
   ALLOCATOR_PROPERTIES InProps;
   IMemAllocator * pInAlloc = NULL;

   hr = m_pInput->GetAllocator(&pInAlloc);
   if(FAILED(hr)) return hr;

   hr = pInAlloc->GetProperties(&InProps);
   if(FAILED(hr))
   {
      SAFE_RELEASE(pInAlloc);
      return hr;
   }

   SAFE_RELEASE(pInAlloc);

   // o͂ɕKvȃTCYZo
   m_nInBufferSize = InProps.cbBuffer * 12 / 10;
   m_nOutBufferSize = m_pAcmFraunhofer->GetOutBufferSize(&m_inFormat, m_nInBufferSize, m_nBitrate);
   if(m_nOutBufferSize <= 0)
      return E_FAIL;

   // o͂̐ݒ
   pProperties->cbBuffer = m_nOutBufferSize;
   pProperties->cBuffers = 1;
   pProperties->cbAlign = 1;
	pProperties->cbPrefix = 0;

   // AP[^͗vɑ΂ĐmɈvłƂ͌Ȃ߁AۂɊmۂꂽ`FbN
   ALLOCATOR_PROPERTIES Actual;
   hr = pAlloc->SetProperties(pProperties, &Actual);
   if(FAILED(hr)) return hr;

   if( (pProperties->cBuffers > Actual.cBuffers) || (pProperties->cbBuffer > Actual.cbBuffer) )
       return E_FAIL;

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CMP3FraunhoferEnc::StartStreaming(void)
{  // Xg[~OJnɌĂ΂

   CAutoLock cObjectLock(m_pLock);

   //acm̏
   HRESULT hr;
   hr = m_pAcmFraunhofer->ConvertInit(&m_inFormat, m_nInBufferSize, &m_outFormat, &m_nOutBufferSize, m_nBitrate);

   if(hr != S_OK)
      return E_FAIL;

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CMP3FraunhoferEnc::Receive(IMediaSample *pInSample)
{  // ϊ

   HRESULT hr;
   BYTE *pbInBuffer = NULL;
   BYTE *pbOutBuffer = NULL;
   IMediaSample *pOutSample = NULL;

   CAutoLock lck(&m_myLock);

   // AM_STREAM_MEDIAȊOȂʉ߂
   AM_SAMPLE2_PROPERTIES * const pProps = m_pInput->SampleProps();
   if(pProps->dwStreamId != AM_STREAM_MEDIA)
      return m_pOutput->Deliver(pInSample);

   // o͂̏
   hr = m_pOutput->GetDeliveryBuffer(&pOutSample, NULL, NULL, 0);
   if FAILED(hr)
      return S_FALSE;

   if(pInSample->IsSyncPoint() == S_OK)
      pOutSample->SetSyncPoint(TRUE);
 
   if(pInSample->IsPreroll() == S_OK)
      pOutSample->SetPreroll(TRUE);

   if(pInSample->IsDiscontinuity() == S_OK || m_bSampleSkipped)
      pOutSample->SetDiscontinuity(TRUE);

   pInSample->GetPointer(&pbInBuffer);
   pOutSample->GetPointer(&pbOutBuffer);

   int nLength = 0;

   // acmŕϊ
   hr = m_pAcmFraunhofer->Convert(pbInBuffer, pInSample->GetActualDataLength(), pbOutBuffer, &nLength);

   if(hr != S_OK)
      return S_FALSE;

   // fBAԂ̐ݒ
   REFERENCE_TIME rtStart, rtStop;
   pInSample->GetTime(&rtStart, &rtStop);
   pOutSample->SetTime(&rtStart, &rtStop);

   // o̓f[^̒ݒ
   pOutSample->SetActualDataLength(nLength);

   // f[^o
   hr = m_pOutput->Deliver(pOutSample);

   // ㏈
   SAFE_RELEASE(pOutSample);

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CMP3FraunhoferEnc::StopStreaming()
{  // ~ꍇɌĂ΂

   CAutoLock lck(&m_myLock);

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CMP3FraunhoferEnc::EndOfStream(void)
{  // Xg[~OIɌĂ΂

   CAutoLock lck(&m_myLock);

   // o͂̏
   HRESULT hr;
   IMediaSample *pOutSample = NULL;
   BYTE *pbOutBuffer = NULL;

   hr = m_pOutput->GetDeliveryBuffer(&pOutSample, NULL, NULL, 0);
   if FAILED(hr)
      return S_FALSE;

   pOutSample->GetPointer(&pbOutBuffer);
   pOutSample->SetSyncPoint(TRUE);
   pOutSample->SetDiscontinuity(FALSE);
   pOutSample->SetPreroll(FALSE);

   // acmŕϊ
   int nLength = 0;
   hr = m_pAcmFraunhofer->ConvertEnd(pbOutBuffer, &nLength);

   // o̓f[^̒ݒ
   pOutSample->SetActualDataLength(nLength);

   // f[^o
   m_pOutput->Deliver(pOutSample);

   // ㏈
   SAFE_RELEASE(pOutSample);

   return CTransformFilter::EndOfStream();
}
// -----------------------------------------------------------------------------------------------------------------------------------
