/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE WAVCONFILTER SOURCE CODE.               *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE WAVCONFILTER SOURCE CODE IS (C) COPYRIGHT 2008 Cocha         *
 * http://sourceforge.jp/projects/directshow/                       *
 *                                                                  *
 ********************************************************************/

#include "WavConFilter.h"


// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CWavConFilter::GetInFormat(WAVEFORMATEX *pwf)
{  // ̓s̃tH[}bg擾

   CheckPointer(pwf, E_POINTER);
   CheckPointer(m_pInput, E_POINTER);

   if(m_pInput->IsConnected() == FALSE)
      return VFW_E_NOT_CONNECTED;

   pwf->cbSize = m_inFormat.cbSize;
   pwf->nAvgBytesPerSec = m_inFormat.nAvgBytesPerSec;
   pwf->nBlockAlign = m_inFormat.nBlockAlign;
   pwf->nChannels = m_inFormat.nChannels;
   pwf->nSamplesPerSec = m_inFormat.nSamplesPerSec;
   pwf->wBitsPerSample = m_inFormat.wBitsPerSample;
   pwf->wFormatTag = m_inFormat.wFormatTag;

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CWavConFilter::GetOutFormat(WAVEFORMATEX *pwf)
{  // o̓tH[}bg擾

   if(m_bSetOutFormat == false)
      return E_FAIL;

   pwf->cbSize = m_outFormat.cbSize;
   pwf->nAvgBytesPerSec = m_outFormat.nAvgBytesPerSec;
   pwf->nBlockAlign = m_outFormat.nBlockAlign;
   pwf->nChannels = m_outFormat.nChannels;
   pwf->nSamplesPerSec = m_outFormat.nSamplesPerSec;
   pwf->wBitsPerSample = m_outFormat.wBitsPerSample;
   pwf->wFormatTag = m_outFormat.wFormatTag;

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CWavConFilter::SetOutFormat(WAVEFORMATEX *pwf)
{  // o̓tH[}bgݒ肷

   CheckPointer(pwf, E_POINTER);

   if(m_pOutput->IsConnected() != FALSE)
   {  // łɏo̓sڑĂꍇ
      return E_FAIL;
   }

   // l̑Ó`FbN
   if(pwf->wFormatTag != WAVE_FORMAT_PCM && pwf->wFormatTag != WAVE_FORMAT_IEEE_FLOAT)
      return E_FAIL;

   if(pwf->nBlockAlign != (pwf->nChannels * pwf->wBitsPerSample / 8))
      return E_FAIL;

   if(pwf->nAvgBytesPerSec != pwf->nBlockAlign * pwf->nSamplesPerSec)
      return E_FAIL;


   // o̓tH[}bgi[Ă
   m_bSetOutFormat = true;
   m_outFormat.cbSize = pwf->cbSize;
   m_outFormat.nAvgBytesPerSec = pwf->nAvgBytesPerSec;
   m_outFormat.nBlockAlign = pwf->nBlockAlign;
   m_outFormat.nChannels = pwf->nChannels;
   m_outFormat.nSamplesPerSec = pwf->nSamplesPerSec;
   m_outFormat.wBitsPerSample = pwf->wBitsPerSample;
   m_outFormat.wFormatTag = pwf->wFormatTag;

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP CWavConFilter::SetQuality(int nAccurate)
{  // TvOgϊ̐xݒ肷

   if(0 <= nAccurate && nAccurate <=4)
   {
      m_nAccurate = nAccurate;
      return S_OK;
   }

   return E_FAIL;
}
// -----------------------------------------------------------------------------------------------------------------------------------
CWavConFilter::CWavConFilter(IUnknown *pUnk, HRESULT *phr) :
   CTransformFilter(FILTER_NAME, pUnk, CLSID_WavConFilter)
{  // RXgN^

   // ϐ̏
   m_pBitConv = NULL;
   m_pChannelConv = NULL;
   m_pSamplingConv = NULL;
   m_smpld = NULL;
   m_nSmplSize = 0;
   m_bSetOutFormat = false;
   m_nAccurate = 4;
   ::ZeroMemory(&m_inFormat, sizeof(m_inFormat));
   ::ZeroMemory(&m_outFormat, sizeof(m_outFormat));

   if(phr)
      *phr = S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
CWavConFilter::~CWavConFilter()
{  // fXgN^

   if(m_pSamplingConv != NULL)
   {
      delete m_pSamplingConv;
      m_pSamplingConv = NULL;
   }

   if(m_smpld != NULL)
   {
      delete []m_smpld;
      m_smpld = NULL;
   }

   if(m_pBitConv != NULL)
   {
      delete m_pBitConv;
      m_pBitConv = NULL;
   }

   if(m_pChannelConv != NULL)
   {
      delete m_pChannelConv;
      m_pChannelConv = NULL;
   }
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CWavConFilter::CheckInputType(const CMediaType *mtIn)
{  // ̓s󂯓邩`FbN

   CAutoLock cObjectLock(m_pLock);

   // kPCM(WAVE_FORMAT_PCM or WAVE_FORMAT_IEEE_FLOAT)ŁA߂ꂽg&rbg&`lȂ󂯓
   if(*mtIn->Subtype() == MEDIASUBTYPE_PCM || *mtIn->Subtype() == MEDIASUBTYPE_IEEE_FLOAT)
   {
      if(*mtIn->FormatType() == FORMAT_WaveFormatEx)
      {
         WAVEFORMATEX *pwf = (WAVEFORMATEX *)mtIn->pbFormat;

         // TvOg`FbN
         int i;
         int nAcceptSamplingRate[] = {8000, 11025, 12000, 16000, 22050, 24000, 32000, 44100, 48000, 88200, 96000, 176400, 192000};

         for(i=0;i<13;i++)
         {
            if(pwf->nSamplesPerSec == nAcceptSamplingRate[i])
               break;
         }

         if(i == 13)
            return S_FALSE;

         // rbg`FbN
         if(pwf->wFormatTag == WAVE_FORMAT_PCM)
         {
            if(pwf->wBitsPerSample != 8 && pwf->wBitsPerSample != 16 && pwf->wBitsPerSample != 24 && pwf->wBitsPerSample != 32)
               return S_FALSE;
         }
         else if(pwf->wFormatTag == WAVE_FORMAT_IEEE_FLOAT)
         {
            if(pwf->wBitsPerSample != 32 && pwf->wBitsPerSample != 64)
               return S_FALSE;
         }
         else
            return S_FALSE;

         // `l`FbN
         if(pwf->nChannels != 1 && pwf->nChannels != 2)
            return S_FALSE;

         // ̓tH[}bgi[Ă
        m_inFormat.cbSize = pwf->cbSize;
        m_inFormat.nAvgBytesPerSec = pwf->nAvgBytesPerSec;
        m_inFormat.nBlockAlign = pwf->nBlockAlign;
        m_inFormat.nChannels = pwf->nChannels;
        m_inFormat.nSamplesPerSec = pwf->nSamplesPerSec;
        m_inFormat.wBitsPerSample = pwf->wBitsPerSample;
        m_inFormat.wFormatTag = pwf->wFormatTag;

         return S_OK;
      }
   }

   return S_FALSE;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CWavConFilter::GetMediaType(int iPosition, CMediaType *pOutMediaType)
{  // o̓s̃fBA^Cvݒ肷

   if(iPosition<0)
      return E_INVALIDARG;

   if(iPosition>0)
      return VFW_S_NO_MORE_ITEMS;

   if(m_bSetOutFormat == false)
   {  // o͂̃tH[}bgݒ肳ĂȂꍇ

      // ̓sƓfBA^Cvw肷
      *pOutMediaType = m_pInput->CurrentMediaType();
   }
   else
   {  // o͂̃tH[}bgݒ肳Ăꍇ

      pOutMediaType->SetType(&MEDIATYPE_Audio);
      pOutMediaType->SetSubtype(&MEDIASUBTYPE_PCM);
      pOutMediaType->SetTemporalCompression(FALSE);
      pOutMediaType->SetSampleSize(0);
      pOutMediaType->SetFormatType(&FORMAT_WaveFormatEx);
      pOutMediaType->SetFormat((BYTE*)&m_outFormat, sizeof(m_outFormat));
   }

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CWavConFilter::CheckTransform(const CMediaType *mtIn, const CMediaType *mtOut)
{  // ̓fBA^Cvo̓fBA^Cvƌ݊邩`FbN

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CWavConFilter::DecideBufferSize(IMemAllocator *pAlloc, ALLOCATOR_PROPERTIES *pProperties)
{  // o̓s̃obt@vݒ肷

   CheckPointer(pAlloc, E_POINTER);
   CheckPointer(pProperties, E_POINTER);
   CheckPointer(m_pInput, E_FAIL);

   if(m_pInput->IsConnected()==FALSE)
      return E_FAIL;

   int nInBufferSize, nOutBufferSize;

   // ͂̃f[^TCY擾
   HRESULT hr;
   ALLOCATOR_PROPERTIES InProps;
   IMemAllocator * pInAlloc = NULL;

   hr = m_pInput->GetAllocator(&pInAlloc);
   if(FAILED(hr)) return hr;

   hr = pInAlloc->GetProperties(&InProps);
   if(FAILED(hr))
   {
      SAFE_RELEASE(pInAlloc);
      return hr;
   }

   nInBufferSize = InProps.cbBuffer;
   SAFE_RELEASE(pInAlloc);

   // o̓TCYZo
   if(m_bSetOutFormat == false)
   {  // o̓tH[}bgw肳ĂȂꍇAo̓tH[}bg͓̓tH[}bgƓɂ
      ::CopyMemory(&m_outFormat, &m_inFormat, sizeof(m_inFormat));
   }

   // KvTCY1.1{mۂ
   nOutBufferSize = (int)((double)nInBufferSize / (double)m_inFormat.nAvgBytesPerSec * (double)m_outFormat.nAvgBytesPerSec * 1.1);

   pProperties->cbBuffer = nOutBufferSize;
   pProperties->cBuffers = 1;
   pProperties->cbAlign = 1;
	pProperties->cbPrefix = 0;

   // AP[^͗vɑ΂ĐmɈvłƂ͌Ȃ߁AۂɊmۂꂽ`FbN
   ALLOCATOR_PROPERTIES Actual;
   hr = pAlloc->SetProperties(pProperties, &Actual);
   if(FAILED(hr)) return hr;

   if( (pProperties->cBuffers > Actual.cBuffers) || (pProperties->cbBuffer > Actual.cbBuffer) )
       return E_FAIL;

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CWavConFilter::StartStreaming(void)
{  // Xg[~OJnɌĂ΂

   CAutoLock cObjectLock(m_pLock);

   // KvȃNX̍쐬
   if(m_inFormat.nSamplesPerSec == m_outFormat.nSamplesPerSec)
   {
      if( (m_inFormat.wFormatTag != m_outFormat.wFormatTag) || (m_inFormat.wBitsPerSample != m_outFormat.wBitsPerSample) )
      {
         if(m_pBitConv == NULL)
            m_pBitConv = new CBitConv();

         m_pBitConv->SetFormat(&m_inFormat, &m_outFormat);
      }

      if(m_inFormat.nChannels != m_outFormat.nChannels)
      {
         if(m_pChannelConv == NULL)
            m_pChannelConv = new CChannelConv();

         m_pChannelConv->SetFormat(&m_inFormat, &m_outFormat);
      }
   }
   else
   {
      if(m_pSamplingConv == NULL)
         m_pSamplingConv = new CSamplingConv();

      if(m_smpld == NULL)
         m_smpld = new SmplData[131072];

      // m_smpldz̏
      for(int i=0;i<0x20000;i++)
      {
         m_smpld[i].l = 0.0;
         m_smpld[i].r = 0.0;
      }

      m_nSmplSize = 0;
      m_pSamplingConv->ConvertInit(&m_inFormat, &m_outFormat, m_nAccurate);
   }

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
void CWavConFilter::ConvertToDouble(BYTE *pbData, int nDataLength)
{  // pbDatam_smpldzɊi[

   int i, j, nPos;

   int nInSmplSize = nDataLength / m_inFormat.nBlockAlign;


   if(m_inFormat.wFormatTag == WAVE_FORMAT_PCM)
   {
      if(m_inFormat.wBitsPerSample == 16)
      {
         j = 0;

         for(i=0;i<nInSmplSize;i++)
         {
            nPos = (m_nSmplSize + i) & 0x1ffff;

            m_smpld[nPos].l = short((pbData[j+1] << 8) | pbData[j]);
            j += 2;

            if(m_inFormat.nChannels == 2)
            {
               m_smpld[nPos].r = short((pbData[j+1] << 8) | pbData[j]);
               j += 2;
            }
         }
      }
      else if(m_inFormat.wBitsPerSample == 8)
      {
         j = 0;

         for(i=0;i<nInSmplSize;i++)
         {
            nPos = (m_nSmplSize + i) & 0x1ffff;

            m_smpld[nPos].l = pbData[j] - 128.0;
            j++;

            if(m_inFormat.nChannels == 2)
            {
               m_smpld[nPos].r = pbData[j] - 128.0;
               j++;
            }
         }
      }
      else if(m_inFormat.wBitsPerSample == 24)
      {
         j = 0;

         for(i=0;i<nInSmplSize;i++)
         {
            nPos = (m_nSmplSize + i) & 0x1ffff;

            if(pbData[j+2] & 0x80)
               m_smpld[nPos].l = int(0xff000000 | (pbData[j+2] << 16) | (pbData[j+1] << 8) | pbData[j]);
            else
               m_smpld[nPos].l = (pbData[j+2] << 16) | (pbData[j+1] << 8) | pbData[j];

            j += 3;

            if(m_inFormat.nChannels == 2)
            {
               if(pbData[j+2] & 0x80)
                  m_smpld[nPos].r = int(0xff000000 | (pbData[j+2] << 16) | (pbData[j+1] << 8) | pbData[j]);
               else
                  m_smpld[nPos].r = (pbData[j+2] << 16) | (pbData[j+1] << 8) | pbData[j];
   
               j += 3;
            }
         }
      }
      else if(m_inFormat.wBitsPerSample == 32)
      {
         j = 0;

         for(i=0;i<nInSmplSize;i++)
         {
            nPos = (m_nSmplSize + i) & 0x1ffff;

            m_smpld[nPos].l = int((pbData[j+3] << 24) | (pbData[j+2] << 16) | (pbData[j+1] << 8) | pbData[j]);
            j += 4;

            if(m_inFormat.nChannels == 2)
            {
               m_smpld[nPos].r = int((pbData[j+3] << 24) | (pbData[j+2] << 16) | (pbData[j+1] << 8) | pbData[j]);
               j += 4;
            }
         }
      }
   }
   else // if(m_inFormat.wFormatTag == WAVE_FORMAT_IEEE_FLOAT)
   {
      if(m_inFormat.wBitsPerSample == 32)
      {
         float fData;

         j = 0;

         for(i=0;i<nInSmplSize;i++)
         {
            nPos = (m_nSmplSize + i) & 0x1ffff;
            ::CopyMemory(&fData, &pbData[j], 4);
            m_smpld[nPos].l = fData;
            j += 4;

            if(m_inFormat.nChannels == 2)
            {
               ::CopyMemory(&fData, &pbData[j], 4);
               m_smpld[nPos].r = fData;
               j += 4;
            }
         }
      }
      else if(m_inFormat.wBitsPerSample == 64)
      {
         j = 0;

         for(i=0;i<nInSmplSize;i++)
         {
            nPos = (m_nSmplSize + i) & 0x1ffff;
            ::CopyMemory(&m_smpld[nPos].l, &pbData[j], 8);
            j += 8;

            if(m_inFormat.nChannels == 2)
            {
               ::CopyMemory(&m_smpld[nPos].r, &pbData[j], 8);
               j += 8;
            }
         }
      }
   }

   m_nSmplSize = (m_nSmplSize + nInSmplSize) & 0x1ffff;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CWavConFilter::Receive(IMediaSample *pInSample)
{  // ϊ

   HRESULT hr;
   BYTE *pbInBuffer = NULL;
   BYTE *pbOutBuffer = NULL;
   IMediaSample *pOutSample = NULL;

   CAutoLock lck(&m_myLock);

   // AM_STREAM_MEDIAȊOȂʉ߂
   AM_SAMPLE2_PROPERTIES * const pProps = m_pInput->SampleProps();
   if(pProps->dwStreamId != AM_STREAM_MEDIA)
      return m_pOutput->Deliver(pInSample);

   // o͂̏
   hr = m_pOutput->GetDeliveryBuffer(&pOutSample, NULL, NULL, 0);
   if FAILED(hr)
      return S_FALSE;

   if(pInSample->IsSyncPoint() == S_OK)
      pOutSample->SetSyncPoint(TRUE);
 
   if(pInSample->IsPreroll() == S_OK)
      pOutSample->SetPreroll(TRUE);

   if(pInSample->IsDiscontinuity() == S_OK || m_bSampleSkipped)
      pOutSample->SetDiscontinuity(TRUE);

   pInSample->GetPointer(&pbInBuffer);
   pOutSample->GetPointer(&pbOutBuffer);

   int nLength = 0;

   if(m_inFormat.nSamplesPerSec == m_outFormat.nSamplesPerSec)
   {  // TvO[gϊȂꍇ

      if(m_inFormat.wFormatTag == m_outFormat.wFormatTag && m_inFormat.nChannels == m_outFormat.nChannels && m_inFormat.wBitsPerSample == m_outFormat.wBitsPerSample)
      {  // ύXȂꍇA̓f[^̂܂܏o̓f[^ɓn

         ::CopyMemory(pbOutBuffer, pbInBuffer, (int)pInSample->GetActualDataLength());
         nLength = (int)pInSample->GetActualDataLength();
      }
      else if( (m_inFormat.wFormatTag != m_outFormat.wFormatTag) || (m_inFormat.wBitsPerSample != m_outFormat.wBitsPerSample) )
      {  // rbgϊꍇ

         if(m_inFormat.nChannels == m_outFormat.nChannels)
         {  // rbĝݕϊꍇ
            hr = m_pBitConv->Convert(pbInBuffer, (int)pInSample->GetActualDataLength(), pbOutBuffer, &nLength);
            if(FAILED(hr)) return S_FALSE;
         }
         else if(m_inFormat.nChannels == 1 && m_outFormat.nChannels == 2)
         {  // mXeIϊƃrbgϊsꍇ

            hr = m_pBitConv->Convert(pbInBuffer, (int)pInSample->GetActualDataLength(), pbOutBuffer, &nLength);
            if(FAILED(hr)) return S_FALSE;

            hr = m_pChannelConv->Convert(pbOutBuffer, nLength, &nLength);
            if(FAILED(hr)) return S_FALSE;
         }
         else if(m_inFormat.nChannels == 2 && m_outFormat.nChannels == 1)
         {  // XeImϊƃrbgϊsꍇ

            hr = m_pChannelConv->Convert(pbInBuffer, (int)pInSample->GetActualDataLength(), &nLength);
            if(FAILED(hr)) return S_FALSE;

            hr = m_pBitConv->Convert(pbInBuffer, nLength, pbOutBuffer, &nLength);
            if(FAILED(hr)) return S_FALSE;
         }
      }
      else if(m_inFormat.nChannels != m_outFormat.nChannels)
      {  // `l̂ݕϊꍇ
         hr = m_pChannelConv->Convert(pbInBuffer, (int)pInSample->GetActualDataLength(), pbOutBuffer, &nLength);
      }
   }
   else  // TvO[gϊꍇ
   {
      // ̓f[^Am_smpldzɊi[
      ConvertToDouble(pbInBuffer, (int)pInSample->GetActualDataLength());

      // TvO[gϊ
      m_pSamplingConv->Convert(m_smpld, (int)pInSample->GetActualDataLength() / m_inFormat.nBlockAlign, pbOutBuffer, &nLength);
   }

   // fBAԂ̐ݒ
   REFERENCE_TIME rtStart, rtStop;
   pInSample->GetTime(&rtStart, &rtStop);
   pOutSample->SetTime(&rtStart, &rtStop);

   // o̓f[^̒ݒ
   pOutSample->SetActualDataLength(nLength);

   // f[^o
   m_pOutput->Deliver(pOutSample);

   // ㏈
   SAFE_RELEASE(pOutSample);
  
   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CWavConFilter::StopStreaming()
{  // ~ꍇɌĂ΂

   CAutoLock lck(&m_myLock);

   return S_OK;
}
// -----------------------------------------------------------------------------------------------------------------------------------
HRESULT CWavConFilter::EndOfStream(void)
{  // Xg[~OIɌĂ΂

   CAutoLock lck(&m_myLock);

   if(m_inFormat.nSamplesPerSec != m_outFormat.nSamplesPerSec)
   {  // TvOgϊꍇ
      // ܂ϊĂȂf[^ϊďo͂

      int nLength;

      // f[^Iʒu1000̃f[^𖳉ɂĂ
      for(int i=0;i<1000;i++)
      {
         m_smpld[(m_nSmplSize + i) & 0x1ffff].l = 0.0;
         m_smpld[(m_nSmplSize + i) & 0x1ffff].r = 0.0;
      }


      // o͂̏
      HRESULT hr;
      IMediaSample *pOutSample = NULL;
      BYTE *pbOutBuffer = NULL;

      hr = m_pOutput->GetDeliveryBuffer(&pOutSample, NULL, NULL, 0);
      if FAILED(hr)
         return S_FALSE;

      pOutSample->GetPointer(&pbOutBuffer);
      pOutSample->SetSyncPoint(TRUE);
      pOutSample->SetDiscontinuity(FALSE);
      pOutSample->SetPreroll(FALSE);

      // TvOϊ
      m_pSamplingConv->ConvertEnd(m_smpld, m_nSmplSize, pbOutBuffer, &nLength);

      // o̓f[^̒ݒ
      pOutSample->SetActualDataLength(nLength);

      // f[^o
      m_pOutput->Deliver(pOutSample);

      // ㏈
      SAFE_RELEASE(pOutSample);
   }

   return CTransformFilter::EndOfStream();
}
// -----------------------------------------------------------------------------------------------------------------------------------
