/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.inaver.task;

import java.io.PrintWriter;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Vector;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.aibonware.inaver.*;
import com.aibonware.inaver.model.*;
import com.aibonware.nvrlib.model.*;
import com.aibonware.inaver.store.*;
import com.aibonware.inaver.url.SearchUrl;


public class SearchThreadTitleTask extends UserTask {
    private final String author;
    private final Vector<String> keywords;
    private final int boardId;
    private final int country;
    private final int start;
    private final int end;
    private final int range;
    private final String userName;
    private final SearchUrl searchUrl;

    private SQLCanceler canceler;
    volatile public boolean canceled = false;
     volatile public Vector<ThreadFullPath> threads = null;

    public SearchThreadTitleTask(
            String author,
            Vector<String> keywords,
            int boardId,
            int country,
            int start,
            int end,
            int range,
            SearchUrl searchUrl,
            UserConfig user) {
        super("ٌ", user);
        this.author = author;
        this.keywords = keywords;
        this.boardId = boardId;
        this.country = country;
        this.start = start;
        this.end = end;
        this.range = range;
        this.searchUrl = searchUrl;
        this.userName = user.userName;

        canceler = new SQLCanceler() {
            private Statement sql;
            
            public void setStatement(Statement sql) {
                this.sql = sql;
            }

            public void cancel() {
                try {
                    sql.cancel();
                } catch(SQLException e) {
                    Log.err("cancel operation failed");
                    Log.err(e);
                }
            }
        };
    }

    public void execute() {
    	startTime = System.currentTimeMillis();
    	   
    	try {
            if(!canceled) {
         	
               Log.info(userName + " search thread-title author=" + author + " keywords=" + keywords + " b=" + boardId + " c=" + searchUrl.country + " s=" + searchUrl.start + " e=" + searchUrl.end + " r=" + range);
              	BackupStore store = INaver.getInstance().getStoreProvider().openBackupStore();
                threads = store.searchThreadTitle(author, keywords, boardId, country, start, end - start + 1 + 1, range, canceler);
                store.close();
                Log.info(userName + " search done");
                
            }
        } catch(StoreException e) {
            exception = e;
        }

        endTime = System.currentTimeMillis();
        completed = true;
   }

    public SearchThreadTitleResult getResult(UserConfig user) {
    	boolean searchDone = false;
        if(waitDone(Env.getEnv().searchLimitTime * 1000)) searchDone = true;

        if(searchDone) {
        	return new SearchThreadTitleResult(searchUrl, threads);
        }

        return null;
    }
    
    public void dispResult(UserConfig user, PrintWriter out, HttpServletRequest request, HttpServletResponse response) throws StoreException {
    	SearchThreadTitleResult result = getResult(user);
    	result.dispResultContents(out, request, response);
    }

   public void cancel() {
        canceler.cancel();
        canceled = true;
    }
}
