/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.inaver.store;

import java.io.*;
import java.sql.*;
import java.text.*;
import java.util.*;

import org.w3c.dom.*;

import com.aibonware.inaver.INaver;
import com.aibonware.inaver.Log;
import com.aibonware.nvrlib.model.*;

public class BackupStore extends StoreBase {
	public static final int THREAD_UPDATE = 0;
	public static final int THREAD_DELETE = 1;

	public static final Object ThreadLogSync = new Object();

	/**
	 * RXgN^B
	 */
	public BackupStore(String dbHost, String dbPass, String dbname) throws StoreException {
		super("backup", dbHost, dbPass, dbname);
	}

	public ArticleFullPathList searchArticle(String author, Vector<String> keywords, int boardId, int country, int start, int num, int range, SQLCanceler canceler) throws StoreException {
		ArticleFullPathList v = new ArticleFullPathList();
        canceler.setStatement(sql);
        
		try {
			String where = "";
			String delim = "";

			if(boardId != 0) {
				where = where + delim + "BoardID=" + boardId;
				delim = " and ";
			}

            if(range > 0) {
                ResultSet rs = sql.executeQuery("select MaxArticleID from Boards where BoardID=" + boardId);
                rs.next();
                int maxArticleId = rs.getInt("MaxArticleID");
                rs.close();
                
                where = where + delim + "ArticleID>" + (maxArticleId - range * 100000);
                delim = " and ";
            }
 
			if(!author.equals("")) {
				where = where + delim + "Author=" + safer.text(author);
				delim = " and ";
			}

            if(country != 0) {
                where = where + delim + "Country=\"" + country + "\"";
                delim = " and ";
            }
            
			for(int i=0; i<keywords.size(); i++) {
				where = where + delim + "Content like " + safer.like(keywords.elementAt(i));
				delim = " and ";
			}

			String useIndex;
			
			if(!author.equals("")) {
				useIndex = "use index (idx_author)";
			} else {
				useIndex = "use index (idx_order2)";
			}
			
			String sqlText = "select * from Articles " + useIndex + " ";
			
			if(!where.equals("")) {
				sqlText = sqlText + "where " + where;
			}

			sqlText = sqlText + " order by BoardID, ArticleID desc limit " + start + "," + num;
			
			ResultSet rs = sql.executeQuery(sqlText);

			while(rs.next()) {
				ArticleFullPath art = new ArticleFullPath(
					rs.getInt("BoardID"),
					rs.getInt("NID"),
					rs.getString("Author"),
					rs.getInt("Country"),
					rs.getString("Content"),
					rs.getInt("ArticleID"),
					safer.sqlDatetimeToDate(rs.getString("PostTime")),
					rs.getInt("ParentArticleID"));
					
				v.addArticle(art);		
			}

			rs.close();
			return v;

		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	public Vector<ThreadFullPath> searchThread(String author, Vector<String> keywords, int boardId, int country, int start, int num, int range, SQLCanceler canceler) throws StoreException {
		Vector<ThreadFullPath> v = new Vector<ThreadFullPath>();
        canceler.setStatement(sql);
        		
		try {
			String where = "";
			String delim = "";
			
			if(boardId != 0) {
				where = where + delim + "BoardID=" + boardId;
				delim = " and ";
			}
            
            if(range > 0) {
                ResultSet rs = sql.executeQuery("select max(NID) as mnid from Threads where BoardID=" + boardId);
                rs.next();
                int maxNID = rs.getInt("mnid");
                rs.close();
                
                where = where + delim + "NID>" + (maxNID- range * 10000);
                delim = " and ";
            }
            
			if(!author.equals("")) {
				where = where + delim + "Poster=" + safer.text(author);
				delim = " and ";
			}

            if(country != 0) {
                where = where + delim + "Country=\"" + country + "\"";
                delim = " and ";
            }

			for(int i=0; i<keywords.size(); i++) {
				where = where + delim + "Content like " + safer.like(keywords.elementAt(i));
				delim = " and ";
			}

			String sqlText = "select * from Threads ";
			
			if(!where.equals("")) {
				sqlText = sqlText + "where " + where;
			}

			sqlText = sqlText + " order by BoardID, NID desc limit " + start + "," + num;
			
			ResultSet rs = sql.executeQuery(sqlText);

			while(rs.next()) {
				ThreadFullPath thread = new ThreadFullPath(
					rs.getInt("BoardID"),
					rs.getInt("Depth"),
					safer.sqlStr(rs.getString("DispID")),
					safer.sqlBool(rs.getString("IsAdmin")),
					rs.getInt("NID"),
					rs.getInt("Country"),
					safer.sqlStr(rs.getString("Title")),
					rs.getInt("ArticleNum"),
					safer.sqlStr(rs.getString("Poster")),
					safer.sqlDatetimeToDate(rs.getString("CreateDate")),
					safer.sqlDatetimeToDate(rs.getString("ModifiedDate")),
					rs.getInt("ViewNum"),
					safer.sqlStr(rs.getString("Mail")),
					safer.sqlDatetimeToDate(rs.getString("LastCrawlTime")),
					safer.sqlBool(rs.getString("IsHot")));
				
				v.addElement(thread);		
			}

			rs.close();
			return v;

		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}
	
	private static class PoemUser {
		public final int poemerId;
		public final String poemerName;
		public final int lastNid;
		
		public PoemUser(int poemerId, String poemerName, int lastNid) {
			this.poemerId = poemerId;
			this.poemerName = poemerName;
			this.lastNid = lastNid;
		}
	}

	private Vector<PoemUser> getPoemUsers(Statement sql) throws SQLException {
		Vector<PoemUser> users = new Vector<PoemUser>();
		
		ResultSet rs = sql.executeQuery("select * from poemdb.Poemers");

		while(rs.next()) {
			users.addElement(new PoemUser(
				rs.getInt("PoemerID"),
				rs.getString("PoemerName"),
				rs.getInt("LastNID")
			));
		}
		
		rs.close();
		return users;
	}
	
	public void updatePoemDB(String poemKeyword) throws StoreException {
		try {
			Vector<PoemUser> users = getPoemUsers(sql);
			
			for(PoemUser user: users) {
				sql.executeUpdate("insert into poemdb.Poems select null, " 
					+ user.poemerId + ", " 
					+ safer.text(user.poemerName) + ", " 
					+ "concat(\"http://bbs.enjoykorea.jp/tbbs/read.php?board_id=thistory&nid=\", NID), "
					+ "CreateDate, Title, RawContent, \"\""
					+ " from inavdb2.Threads use index(idx_poster)"
					+ " where BoardID=8"
					+ " and Poster=" + safer.text(user.poemerName) 
					+ " and NID>" + user.lastNid
					+ " and Title like " + safer.like(poemKeyword));
				
				ResultSet rs = sql.executeQuery("select max(NID) as NewLastNID from Threads use index(idx_poster) where BoardID=8 and Poster=" + safer.text(user.poemerName));
				rs.next();
				int newLastNid = rs.getInt("NewLastNID");
				rs.close();

				sql.executeUpdate("update poemdb.Poemers set LastNID=" + newLastNid + " where PoemerID=" + user.poemerId);
			}

		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}
	
	public Vector<ThreadFullPath> searchThreadTitle(String author, Vector<String> keywords, int boardId, int country, int start, int num, int range, SQLCanceler canceler) throws StoreException {
		Vector<ThreadFullPath> v = new Vector<ThreadFullPath>();
        canceler.setStatement(sql);
        
		try {
			String where = "";
			String delim = "";
			
			if(boardId != 0) {
				where = where + delim + "BoardID=" + boardId;
				delim = " and ";
			}
			
            if(range > 0) {
                ResultSet rs = sql.executeQuery("select max(NID) as mnid from Threads where BoardID=" + boardId);
                rs.next();
                int maxNID = rs.getInt("mnid");
                rs.close();
                
                where = where + delim + "NID>" + (maxNID- range * 10000);
                delim = " and ";
            }

            if(!author.equals("")) {
                where = where + delim + "Poster=" + safer.text(author);
                delim = " and ";
            }

            if(country != 0) {
				where = where + delim + "Country=\"" + country + "\"";
				delim = " and ";
			}

			for(int i=0; i<keywords.size(); i++) {
				where = where + delim + "Title like " + safer.like(keywords.elementAt(i));
				delim = " and ";
			}

			String sqlText = "select * from Threads ";
			
			if(!where.equals("")) {
				sqlText = sqlText + "where " + where;
			}

			sqlText = sqlText + " order by BoardID, NID desc limit " + start + "," + num;
			
			ResultSet rs = sql.executeQuery(sqlText);

			while(rs.next()) {
				ThreadFullPath thread = new ThreadFullPath(
					rs.getInt("BoardID"),
					rs.getInt("Depth"),
					safer.sqlStr(rs.getString("DispID")),
					safer.sqlBool(rs.getString("IsAdmin")),
					rs.getInt("NID"),
					rs.getInt("Country"),
					safer.sqlStr(rs.getString("Title")),
					rs.getInt("ArticleNum"),
					safer.sqlStr(rs.getString("Poster")),
					safer.sqlDatetimeToDate(rs.getString("CreateDate")),
					safer.sqlDatetimeToDate(rs.getString("ModifiedDate")),
					rs.getInt("ViewNum"),
					safer.sqlStr(rs.getString("Mail")),
					safer.sqlDatetimeToDate(rs.getString("LastCrawlTime")),
					safer.sqlBool(rs.getString("IsHot")));
				
				v.addElement(thread);		
			}

			rs.close();
			return v;

		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	private static final SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
	private static final SimpleDateFormat tf = new SimpleDateFormat("MM/dd HH:mm");

	private void outCol(PrintWriter out, String text) {
		out.println("\t\t<td>" + text + "</td>");
	}
	
	private void outCol(PrintWriter out, int i) {
		out.println("\t\t<td>" + i + "</td>");
	}
	
	// XbhX
	public void outputArticlesPerThread(
			String title, 
			java.util.Date firstDate, 
			java.util.Date lastDate, 
			int limitNum, 
			PrintWriter out
	) throws StoreException {
		try {
			Log.info("outputArticlesPerThread " + df.format(firstDate));
			Statement sql2 = con.createStatement();

			String sFirstDate = "\"" + df.format(firstDate) + " 00:00:00\"";
			String sLastDate = "\"" + df.format(lastDate) +  " 00:00:00\"";
			BoardList boards = queryBoardList(false, false);
		
			out.println("<h3>" + title + "</h3>");

			String sqlText = 
				"select Threads.BoardID, Threads.NID, count(*) as cnt "
				+ "from Threads, Articles "
				+ "where "
				+ " Threads.BoardID=Articles.BoardID and Threads.NID=Articles.NID "
				+ " and Threads.CreateDate >= " + sFirstDate + " "
				+ " and Threads.CreateDate < " + sLastDate + " "
				+ " and Threads.Country!='0' "
				+ "group by Threads.BoardID, Threads.NID "
				+ "order by cnt desc limit 0," + limitNum;

			ResultSet rs = sql2.executeQuery(sqlText);

			int rank = 1;
			int prevArticleNum = -1;
			int index = 0;

			while(rs.next()) {
				int boardId = rs.getInt("Threads.BoardID");
				int nid = rs.getInt("Threads.NID");
				int cnt = rs.getInt("cnt");

				NThread thread = queryThread(boardId, nid);
				Board board = boards.getBoardById(boardId);

				index++;
				if(prevArticleNum != cnt) rank = index;

				prevArticleNum = cnt;

				out.print(rank + ".");
				out.print("[" + "<a href=\"" + INaver.getInstance().serviceBaseUrl + "thread?b=" + boardId + "&id=" + nid + "\">" + cnt + "</a>" + "] ");
				out.print(INaver.getCountryHtml(thread.country) + " ");
				out.print("<b>" + thread.poster + "</b> ");
				out.print(thread.title + " ");
				out.print(tf.format(thread.createDate) + " ");
				out.print(board.boardName);
				out.println("<br>");
			}
			
			out.println("<br>");
			
			rs.close();
			sql2.close();
	
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	private boolean existsEnjoyID(String name, int country) throws SQLException {
		Statement state = con.createStatement();
		ResultSet result = state.executeQuery("select PersonID from idlistdb.Persons where Persons.Name=" + safer.text(name) + " and Country=\"" + country + "\"");

		if(!result.next()) {
			result.close();
			result = state.executeQuery("select PersonID from idlistdb.PersonAlias where AliasName=" + safer.text(name) + " and Country=\"" +country + "\"");

			if(!result.next()) {
				result.close();
				state.close();
				return false;
			}
		}

		int personId = result.getInt("PersonID");
		result.close();

		result = state.executeQuery("select Visible from idlistdb.Persons where PersonID=" + personId);

		result.next();
		boolean visible = safer.sqlBool(result.getString("Visible"));
		result.close();
		state.close();

		return visible;
	}
	
	private String escapeNGWord(Vector<String> ngwords, String name) throws SQLException {
		for(int i=0; i<ngwords.size(); i++) {
			int index = name.indexOf(ngwords.elementAt(i));

			if(index >= 0) {
				name = name.substring(0, index+1)
					+ "<i></i>" 
					+ name.substring(index + 1);
			}
		}
		
		return name;
	}

	private Vector<String> getNGWords() throws SQLException {
		Statement state = con.createStatement();
		ResultSet result = state.executeQuery("select * from NGWords");
		Vector<String> ngwords = new Vector<String>();

		while(result.next()) {
			ngwords.addElement(result.getString("Word"));
		}
		
		result.close();
		state.close();
		
		return ngwords;
	}
	
	private void outField(PrintWriter out, String s, int width) {
		out.print(s);
		
		for(int i=s.length(); i<width; i++) out.print(" ");
		if(s.length() > width) out.print(" ");
	}
	
	private void outField(PrintWriter out, int s, int width) {
		outField(out, Integer.toString(s), width);
	}
	
	// lCXbhe
	public void outputPopularPoster(
			String title, 
			java.util.Date firstDate, 
			java.util.Date lastDate, 
			int limitNum, 
			PrintWriter out
	) throws StoreException {
		try {
			Log.info("outputPopularPoster " + df.format(firstDate));

			String sFirstDate = "\"" + df.format(firstDate) + " 00:00:00\"";
			String sLastDate = "\"" + df.format(lastDate) +  " 00:00:00\"";

			Vector<String> ngwords = getNGWords();
			
			sql.executeUpdate("delete from PosterAvg");
			
			sql.executeUpdate(
				"insert into PosterAvg "
				+ "select if(GroupName is null, Threads.Poster, GroupName) as Poster, Threads.Country, avg(ArticleNum) as cnt "
				+ "from Threads left join PosterAliases "
				+ "on Threads.Poster = PosterAliases.Poster and Threads.Country = PosterAliases.Country "
				+ "where CreateDate >= " + sFirstDate + " and CreateDate < " + sLastDate + " "
				+ "group by Poster, Country having cnt>=10 ");
						
			sql.executeUpdate("delete from PosterThreadNum");
			
			sql.executeUpdate(
				"insert into PosterThreadNum "
				+ "select if(GroupName is null, Threads.Poster, GroupName) as Poster, Threads.Country, count(*) as cnt "
				+ "from Threads left join PosterAliases "
				+ "on Threads.Poster = PosterAliases.Poster and Threads.Country = PosterAliases.Country "
				+ "where CreateDate >= " + sFirstDate + " and CreateDate < " + sLastDate + " "
				+ "group by Poster, Country having cnt>=5");						

			ResultSet rs = sql.executeQuery(
				"select PosterAvg.Poster, PosterAvg.Country, PosterAvg.ArtAvg, PosterThreadNum.ThreadNum "
				+ "from PosterAvg, PosterThreadNum "
				+ "where PosterAvg.Poster = PosterThreadNum.Poster and PosterAvg.Country = PosterThreadNum.Country "
				+ "and PosterAvg.Country!='0' "
				+ "order by PosterAvg.ArtAvg desc limit 0," + limitNum);

			out.println("" + title + "<br><br>");

			out.println("Wvԓ5ȏ̃Xbh𓊍ee҂ɂāAeXbhɂReply̕ϒl̑ɃLOWvBNaver^c҂ɂƂĂ͏Wq͂̂A肪݁B邢͏㋉ނtLOB<br>");
/*
			out.println("<table border=\"1\">");

			out.println("\t<tr>");
			outCol(out, "");
			outCol(out, "");
			outCol(out, "e");
			outCol(out, "ϔReply");
			outCol(out, "Xbh");
			out.println("\t</tr>");
*/
			out.print("<pre>");
	
			int rank = 1;
			double prevArtAvg = -1;
			int index = 0;

			DecimalFormat nf = new DecimalFormat("###.##");

			while(rs.next()) {
				String poster = rs.getString("PosterAvg.Poster");
				int country = rs.getInt("PosterAvg.Country");
				double artAvg = rs.getDouble("PosterAvg.ArtAvg");
				int threadNum = rs.getInt("PosterThreadNum.ThreadNum");
				
				index++;
				if(prevArtAvg != artAvg) rank = index;

				prevArtAvg = artAvg;

//				out.println("\t<tr>");
				// outCol(out, rank);
				outField(out, rank, 4);
				
				if(poster.charAt(0) == '*') {
//					outCol(out, "*");
					outField(out, "*", 2);
				} else {
//					outCol(out, INaver.getCountryHtml(country));
					outField(out, INaver.getShortCountryHtml(country), 2);
				}
				
//				if(existsEnjoyID(poster, country)) {
//					outCol(out, 
//						"<b><a href=\"http://enjoyid.net/detail?"
//						+ "id=" + poster + "&c=" + country + "\"><u>" + escapeNGWord(ngwords, poster) + "</u></a></b>");
//				} else {
//					outCol(out, "<b>" + poster + "</b>");
					outField(out, escapeNGWord(ngwords, poster), 17);
//				}
				
				
//				outCol(out, nf.format(artAvg));
//				outCol(out, threadNum);
//				out.println("\t</tr>");
				
				outField(out, nf.format(artAvg), 8);
				out.println(threadNum);
			}

//			out.println("</table>");
//			out.println("<br>");
			out.println("</pre>");
			rs.close();
	
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}
	
	// eҖX
	public void outputArticlesPerPoster(
			String title, 
			java.util.Date firstDate, 
			java.util.Date lastDate, 
			int limitNum, 
			PrintWriter out,
			int uniqueLimitNum
	) throws StoreException {
		try {
			Log.info("outputArticlesPerPoster " + df.format(firstDate));

			String sFirstDate = "\"" + df.format(firstDate) + " 00:00:00\"";
			String sLastDate = "\"" + df.format(lastDate) +  " 00:00:00\"";

			Vector<String> ngwords = getNGWords();
			
			out.println("" + title + "<br>");

//			out.println("ʖANAVERplXgBQlURL: <a href=\"http://bbs.enjoykorea.jp/tbbs/read.php?board_id=thistory&nid=1313625\"><u>TNAVERLOA҂̃KChC</u></a><br>Rsy25%𒴂ꍇ͂̊tLĂ܂B<br><br>");

			ResultSet rs = sql.executeQuery(
				"select if(GroupName IS NULL, Author, GroupName) as Author, Articles.Country, count(*) as cnt "
				+ "from Articles left join PosterAliases "
				+ "on Articles.Author = PosterAliases.Poster and Articles.Country = PosterAliases.Country "
				+ "where PostTime >= " + sFirstDate + " and PostTime < " + sLastDate + " "
				+ "group by Author, Country order by cnt desc limit 0," + limitNum);

			int rank = 1;
			int prevArtNum = -1;
			int index = 0;

//			DecimalFormat nf = new DecimalFormat("###.##");

			Statement state2 = con.createStatement();
			
			int count = 0;
			
//			out.println("<table border=\"0\"><tr>");
			out.print("<pre>");

			while(rs.next()) {
/*				if((count % 50) == 0) {
					out.println("<td><table border=\"1\">");

					out.println("\t<tr>");
					outCol(out, "");
					outCol(out, "");
					outCol(out, "e");
					outCol(out, "Reply");
					out.println("\t</tr>");
				}
*/				
				String poster = rs.getString("Author");
				int country = rs.getInt("Country");
				int artNum = rs.getInt("cnt");
				
				index++;
				if(prevArtNum != artNum) rank = index;

				prevArtNum = artNum;

				ResultSet rs2 = state2.executeQuery(
					"select count(distinct Content) from Articles  left join PosterAliases "
					+ "on Articles.Author = PosterAliases.Poster and Articles.Country = PosterAliases.Country "
					+ "where PostTime >= " + sFirstDate + " and PostTime < " + sLastDate + " "
					+ "and Author = " + safer.text(poster) + " and Articles.Country=\"" + country + "\"");

				rs2.next();
				int dupNum = artNum - rs2.getInt(1) + 1;
				rs2.close();

				String numText = "" + artNum;

				if(100 * dupNum / artNum >= uniqueLimitNum) {
					numText = numText + " (" + (100 * dupNum / artNum) + "%)";
				}

//				out.println("\t<tr>");
//				outCol(out, rank);
				
				outField(out, rank, 4);
				
				if(poster.charAt(0) == '*') {
//					outCol(out, "*");
					outField(out, "*", 2);
				} else {
//					outCol(out, INaver.getCountryHtml(country));
					outField(out, INaver.getShortCountryHtml(country), 2);
				}
				
//				if(existsEnjoyID(poster, country)) {
//					outCol(out, 
//						"<b><a href=\"http://enjoyid.net/detail?"
//						+ "id=" + poster + "&c=" + country + "\"><u>" + escapeNGWord(ngwords, poster) + "</u></a></b>");
//				} else {
//					outCol(out, "<b>" + escapeNGWord(ngwords, poster) + "</b>");
					outField(out, escapeNGWord(ngwords, poster), 17);
//				}

//				outCol(out, numText);
//				out.println("\t</tr>");
				out.println(numText);
				
				count++;
				
//				if((count % 50) == 0) {
//					out.println("</table></td>");
//				}
			}
			
			state2.close();

//			out.println("</tr></table>");
			out.println("</pre>");

			out.println("<br>");
			rs.close();
	
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	// eҖXbh
	public void outputThreadsPerPoster(
			String title, 
			java.util.Date firstDate, 
			java.util.Date lastDate, 
			int limitNum, 
			PrintWriter out
	) throws StoreException {
		try {
			Log.info("outputThreadsPerPoster " + df.format(firstDate));

			String sFirstDate = "\"" + df.format(firstDate) + " 00:00:00\"";
			String sLastDate = "\"" + df.format(lastDate) +  " 00:00:00\"";
			
			Vector<String> ngwords = getNGWords();
			
			ResultSet rs = sql.executeQuery(
				"select if(GroupName is null, Threads.Poster, GroupName) as Poster, Threads.Country, count(*) as cnt, avg(ArticleNum) as av "
				+ "from Threads left join PosterAliases "
				+ "on Threads.Poster = PosterAliases.Poster and Threads.Country = PosterAliases.Country "
				+ "where CreateDate >= " + sFirstDate + " and CreateDate < " + sLastDate + " "
				+ "group by Poster, Country order by cnt desc limit 0," + limitNum);
			
			out.println("" + title + "<br>");
			
//			out.println("eXbhŃLOBReply1悤ȁAfȕǓ\sׂ͂߂܂傤B<br><br>");

/*			out.println("<table border=\"1\">");

			out.println("\t<tr>");
			outCol(out, "");
			outCol(out, "");
			outCol(out, "e");
			outCol(out, "گސ");
			outCol(out, "ϔReply");
			out.println("\t</tr>");
*/
			out.println("<pre>");
			
			int rank = 1;
			int prevArtNum = -1;
			int index = 0;

			DecimalFormat nf = new DecimalFormat("###.##");

			while(rs.next()) {
				String poster = rs.getString("Poster");
				int country = rs.getInt("Country");
				int artNum = rs.getInt("cnt");
				double artAvg = rs.getDouble("av");
				
				index++;
				if(prevArtNum != artNum) rank = index;

				prevArtNum = artNum;

//				out.println("\t<tr>");
//				outCol(out, rank);
				outField(out, rank, 4);
				
				if(poster.charAt(0) == '*') {
//					outCol(out, "*")
					outField(out, "*", 2);
				} else {
//					outCol(out, INaver.getCountryHtml(country));
					outField(out, INaver.getShortCountryHtml(country), 2);
				}
				
//				if(existsEnjoyID(poster, country)) {
//					outCol(out, 
//						"<b><a href=\"http://enjoyid.net/detail?"
//						+ "id=" + poster + "&c=" + country + "\"><u>" + escapeNGWord(ngwords, poster) + "</u></a></b>");
//				} else {
//					outCol(out, "<b>" + escapeNGWord(ngwords, poster) + "</b>");
					outField(out, escapeNGWord(ngwords, poster), 17);
//				}
				
//				outCol(out, artNum);
//				outCol(out, nf.format(artAvg));
//				out.println("\t</tr>");
				
				outField(out, artNum, 4);
				out.println(nf.format(artAvg));
			}

//			out.println("</table>");
			out.println("</pre>");
			
			out.println("<br>");
			rs.close();
	
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	// ԖX
	public void outputArticlesPerHour(
			String title, 
			java.util.Date firstDate, 
			java.util.Date lastDate, 
			PrintWriter out
	) throws StoreException {
		try {
			Log.info("outputArticlesPerHour " + df.format(firstDate));

			String sFirstDate = "\"" + df.format(firstDate) + " 00:00:00\"";
			String sLastDate = "\"" + df.format(lastDate) +  " 00:00:00\"";
			
			ResultSet rs = sql.executeQuery(
				"select hour(PostTime) as h, count(hour(PostTime)) as cnt "
				+ "from Articles "
				+ "where PostTime >= " + sFirstDate + " and PostTime < " + sLastDate + " "
				+ "group by hour(PostTime) order by h");
			
			out.println("<h3>" + title + "</h3>");

			out.println("<table border=\"1\">");

			int total = 0;

			out.println("\t<tr>");
			outCol(out, "");
			outCol(out, "Reply");
			out.println("\t</tr>");

			while(rs.next()) {
				int h = rs.getInt("h");
				int cnt = rs.getInt("cnt");
				total += cnt;
				
				out.println("\t<tr>");
				outCol(out, h);
				outCol(out, cnt);
				out.println("\t</tr>");
			}

			out.println("\t<tr>");
			outCol(out, "Total");
			outCol(out, total);
			out.println("\t</tr>");

			out.println("</table>");
			out.println("<br>");
			rs.close();
	
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}
	
	public void outputIDList( PrintWriter out) throws StoreException {
		try {
			ResultSet rs;

			out.println("\nXbhe()F");
			rs = sql.executeQuery("select distinct Poster from Threads where Country=\"1\" order by Poster");
			while(rs.next()) out.println(rs.getString("Poster"));
			rs.close();

			out.println("\nXbhe()F");
			rs = sql.executeQuery("select distinct Poster from Threads where Country=\"2\" order by Poster");
			while(rs.next()) out.println(rs.getString("Poster"));
			rs.close();

			out.println("\nXe()F");
			rs = sql.executeQuery("select distinct Author from Articles where Country=\"1\" order by Author");
			while(rs.next()) out.println(rs.getString("Author"));
			rs.close();

			out.println("\nXe()F");
			rs = sql.executeQuery("select distinct Author from Articles where Country=\"2\" order by Author");
			while(rs.next()) out.println(rs.getString("Author"));
			rs.close();

		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	public void updateMaxArticleId(int bid, int aid) throws StoreException {
        try {
            ResultSet rs;

            rs = sql.executeQuery("select MaxArticleID from Boards where BoardID=" + bid);
            rs.next();
            
            int curMaxArticleId = rs.getInt("MaxArticleID");
            rs.close();
            
            if(curMaxArticleId < aid) {
                sql.executeUpdate("update Boards set MaxArticleID=" + aid + " where BoardID=" + bid);
            }

        } catch(SQLException e) {
            throw new StoreException(e);
        }
	}

    // for NDES
    
    public Vector<Element> queryNDESThreadList(
            Document doc,
            Vector<String> attrs,
            int boardId, 
            int start, 
            int end,
            int num) throws StoreException {
        try {
            final int maxThreadNum = 100;
            Vector<Element> elements = new Vector<Element>();

            if(num > maxThreadNum) num = maxThreadNum;

            // SQL̍쐬
            String sqlText = "select * from Threads where boardID=" + boardId + " ";

            if(start >= 0) sqlText = sqlText + " and NID >= " + start;
            if(end >= 0) sqlText = sqlText + " and NID <= " + end;
            sqlText = sqlText + " order by NID desc ";

            sqlText = sqlText + " limit 0," + num;

            // w肳ꂽw肳ꂽXbhe擾
            ResultSet rs = sql.executeQuery(sqlText);

            // 擾ʂ̊eXbhɂāAXbh̓eɒǉ
            while(rs.next()) {
                Element thread = doc.createElement("Thread");

                ResultSetMetaData metaData = rs.getMetaData();
                
                for(int i=0; i<metaData.getColumnCount(); i++) {
                    String ndesName = toNDESColumnName(metaData.getColumnName(i+1), true);
                    if(attrs.contains(ndesName)) {
                        addNDESAttr(doc, rs.getString(i+1), thread, ndesName);
                    }
                }
                
                
                elements.addElement(thread);
            }

            rs.close();
            return elements;

        } catch(SQLException e) {
            throw new StoreException(e);
        }
    }
    
    public Vector<Element> queryNDESCommentList(
            Document doc,
            Vector<String> attrs,
            int boardId, 
            int nid,
            int start, 
            int end,
            int num) throws StoreException {
        try {
            final int maxCommentNum = 100;
            Vector<Element> elements = new Vector<Element>();

            if(num > maxCommentNum) num = maxCommentNum;

            // SQL̍쐬
            String sqlText = "select * from Articles where boardID=" + boardId + " and nid=" + nid;

            if(start >= 0) sqlText = sqlText + " and ArticleID >= " + start;
            if(end >= 0) sqlText = sqlText + " and ArticleID <= " + end;
            sqlText = sqlText + " order by ArticleID asc ";

            sqlText = sqlText + " limit 0," + num;

            ResultSet rs = sql.executeQuery(sqlText);

            while(rs.next()) {
                Element comment = doc.createElement("Comment");

                ResultSetMetaData metaData = rs.getMetaData();
                
                for(int i=0; i<metaData.getColumnCount(); i++) {
                    String ndesName = toNDESColumnName(metaData.getColumnName(i+1), false);
                    if(attrs.contains(ndesName)) {
                        addNDESAttr(doc, rs.getString(i+1), comment, ndesName);
                    }
                }
               
                elements.addElement(comment);
            }

            rs.close();
            return elements;

        } catch(SQLException e) {
            throw new StoreException(e);
        }
    }
    
    private void addNDESAttr(Document doc, String value, Element parent, String ndesName) {
        final SimpleDateFormat srcFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        final SimpleDateFormat destFormat = new SimpleDateFormat("yy/MM/dd HH:mm:ss");

        Element attr = doc.createElement(ndesName);

        if(ndesName.equals("Country")) {
            if(value.equals("1")) attr.setTextContent("JP");
            else if(value.equals("2")) attr.setTextContent("KR");
            else attr.setTextContent("NA");
        } else if(ndesName.endsWith("Date")) {
            try {
                attr.setTextContent(destFormat.format(srcFormat.parse(value)));
            } catch(ParseException e) {
                attr.setTextContent("00/00/00 00:00:00");
            }
        } else {
            attr.setTextContent(value);
        }
        parent.appendChild(attr);
    }

    private String toNDESColumnName(String colName, boolean nowThread) {
        final String[][] map = {
                {"NID",           "NID"},
                {"BoardID",       "BoardID"},
                {"DispID",        "DispID"},
                {"Depth",         "Depth"},
                {"Country",       "Country"},
                {"ArticleNum",    "NumberOfComments"},
                {"Poster",        "Writer"},
                {"CreateDate",    "CreateDate"},
                {"ViewNum",       "NumberOfRead"},
                {"Mail",           "Mail"},
                {"LastCrawlTime", "RegisterDate"},
                {"Title",         "Subject"},
                {"ModifiedDate",  "ModifyDate"},
                {"RawContent",    "Content"},
                {"ArticleID",     "CommentID"},
                {"Author",        "Writer"},
                {"PostTime",      "CreateDate"},
                {"Content",       "Content"},
        };

        int len = map.length;
        if(nowThread) len--;
        
        for(int i=0; i<len; i++) {
            if(map[i][0].equals(colName)) return map[i][1];
        }

        return "unknown column";
    }


//  O

 	public void addThreadLog(int verb, int boardId, int nid) throws StoreException {
 		synchronized(ThreadLogSync) {
 			try {
 				String sqlText = "insert into ThreadLog values(" + verb + ", " + boardId + ", " + nid + ")";
 				sql.executeUpdate(sqlText);

 			} catch(SQLException e) {
 				throw new StoreException(e);
 			}
 		}
 	}

 	class BNID {
 		public final int boardId;
 		public final int nid;
 		
 		public BNID(int boardId, int nid) {
 			this.boardId = boardId;
 			this.nid = nid;
 		}
 	}
 	
    public void syncStore(MasterStore master) throws StoreException {
 		synchronized(ThreadLogSync) {
 			try {
 				Log.info("begin sync");
 				
	    		// 폜wNG
	    		String sqlText = "select BoardID, NID from ThreadLog where Verb=" + THREAD_DELETE;    		
	    		ResultSet logs = sql.executeQuery(sqlText);
	    		
	    		// 폜w݂ꍇÃXbh폜
	    		while(logs.next()) {
	    			int boardId = logs.getInt("BoardID");
	    			int nid = logs.getInt("NID");

	    			deleteThread(boardId, nid);
	    		}
	
	    		logs.close();
	    		
	    		// XbhONG
		    	sqlText = "select distinct * from ThreadLog limit 0,1000";
		
		    	logs = sql.executeQuery(sqlText);
	
	    		Vector<BNID> bnids = new Vector<BNID>();
	    		
		    	while(logs.next()) {
	    			int boardId = logs.getInt("BoardID");
	    			int nid = logs.getInt("NID");
	
	    			bnids.addElement(new BNID(boardId, nid));
		    	}
		    	logs.close();

		    	for(int i=0; i<bnids.size(); i++) {
		    		BNID bnid = bnids.elementAt(i);
		    		syncThread(master, bnid.boardId, bnid.nid);
		    	}
		    	
		    	Log.info("end sync");
	    	} catch(SQLException e) {
	    		throw new StoreException(e);
	    	}
 		}
    }
    
    // ͓T[ôcâ݃T|[g
    private void syncThread(MasterStore master, int boardId, int nid) throws SQLException, StoreException {
    	// w肳ꂽXbh݂邩ǂ`FbN
    	Statement sql2 = con.createStatement();

    	ResultSet rs = sql2.executeQuery("select * from Threads where BoardID=" + boardId + " and NID=" + nid);
    	boolean exists = rs.next();
    	rs.close();

    	// Xbh̏
    	if(!exists) {
        	// ݂ĂȂAVK}
    		String sqlText = "insert into " + this.dbname + ".Threads select * from " + master.getDBName() + ".Threads "
	    		+ "where " + master.getDBName() + ".Threads.BoardID=" + boardId 
	    		+ " and " + master.getDBName() + ".Threads.NID=" + nid;

        	sql2.executeUpdate(sqlText);
        	Log.info("sync insert thread b=" + boardId + ", nid=" + nid);

    	} else {
    		// ݂ĂAXV
    		NThread thread = master.queryThread(boardId, nid);
    		String rawContents = master.queryRawThreadContents(boardId, nid);
    		String contents = master.queryThreadContents(boardId, nid);
  
    		String sqlText = "update " + this.dbname + ".Threads set "
	    		+ "Content=" + safer.text(contents) 
	    		+ ", RawContent=" + safer.text(rawContents) 
	    		+ ", ImageURL=" + safer.text(thread.imageUrl)
	    		+ ", IsHot=" + safer.bool(thread.isHot)
	    		+ ", ArticleNum= " + thread.articleNum
	    		+ " where BoardID=" + boardId + " and NID=" + nid;

    		sql2.executeUpdate(sqlText);
        	Log.info("sync update thread b=" + boardId + ", nid=" + nid);
    	}
    	
    	// Rg̏
     	// ݂ĂAXV
		int maxArticleID = getLastArticleId(boardId, nid);
    	
		String sqlText = "insert into " + this.dbname + ".Articles select * from " + master.getDBName() + ".Articles "
    		+ "where " + master.getDBName() + ".Articles.BoardID=" + boardId 
    		+ " and " + master.getDBName() + ".Articles.NID=" + nid 
    		+" and " + master.getDBName()+ ".Articles.ArticleID>" + maxArticleID;

    	sql2.executeUpdate(sqlText);
//		Log.info("sync articles b=" + boardId + ", nid=" + nid);

    	// OYXbhID폜
    	sql2.executeUpdate("delete from ThreadLog where BoardID=" + boardId + " and NID=" + nid);
    	sql2.close();
    }
}
