/*
 * token operation program copyright (C) 2009 H.Niwa
 */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include <string>
#include <complex>

#include "syserr.h"

#include "bin_node.h"
#include "gc.h"
#include "var.h"
#include "pred.h"
#include "context.h"
#include "unify.h"
#include "builtin.h"
#include "sysmodule.h"
#include "context.h"
#include "code.h"
#include "token.h"

extern void PushStack(Context* cx, Node* goals, Node* md, Node* env);

void tokenSkipSpace(Context* cx);

int token(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals, List* module);
int tokenC(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenEof(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenN(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenA(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenAN(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenLineHead(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenLineEnd(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenCR(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenCNTL(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenSpace(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenPunct(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenStrings(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenWildcard(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenWORD(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenNum(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenFNum(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenID(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenRange(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals,
							List* module);
int tokenNonRange(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals,
							List* module);
int tokenGettoken(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenSkip(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals,
							List* module);
int tokenSkipCR(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenNullLine(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenNextChar(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenNextStr(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenNotNextStr(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenNextSyntax(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);
int tokenNotNextSyntax(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals);

std::string	Tokenbuf = "";

int TokenVar(Context* cx, Node* goalscar)
{
	goalscar = goalscar->Cdr();

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() == UNDEF) {
		
		Node* env = Nil->Cons(Nil);
		SetEnv(env, nvar);
		((Undef*)nvar)->Set(mka(Tokenbuf.c_str()));

		if (cx->tokenflag) cx->token = cx->token + Tokenbuf;
		Tokenbuf = "";

		PushStack(cx, Nil, Nil, env);
	
		return 1;
	}

	return -1;
	
}

int isSpace(FILE* fd, int c)
{
	int c1, c2;
	extern std::string code;
	
	if (c == ' ') {
		return 1;
	}

	if (code == "EUC") {
		if (c == 0xa1){	// EUC space
			c1 = fgetc(fd);
			if (c1 == 0xa1) {
				return 1;	
			} else {
				ungetc(c1, fd);
				return 0;
			}
		}			
	} else if (code == "SJIS") {
		if (c == 0x81){	// EUC space
			c1 = fgetc(fd);
			if (c1 == 0x40) {
				return 1;	
			} else {
				ungetc(c1, fd);
				return 0;
			}
		}			
	} else {	// UTF8
		if (c == 0xe3){	// UTF8 space
			c1 = fgetc(fd);
			if (c1 == 0x80) {
				c2 = fgetc(fd);
				if (c2 == 0x80) {
					return 1;
				} else {
					ungetc(c2, fd);
					ungetc(c1, fd);
					return 0;
				}
			} else {
				ungetc(c1, fd);
				return 0;
			}
		}			
	}
	return 0;
}


int isSpace(Context* cx, int c)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	return isSpace(cx->ioin, c);
}

void tokenSkipSpace(Context* cx)
{
	int c = 0;

	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return;
	}

	if (cx->tokenflag) {
		return;
	}
	
	for( ; c != EOF; ) {
		c = fgetc(cx->ioin);
		if ((c == '\n') || (c == '\t')) {
			continue;
		}
		if ((c < ' ') || (c == EOF)) {
			break;
		}
		if (!isSpace(cx, c)) {
			break;
		}
	}
	
	ungetc(c, cx->ioin);
}

int token(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals, List* module)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if (ll < 2) {
		syserr("usage : <TOKEN VAR pred> \n");
		return 0;
	}

	if (goalscar->Cdr()->Car()->kind() != UNDEF) {
		syserr("usage : <TOKEN VAR pred> \n");
		return 0;
	}

	tokenSkipSpace(cx);
	
	Context *cx2 = new Context(module);
	cx2->inherit = cx->inherit;
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;

	cx2->token = "";
	cx2->tokenflag = 1;

	cxpush(cx2, goalscar);
	cxpush(cx2, module);

	int r = Unify(cx2, goalscar->Cdr()->Cdr(), module);
	if (r == 1) {
		if (cx->tokenflag) cx->token = cx2->token;
		Tokenbuf = cx2->token;

		cxpop(cx2);
		cxpop(cx2);
		cx2->tokenflag = 0;

		delete cx2;
		cx2 = 0;

		if (Tokenbuf != "") {		
			return TokenVar(cx, goalscar);
		} else {
			return -1;
		}
	}

	cxpop(cx2);
	cxpop(cx2);
	cx2->tokenflag = 0;

	delete cx2;
	cx2 = 0;

	return r;
}


int tokenCmpDirect(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals, Node* md)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}


//printf("tokenCmpDirect >goals "); goals->print(); printf("\n");
	
	Node* nstr = goalscar->Val();
//PrintNode("tokenCmpDirect nstr ", nstr);

	if (nstr->kind() != ATOM) {
//printf("tokenCmpDirect trace 1\n");
		syserr("token is not Atom \n");
		return 0;
	}


	long tellsave = ftell(cx->ioin);
	
	tokenSkipSpace(cx);

	std::string s;
	
	((Atom*)nstr)->toString(s);
//printf("tokenCmpDirect >>> %s\n", s.c_str());
//printf("tokenCmpDirect >nstr "); nstr->print(); printf("\n");

	int c;
	int i;
	Tokenbuf = "";

	for (i=0; i <s.length(); i++) {
		c = fgetc(cx->ioin);
//printf("tokencmpdirect loop %x %x \n", c, 0xff & s[i]);
		if (c != (0xff & s[i])) {
//printf("tokencmpdirect if %x %x \n", c, 0xff & s[i]);
			fseek(cx->ioin, tellsave, SEEK_SET);

			Tokenbuf = "";
//printf("tokencmpdirect return -1 %x %x \n", c, 0xff & s[i]);
			return -1;
		}
		Tokenbuf = Tokenbuf + (char)c;
	}
	
//printf("tokencmpdirect return 1 %x %x \n", c, 0xff & s[i]);

	Node* env=Nil->Cons(Nil);
	PushStack(cx, goals, md, env);

	if (cx->tokenflag) cx->token = cx->token + Tokenbuf;
	
	return 1;
}



int tokenC(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int	i, n;
	std::string s;
	
	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <C [VAR]> \n");
		return 0;
	}

	int c = fgetc(cx->ioin);
	if (c == EOF) {
		ungetc(c, cx->ioin);
		return -1;
	}
	Tokenbuf = (char)c;

	n = CharLen(c);
	if (n > 1) {
		for (i=1; i < n; i++) {
			c = fgetc(cx->ioin);
			if (c == EOF) {
				ungetc(c, cx->ioin);
				return -1;
			}
			s = (char)c;
			Tokenbuf = Tokenbuf + s;
		}
	}

	if (ll == 1) {
		return 1;
	}
	
	if (goalscar->Cdr()->Car()->kind() != UNDEF) {
		syserr("usage : <C [VAR]> \n");
		return 0;
	}

	Node* env=Nil->Cons(Nil);
	PushStack(cx, goalscar, Nil, env);

	return TokenVar(cx, goalscar);
}

int tokenEof(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if (ll != 1) {
		syserr("usage : <EOF> \n");
		return 0;
	}

	int c = fgetc(cx->ioin);
	if (c != EOF) {
		ungetc(c, cx->ioin);
		return -1;
	}

	Tokenbuf = (char)c;

	return 1;

}

int tokenN(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <N [VAR]> \n");
		return 0;
	}

	int c = fgetc(cx->ioin);
	if (c == EOF) {
		ungetc(c, cx->ioin);
		return -1;
	}
	if ((c < '0') || (c > '9')) {
		ungetc(c, cx->ioin);
		return -1;
	}

	Tokenbuf = (char)c;

	if (ll == 1) {
		return 1;
	}

	if (goalscar->Cdr()->Car()->kind() != UNDEF) {
		syserr("usage : <N [VAR]> \n");
		return 0;
	}

	Node* env=Nil->Cons(Nil);
	PushStack(cx, goalscar, Nil, env);

	return TokenVar(cx, goalscar);
}

int tokenA(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <A [VAR]> \n");
		return 0;
	}


	int c = fgetc(cx->ioin);
	if (c == EOF) {
		ungetc(c, cx->ioin);
		return -1;
	}
	if (!(((c >= 'a') && (c <= 'z')) ||
	      ((c >= 'A') && (c <= 'Z')) ||
	       (c == '_'))) {
		ungetc(c, cx->ioin);
		return -1;
	}
		
	if (ll == 1) {
		return 1;
	}

	Tokenbuf = (char)c;

	if (goalscar->Cdr()->Car()->kind() != UNDEF) {
		syserr("usage : <A [VAR]> \n");
		return 0;
	}

	Node* env=Nil->Cons(Nil);
	PushStack(cx, goalscar, Nil, env);

	return TokenVar(cx, goalscar);
}

int tokenAN(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <AN [VAR]> \n");
		return 0;
	}


	int c = fgetc(cx->ioin);
	if (c == EOF) {
		ungetc(c, cx->ioin);
		return -1;
	}
	if (!(((c >= 'a') && (c <= 'z')) ||
	      ((c >= 'A') && (c <= 'Z')) ||
	       (c == '_') ||
	      ((c >= '0') && (c <= '9')))) {
		ungetc(c, cx->ioin);
		return -1;
	}
		
	Tokenbuf = (char)c;

	if (ll == 1) {
		return 1;
	}

	if (goalscar->Cdr()->Car()->kind() != UNDEF) {
		syserr("usage : <AN [VAR]> \n");
		return 0;
	}

	Node* env=Nil->Cons(Nil);
	PushStack(cx, goalscar, Nil, env);

	return TokenVar(cx, goalscar);
}


int tokenLineHead(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if (ll != 1) {
		syserr("usage : <^> \n");
		return 0;
	}


	long fpos = ftell(cx->ioin);
	if (fpos == 0L) {
		return 1;
	}

	fseek(cx->ioin, -1L, SEEK_CUR);
	
	int c = fgetc(cx->ioin);
	if (c == '\n') {
		return 1;
	} else {
		return -1;
	}
}

int tokenLineEnd(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if (ll != 1) {
		syserr("usage : <$> \n");
		return 0;
	}

	int c = fgetc(cx->ioin);
	if (c == EOF) {
		c = '\n';
	}
	
	if (c != '\n') {
		ungetc(c, cx->ioin);
		return -1;
	}

	Tokenbuf = (char)c;

	return 1;
}

int tokenCR(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if (ll != 1) {
		syserr("usage : <CR> \n");
		return 0;
	}

	int c = fgetc(cx->ioin);
	if (c != '\n') {
		ungetc(c, cx->ioin);
		return -1;
	}

	Tokenbuf = (char)c;

	return 1;
}

int tokenCNTL(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <CNTL [VAR]> \n");
		return 0;
	}

	int c = fgetc(cx->ioin);
	if (!((c < ' ') || (c == 0x7f))) {
		ungetc(c, cx->ioin);
		return -1;
	}

	Tokenbuf = (char)c;

	if (ll == 1) {
		return 1;
	}

	if (goalscar->Cdr()->Car()->kind() != UNDEF) {
		syserr("usage : <CNTL [VAR]> \n");
		return 0;
	}

	Node* env=Nil->Cons(Nil);
	PushStack(cx, goalscar, Nil, env);

	Tokenbuf = (char)c;
	return TokenVar(cx, goalscar);
}

int tokenSpace(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int	n;
	int ll = ListLength(goalscar);
	if (ll != 1) {
		syserr("usage : <SPACE> \n");
		return 0;
	}

	long fpos = ftell(cx->ioin);

	int c = fgetc(cx->ioin);
	if (isSpace(cx, c)) {
		Node* env=Nil->Cons(Nil);
		PushStack(cx, goalscar, Nil, env);
		return 1;
	}

	char* spc = CharSpace();
	int  nspc = strlen(spc);
	if (spc[0] != c) {
		fseek(cx->ioin, fpos, SEEK_SET);
		return -1;
	}
	
	char buf[nspc+1];
	int i;
	for (i = 1; i < nspc; i++) {
		c = fgetc(cx->ioin);
		if (c == EOF) {
			fseek(cx->ioin, fpos, SEEK_SET);
			return -1;
		} else if (c != spc[i]) {
			fseek(cx->ioin, fpos, SEEK_SET);
			return -1;
		}
	}

	return 1;
}

int tokenPunct(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <PUNCT [VAR]> \n");
		return 0;
	}

	int c = fgetc(cx->ioin);
	if (!ispunct(c)) {
		ungetc(c, cx->ioin);
		return -1;
	}

	Tokenbuf = (char)c;

	if (ll == 1) {
		return 1;
	}

	if (goalscar->Cdr()->Car()->kind() != UNDEF) {
		syserr("usage : <PUNCT [VAR]> \n");
		return 0;
	}

	Node* env=Nil->Cons(Nil);
	PushStack(cx, goalscar, Nil, env);

	Tokenbuf = (char)c;
	return TokenVar(cx, goalscar);

}

int tokenStrings(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <STRINGS [VAR]> \n");
		return 0;
	}

	if ((ll == 2) && (goalscar->Cdr()->Car()->kind() != UNDEF)) {
		syserr("usage : <STRINGS [VAR]> \n");
		return 0;
	}

	char	qt = 0;

	Tokenbuf = "";
	
	tokenSkipSpace(cx);
	int c = fgetc(cx->ioin);
	if ((c != '\"') && (c != '\'')) {
//printf("tokenStrings fail 1 \n");
		ungetc(c, cx->ioin);
		Tokenbuf = "";
		return -1;
	}
	qt = c;

	for (;;) {
		c = fgetc(cx->ioin);
		int n = CharLen(c);
		if (n > 1) {
			Tokenbuf = Tokenbuf + (char)(c & 0xff);
			do {
				c = fgetc(cx->ioin);
				Tokenbuf = Tokenbuf + (char)(c & 0xff);
				n--;
			} while(n > 1);
			continue;
		}

		if (c == qt) {
			Node* env=Nil->Cons(Nil);
			PushStack(cx, goalscar, Nil, env);

			if (ll == 1) {
				return 1;
			}
			return TokenVar(cx, goalscar);
		} else if (c == EOF) {
			syserr("unclosed %c \n", qt);
			break;
		} else {
		 	Tokenbuf = Tokenbuf + (char)c;
		}
	}
	Tokenbuf = "";
	return -1;
}

int tokenWORD(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int c, i, n;
	
	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <WORD [VAR]> \n");
		return 0;
	}

	if ((ll == 2) && (goalscar->Cdr()->Car()->kind() != UNDEF)) {
		syserr("usage : <WORD [VAR]> \n");
		return 0;
	}


	Tokenbuf = "";
	
	tokenSkipSpace(cx);

	for (;;) {
		c = fgetc(cx->ioin);
		n = CharLen(c);

		if (c == EOF) {
			if (Tokenbuf == "") {
				break;
			}
			
			Node* env=Nil->Cons(Nil);
			PushStack(cx, goalscar, Nil, env);

			if (ll == 1) {
				return 1;
			}
			return TokenVar(cx, goalscar);
		} if (((c >= 'a') && (c <= 'z')) ||
		    ((c >= 'A') && (c <= 'Z')) ||
		     (c == '_') ||
		    ((c >= '0') && (c <= '9'))) {
		 	Tokenbuf = Tokenbuf + (char)c;
		} else if (n >= 2) {
			if (isSpace(cx, c)) {
				if (Tokenbuf == "") {
					break;
				}
			
				Node* env=Nil->Cons(Nil);
				PushStack(cx, goalscar, Nil, env);

				if (ll == 1) {
					return 1;
				}
				return TokenVar(cx, goalscar);
			}
			
		 	Tokenbuf = Tokenbuf + (char)c;
			for (i=1; i<n; i++) {
				c = fgetc(cx->ioin);
				if (c == EOF) {
					break;
				}
			 	Tokenbuf = Tokenbuf + (char)c;
			}
		} else {
			ungetc(c, cx->ioin);
			
			if (Tokenbuf == "") {
				break;
			}
			
			Node* env=Nil->Cons(Nil);
			PushStack(cx, goalscar, Nil, env);

			if (ll == 1) {
				return 1;
			}
			return TokenVar(cx, goalscar);
		}
	}
	Tokenbuf = "";
	return -1;
}

int tokenWildcardSub(Context* cx, int len)
{
	int c, i, j, n;
	
	for (i = 0; i < len; i++) {
		c = fgetc(cx->ioin);
		n = CharLen(c);

		if (c == EOF) {
			return 0;
		}

		if (n == 1) {
		 	Tokenbuf += (c & 0xff);
		} else {
		 	Tokenbuf += (c & 0xff);
			for (j=1; j<n; j++) {
				c = fgetc(cx->ioin);
				if (c == EOF) {
					return 1;
				}
				Tokenbuf += (c & 0xff);
			}
		}
	}
	if (TraceFlag) {
		printf("* : %s\n", Tokenbuf.c_str());
	}
	return 1;
}

int tokenWildcard(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	Node* env = Nil->Cons(Nil);

	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <* [VAR]> \n");
		return 0;
	}

	if ((ll == 2) && (goalscar->Cdr()->Car()->kind() != UNDEF)) {
		syserr("usage : <* [VAR]> \n");
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car();
	
	tokenSkipSpace(cx);

	long fpos = ftell(cx->ioin);
	int len;
	std::string savecxtoken = cx->token;
	
	for(len = 1;; len++) {
		fseek(cx->ioin, fpos, SEEK_SET);
		cx->token = savecxtoken;
		Tokenbuf = "";

		UnsetEnv(env);
		
		if (!tokenWildcardSub(cx, len)) {
			fseek(cx->ioin, fpos, SEEK_SET);
			break;
		}

		std::string Savetoken = Tokenbuf;
		if (cx->tokenflag) cx->token = cx->token + Tokenbuf;
		
		Context *cx2 = new Context(cx->module);
		cx2->ioin = cx->ioin;
		cx2->ioout = cx->ioout;
		cx2->tokenflag = cx->tokenflag;
		cx2->token = cx->token;

		if (ll == 2) {
			env=Nil->Cons(Nil);
			SetEnv(env, nvar);
			((Undef*)nvar)->Set(mka((char*)Tokenbuf.c_str()));
//			PushStack(cx, goalscar, Nil, env);
		}
		
		cxpush(cx2, env);
		cxpush(cx2, nvar);
		cxpush(cx2, goalscar);
		cxpush(cx2, goalscdr);

		int r = Unify(cx2, goalscdr, cx2->module);

		if (r == 1) {
			if (cx->tokenflag) cx->token = cx2->token;
			cx->Merge(cx2);

			cxpop(cx2);
			cxpop(cx2);
			cxpop(cx2);
			cxpop(cx2);
		
		} else {
			cx2->Clear();

			cxpop(cx2);
			cxpop(cx2);
			cxpop(cx2);
			cxpop(cx2);
		
			UnsetEnv(env);
		}
		
		
		delete cx2;
		cx2 = 0;

		if (r == 1) {
			return 1;
		} else if (r == 0) {
			return 0;
		}
			
	}


	Tokenbuf = "";
	return -1;
}

int tokenNum(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int c;
	
	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <NUM [VAR]> \n");
		return 0;
	}

	if ((ll == 2) && (goalscar->Cdr()->Car()->kind() != UNDEF)) {
		syserr("usage : <NUM [VAR]> \n");
		return 0;
	}

	Tokenbuf = "";
	
	tokenSkipSpace(cx);

	for (;;) {
		c = fgetc(cx->ioin);

		if (isdigit(c)) {
		 	Tokenbuf = Tokenbuf + (char)c;
		} else {
			ungetc(c, cx->ioin);
			
			if (Tokenbuf == "") {
				break;
			}
			
			Node* env=Nil->Cons(Nil);
			PushStack(cx, goalscar, Nil, env);

			if (ll == 1) {
				return 1;
			}
			return TokenVar(cx, goalscar);
		}
	}
	Tokenbuf = "";
	return -1;
}

int tokenFNum(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <FNUM [VAR]> \n");
		return 0;
	}

	if ((ll == 2) && (goalscar->Cdr()->Car()->kind() != UNDEF)) {
		syserr("usage : <FNUM [VAR]> \n");
		return 0;
	}

	int	dot = 0;

	tokenSkipSpace(cx);

	Tokenbuf = "";
	int c = fgetc(cx->ioin);
	if (!isdigit(c) && (c != '.')) {
		Tokenbuf = "";
		ungetc(c, cx->ioin);
		return -1;
	}
	if (c == '.') {
		dot++;
	}
	Tokenbuf = (char)c;

	for (;;) {
		c = fgetc(cx->ioin);
		if (c == '.') {
			if (dot) {
				ungetc(c, cx->ioin);
				int i;
				for (i = Tokenbuf.length()-1; i >= 0; i--) {
					ungetc(Tokenbuf.c_str()[i], cx->ioin);
				}
				Tokenbuf = "";
				return -1;
			}
			dot++;
			Tokenbuf = Tokenbuf + (char)c;
		} else if ((c == 'e') || (c == 'E')) {
			Tokenbuf = Tokenbuf + 'E';
			c = fgetc(cx->ioin);
			if (isdigit(c) || (c == '+') || (c == '-')) {
				Tokenbuf = Tokenbuf + (char)c;
				for (;;) {
					c = fgetc(cx->ioin);
					if (!isdigit(c)) {
						ungetc(c, cx->ioin);

						if (ll == 1) {
							return 1;
						}
						return TokenVar(cx, goalscar);
					}
					Tokenbuf = Tokenbuf + (char)c;
				}
			} else {
				int i;
				ungetc(c, cx->ioin);
				for (i = Tokenbuf.length()-1; i >= 0; i--) {
					ungetc(Tokenbuf.c_str()[i], cx->ioin);
				}
				Tokenbuf = "";
				return -1;
			}
		} else if (!isdigit(c)) {
			ungetc(c, cx->ioin);

			Node* env=Nil->Cons(Nil);
			PushStack(cx, goalscar, Nil, env);

			if (ll == 1) {
				return 1;
			}
			return TokenVar(cx, goalscar);
		} else {
			Tokenbuf = Tokenbuf + (char)c;
		}
	}
	return -1;

}


int tokenID(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int c, i, n;
	
	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <ID [VAR]> \n");
		return 0;
	}

	if ((ll == 2) && (goalscar->Cdr()->Car()->kind() != UNDEF)) {
		syserr("usage : <ID [VAR]> \n");
		return 0;
	}

	Tokenbuf = "";
	
	tokenSkipSpace(cx);

	// check first character
	c = fgetc(cx->ioin);
	n = CharLen(c);
	ungetc(c, cx->ioin);
	if (n == 1) {
		if (!(((c >= 'a') && (c <= 'z')) ||
		    ((c >= 'A') && (c <= 'Z')) ||
		     (c == '_'))) {
		     	return -1;
		}
	}

	for (;;) {
		c = fgetc(cx->ioin);
		n = CharLen(c);

		if (c == EOF) {
			if (Tokenbuf == "") {
				break;
			}
			
			Node* env=Nil->Cons(Nil);
			PushStack(cx, goalscar, Nil, env);

			if (ll == 1) {
				return 1;
			}
			return TokenVar(cx, goalscar);
		} else if (((c >= 'a') && (c <= 'z')) ||
		    ((c >= 'A') && (c <= 'Z')) ||
		     (c == '_') ||
		    ((c >= '0') && (c <= '9'))) {
		 	Tokenbuf = Tokenbuf + (char)c;
		} else if (n >= 2) {
			if (isSpace(cx, c)) {
				if (Tokenbuf == "") {
					break;
				}
			
				Node* env=Nil->Cons(Nil);
				PushStack(cx, goalscar, Nil, env);

				if (ll == 1) {
					return 1;
				}
				return TokenVar(cx, goalscar);
			}
			
		 	Tokenbuf = Tokenbuf + (char)c;
			for (i=1; i<n; i++) {
				c = fgetc(cx->ioin);
				if (c == EOF) {
					break;
				}
			 	Tokenbuf = Tokenbuf + (char)c;
			}
		} else {
			ungetc(c, cx->ioin);
			
			if (Tokenbuf == "") {
				break;
			}
			
			Node* env=Nil->Cons(Nil);
			PushStack(cx, goalscar, Nil, env);

			if (ll == 1) {
				return 1;
			}
			return TokenVar(cx, goalscar);
		}
	}
	Tokenbuf = "";
	return -1;
}

int tokenRangeSub(Context* cx, Node* goalscar, int flg, List* module)
{
	int		c, i, n;
	long long	cl, cl1, cl2;
	std::string	s;
	
	long tp = ftell(cx->ioin);

	int ll = ListLength(goalscar);
	if (ll != 4) {
		return -1;
	}

	if (goalscar->Cdr()->Car()->kind() != UNDEF) {
		if (flg) {
			syserr("usage : <RANGE char char> \n");
		} else {
			syserr("usage : <NONRANGE char char> \n");
		}
		return 0;
	}


	// 2nd args
	Node* arg2 = goalscar->Cdr()->Cdr()->Car();
	int rn;

	if ((rn = FuncArg(cx, arg2, goalscar, module)) <= 0) {
		if (flg) {
			syserr("RANGE: failed in the evaluation of the argument. \n");
		} else {
			syserr("NONRANGE: failed in the evaluation of the argument. \n");
		}
		return 0;
	}

	if (arg2->kind() != ATOM) {
		if (flg) {
			syserr("usage : <RANGE char char> \n");
		} else {
			syserr("usage : <NONRANGE char char> \n");
		}
		return 0;
	}

	((Atom*)arg2)->toString(s);
	c = s[0];
	n = CharLen(c);
	if (n > s.length()) {
		return -1;
	}
	
	cl1 = c & 0xff;
	for (i = 1; i < n; i++) {
		cl1 <<= 8;
		c = s[i];
		cl1 += (0xff & c);
	}
//printf("RANGE args2 %llx \n", cl1);

	// 3rd args
	Node* arg3 = goalscar->Cdr()->Cdr()->Cdr()->Car();

	if ((rn = FuncArg(cx, arg3, goalscar, module)) <= 0) {
		if (flg) {
			syserr("RANGE: failed in the evaluation of the argument. \n");
		} else {
			syserr("NONRANGE: failed in the evaluation of the argument. \n");
		}
		return 0;
	}

	if (arg3->kind() != ATOM) {
		if (flg) {
			syserr("usage : <RANGE char char> \n");
		} else {
			syserr("usage : <NONRANGE char char> \n");
		}
		return 0;
	}
	((Atom*)arg3)->toString(s);

	c = s[0];
	n = CharLen(c);
	if (n > s.length()) {
		return -1;
	}
	
	cl2 = c & 0xff;
	for (i = 1; i < n; i++) {
		cl2 <<= 8;
		c = s[i];
		cl2 += (0xff & c);
	}
//printf("RANGE args3 %llx \n", cl2);

	if (cl2 < cl1) {
		if (flg) {
			syserr("usage : <RANGE char char> \n");
		} else {
			syserr("usage : <NONRANGE char char> \n");
		}
		return 0;
	}	

	c = fgetc(cx->ioin);
	n = CharLen(c);
	Tokenbuf = (char)c;
	cl = 0xff & c;
	for (i = 1; i < n; i++) {
		cl <<= 8;
		c = fgetc(cx->ioin);
		cl += (0xff & c);
		Tokenbuf = Tokenbuf + (char)c;
	}
//printf("RANGE cl %llx \n", cl);

	// flg: 1 RANGE, :0 NONRANGE
	if (flg) {
		if ((cl >= cl1) && (cl <= cl2)) {
			Node* env=Nil->Cons(Nil);
			PushStack(cx, goalscar, Nil, env);
			return TokenVar(cx, goalscar);
		}
	} else {
		if ((cl < cl1) || (cl > cl2)) {
			Node* env=Nil->Cons(Nil);
			PushStack(cx, goalscar, Nil, env);
			return TokenVar(cx, goalscar);
		}
	}

	fseek(cx->ioin, tp, 0);

	return -1;
}

int tokenRange(Context* cx, Node* goalscar, Node* &goalscdr, 
					Node* &goals, List* module)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}
	return tokenRangeSub(cx, goalscar, 1, module);
}

int tokenNonRange(Context* cx, Node* goalscar, Node* &goalscdr, 
					Node* &goals, List* module)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}
	return tokenRangeSub(cx, goalscar, 0, module);
}

int tokenGettoken(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if (ll != 2) {
		syserr("usage : <GETTOKEN VAR>\n");
		return 0;
	}

	if (goalscar->Cdr()->Car()->kind() != UNDEF) {
		syserr("usage : <GETTOKEN VAR>\n");
		return 0;
	}
//printf("GETTOKEN %s \n", Tokenbuf.c_str());

	return TokenVar(cx, goalscar);
}

int tokenSkip(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals,	
							List* module)
{
	int c;
	
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if (ll != 2) {
		syserr("usage : <SKIP STR>\n");
		return 0;
	}

	Node* ag = goalscar->Cdr()->Car();
	int rn;

	if ((rn = FuncArg(cx, ag, goalscar, module)) <= 0) {
		syserr("SKIP: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (ag->kind() != ATOM) {
		syserr("usage : <SKIP STR>\n");
		return 0;
	}

	std::string s, subs;

	((Atom*)ag)->toString(s);

	int i, n;
	for (;;) {
		c = fgetc(cx->ioin);
		if (c == EOF) {
			return 0;
		}
		n = CharLen(c);
		if ((char)(s[0] & 0xff) != (char)(c & 0xff)) {
			for (i = 1; i < n; i++) {
				c=fgetc(cx->ioin);
			}
		} else {
			subs = "";
			subs = subs + (char)c;
			for (i = 1; i < n; i++) {
				c=fgetc(cx->ioin);
				subs = subs + (char)c;
			}
			long fpo = ftell(cx->ioin);
			
			for (i = n; i < s.length(); i++) {
				c = fgetc(cx->ioin);
				subs = subs + (char)c;
			}
			if (s == subs) {
				break;
			}
			fseek(cx->ioin, fpo, 0);
		} 
	}
		
		
	return 1;
}

int tokenSkipCR(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	int c;
	
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if (ll != 1) {
		syserr("usage : <SKIPCR>\n");
		return 0;
	}

	if (goalscar->Cdr()->Car()->kind() != ATOM) {
		syserr("usage : <SKIPCR>\n");
		return 0;
	}

	int i, n;
	for (;;) {
		c = fgetc(cx->ioin);
		if (c == EOF) {
			return 1;
		}
		if (c == '\n') {
			break;
		} 
	}
		
		
	return 1;
}


int tokenNullLine(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	int c;
	
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int ll = ListLength(goalscar);
	if (ll != 1) {
		syserr("usage : <NULLLINE>\n");
		return 0;
	}

	tokenSkipSpace(cx);
	c = fgetc(cx->ioin);
	if ((c == EOF) || (c == '\n')) {
		return 1;
	} 
	return -1;
}


int tokenNextChar(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int	i, n;
	std::string s;
	
	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 1)) {
		syserr("usage : <NEXTCHAR [VAR]> \n");
		return 0;
	}

	long fpo = ftell(cx->ioin);

	int c = fgetc(cx->ioin);
	if (c == EOF) {
		ungetc(c, cx->ioin);
		return -1;
	}
	
	std::string buf;

	buf = (char)c;
	n = CharLen(c);
	if (n > 1) {
		for (i=1; i < n; i++) {
			c = fgetc(cx->ioin);
			if (c == EOF) {
				ungetc(c, cx->ioin);

				fseek(cx->ioin, fpo, 0);
				return -1;
			}
			s = (char)c;
			buf = buf + s;
		}
	}

	fseek(cx->ioin, fpo, 0);

	if (ll == 1) {
		return 1;
	}
	
	if (goalscar->Cdr()->Car()->kind() != UNDEF) {
		syserr("usage : <NEXTCHAR [VAR]> \n");
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car();

	Node* env=Nil->Cons(Nil);
	PushStack(cx, goalscar, Nil, env);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(buf.c_str()));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}



int tokenNextStr(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int	n;
	std::string s;
		
	int ll = ListLength(goalscar);
	if (ll != 2) {
		syserr("usage : <NEXTSTR STRINGS> \n");
		return 0;
	}

	if (goalscar->Cdr()->Car()->Val()->kind() != ATOM) {
		syserr("usage : <NEXTSTR STRINGS> \n");
		return 0;
	}

	std::string str;
	((Atom*)(goalscar->Cdr()->Car()->Val()))->toString(str);

	std::string savecxtoken = cx->token;
	std::string saveTokenbuf = Tokenbuf;
	
	long fpo = ftell(cx->ioin);

	tokenSkipSpace(cx);

	int i, f = 1;
	for (i = 0; i < str.length(); i++) {
		int c = fgetc(cx->ioin);
		if (c == EOF) {
			f = -1;
			break;
		} else if (c != (str[i] & 0xff)) {
			f = -1;
			break;
		}
	}
			
	fseek(cx->ioin, fpo, 0);
	
	cx->token = savecxtoken;
	Tokenbuf = saveTokenbuf;
		
	return f;
}

int tokenNotNextStr(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int	n;
	std::string s;
		
	int ll = ListLength(goalscar);
	if (ll != 2) {
		syserr("usage : <NOTNEXTSTR STRINGS> \n");
		return 0;
	}

	if (goalscar->Cdr()->Car()->Val()->kind() != ATOM) {
		syserr("usage : <NOTNEXTSTR STRINGS> \n");
		return 0;
	}

	std::string str;
	((Atom*)(goalscar->Cdr()->Car()->Val()))->toString(str);

	std::string savecxtoken = cx->token;
	std::string saveTokenbuf = Tokenbuf;
	
	long fpo = ftell(cx->ioin);

	tokenSkipSpace(cx);

	int i, f = -1;
	for (i = 0; i < str.length(); i++) {
		int c = fgetc(cx->ioin);
		if (c == EOF) {
			f = 1;
			break;
		} else if (c != (str[i] & 0xff)) {
			f = 1;
			break;
		}
	}
			
	fseek(cx->ioin, fpo, 0);
	
	cx->token = savecxtoken;
	Tokenbuf = saveTokenbuf;
		
	return f;
}


int tokenNextSyntax(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int	n;
	std::string s;
	
	int ll = ListLength(goalscar);
	if (ll < 2) {
		syserr("usage : <NEXTSYNTAX PRED...> \n");
		return 0;
	}

	Node* npred = goalscar->Cdr()->Val();

	std::string savecxtoken = cx->token;
	std::string saveTokenbuf = Tokenbuf;
	
	long fpo = ftell(cx->ioin);

	int rn;

	Context *cx2 = new Context(cx->module);
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, npred);

	if ((rn=Unify(cx2, npred, cx2->module))>0) {
		cx->Merge(cx2);
	}

	cxpop(cx2);

	delete cx2;
	cx2 = 0;

	fseek(cx->ioin, fpo, 0);

	cx->token = savecxtoken;
	Tokenbuf = saveTokenbuf;
	
	return rn;
}


int tokenNotNextSyntax(Context* cx, Node* goalscar, Node* &goalscdr, Node* &goals)
{
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	int	n;
	std::string s;
	
	int ll = ListLength(goalscar);
	if (ll < 2) {
		syserr("usage : <NOTNEXTSYNTAX PRED...> \n");
		return 0;
	}

	Node* npred = goalscar->Cdr()->Val();

	std::string savecxtoken = cx->token;
	std::string saveTokenbuf = Tokenbuf;
	
	long fpo = ftell(cx->ioin);

	int rn;

	Context *cx2 = new Context(cx->module);
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, npred);

	if ((rn=Unify(cx2, npred, cx2->module))>0) {
		// cx->Merge(cx2);
	}

	cxpop(cx2);

	delete cx2;
	cx2 = 0;

	fseek(cx->ioin, fpo, 0);

	cx->token = savecxtoken;
	Tokenbuf = saveTokenbuf;

	rn = -rn;	
	return rn;
}



