/*
 * Copyright (C) 2007 uguu at users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.deployer;

import java.io.File;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

/**
 * <p>
 * アーカイブ・ファイル配置後のクラスやリソースを読み込むクラスローダーです。
 * </p>
 * <p>
 * アーカイブ・ファイルにはクラス・ファイル、リソース、jarファイルを含めることが出来、コンストラクタでディレクトリを指定することでクラスローダーの管理下に置くことができます。例えば、classes/からクラス・ファイル、lib/からjarファイルを検索するように初期化することが出来ます。ディレクトリは複数指定することが出来ます。
 * </p>
 * <p>
 * 内部的には{@link URLClassLoader}クラスを使用しています。コンストラクタで指定された検索対象ディレクトリ、及びjarディレクトリに含まれるjarファイルは、{@link URLClassLoader}クラスにURLで渡されます。{@link #dispose()}メソッドが呼び出されると、{@link URLClassLoader}インスタンスは破棄され、強制的にガベージコレクタを動かすことで完全に破棄します(破棄しようとします。クラスローダーが読み込んだクラス内でリソースリークが発生している場合、破棄は失敗し、ブロックしてしまいます)。
 * </p>
 * <p>
 * ホット・デプロイを前提としているので、クラスローダーが読み込んだリソースなどを破棄する{@link #dispose()}メソッドが用意されています。例えば{@link Deployer}クラスと連携する場合、{@link DeployerListener#undeployEnd(Deployer, File)}メソッドの呼び出しでこのクラスローダーを破棄することが出来ます。
 * </p>
 * 
 * @author $Author: uguu $
 * @version $Rev: 20 $ $Date: 2007-05-31 06:02:01 +0900 (木, 31 5 2007) $
 */
public final class DeployerClassLoader extends ClassLoader {

    /**
     * 内部クラスローダの弱参照。参照の有効性を調べることで、クラスローダの破棄が成功したかどうかを判定するために使用しています。
     */
    private WeakReference<InternalURLClassLoader> wrCL;

    /**
     * 内部クラスローダ。クラスローダの実際の処理は、この内部クラスローダが行っています。
     */
    private InternalURLClassLoader                cl;

    /**
     * <p>
     * {@link DeployerClassLoader}インスタンスを初期化します。
     * </p>
     * 
     * @param fileDirectories
     *            クラス・ファイル、リソースなどを検索するディレクトリ。複数指定することが出来ます(例えば、classes/、conf/のように)。<br>
     *            nullの場合は無視します。配列中のnullも無視します。
     * @param jarDirectories
     *            jarファイルを検索するディレクトリ。このディレクトリ以下に存在するjarファイルを読み込みます。<br>
     *            nullの場合は無視します。配列中のnullも無視します。
     * @param parent
     *            親クラスローダー。
     * @throws MalformedURLException
     *             指定されたディレクトリの形式が不正である場合。
     */
    public DeployerClassLoader(File[] fileDirectories, File[] jarDirectories, ClassLoader parent) throws MalformedURLException {
        super();

        /*
         * クラスローダの検索パスとして追加するURLを収集します。
         */
        List<URL> urlList = new ArrayList<URL>();

        // クラス・ファイル、リソースなどを検索するディレクトリのパスを追加します。
        if (fileDirectories != null) {
            for (File f : fileDirectories) {
                if (f != null) {
                    urlList.add(f.toURL());
                }
            }
        }

        // jarファイルのパスを追加します。
        if (jarDirectories != null) {
            for (File f : jarDirectories) {
                if (f != null) {
                    urlList.addAll(this.getJarFileList(f));
                }
            }
        }

        /*
         * 内部クラスローダを初期化します。
         */
        if (parent == null) {
            this.cl = new InternalURLClassLoader(urlList.toArray(new URL[0]));
        } else {
            this.cl = new InternalURLClassLoader(urlList.toArray(new URL[0]), parent);
        }
        // 内部クラスローダの破棄を確認するため、弱参照を保持します。
        this.wrCL = new WeakReference<InternalURLClassLoader>(this.cl);
    }

    /**
     * <p>
     * {@link DeployerClassLoader}インスタンスを初期化します。
     * </p>
     * 
     * @param fileDirectories
     *            クラス・ファイル、リソースなどを検索するディレクトリ。複数指定することが出来ます(例えば、classes/、conf/のように)。<br>
     *            nullの場合は無視します。配列中のnullも無視します。
     * @param jarDirectories
     *            jarファイルを検索するディレクトリ。このディレクトリ以下に存在するjarファイルを読み込みます。<br>
     *            nullの場合は無視します。配列中のnullも無視します。
     * @throws MalformedURLException
     *             指定されたディレクトリの形式が不正である場合。
     */
    public DeployerClassLoader(File[] fileDirectories, File[] jarDirectories) throws MalformedURLException {
        this(fileDirectories, jarDirectories, null);
    }

    /**
     * <p>
     * 指定されたディレクトリ以下のjarファイルを検索し、そのURLをリストとして返します。
     * </p>
     * 
     * @param dir
     *            検索するディレクトリ。<br>
     *            nullの場合は空のリストを返します。
     * @return 検索結果のURLのリスト。
     * @throws MalformedURLException
     *             URLの形式が不正である場合。
     */
    private List<URL> getJarFileList(File dir) throws MalformedURLException {
        List<URL> urlList = new ArrayList<URL>();

        if (dir == null) {
            return urlList;
        }

        File[] files = dir.listFiles();
        if (files == null) {
            return urlList;
        }

        for (File f : files) {
            if (f.isFile() && f.getName().toLowerCase().endsWith(".jar")) {
                urlList.add(f.toURL());
            } else if (f.isDirectory()) {
                urlList.addAll(this.getJarFileList(f));
            }
        }

        return urlList;
    }

    /**
     * <p>
     * クラスローダーを破棄します。これにより、クラスローダーは使用不可状態になり、クラスローダーに属していたクラスは解放されます。
     * </p>
     * <p>
     * このメソッド呼び出しはタイムアウトしません。すなわち、以下のコードと等価です。
     * </p>
     * 
     * <pre>
     * cl.dispose(0);
     * </pre>
     */
    public void dispose() {
        this.dispose(0);
    }

    /**
     * <p>
     * クラスローダーを破棄します。これにより、クラスローダーは使用不可状態になり、クラスローダーに属していたクラスは解放されます。
     * </p>
     * <p>
     * メソッド呼び出しの結果、クラスローダーはすぐに使用不可状態になりますが、属しているクラスの解放はすぐには行われない可能性があります。これは、クラスローダーのリソースの解放を強制ガベージコレクタに頼っているためです。このため、タイムアウト時間を指定することが出来ます。タイムアウト時間は破棄を試みますが、それでも破棄出来ない場合は{@link ClassLoaderUnloadFailException}例外をスローします。この場合、クラスローダーは使用不可状態になっていますが、クラスは破棄されていません。完全に破棄するために、このメソッド呼び出しを何度も試みることが出来ます。
     * </p>
     * 
     * @param timeout
     *            破棄を試みるタイムアウト時間。0以下の場合、タイムアウトしない。
     */
    public void dispose(int timeout) {
        this.cl = null;

        // タイムアウト時間が経過するまで、破棄を試みます。
        long t = System.currentTimeMillis();
        while (((System.currentTimeMillis() - t) < timeout) || timeout <= 0) {
            // 内部クラスローダを破棄するため、強制的にガベージコレクションを試みます。
            System.gc();

            // 内部クラスローダへの弱参照がnullになっている場合、破棄されたことを表します。
            // 従って、処理を抜けます。
            if (this.wrCL.get() == null) {
                break;
            }

            // 処理の間隔をあけるため、単位時間の間、処理を中断します。
            try {
                Thread.sleep(Consts.SLEEP_UNIT_TIME);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        // 弱参照がnullではない場合、タイムアウトしたことを表します。
        // 従って、例外をスローします。
        if (this.wrCL.get() != null) {
            throw new ClassLoaderUnloadFailException();
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected Package definePackage(String name, String specTitle, String specVersion, String specVendor, String implTitle, String implVersion, String implVendor, URL sealBase) {
        if (this.cl == null) {
            throw new ClassLoaderUnloadedException();
        }
        return this.cl.definePackage(name, specTitle, specVersion, specVendor, implTitle, implVersion, implVendor, sealBase);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected Class<?> findClass(String name) throws ClassNotFoundException {
        if (this.cl == null) {
            throw new ClassLoaderUnloadedException();
        }
        return this.cl.findClass(name);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected String findLibrary(String libname) {
        if (this.cl == null) {
            throw new ClassLoaderUnloadedException();
        }
        return this.cl.findLibrary(libname);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected URL findResource(String name) {
        if (this.cl == null) {
            throw new ClassLoaderUnloadedException();
        }
        return this.cl.findResource(name);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected Enumeration<URL> findResources(String name) throws IOException {
        if (this.cl == null) {
            throw new ClassLoaderUnloadedException();
        }
        return this.cl.findResources(name);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected Package getPackage(String name) {
        if (this.cl == null) {
            throw new ClassLoaderUnloadedException();
        }
        return this.cl.getPackage(name);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected Package[] getPackages() {
        if (this.cl == null) {
            throw new ClassLoaderUnloadedException();
        }
        return this.cl.getPackages();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected synchronized Class<?> loadClass(String name, boolean resolve) throws ClassNotFoundException {
        if (this.cl == null) {
            throw new ClassLoaderUnloadedException();
        }
        return this.cl.loadClass(name, resolve);
    }

    /**
     * <p>
     * 内部クラスローダ。必要なメソッドのアクセス修飾子をpublicに変更するために定義します。
     * </p>
     * 
     * @author uguu
     */
    private class InternalURLClassLoader extends URLClassLoader {

        /**
         * <p>
         * インスタンスを初期化します。
         * </p>
         * 
         * @param urls
         *            検索パスに含めるURLの配列。
         */
        public InternalURLClassLoader(URL[] urls) {
            super(urls);
        }

        /**
         * <p>
         * インスタンスを初期化します。
         * </p>
         * 
         * @param urls
         *            検索パスに含めるURLの配列。
         * @param parent
         *            親クラスローダー。
         */
        public InternalURLClassLoader(URL[] urls, ClassLoader parent) {
            super(urls, parent);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Package definePackage(String name, String specTitle, String specVersion, String specVendor, String implTitle, String implVersion, String implVendor, URL sealBase) {
            return super.definePackage(name, specTitle, specVersion, specVendor, implTitle, implVersion, implVendor, sealBase);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Class<?> findClass(String name) throws ClassNotFoundException {
            return super.findClass(name);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String findLibrary(String libname) {
            return super.findLibrary(libname);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Package getPackage(String name) {
            return super.getPackage(name);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public Package[] getPackages() {
            return super.getPackages();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public synchronized Class<?> loadClass(String name, boolean resolve) throws ClassNotFoundException {
            return super.loadClass(name, resolve);
        }

    }

}
