/**
 * Copyright (c) 2004, yher2 project
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice, 
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * * Neither the name of the nor the names of its contributors may be used to endorse or 
 *   promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 */
package net.yher2.junit.db.excel;

import java.io.IOException;

import net.yher2.commons.io.Classpath;
import net.yher2.commons.lang.DateUtils;
import net.yher2.junit.db.data.ColumnData;
import net.yher2.junit.db.data.Header;
import net.yher2.junit.db.data.Row;
import net.yher2.junit.db.data.Table;
import net.yher2.junit.db.excel.ExcelReader;

import junit.framework.TestCase;

/**
 * @auther  hisaboh
 */
public class ExcelReaderTest extends TestCase {

	/*
	 * @see TestCase#setUp()
	 */
	protected void setUp() throws Exception {
		super.setUp();
	}

	/*
	 * @see TestCase#tearDown()
	 */
	protected void tearDown() throws Exception {
		super.tearDown();
	}
	
	public void testRead() throws IOException {
		ExcelReader reader = new ExcelReader(new Classpath("net/yher2/junit/db/excel/ExcelReaderTestNormal.xls"));
		Table table[] = reader.read();
		
		// Table
		assertEquals(1, table.length);
		assertEquals("WORKBOOK", table[0].getName());
		
		// Header
		Header header[] = table[0].getHeader();
		Header expectHeader[] = new Header[]{
				new Header("ID", true)
				, new Header("String")
				, new Header("NUMBER")
				, new Header("BOOLEAN")
				, new Header("BLANK")
				, new Header("DATE")
		};
		assertEquals(6, header.length);
		for (int i=0; i<header.length; i++) {
			assertEquals(expectHeader[i].isPrimaryKey(), header[i].isPrimaryKey());
			assertEquals(expectHeader[i].getName(), header[i].getName());
		}
		
		// Row
		Row row[] = table[0].getRow();
		Object expectRow[][] = new Object[][]{
			new Object[]{new Integer(1), "{", new Integer(1234567890), new Boolean(true), null, DateUtils.parse(2004, 12, 24)}
			, new Object[]{new Integer(2), "English", new Double(123.456), new Boolean(false),null,  DateUtils.parse(2004, 12, 24, 10, 20,30)}
			, new Object[]{new Integer(3), "'\"/\\`", new Integer(123), new Integer(1), null, DateUtils.parse(2004, 12,24, 10, 20, 30)}
			, new Object[]{new Integer(4), null, null, new Integer(0), null, DateUtils.parse(2004, 12, 1)}
			, new Object[]{new Integer(5), null, null, null, null, DateUtils.parse(2004, 1, 1)}
			, new Object[]{new Integer(6), null, null, null, null, null}
			, new Object[]{new Integer(7), null, null, null, null, null}
			, new Object[]{new Integer(8), null, null, null, null, null}
		};
		assertEquals(expectRow.length, row.length);
		for (int i=0; i<expectRow.length; i++) {
			ColumnData column[] = row[i].getColumn();
			assertEquals(expectRow[i].length, column.length);
			for (int j=0; j<column.length; j++) {
				assertEquals(expectRow[i][j], column[j].getValue());
			}
		}
	}

	public void testReadNoColumnData() throws IOException {
		ExcelReader reader = new ExcelReader(new Classpath("net/yher2/junit/db/excel/ExcelReaderTestNoColumnData.xls"));
		Table table[] = reader.read();
		
		// Table
		assertEquals(1, table.length);
		assertEquals("WORKBOOK", table[0].getName());
		
		// Header
		Header header[] = table[0].getHeader();
		Header expectHeader[] = new Header[]{
				new Header("ID", true)
				, new Header("String")
				, new Header("NUMBER")
				, new Header("BOOLEAN")
				, new Header("BLANK")
				, new Header("DATE")
		};
		assertEquals(6, header.length);
		for (int i=0; i<header.length; i++) {
			assertEquals(expectHeader[i].isPrimaryKey(), header[i].isPrimaryKey());
			assertEquals(expectHeader[i].getName(), header[i].getName());
		}
		
		// Row
		Row row[] = table[0].getRow();
		assertEquals(0, row.length);
	}
	public void testReadMultiTable() throws IOException {
		ExcelReader reader = new ExcelReader(new Classpath("net/yher2/junit/db/excel/ExcelReaderTestMultiTable.xls"));
		Table table[] = reader.read();
		
		// Table
		assertEquals(3, table.length);
		assertEquals("Table1", table[0].getName());
		assertEquals("Table2", table[1].getName());
		assertEquals("Table3", table[2].getName());
	}

	public void testReadNoData() throws IOException {
		ExcelReader reader = new ExcelReader(new Classpath("net/yher2/junit/db/excel/ExcelReaderTestNoData.xls"));
		try {
			Table table[] = reader.read();
		} catch (IllegalStateException e) {
			return;
		}
		fail();
	}
	public void testReadNoHeaderData() throws IOException {
		ExcelReader reader = new ExcelReader(new Classpath("net/yher2/junit/db/excel/ExcelReaderTestNoHeaderData.xls"));
		try {
			Table table[] = reader.read();
		} catch (IllegalStateException e) {
			return;
		}
		fail();
	}
	public void testReadIllegalNumOfCoumn() throws IOException {
		ExcelReader reader = new ExcelReader(new Classpath("net/yher2/junit/db/excel/ExcelReaderTestIllegalNumOfColumn.xls"));
		try {
			Table table[] = reader.read();
		} catch (IllegalStateException e) {
			return;
		}
		fail();
	}
}
