package daruma.storage_manager.type_definition.types;

import daruma.storage_manager.type_definition.TypeDefinition;
import daruma.storage_manager.type_definition.TypeException;
import daruma.storage_manager.type_definition.TypedInstance;
import daruma.storage_manager.type_definition.ElementName;
import daruma.storage_manager.type_definition.InstanceParseContext;
import daruma.storage_manager.type_definition.ColumnNameFactory;
import daruma.storage_manager.StorageManager;
import daruma.storage_manager.StorageException;
import daruma.storage_manager.ElementInfo;

import daruma.geometry.TransformationContext;
import daruma.geometry.EulerAngles;
import daruma.geometry.CoordinateSystem;
import daruma.geometry.TransformationException;

import daruma.sql.SQLDataType;
import daruma.sql.SQLDataTypeConstant;
import daruma.sql.TableColumnDefinition;
import daruma.sql.TableColumn;

import daruma.xml.UniversalName;
import daruma.xml.SimpleXPath;
import daruma.xml.URI;
import daruma.xml.util.ElementUtil;

import org.w3c.dom.Element;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;

import daruma.util.LogWriter;
import daruma.util.Pair;
import daruma.util.EqualChecker;

import java.util.List;
import java.util.ArrayList;


public class EulerAnglesTypeDefinition extends TypeDefinition
{
    private final List<String> ANGLE_COLUMN_NAMES = new ArrayList<String>();
    {
	ANGLE_COLUMN_NAMES.add( "alpha" );
	ANGLE_COLUMN_NAMES.add( "beta" );
	ANGLE_COLUMN_NAMES.add( "gamma" );
    }
    private final String SRS_NAME_COLUMN_NAME_SUFFIX = "_srs_name_";

    private List<UniversalName> tagNames;
    private UniversalName SRS_NAME_ATTRIBUTE
	= new UniversalName( null, "srsName" );

    public EulerAnglesTypeDefinition( boolean hasRollPitchYaw )
    {
	super( true, true );

	this.tagNames = new ArrayList<UniversalName>();

	if ( hasRollPitchYaw )
	{
	    this.tagNames.add( new UniversalName( URI.MISP, "roll" ) );
	    this.tagNames.add( new UniversalName( URI.MISP, "pitch" ) );
	    this.tagNames.add( new UniversalName( URI.MISP, "yaw" ) );
	}
	else
	{
	    this.tagNames.add( new UniversalName( URI.MISP, "alpha" ) );
	    this.tagNames.add( new UniversalName( URI.MISP, "beta" ) );
	    this.tagNames.add( new UniversalName( URI.MISP, "gamma" ) );
	}
    }

    @Override
    public SQLDataType getSingleSQLDataType()
    {
	return null;
    }

    @Override
    public List<TableColumnDefinition>
	getCompositeSQLDataType( StorageManager storage,
				 SimpleXPath path,
				 ElementInfo elementInfo ) throws TypeException
    {
	List<TableColumnDefinition>
	    ret = new ArrayList<TableColumnDefinition>();

	String columnNameBase = null;

	if ( path == null )
	{
	    columnNameBase = ColumnNameFactory.getTopLevelElementColumnName();

	    if ( elementInfo != null )
	    {
		elementInfo.registerSubElement( new SimpleXPath(), this );
	    }
	}
	else
	{
	    SimpleXPath newPath = new SimpleXPath( path );

	    try
	    {
		columnNameBase = storage.getShortXPathStringForDB( newPath );
	    }
	    catch( StorageException e )
	    {
		throw new TypeException( e );
	    }

	    if ( elementInfo != null )
	    {
		elementInfo.registerSubElement( newPath, this );
	    }
	}

	for ( int i = 0 ; i < ANGLE_COLUMN_NAMES.size() ; ++ i )
	{
	    ret.add( new TableColumnDefinition
		     ( ColumnNameFactory.getColumnName
		       ( columnNameBase, ANGLE_COLUMN_NAMES.get( i ) ),
		       SQLDataTypeConstant.DOUBLE ) );
	}

	ret.add( new TableColumnDefinition
		 ( ColumnNameFactory.getColumnName
		   ( columnNameBase, SRS_NAME_COLUMN_NAME_SUFFIX ),
		   SQLDataTypeConstant.BLOB ) );

	return ret;
    }

    @Override
    public Pair<TypedInstance, Integer>
	createInstance( Element element,
			ElementName topLevelElement,
			SimpleXPath path,
			StorageManager storage,
			int elementIndex,
			InstanceParseContext parseContext )
	throws TypeException
    {
	SimpleXPath p = path;
	if ( p != null )
	{
	    p = new SimpleXPath( path, new UniversalName( element ) );
	}

	List<Element> childElements = ElementUtil.getChildElements( element );

	if ( childElements.size() != this.tagNames.size() )
	{
	    throw new TypeException( "invalid number of child elements found"
				     + " in " + new UniversalName( element )
				     + ", expected 3 child elements" );
	}

	List<TableColumn> columns = new ArrayList<TableColumn>();

	List<TableColumnDefinition> def
	    = this.getCompositeSQLDataType( storage, p, null );
	final FloatTypeDefinition floatDef = new FloatTypeDefinition();

	for ( int i = 0 ; i < tagNames.size() ; ++ i )
	{
	    if ( ! new UniversalName( childElements.get( i ) )
		   .equals( this.tagNames.get( i ) ) )
	    {
		throw new TypeException
		    ( "invalid angle element "
		      + new UniversalName( childElements.get( i ) )
		      + " found in " + new UniversalName( element )
		      + ", expected was " + this.tagNames.get( i ) );
	    }

	    String valueString
	    = ElementUtil.getChildNodesWholeText( childElements.get( i ) );

	    try
	    {
		valueString = floatDef.parseValue( valueString );
	    }
	    catch( TypeException e )
	    {
		throw new TypeException( "invalid Euler angle: "
					 + e.getMessage() );
	    }

	    columns.add( new TableColumn( def.get( i ), valueString ) );
	}

	final String srsName
	    = element.getAttributeNS( SRS_NAME_ATTRIBUTE.getNamespace(),
				      SRS_NAME_ATTRIBUTE.getLocalName() );
	columns.add( new TableColumn( def.get( tagNames.size() ), srsName ) );

	return new Pair<TypedInstance, Integer>
		   ( new TypedInstance( columns, this ),
		     element.getChildNodes().getLength() );
    }

    @Override
    public int convertToXMLElement
	( Element element,
	  Document doc,
	  StorageManager storage,
	  TransformationContext trans,
	  List<TableColumn> columns,
	  int index,
	  long id ) throws TypeException
    {
	//
	// check
	//
	assert ANGLE_COLUMN_NAMES.size() == 3;

	int numColumns = ANGLE_COLUMN_NAMES.size() + 1/* srsName */;

	if ( index + numColumns - 1 >= columns.size() )
	{
	    throw new TypeException( "internal error in "
				     + this.getClass().getName()
				     + ": unexpected instantiation" );
	}


	//
	// get srsName
	//
	final String srsName
	    = columns.get( index + tagNames.size() ).getValue();


	//
	// get angle values
	//
	List<Double> values = new ArrayList<Double>();

	for ( int i = 0 ; i < ANGLE_COLUMN_NAMES.size() ; ++ i )
	{
	    final String valueString = columns.get( index + i ).getValue();

	    try
	    {
		values.add( DoubleTypeDefinition.parseDouble( valueString ) );
	    }
	    catch( TypeException e )
	    {
		throw new TypeException( "internal error in "
					 + this.getClass().getName()
					 + ": " + e.getMessage() );
	    }
	}


	//
	// transform EulerAngles
	//
	EulerAngles transformedAngles
	    = transformEulerAngles( new EulerAngles( srsName,
						     values.get( 0 ),
						     values.get( 1 ),
						     values.get( 2 ) ),
				    trans );

	List<Double> returnValues = new ArrayList<Double>();
	returnValues.add( transformedAngles.getAlpha() );
	returnValues.add( transformedAngles.getBeta() );
	returnValues.add( transformedAngles.getGamma() );


	//
	// create XML
	//
	for ( int i = 0 ; i < returnValues.size() ; ++ i )
	{
	    final UniversalName name = this.tagNames.get( i );
	    Element tag = doc.createElementNS( name.getNamespace(),
					       name.getLocalName() );
	    tag.appendChild( doc.createTextNode
			     ( returnValues.get( i ).toString() ) );
	    element.appendChild( tag );
	}

	if ( transformedAngles.getSrsName() != null )
	{
	    element.setAttributeNS( SRS_NAME_ATTRIBUTE.getNamespace(),
				    SRS_NAME_ATTRIBUTE.getLocalName(),
				    transformedAngles.getSrsName() );
	}

	return index + numColumns;
    }

    private EulerAngles transformEulerAngles( EulerAngles angles,
					      TransformationContext trans )
    {
	switch( trans.getTransformationType() )
	{
	case Selective:
	case Conv:
	    final CoordinateSystem target = trans.getTargetCoordinateSystem();
	    final String srsName = angles.getSrsName();

	    if ( ! EqualChecker.equals( target.getSrsName(), srsName ) )
	    {
		try
		{
		    angles = trans.transform( angles );
		}
		catch( TransformationException e )
		{
		    LogWriter.qwrite( "DEBUG",
				      "transform failed: " + e.getMessage() );
		}
	    }
	    break;


	case NoConv:
	    break;
	}

	return angles;
    }
}
