package daruma.geometry;

import java.lang.Math;
import java.lang.StringBuilder;

public class ThreeDementionalTransformMatrix
{
    private double[][] m;

    public ThreeDementionalTransformMatrix( double[][] matrix )
	throws GeometryException
    {
	if ( matrix.length != 4 && matrix.length != 3 )
	{
	    throw new GeometryException( "invalid matrix size" );
	}

	this.m = new double[4][4];

	for ( int y = 0 ; y < matrix.length ; ++ y )
	{
	    if ( matrix[y].length != 4 )
	    {
		throw new GeometryException( "invalid matrix size "
					     + matrix[y].length + "x"
					     + matrix.length );
	    }

	    for ( int x = 0 ; x < matrix[y].length ; ++ x )
	    {
		this.m[y][x] = matrix[y][x];
	    }
	}

	if ( matrix.length < 4 )
	{
	    this.m[3][0] = 0.0;
	    this.m[3][1] = 0.0;
	    this.m[3][2] = 0.0;
	    this.m[3][3] = 1.0;
	}
    }

    public ThreeDementionalTransformMatrix()
    {
	this.m = new double[4][4];

	for ( int x = 0 ; x < 4 ; ++ x )
	{
	    for ( int y = 0 ; y < 4 ; ++ y )
	    {
		if ( x == y )
		{
		    this.m[y][x] = 1.0;
		}
		else
		{
		    this.m[y][x] = 0.0;
		}
	    }
	}
    }

    public ThreeDementionalTransformMatrix
	( double v00, double v01, double v02, double v03,
	  double v10, double v11, double v12, double v13,
	  double v20, double v21, double v22, double v23,
	  double v30, double v31, double v32, double v33 )
    {
	this.setMatrix( v00, v01, v02, v03,
			v10, v11, v12, v13,
			v20, v21, v22, v23,
			v30, v31, v32, v33 );
    }


    public ThreeDementionalTransformMatrix
	( double x, double y, double z,
          double roll, double pitch, double yaw )
    {
	this.setMatrixByShiftAndRotate( x, y, z, roll, pitch, yaw );
    }

    public ThreeDementionalTransformMatrix( EulerAngles angles )
    {
	this.setMatrixByShiftAndRotate
	    ( 0.0, 0.0, 0.0,
	      angles.getAlpha(), angles.getBeta(), angles.getGamma() );
    }

    public void setMatrix( double v00, double v01, double v02, double v03,
			   double v10, double v11, double v12, double v13,
			   double v20, double v21, double v22, double v23,
			   double v30, double v31, double v32, double v33 )
    {
	this.m = new double[4][4];

	this.m[0][0] = v00;
	this.m[0][1] = v01;
	this.m[0][2] = v02;
	this.m[0][3] = v03;
	this.m[1][0] = v10;
	this.m[1][1] = v11;
	this.m[1][2] = v12;
	this.m[1][3] = v13;
	this.m[2][0] = v20;
	this.m[2][1] = v21;
	this.m[2][2] = v22;
	this.m[2][3] = v23;
	this.m[3][0] = v30;
	this.m[3][1] = v31;
	this.m[3][2] = v32;
	this.m[3][3] = v33;
    }

    public void setMatrixByShiftAndRotate
	( double x, double y, double z,
          double roll, double pitch, double yaw )
    {
        double alpha = yaw * Math.PI / 180.0;
        double beta  = pitch * Math.PI / 180.0;
        double gamma = roll * Math.PI / 180.0;

	this.setMatrix
	     ( Math.cos( alpha ) * Math.cos( beta ),
	       Math.cos( alpha ) * Math.sin( beta ) * Math.sin( gamma ) - Math.sin( alpha ) * Math.cos( gamma ),
	       Math.cos( alpha ) * Math.sin( beta ) * Math.cos( gamma ) + Math.sin( alpha ) * Math.sin( gamma ),
	       x,

	       Math.sin( alpha ) * Math.cos( beta ),
	       Math.sin( alpha ) * Math.sin( beta ) * Math.sin( gamma ) + Math.cos( alpha ) * Math.cos( gamma ),
	       Math.sin( alpha ) * Math.sin( beta ) * Math.cos( gamma ) - Math.cos( alpha ) * Math.sin( gamma ),
	       y,

	       -Math.sin( beta ),
	       Math.cos( beta ) * Math.sin( gamma ),
	       Math.cos( beta ) * Math.cos( gamma ),
	       z,

	       0.0,
	       0.0,
	       0.0,
	       1.0 );
    }

    public boolean equals( ThreeDementionalTransformMatrix matrix )
    {
	final double[][] m1 = this.m;
	final double[][] m2 = matrix.getMatrix();

	if ( m1.length != m2.length )
	{
	    return false;
	}

	for ( int y = 0 ; y < m1.length ; ++ y )
	{
	    for ( int x = 0 ; x < m1[y].length ; ++ x )
	    {
		if ( m1[y][x] != m2[y][x] )
		{
		    return false;
		}
	    }	    
	}

	return true;
    }

    public String toString()
    {
	StringBuilder buf = new StringBuilder();

	for ( int y = 0 ; y < this.m.length ; ++ y )
	{
	    for ( int x = 0 ; x < this.m[y].length ; ++ x )
	    {
		if ( x != 0 )
		{
		    buf.append( " " );
		}

		buf.append( this.m[y][x] );
	    }

	    buf.append( "\n" );
	}

	return buf.toString();
    }

    public double[][] getMatrix()
    {
	return this.m;
    }


    public ThreeDementionalTransformMatrix
	multiply( ThreeDementionalTransformMatrix matrix )
    {
	double[][] ret = new double[4][4];

	for ( int y = 0 ; y < 4 ; ++ y )
	{
	    for ( int x = 0 ; x < 4 ; ++ x )
	    {
		double value = 0.0;

		for ( int i = 0 ; i < 4 ; ++ i )
		{
		    value += (this.m[y][i] * matrix.getMatrix()[i][x]);
		}

		ret[y][x] = value;
	    }
	}

	try
	{
	    return new ThreeDementionalTransformMatrix( ret );
	}
	catch( GeometryException e )
	{
	    System.err.println( "internal error in "
				+ this.getClass().getName()
				+ ": " + e.getMessage() );

	    return new ThreeDementionalTransformMatrix();
	}
    }

    public double getRoll()
    {
	return Math.atan2( this.m[2][1], this.m[2][2] ) / Math.PI * 180.0;
    }

    public double getPitch()
    {
	return Math.atan2( - this.m[2][0],
			   Math.sqrt( Math.pow( this.m[2][1], 2 )
				      + Math.pow( this.m[2][2], 2 ) ) )
	    / Math.PI * 180.0;
    }

    public double getYaw()
    {
	return Math.atan2( this.m[1][0], this.m[0][0] ) / Math.PI * 180.0;
    }
}
