#include  "system_call_wrapper.h"

#include  <sys/types.h>
#include  <unistd.h>
#include  <errno.h>
#include  <iostream>
#include  <cstddef>
#include  <cstdio>
#include  <cstring>

namespace System_Call_Wrapper
{


namespace Internal
{

class  C_Style_String_Vector
{
private:
	char **	v;

private:
	// no implementation (not allowed)
	C_Style_String_Vector( const C_Style_String_Vector & );
	// no implementation (not allowed)
	C_Style_String_Vector &	operator= ( const C_Style_String_Vector & );

public:
	C_Style_String_Vector( const std::vector<std::string> &  org )
		: v( static_cast<char **>(0) )
	{
		try
		{
			this -> v = new char * [org.size() + 1];

			for ( size_t  i = 0  ;  i < org.size() + 1  ;  i ++ )
			{
				this -> v[i] = static_cast<char *>(0);
			}

			for ( size_t  i = 0  ;  i < org.size()  ;  i ++ )
			{
				this -> v[i] = new char [org[i].length() + 1];
				std::strcpy( this -> v[i] , org[i].c_str() );
			}
		}
		catch(...)
		{
			this -> clear();
		}
	}

	~C_Style_String_Vector()
	{
		this -> clear();
	}

	char **	get() const
	{
		return( this -> v );
	}

	void	clear()
	{
		if ( this -> v )
		{
			for ( char ** p = this -> v  ;
			      *p != static_cast<char *>(0)  ;
			      p ++ )
			{
				delete[] p;
			}

			delete[] this -> v;
			this -> v = static_cast<char **>(0);
		}
	}
};

} // end of namespace Internal


int    execv( const std::string &  path ,
	      const std::vector<std::string> &  argv )
{
	Internal::C_Style_String_Vector	c_argv( argv );

	return( ::execv( path.c_str() , c_argv.get() ) );
}


int    execvp( const std::string &  file_name ,
	       const std::vector<std::string> &  argv )
{
	Internal::C_Style_String_Vector	c_argv( argv );

	return( ::execvp( file_name.c_str() , c_argv.get() ) );
}


#if ! defined(TENTATIVE_HACK_FOR_ONLY_MINGW_COMPILE)
pid_t  fork_execvp( const std::string &  file_name ,
		    const std::vector<std::string> &  argv )
{
	pid_t	child_pid;

	switch( (child_pid = ::fork()) )
	{
	case  -1:
		// error
		std::perror( "fork" );
		return( -1 );
		break;

	case  0:
	      {
		// child process

		Internal::C_Style_String_Vector	c_argv( argv );

		if ( ::execvp( file_name.c_str() , c_argv.get() ) == -1 )
		{
			if ( errno == ENOENT )
			{
				std::cerr << file_name.c_str()
					  << " not found" << std::endl;
			}
			else
			{
				std::perror( "execvp" );
			}
		}

		::_exit( 1 );
		break;
	      }

	default:
		// parent process
		return( child_pid );
		break;
	}
}

#include  <sys/wait.h>

pid_t  fork_execvp_wait( const std::string &  file_name ,
			 const std::vector<std::string> &  argv ,
			 int * status )
{
    pid_t child_pid = fork_execvp( file_name , argv );

    if ( child_pid == -1 )
    {
	return -1;
    }

    pid_t ret;
    if ( (ret = waitpid( child_pid , status , 0 )) == -1 )
    {
	std::perror( "waitpid" );
    }

    return( ret );
}
#endif

} // end of namespace System_Call_Wrapper




// for mkdir()
#include  <sys/types.h>
#include  <sys/stat.h>

// for unlink()
#include  <unistd.h>

// for lstat()
#include  <sys/types.h>
#include  <sys/stat.h>

#include  "posix_compat_sys_stat.h"
#include  <errno.h>
#include  <cstdio>
#include  <iostream>
#include  "path.h"

namespace System_Call_Wrapper
{

int    mkdir( const Path &  path ,  mode_t  mode ,  bool  print_error )
{
	std::string	new_path;

	for ( size_t  i = 0  ;  i < path.size()  ;  i ++ )
	{
		if ( i != 0 )
		{
			new_path += Path::SEPARATOR;
		}

		new_path += path[i];

		if ( ::mkdir( path.native_string().c_str() , mode ) == -1 )
		{
			if ( print_error )
			{
				std::perror( "mkdir" );
				std::cerr << "mkdir \""
					  << new_path.c_str() << "\""
					  << " failed" << std::endl;
			}

			return( -1 );
		}
	}

	return( 0 );
}


int    mkdir_p( const Path &  path ,  mode_t  mode ,  bool  print_error )
{
	std::string	new_path;

	for ( size_t  i = 0  ;  i < path.size()  ;  i ++ )
	{
		if ( i != 0 || path.absolute() )
		{
			new_path += Path::SEPARATOR;
		}

		new_path += path[i];

#ifdef MINGW
		static_cast<void>(mode);
		if ( ::mkdir( new_path.c_str() ) == -1 )
#else
		if ( ::mkdir( new_path.c_str() , mode ) == -1 )
#endif

		{
			if ( errno == EEXIST )
			{
				/* file already exists: ignore */
			}
			else
			{
				if ( print_error )
				{
					std::perror( "mkdir" );
					std::cerr << "mkdir \""
						  << new_path.c_str() << "\""
						  << " failed" << std::endl;
				}

				return( -1 );
			}
		}
	}

	return( 0 );
}


int    unlink( const Path &  path ,  bool  print_error )
{
	if ( ::unlink( path.native_string().c_str() ) == -1 )
	{
		if ( print_error )
		{
			std::perror( "unlink" );
		}

		return( -1 );
	}

	return( 0 );
}

bool   is_directory( const Path &  p )
{
	struct stat	buf;

#ifdef TENTATIVE_HACK_FOR_ONLY_MINGW_COMPILE
	if ( ::stat( p.native_string().c_str() , &buf ) == -1 )
#else
	if ( ::lstat( p.native_string().c_str() , &buf ) == -1 )
#endif
	{
		return( false );
	}

	return( S_ISDIR( buf.st_mode ) );
}


} // end of namespace System_Call_Wrapper




#include  <iostream>
#include  <sys/types.h>
#include  <unistd.h>
#include  <string>

namespace System_Call_Wrapper
{


size_t  read_n_bytes( int  fd ,  void *  result ,  size_t  n )
{
	size_t	read_count = 0;

	while( read_count < n )
	{
		int	n_read;
		switch( (n_read = ::read( fd ,
					  static_cast<char *>(result)
							      + read_count ,
					  n - read_count )) )
		{
		case -1:
			return( read_count );
			break;

		case 0:
			return( read_count );
			break;

		default:
			read_count += n_read;
			break;
		}
	}

	return( read_count );
}

size_t  read_n_bytes( int  fd ,  std::string *  result ,  size_t  n )
{
	result -> erase();

	char *	buf = new char[n];

	size_t	ret = read_n_bytes( fd , buf , n );
	result -> append( buf , ret );

	delete buf;

	return( ret );
}

size_t  read_n_bytes_with_timeout( int  fd ,  void *  result ,  size_t  n ,
				   long  timeout_msec ,  bool *  timeout )
{
	if ( timeout )
	{
		*timeout = false;
	}

	size_t	read_count = 0;

	while( read_count < n )
	{
		if ( ! System_Call_Wrapper::poll_fd( fd , timeout_msec ) )
		{
			if ( timeout )
			{
				*timeout = true;
			}

			return( read_count );
		}

		int	n_read;
		switch( (n_read = ::read( fd ,
					  static_cast<char *>(result)
							      + read_count ,
					  n - read_count )) )
		{
		case -1:
			return( read_count );
			break;

		case 0:
			return( read_count );
			break;

		default:
			read_count += n_read;
			break;
		}
	}

	return( read_count );
}

size_t  read_n_bytes_with_timeout
		( int  fd ,  std::string *  result ,  size_t  n ,
		  long  timeout_msec ,  bool *  timeout )
{
	result -> erase();

	char *	buf = new char[n];

	size_t	ret = read_n_bytes_with_timeout( fd , buf , n ,
						 timeout_msec , timeout );
	result -> append( buf , ret );

	delete buf;

	return( ret );
}


size_t  write_n_bytes( int  fd ,  const void *  data ,  size_t  n )
{
	size_t	write_count = 0;

	while( write_count < n )
	{
		int	n_write;

		switch( (n_write = ::write( fd ,
					    static_cast<const char *>(data)
							      + write_count ,
					    n - write_count )) )
		{
		case -1:
			return( write_count );

		case 0:
			return( write_count );
			break;

		default:
			write_count += n_write;
			break;
		}
	}

	return( write_count );
}


ssize_t  read( int  fd ,  void *  result ,  size_t  n )
{
	return( ::read( fd , static_cast<char *>(result) , n ) );
}

ssize_t  write( int  fd ,  const void *  data ,  size_t  n )
{
	return( ::write( fd , static_cast<const char *>(data) , n ) );
}


void   read_all( int  fd ,  std::string *  result )
{
	size_t	read_count = 0;

	char	buf[BUFSIZ];

	for(;;)
	{
		int	n_read;
		switch( (n_read = ::read( fd , buf , sizeof(buf) )) )
		{
		case -1:
		case 0:
			return;
			break;

		default:
			read_count += n_read;
			break;
		}

		result -> append( buf , n_read );
	}
}


} // end of namespace System_Call_Wrapper




#include  <sys/types.h>
#include  <unistd.h>
#include  "posix_compat_sys_time.h"
#include  "posix_compat_sys_select.h"

#include  <cstdio> // for perror()

namespace System_Call_Wrapper
{

int    poll( int  fd ,  long  timeout_msec )
{
	fd_set		readfd;
	FD_ZERO( &readfd );
	FD_SET( static_cast<unsigned int>(fd) , &readfd );

	struct timeval	tv;
	tv.tv_sec  = timeout_msec / 1000;
	tv.tv_usec = (timeout_msec - timeout_msec / 1000) * 1000;

	return( select( fd + 1 ,
                        &readfd ,
                        static_cast<fd_set *>(0) ,
                        static_cast<fd_set *>(0) ,
                        &tv ) );
}

int    poll( int  fd )
{
	fd_set		readfd;
	FD_ZERO( &readfd );
	FD_SET( static_cast<unsigned int>(fd) , &readfd );

	struct timeval	tv;
	tv.tv_sec  = 0;
	tv.tv_usec = 0;

	return( select( fd + 1 , &readfd ,
                        static_cast<fd_set *>(0) ,
                        static_cast<fd_set *>(0) ,
                        &tv ) );
}

bool   poll_fd( int  fd ,  long  timeout_msec )
{
	int	ret;

	if ( (ret = System_Call_Wrapper::poll( fd , timeout_msec )) == -1 )
	{
		std::perror( "select" );

		return( false );
	}

	return( ret >= 1 );
}

bool   poll_fd( int  fd )
{
	int	ret;

	if ( (ret = System_Call_Wrapper::poll( fd )) == -1 )
	{
		std::perror( "select" );

		return( false );
	}

	return( ret >= 1 );
}

} // end of namespace System_Call_Wrapper
