#include "fuse_xml.h"
#include "universal_name.h"
#include "path_converter.h"
#include "string_extension.h"

#include <sys/stat.h>
#include <cstring>
#include <cerrno>
#include <sstream>

#include <iostream>
#include <algorithm>

#include "ref_count_ptr.h"
#include "xml_data_tree.h"

extern ref_count_ptr<fuse_xml::XMLDataTree> xml_data_tree;

namespace fuse_xml {

int FuseXML::getattr( const char * path,
                      struct stat * stat_buf )
{
    std::cerr << "getattr [" << path << "]" << std::endl;

    std::memset( stat_buf, 0, sizeof(struct stat) );

    std::map<std::string, ref_count_ptr<XMLDataNode> >::const_iterator
        it = xml_data_tree -> getFiles().find( path );

    if ( it == xml_data_tree -> getFiles().end() )
    {
        return -ENOENT;
    }

    ref_count_ptr<XMLDataNode> n = (*it).second;

    if ( n -> isText() )
    {
        stat_buf -> st_mode = S_IFREG | 0444;
        stat_buf -> st_nlink = 1;
        stat_buf -> st_size = n -> getValue().size();
    }
    else
    {
        stat_buf -> st_mode = S_IFDIR | 0755;
        stat_buf -> st_nlink = 2;
    }

    return 0;
}

int FuseXML::open( const char * path,
                   struct fuse_file_info * file_info )
{
    std::cerr << "open: path = [" << path << "]" << std::endl;

    std::map<std::string, ref_count_ptr<XMLDataNode> >::const_iterator
        it = xml_data_tree -> getFiles().find( path );

    if ( it == xml_data_tree -> getFiles().end() )
    {
        return -ENOENT;
    }

    if ( (*it).second -> isText() )
    {
        return 0;
    }
    else
    {
        if ( file_info -> flags & 0x11 != O_RDONLY )
        {
            return -EACCES;
        }
        else
        {
            return 0;
        }
    }
}

int FuseXML::readdir( const char * path,
                         void * buf,
                         fuse_fill_dir_t filler,
                         off_t offset,
                         struct fuse_file_info * file_info )
{
    std::cerr << "readdir: path = [" << path << "]" << std::endl;

    static_cast<void>(offset);
    static_cast<void>(file_info);

    static_cast<void>(path);
    static_cast<void>(buf);
    static_cast<void>(filler);

    std::map<std::string, ref_count_ptr<XMLDataNode> >::const_iterator
        it = xml_data_tree -> getFiles().find( path );

    if ( it == xml_data_tree -> getFiles().end() )
    {
        return -ENOENT;
    }

    ref_count_ptr<XMLDataNode> n = (*it).second;

    if ( n -> isElement() )
    {
        filler( buf, ".", NULL, 0 );
        filler( buf, "..", NULL, 0 );

        for ( size_t i = 0 ; i < n -> getChildElements().size() ; ++ i )
        {
            filler( buf,
                    Path_Converter::escape_slash
                    ( n -> getChildElements()[i].toPathString() ).c_str(),
                    NULL, 0 );
        }
    }

    return 0;
}

int FuseXML::read( const char * path,
                   char * buf, size_t size,
                   off_t offset,
                   struct fuse_file_info * file_info )
{
    std::cerr << "read: path = [" << path << "], "
              << "size = " << size << ", "
              << "offset = " << offset << std::endl;

    static_cast<void>(file_info);

    static_cast<void>(buf);

    std::map<std::string, ref_count_ptr<XMLDataNode> >::const_iterator
        it = xml_data_tree -> getFiles().find( path );

    if ( it != xml_data_tree -> getFiles().end() )
    {
        ref_count_ptr<XMLDataNode> n = (*it).second;

        const std::string & output = n -> getValue();
        size_t write_size = std::min( static_cast<size_t>(output.size() - offset),
                                      size );
        std::memcpy( buf, output.data() + offset, write_size );

        return write_size;
    }

    return -ENOENT;
}

void FuseXML::getPathInfo( std::vector<std::string> * hier,
                           const std::string & path )
{
    std::string p = path;
    String_Extension::chomp( &p, '/' );

    String_Extension::split_by_char_set( hier, p, "/" );

    for ( size_t i = 0; i < hier -> size(); ++ i )
    {
        std::cerr << "path[" << i << "] = [" << (*hier)[i] << "]"
                  << std::endl;
    }
}

struct fuse_operations FuseXML::getFuseOperations()
{
    struct fuse_operations operations;
    memset( &operations, 0, sizeof(operations) );

    operations.read    = FuseXML::read;
    operations.open    = FuseXML::open;
    operations.readdir = FuseXML::readdir;
    operations.getattr = FuseXML::getattr;

    return operations;
}


} // end of namespace fuse_xml
