#include  "udp_pseudo_connection.h"
#include  "ref_count_ptr.h"
#include  "ref_count_array_ptr.h"
#include  "debugstream.h"
#include  <cstring>
#include  <iostream>
#include  <cstdio>

extern "C"
{
	#include  <stdio.h>
	#include  <string.h>
	#include  <errno.h>
	#include  <sys/types.h>

	#ifdef HAVE_CONFIG_H
	    #if HAVE_UNISTD_H
		#include  <unistd.h>
	    #endif
	#else
	    #include  <unistd.h>
	#endif

	#ifdef HAVE_SYS_IOCTL_H
	    #if HAVE_SYS_IOCTL_H
		#include  <sys/ioctl.h>
	    #endif
	#else
	  #if !defined(TENTATIVE_HACK_FOR_ONLY_MINGW_COMPILE)
	    #include  <sys/ioctl.h>
	  #endif
	#endif

	#ifdef HAVE_FCNTL_H
	    #if HAVE_FCNTL_H
		#include  <fcntl.h>
	    #endif
	#else
	    #include  <fcntl.h>
	#endif

	#include  "posix_compat_sys_time.h"

	#ifdef sgi
	    #include <bstring.h>
	#endif
}

using namespace std;

UDP_Pseudo_Connection::UDP_Pseudo_Connection( const string &  host ,
					      port_number_t  port_num ,
					      bool  debug ,
					      size_t  buf_size ,
					      long  max_resp ,
					      bool  close_on_exec )
	: recv_time( 0 , 0 ) , received( false ) ,
	  buffer_size( buf_size ) , max_response( max_resp ) ,
	  debug_flag( debug )
{
	memset( &(this -> addr) , 0 , sizeof( this -> addr ) );

	if ( (this -> sfd = ::socket( AF_INET , SOCK_DGRAM  , 0 ))
	     == BAD_SOCKET_FD )
	{
		this -> sfd = BAD_SOCKET_FD;

		perror( "socket" );
		return;
	}

	if ( close_on_exec )
	{
#if !defined(TENTATIVE_HACK_FOR_ONLY_MINGW_COMPILE)
		if ( ::fcntl( this -> sfd , F_SETFD , 1 ) == -1 )
		{
			perror( "fcntl" );
			this -> sfd = BAD_SOCKET_FD;
			this -> close();
			return;
		}
#endif
	}


	struct sockaddr_in	my_addr;
	memset( &my_addr , 0 , sizeof( my_addr ) );
	my_addr.sin_family      = AF_INET;
	my_addr.sin_addr.s_addr = htonl( INADDR_ANY );
	my_addr.sin_port        = htons( 0 );

	if ( ::bind( this -> sfd , (const struct sockaddr *)( &my_addr ) ,
				   sizeof( my_addr ) )  ==  -1 )
	{
		perror( "bind" );
		this -> sfd = BAD_SOCKET_FD;
		this -> close();
		return;
	}


	struct hostent *	hp;
	if ( (hp = ::gethostbyname( host.c_str() ))
	     ==  static_cast<struct hostent *>(0) )
	{
		perror( "gethostbyname" );
		this -> sfd = BAD_SOCKET_FD;
		this -> close();
		return;
	}

	this -> addr.sin_family = AF_INET;
	memcpy( &(this -> addr.sin_addr.s_addr) ,
		hp -> h_addr , hp -> h_length );
	this -> addr.sin_port = htons( port_num );


	recv_time.set_current_time();
}


UDP_Pseudo_Connection::UDP_Pseudo_Connection( port_number_t  port ,
					      bool  debug ,
					      size_t  buf_size ,
					      long  max_resp ,
					      bool  close_on_exec )
	: recv_time( 0 , 0 ) , received( false ) ,
	  buffer_size( buf_size ) , max_response( max_resp ) ,
	  debug_flag( debug )
{
	memset( &(this -> addr) , 0 , sizeof( this -> addr ) );

	if ( (this -> sfd = ::socket( AF_INET , SOCK_DGRAM  , 0 ))
	     == BAD_SOCKET_FD )
	{
		perror( "socket" );
		this -> sfd = BAD_SOCKET_FD;
		return;
	}

	if ( close_on_exec )
	{
#if !defined(TENTATIVE_HACK_FOR_ONLY_MINGW_COMPILE)
		if ( ::fcntl( this -> sfd , F_SETFD , 1 ) == -1 )
		{
			perror( "fcntl" );
			this -> sfd = BAD_SOCKET_FD;
			this -> close();
			return;
		}
#endif
	}


	struct sockaddr_in	my_addr;
	memset( &my_addr , 0 , sizeof( my_addr ) );
	my_addr.sin_family      = AF_INET;
	my_addr.sin_addr.s_addr = htonl( INADDR_ANY );
	my_addr.sin_port        = htons( port );

	if ( ::bind( this -> sfd , (const struct sockaddr *)( &my_addr ) ,
				   sizeof( my_addr ) )  ==  -1 )
	{
		perror( "bind" );
		this -> sfd = BAD_SOCKET_FD;
		this -> close();
		return;
	}

	this -> addr.sin_family = AF_INET;


	recv_time.set_current_time();
}


UDP_Pseudo_Connection::~UDP_Pseudo_Connection()
{
	this -> close();
}



UDP_Pseudo_Connection::operator bool() const
{
	return( this -> fd() != BAD_SOCKET_FD );
}

bool   UDP_Pseudo_Connection::responsive() const
{
	if ( ! (this -> operator bool()) )
	{
		return( false );
	}

	if ( ! received )
	{
		return( false );
	}

	if ( this -> max_response == 0 )
	{
		return( true );
	}
	else
	{
		Time_Stamp	t;
		t.set_current_time();

		return( (t - recv_time).usec() < (this -> max_response) );
	}
}

long   UDP_Pseudo_Connection::get_max_response() const
{
	return( this -> max_response );
}

void   UDP_Pseudo_Connection::set_max_response( long  max_resp )
{
	this -> max_response = max_resp;
}

SOCKET_FD_TYTE  UDP_Pseudo_Connection::fd() const
{
	return( this -> sfd );
}


int    UDP_Pseudo_Connection::close()
{
	// if already closed or not opened
	if ( this -> sfd == BAD_SOCKET_FD )
	{
		// return with no error
		return( 0 );
	}
	else
	{
		// close the file descriptor
		int	ret;

		if ( (ret = ::close( this -> sfd ))  ==  -1 )
		{
			perror( "close" );
		}

		this -> sfd = BAD_SOCKET_FD;

		return( ret );
	}
}


int    UDP_Pseudo_Connection::setsockopt( int  level ,
					  int  option ,
					  const void *  option_value ,
					  socklen_t  option_length )
{
	int	ret;
	if ( (ret = ::setsockopt
		      ( this -> sfd ,
			level ,
			option ,
			static_cast<SOCKOPT_OPTION_VALUE_POINTER_TYPE>
			( const_cast<void *>( option_value ) ) ,
			option_length )) == -1 )
	{
		perror( "setsockopt" );
	}

	return( ret );
}

int    UDP_Pseudo_Connection::send( const string &  str )
{
	if ( this -> sfd == BAD_SOCKET_FD )
	{
		return( -1 );
	}

	int	n;
	if ( (n = ::sendto( this -> sfd , str.c_str() , str.length() , 0 ,
			    (struct sockaddr *)( &(this -> addr) ) ,
			    sizeof( this -> addr ) ))  ==  -1 )
	{
		perror( "sendto" );
		return( -1 );
	}

	if ( this -> debug_flag && Debug_Stream::dbg )
	{
		Debug_Stream::dbg << "send[" << str << "]" << endl;
	}


	// XXX
	// compare int and size_t
	// It may cause problem.
	//
	if ( static_cast<size_t>(n) == str.length() )
	{
		return( n );
	}
	else
	{
		return( -1 );
	}
}


int    UDP_Pseudo_Connection::recv( ref_count_ptr<string> *  buf ,
				    long  usec ,  bool  reply )
{
	if ( ! buf )
	{
		return( -1 );
	}

	ref_count_array_ptr<char>	char_buf;

	int	ret = this -> recv( &char_buf , /*append_nul*/ false ,
				    usec , reply );

	if ( ret == -1 )
	{
		*buf = new string();
	}
	else
	{
		*buf = new string( char_buf.get() , ret );
	}

	return( ret );
}


int    UDP_Pseudo_Connection::recv( ref_count_ptr<string> *  buf ,
				    bool  block ,  bool  reply )
{
	if ( ! buf )
	{
		return( -1 );
	}

	ref_count_array_ptr<char>	char_buf;

	int	ret = this -> recv( &char_buf , /*append_nul*/ false ,
				    block , reply );

	if ( ret == -1 )
	{
		*buf = new string();
	}
	else
	{
		*buf = new string( char_buf.get() , ret );
	}

	return( ret );
}


int    UDP_Pseudo_Connection::recv( string *  str ,
				    long  usec ,  bool  reply )
{
	ref_count_ptr<string>	s;
	int			ret;

	if ( (ret = this -> recv( &s , usec , reply ))  ==  -1 )
	{
		*str = "";
	}
	else
	{
		*str = *s;
	}

	return( ret );
}


int    UDP_Pseudo_Connection::recv( string *  str ,
				    bool  block ,  bool  reply )
{
	ref_count_ptr<string>	s;
	int			ret;

	if ( (ret = this -> recv( &s , block , reply ))  ==  -1 )
	{
		*str = "";
	}
	else
	{
		*str = *s;
	}

	return( ret );
}


int    UDP_Pseudo_Connection::recv( ref_count_array_ptr<char> *  buf ,
				    bool  append_nul ,
				    long  usec ,  bool  reply )
{
	if ( ! buf )
	{
		return( -1 );
	}

	size_t	len = this -> buffer_size + (append_nul ? 1 : 0);

	*buf = new char[ len ];

	int	ret = this -> recv( buf -> get() , len , append_nul ,
				    usec , reply );

	if ( ret == -1 )
	{
#if 0
		// can't create char[0], instead use char[1].
		*buf = new char[1];
		**buf = '\0';
#else
		*buf = static_cast<char *>(0);
#endif
	}

	return( ret );
}


int    UDP_Pseudo_Connection::recv( ref_count_array_ptr<char> *  buf ,
				    bool  append_nul ,
				    bool  block ,  bool  reply )
{
	if ( ! buf )
	{
		return( -1 );
	}

	size_t	len = this -> buffer_size + (append_nul ? 1 : 0);

	*buf = new char[ len ];

	int	ret = this -> recv( buf -> get() , len , append_nul ,
				    block , reply );

	if ( ret == -1 )
	{
#if 0
		// can't create char[0], instead use char[1].
		*buf = new char[1];
		**buf = '\0';
#else
		*buf = static_cast<char *>(0);
#endif
	}

	return( ret );
}


int    UDP_Pseudo_Connection::recv( char *  buf ,  size_t  len ,
				    bool  append_nul ,
				    long  usec ,  bool  reply )
{
	if ( this -> sfd == BAD_SOCKET_FD )
	{
		return( -1 );
	}


	struct sockaddr_in	from;
	SOCKLEN_T		from_size;
	int			n = 0;
	int			ret = 0;

	from_size = sizeof( from );

	struct timeval	tv;
	fd_set		readfd;
	FD_ZERO( &readfd );
	FD_SET( this -> sfd , &readfd );

	tv.tv_sec  = usec / (1000 * 1000);
	tv.tv_usec = usec % (1000 * 1000);

	if ( ::select( this -> sfd + 1 , &readfd ,
		       static_cast<fd_set *>(0) , static_cast<fd_set *>(0) ,
		       &tv )  ==  1 )
	{
		if ( (ret = ::recvfrom( this -> sfd , buf , len , 0 ,
					(struct sockaddr *)( &from ) ,
					&from_size ))  ==  -1 )
		{
			perror( "recvfrom" );
			n = 0;
		}
		else
		{
			recv_time.set_current_time();
			received = true;
			n = ret;
		}
	}
	else
	{
		ret = -1;
		n = 0;
	}

	if ( append_nul )
	{
		buf[n] = '\0';
	}

	if ( this -> debug_flag && Debug_Stream::dbg )
	{
		if ( ret != -1 )
		{
			Debug_Stream::dbg << "recv[";

			for ( int  i = 0  ;  i < n  ;  i ++ )
			{
				Debug_Stream::dbg << buf[i];
			}

			Debug_Stream::dbg << "]" << endl;
		}
	}

	if ( reply )
	{
		if ( from.sin_port != static_cast<port_number_t>(0)
		  && this -> addr.sin_port != from.sin_port )
		{
			if ( this -> debug_flag && Debug_Stream::dbg )
			{
				Debug_Stream::dbg << "changed port from "
						  << this -> addr.sin_port
						  << " to " << from.sin_port
						  << std::endl;
			}

			this -> addr.sin_port = from.sin_port;
		}
	}

	return( ret );
}


int    UDP_Pseudo_Connection::recv( char *  buf ,  size_t  len ,
				    bool  append_nul ,
				    bool  block ,  bool  reply )
{
	if ( this -> sfd == BAD_SOCKET_FD )
	{
		return( -1 );
	}

	if ( this -> max_response != 0L )
	{
		return( this -> recv( buf , len , append_nul ,
				      (block ? (this -> max_response) : 0L) ,
				      reply ) );
	}


	struct sockaddr_in	from;
	SOCKLEN_T		from_size;
	int			n = 0;
	int			ret = 0;

	from_size = sizeof( from );

	if ( block )
	{
		if ( (ret = ::recvfrom( this -> sfd , buf , len , 0 ,
					(struct sockaddr *)(&from) ,
					&from_size ))  ==  -1 )
		{
#if !defined(TENTATIVE_HACK_FOR_ONLY_MINGW_COMPILE)
			if ( errno != EWOULDBLOCK )
			{
				perror( "recvfrom" );
			}
#endif
			n = 0;
		}
		else
		{
			recv_time.set_current_time();
			received = true;
			n = ret;
		}
	}

	if ( append_nul )
	{
		buf[n] = '\0';
	}

	if ( reply )
	{
		if ( from.sin_port != static_cast<port_number_t>(0)
		  && this -> addr.sin_port != from.sin_port )
		{
			if ( this -> debug_flag && Debug_Stream::dbg )
			{
				Debug_Stream::dbg << "changed port from "
						  << this -> addr.sin_port
						  << " to " << from.sin_port
						  << std::endl;
			}

			this -> addr.sin_port = from.sin_port;
		}
	}

	return( ret );
}
