#ifndef	   OPTION_ANALYZER_H_INCLUDED
#define	   OPTION_ANALYZER_H_INCLUDED

// Author:		H. Shimora
// Created:		Jan 27 2000
// Last-Modified:	Jan 27 2000
// Version:		0.00

//------------------------------------------------
// Change Log:
//------------------------------------------------
// version 0.00  Jan 27 2000    test version.
//
//

#ifdef HAVE_CONFIG_H
# include  "config.h"
#endif

#include  <string>
#include  <vector>
#include  <map>
#include  <cstdlib>
#include  <iostream>
#include  "ref_count_ptr.h"
#include  "ip_address.h"
#include  "path.h"
#include  "string_extension.h"

class  Option_Analyzer
{
public:
	class  Option_Descriptor
	{
	public:
		virtual	~Option_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			static_cast<void>(arg);
			return( 0 );
		}

		virtual	bool	check( std::string *  error_message ) const
		{
			static_cast<void>(error_message);

			return( true );
		}
	};

public:
	class  Flag_On_Option_Descriptor : public Option_Descriptor
	{
	protected:
		bool *	flag;

	public:
			 Flag_On_Option_Descriptor( bool *  f )
				 : flag( f ) {}

		virtual	~Flag_On_Option_Descriptor() {}

		virtual	int	execute( const std::vector<std::string> & )
									const
		{
			if ( flag )
			{
				*flag = true;
			}

			return( 0 );
		}
	};

	class  Flag_Off_Option_Descriptor : public Option_Descriptor
	{
	protected:
		bool *	flag;

	public:
			 Flag_Off_Option_Descriptor( bool *  f )
				 : flag( f ) {}

		virtual	~Flag_Off_Option_Descriptor() {}

		virtual	int	execute( const std::vector<std::string> & )
									const
		{
			if ( flag )
			{
				*flag = false;
			}

			return( 0 );
		}
	};

	class  Flag_Toggle_Option_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	protected:
		bool *	flag;

	public:
			 Flag_Toggle_Option_Descriptor( bool *  f )
				 : flag( f ) {}
		virtual	~Flag_Toggle_Option_Descriptor() {}

		virtual	int	execute( const std::vector<std::string> & )
									const
		{
			if ( flag )
			{
				*flag = ! *flag;
			}

			return( 0 );
		}
	};

	class  Integer_Option_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	protected:
		int *	integer;

	public:
			 Integer_Option_Descriptor( int *  i )
				 : integer( i ) {}

		virtual	~Integer_Option_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			if ( arg.size() < 1 )
			{
				return( -1 );
			}

			if ( integer )
			{
				char *	c;
				long	long_num;
				long_num
				    = std::strtol( arg[0].c_str() , &c , 10 );

				if ( *c != '\0'
				  || long_num == LONG_MIN
				  || long_num == LONG_MAX
				  || long_num < INT_MIN
				  || long_num > INT_MAX )
				{
					std::cerr << "integer number option"
						  << " error ["
						  << arg[0] << "]"
						  << std::endl;

					return( -1 );
				}
				else
				{
					*integer = static_cast<int>(long_num);
				}
			}

			return( 1 );
		}
	};

	class  Long_Integer_Option_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	protected:
		long *	integer;

	public:
			 Long_Integer_Option_Descriptor( long *  i )
				 : integer( i ) {}
		virtual	~Long_Integer_Option_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			if ( arg.size() < 1 )
			{
				return( -1 );
			}

			if ( integer )
			{
				char *	c;
				long	long_num;
				long_num
				    = std::strtol( arg[0].c_str() , &c , 10 );

				if ( *c != '\0'
				  || long_num == LONG_MIN
				  || long_num == LONG_MAX )
				{
					std::cerr << "integer number option"
						  << " error ["
						  << arg[0] << "]"
						  << std::endl;

					return( -1 );
				}
				else
				{
					*integer = long_num;
				}
			}

			return( 1 );
		}
	};

	class  Unsigned_Long_Option_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	protected:
		unsigned long *	integer;

	public:
			 Unsigned_Long_Option_Descriptor( unsigned long *  i )
				 : integer( i ) {}
		virtual	~Unsigned_Long_Option_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			if ( arg.size() < 1 )
			{
				return( -1 );
			}

			if ( integer )
			{
				char *	c;
				long	long_num;
				long_num
				    = std::strtol( arg[0].c_str() , &c , 10 );

				if ( *c != '\0'
				  || long_num == LONG_MAX
				  || long_num < 0 )
				{
					std::cerr
					    << "unsigned integer number option"
					    << " error ["
					    << arg[0] << "]"
					    << std::endl;

					return( -1 );
				}
				else
				{
					*integer = long_num;
				}
			}

			return( 1 );
		}
	};

	class  Float_Option_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	protected:
		double *	number;

	public:
			 Float_Option_Descriptor( double *  n )
				 : number( n ) {}
		virtual	~Float_Option_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			if ( arg.size() < 1 )
			{
				return( -1 );
			}

			if ( number )
			{
				char *	c;
				double	dbl_num;
				dbl_num = std::strtod( arg[0].c_str() , &c );

				if ( *c != '\0' )
				{
					std::cerr
					     << "float point number option"
					     << " error [" << arg[0] << "]"
					     << std::endl;

					return( -1 );
				}
				else
				{
					*number = dbl_num;
				}
			}

			return( 1 );
		}
	};

	class  String_Option_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	protected:
		std::string *	str;

	public:
			 String_Option_Descriptor( std::string *  s )
				 : str( s ) {}
		virtual	~String_Option_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			if ( arg.size() < 1 )
			{
				return( -1 );
			}

			if ( str )
			{
				*str = arg[0];
			}

			return( 1 );
		}
	};

	class  Port_Number_Option_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	protected:
		port_number_t *		port;

	public:
			 Port_Number_Option_Descriptor( port_number_t *  p )
				 : port( p ) {}

		virtual	~Port_Number_Option_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			if ( arg.size() < 1 )
			{
				return( -1 );
			}

			if ( port )
			{
				char *	c;
				long	long_num;
				long_num
				    = std::strtol( arg[0].c_str() , &c , 10 );

				if ( *c != '\0'
				  || long_num < 0
				  || long_num > 0xffff )
				{
					std::cerr << "port number option"
						  << " error ["
						  << arg[0] << "]"
						  << std::endl;

					return( -1 );
				}
				else
				{
					*port = static_cast<port_number_t>
								    (long_num);
				}
			}

			return( 1 );
		}
	};

	class  Path_Option_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	protected:
		Path *	path;

	public:
			 Path_Option_Descriptor( Path *  p )
				 : path( p ) {}

		virtual	~Path_Option_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			if ( arg.size() < 1 )
			{
				return( -1 );
			}

			if ( path )
			{
				Path	p( arg[0] );

				if ( ! p.valid() )
				{
					std::cerr << "invalid path"
						  << " [" << arg[0] << "]"
						  << std::endl;

					return( -1 );
				}

				*path = p;
			}

			return( 1 );
		}
	};

	class  Path_List_Option_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	protected:
		std::vector<Path> *	path_list;
		bool			push_front;

	public:
			 Path_List_Option_Descriptor( std::vector<Path> *  pl,
						      bool push_front = false )
				: path_list( pl ), push_front( push_front ) {}

		virtual	~Path_List_Option_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			if ( arg.size() < 1 )
			{
				return( -1 );
			}

			if ( ! path_list )
			{
				return( 1 );
			}

			std::vector<std::string> pl;

			String_Extension::split_by_char_set
				( &pl , arg[0] , ":" );

			std::vector<Path> new_path_list;

			for ( size_t  i = 0  ;  i < pl.size()  ;  ++ i )
			{
				Path	p( pl[i] );

				if ( ! p.valid() )
				{
					std::cerr << "invalid path"
						  << " [" << pl[i] << "]"
						  << std::endl;

					return( -1 );
				}

				new_path_list.push_back( p );
			}

			if ( push_front )
			{
				path_list -> insert
				    ( this -> path_list -> begin() ,
				      new_path_list.begin() ,
				      new_path_list.end() );
			}
			else
			{
				path_list -> insert
				    ( this -> path_list -> end() ,
				      new_path_list.begin() ,
				      new_path_list.end() );
			}

			return( 1 );
		}
	};

	class  Raw_Arguments_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	protected:
		std::vector<std::string> *	args;

	public:
		Raw_Arguments_Descriptor( std::vector<std::string> *  a )
				 : args( a ) {}

		virtual	~Raw_Arguments_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			*(this -> args) = arg;

			return( arg.size() );
		}
	};

	class  Option_Analyzer_Option_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	protected:
		Option_Analyzer *	opt;
		std::string		separator;
		bool			valid;

	public:
		Option_Analyzer_Option_Descriptor
			( Option_Analyzer *  opt ,
			  const std::string &  separator )
				 : opt( opt ) ,
				   separator( separator ) ,
				   valid( true ) {}

		virtual	~Option_Analyzer_Option_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			std::vector<std::string>	sub_arg;

			bool	has_separator = false;

			for( size_t  i = 0  ;  i < arg.size()  ;  i ++ )
			{
				if ( arg[i] == separator )
				{
					has_separator = true;

					break;
				}

				sub_arg.push_back( arg[i] );
			}

			if ( ! opt -> analyze( sub_arg ) )
			{
				return( -1 );
			}

			return( sub_arg.size() + (has_separator ? 1 : 0) );
		}
	};


	template<typename  T, typename  Opt_Desc>
	class  Multiple_Option_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	private:
		std::vector<T> *		values;
		mutable std::vector<Opt_Desc>	descriptors;

	public:
		Multiple_Option_Descriptor( std::vector<T> *  values )
			: values( values ) , descriptors()
		{
			values -> clear();
		}

		virtual	~Multiple_Option_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			this -> values -> push_back( T() );

			this -> descriptors.push_back
				( Opt_Desc( &((*values)
					      [values -> size() - 1]) ) );

			return( (*(this -> descriptors.rbegin()))
				.execute( arg ) );
		}

		virtual	bool	check( std::string *  error_message ) const
		{
			for ( size_t  i = 0  ;
			      i < this -> descriptors.size()  ;  ++ i )
			{
				if ( ! this -> descriptors[i]
					       .check( error_message ) )
				{
					return( false );
				}
			}

			return( true );
		}
	};

	class  Deprecated_Option_Descriptor
		: public Option_Analyzer::Option_Descriptor
	{
	protected:
		ref_count_ptr<Option_Analyzer::Option_Descriptor>  descriptor;
		const std::string				   message;

	public:
		Deprecated_Option_Descriptor
			( const ref_count_ptr<Option_Analyzer
					      ::Option_Descriptor> &  d ,
			  const std::string &  message )
			: descriptor( d ), message( message ) {}

		virtual	~Deprecated_Option_Descriptor() {}

		virtual	int	execute
				( const std::vector<std::string> &  arg ) const
		{
			std::cerr << this -> message << std::flush;

			return( this -> descriptor -> execute( arg ) );
		}
	};

private:
	std::map< int , ref_count_ptr<const Option_Descriptor> >
							short_option_map;

	std::map< std::string , ref_count_ptr<const Option_Descriptor> >
							long_option_map;

	bool				valid;
	bool				more_options;

	std::string			argv_0;
	std::vector< std::string >	argv_original;
	std::vector< std::string >	rest_opt;

protected:
	virtual	void	finish_analyze();

protected:
	virtual	int	analyze_non_option( const std::vector<std::string> & );

	virtual	void	after_analyze_hook();

	virtual	bool	check() const;

protected:
	virtual	std::vector<std::string> &	rest();

public:
		 Option_Analyzer();
	virtual	~Option_Analyzer();

	virtual	void	usage( std::ostream *  ostr = &std::cerr ) const;

	virtual	bool	analyze( int  argc ,  const char *const *const  argv );
	virtual	bool	analyze( const std::vector<std::string> &  argv );

	virtual	const std::string &			program_name() const;

	virtual	const std::vector<std::string> &	argv() const;

	virtual	const std::vector<std::string> &	rest_options() const;

	virtual	operator  bool() const;


	//
	// Add Option
	//
	virtual	void	add_short_option
			( int  short_option ,
			  const ref_count_ptr<const Option_Descriptor> & opt );

	virtual	void	add_long_option
			( const std::string &  long_option ,
			  const ref_count_ptr<const Option_Descriptor> & opt );

	virtual	void	add_option
			( int  short_option ,
			  const std::string &  long_option ,
			  const ref_count_ptr<const Option_Descriptor> & opt );


	//
	// Add Flag On Option
	//
	virtual	void	add_short_option_flag_on( int  short_option ,
						  bool *  flag );

	virtual	void	add_long_option_flag_on
					( const std::string &  long_option ,
					  bool *  flag );

	virtual	void	add_option_flag_on( int  short_option ,
					    const std::string &  long_option ,
					    bool *  flag );


	//
	// Add Flag Off Option
	//
	virtual	void	add_option_flag_off( int  short_option ,
					     const std::string &  long_option ,
					     bool *  flag );

	virtual	void	add_short_option_flag_off( int  short_option ,
						   bool *  flag );

	virtual	void	add_long_option_flag_off
					( const std::string &  long_option ,
					  bool *  flag );

	//
	// Add Flag Toggle
	//
	virtual	void	add_short_option_flag_toggle( int  short_option ,
						      bool *  flag );

	virtual	void	add_long_option_flag_toggle
					( const std::string &  long_option ,
					  bool *  flag );

	virtual	void	add_option_flag_toggle
					( int  short_option ,
					  const std::string &  long_option ,
					  bool *  flag );

	//
	// Add Integer Option
	//
	virtual	void	add_short_option_integer( int  short_option ,
						  int *  i );

	virtual	void	add_long_option_integer
					( const std::string &  long_option ,
					  int *  i );

	virtual	void	add_option_integer( int  short_option ,
					    const std::string &  long_option ,
					    int *  i );

	//
	// Add Long Integer Option
	//
	virtual	void	add_short_option_long_integer( int  short_option ,
						       long *  i );

	virtual	void	add_long_option_long_integer
					( const std::string &  long_option ,
					  long *  i );

	virtual	void	add_option_long_integer
					( int  short_option ,
					  const std::string &  long_option ,
					  long *  i );

	//
	// Add Unsigned Long Option
	//
	virtual	void	add_short_option_unsigned_long
					( int  short_option ,
					  unsigned long *  i );

	virtual	void	add_long_option_unsigned_long
					( const std::string &  long_option ,
					  unsigned long *  i );

	virtual	void	add_option_unsigned_long
					( int  short_option ,
					  const std::string &  long_option ,
					  unsigned long *  i );

	//
	// Add Float Option
	//
	virtual	void	add_short_option_float( int  short_option ,
						double *  d );

	virtual	void	add_long_option_float
					( const std::string &  long_option ,
					  double *  d );

	virtual	void	add_option_float( int  short_option ,
					  const std::string &  long_option ,
					  double *  d );

	//
	// Add String Option
	//
	virtual	void	add_short_option_string( int  short_option ,
						 std::string *  str );

	virtual	void	add_long_option_string
					( const std::string &  long_option ,
					  std::string *  str );

	virtual	void	add_option_string( int  short_option ,
					   const std::string &  long_option ,
					   std::string *  str );
	//
	// Add Port Number Option
	//
	virtual	void	add_short_option_port_number( int  short_option ,
						      port_number_t *  p );

	virtual	void	add_long_option_port_number
					( const std::string &  long_option ,
					  port_number_t *  p );

	virtual	void	add_option_port_number
					( int  short_option ,
					  const std::string &  long_option ,
					  port_number_t *  p );

	//
	// Add Path Option
	//
	virtual	void	add_short_option_path
					( int  short_option ,
					  Path *  p );

	virtual	void	add_long_option_path
					( const std::string &  long_option ,
					  Path *  p );

	virtual	void	add_option_path( int  short_option ,
					 const std::string &  long_option ,
					 Path *  p );


	//
	// Add Raw Arguments Option
	//
	virtual	void	add_short_option_raw_arguments
					( int  short_option ,
					  std::vector<std::string> *  a );

	virtual	void	add_long_option_raw_arguments
					( const std::string &  long_option ,
					  std::vector<std::string> *  a );

	virtual	void	add_option_raw_arguments
					( int  short_option ,
					  const std::string &  long_option ,
					  std::vector<std::string> *  a );
	//
	// Add Option_Analyzer Option
	//
	virtual	void	add_short_option_option_analyzer
					( int  short_option ,
					  Option_Analyzer *  opt ,
					  const std::string &  separator );

	virtual	void	add_long_option_option_analyzer
					( const std::string &  long_option ,
					  Option_Analyzer *  opt ,
					  const std::string &  separator );

	virtual	void	add_option_option_analyzer
					( int  short_option ,
					  const std::string &  long_option ,
					  Option_Analyzer *  opt ,
					  const std::string &  separator );
};


#endif	/* OPTION_ANALYZER_H_INCLUDED */
