#include  "d2_straight_line.h"

#include  <cmath>
#include  "math_extension.h"

using namespace std;

D2_Straight_Line::D2_Straight_Line( const D2_Vector &  base ,
				    const Angle &  direction )
{
	a = - direction.sin();
	b =   direction.cos();
	c = - (a * base.x() + b * base.y());
}


D2_Straight_Line::D2_Straight_Line( const D2_Vector &  point_1 ,
				    const D2_Vector &  point_2 )
{
	FLOAT	direction = (point_2 - point_1).theta().radian();

	a = - sin( direction );
	b =   cos( direction );
	c = - (a * point_1.x() + b * point_1.y());
}


D2_Straight_Line::D2_Straight_Line( FLOAT  aa ,  FLOAT  bb ,  FLOAT cc )
	: a(aa) , b(bb) , c(cc)
{
}


D2_Straight_Line::~D2_Straight_Line()
{
}


bool   D2_Straight_Line::in_region( const D2_Vector &  vec ) const
{
	return( (*this)(vec) == 0.0 );
}


FLOAT  D2_Straight_Line::area() const /* throw() */
{
	return( 0.0 );
}


ref_count_ptr<const D2_Region_Entity>  D2_Straight_Line::copy() const
{
	return( ref_count_ptr<const D2_Region_Entity>
		( new D2_Straight_Line( *this ) ) );
}




FLOAT  D2_Straight_Line::operator () ( const D2_Vector &  p ) const
{
	return( a * p.x() + b * p.y() + c );
}

D2_Vector  D2_Straight_Line::cross_point( const D2_Straight_Line &  line )
	const /* throw( D2_Region::No_Region_Error ) */
{
	const	FLOAT	EPS = static_cast<FLOAT>(1.0e-15);

	FLOAT	x;
	FLOAT	y;

	FLOAT	tmp = (*this).a * line.b - line.a * (*this).b;

	if ( float_traits<FLOAT>::fabs( tmp ) < EPS )
	{
		throw D2_Region::No_Region_Error();
	}

	x = ((*this).b * line.c - line.b * (*this).c) / tmp;
	y = (line.a * (*this).c - (*this).a * line.c) / tmp;

	return( D2_Vector( D2_Vector::XY , x , y ) );
}


bool   D2_Straight_Line::parallel( const D2_Straight_Line &  line ) const
{
	const	FLOAT	EPS = static_cast<FLOAT>(1.0e-15);

	return( ! (float_traits<FLOAT>::fabs
		   ( (*this).a * line.b - line.a * (*this).b ) < EPS) );
}


Angle  D2_Straight_Line::direction() const
{
	D2_Vector	v( D2_Vector::Pole , 1.0 , atan2( - a , b ) );

	v.normalize_theta();

	return( Angle( Angle::Radian , v.theta().radian() ) );
}


FLOAT  D2_Straight_Line::distance( const D2_Vector &  p ) const
{
	D2_Straight_Line	cross_line
				  ( p ,
				    this -> direction()
				    + Radian(Math_Extension::PI / 2.0) );

	return( (this -> cross_point( cross_line ) - p).r() );
}

D2_Vector  D2_Straight_Line::sample_point() const
{
	// XXX
	if ( float_traits<FLOAT>::fabs(a) >= fabs(b) )
	{
		return( D2_Vector( D2_Vector::XY ,  - c / a   ,  0.0 ) );
	}
	else
	{
		return( D2_Vector( D2_Vector::XY ,  0.0  ,  - c / b ) );
	}
}

D2_Vector  D2_Straight_Line::project_point( const D2_Vector &  p ) const
{
	D2_Straight_Line	cross_line
				  ( p ,
				    this -> direction()
				    + Radian(Math_Extension::PI / 2.0) );

	return( this -> cross_point( cross_line ) );
}
