#include  "d2_vector.h"
#include  <cmath>
#include  "math_extension.h"
#include  <cassert>

#ifndef D2_VECTOR_INLINE
# define D2_VECTOR_INLINE /**/
#endif


D2_VECTOR_INLINE
D2_Vector::D2_Vector()
	:
#if 0
	  xy_calculated( true ) ,
	  pole_calculated( true ) ,
#endif
	  x_value( 0.0 ) ,
	  y_value( 0.0 )
#if 0
	  ,
	  r_value    ( 0.0 ) ,
	  theta_value( 0.0 )
#endif
{
}

D2_VECTOR_INLINE
D2_Vector::D2_Vector( XY_or_Pole  type ,
		      FLOAT  x_or_r ,  FLOAT  y_or_theta )
	:
#if 0
	  xy_calculated( true ) ,
	  pole_calculated( true ) ,
#endif
	  x_value( 0.0 ) ,
	  y_value( 0.0 )
#if 0
	  ,
	  r_value    ( 0.0 ) ,
	  theta_value( 0.0 )
#endif
{
	this -> set( type , x_or_r , y_or_theta );
}

D2_VECTOR_INLINE
D2_Vector::D2_Vector( XY_or_Pole  type ,
		      FLOAT  r_val ,  const Angle &  theta_val )
	:
#if 0
	  xy_calculated( true ) ,
	  pole_calculated( true ) ,
#endif
	  x_value( 0.0 ) ,
	  y_value( 0.0 )
#if 0
	  ,
	  r_value    ( 0.0 ) ,
	  theta_value( 0.0 )
#endif
{
	this -> set( type , r_val , theta_val );
}

D2_VECTOR_INLINE
D2_Vector::D2_Vector( FLOAT  x_val ,  FLOAT  y_val )
	:
#if 0
	  xy_calculated( true ) ,
	  pole_calculated( true ) ,
#endif
	  x_value( 0.0 ) ,
	  y_value( 0.0 )
#if 0
	  ,
	  r_value    ( 0.0 ) ,
	  theta_value( 0.0 )
#endif
{
	this -> set( D2_Vector::XY , x_val , y_val );
}

D2_VECTOR_INLINE
D2_Vector::D2_Vector( const D2_Vector &  vec )
	:
#if 0
	  xy_calculated( true ) ,
	  pole_calculated( true ) ,
#endif
	  x_value( 0.0 ) ,
	  y_value( 0.0 )
#if 0
	  ,
	  r_value    ( 0.0 ) ,
	  theta_value( 0.0 )
#endif
{
	this -> set( vec );
}

D2_VECTOR_INLINE
D2_Vector::~D2_Vector()
{
}


D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set( XY_or_Pole  type ,
			     FLOAT  x_or_r ,  FLOAT  y_or_theta )
{
	if ( type == XY )
	{
		x_value = x_or_r;
		y_value = y_or_theta;
#if 0
		xy_calculated   = true;
		pole_calculated = false;
#endif
	}
	else
	{
#if 0
		r_value     = x_or_r;
		theta_value = y_or_theta;
		pole_calculated = true;
		xy_calculated   = false;
#else
		this -> set_xy_from_pole( x_or_r , y_or_theta );
#endif
	}

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set( XY_or_Pole  type ,
			     FLOAT  r_val ,  const Angle &  theta_val )
{
	assert( type == Pole );

	return( this -> set( D2_Vector::Pole , r_val , theta_val.radian() ) );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set( FLOAT  r_val ,  const Angle &  theta_val )
{
	return( this -> set( D2_Vector::Pole , r_val , theta_val.radian() ) );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set( FLOAT  x_val ,  FLOAT  y_val )
{
	return( this -> set( D2_Vector::XY , x_val , y_val ) );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set( const D2_Vector &  vec )
{
#if 0
	if ( vec.xy_calculated )
	{
		this -> xy_calculated = true;
		this -> x_value       = vec.x_value;
		this -> y_value       = vec.y_value;
	}
	else
	{
		this -> xy_calculated = false;
	}

	if ( vec.pole_calculated )
	{
		this -> pole_calculated = true;
		this -> r_value         = vec.r_value;
		this -> theta_value     = vec.theta_value;
	}
	else
	{
		this -> pole_calculated = false;
	}
#else
	this -> x_value = vec.x_value;
	this -> y_value = vec.y_value;
#endif

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set_x( FLOAT  x_val )
{
#if 0
	if ( ! xy_calculated )
	{
		calculate_xy();
	}
#endif

	x_value = x_val;

#if 0
	pole_calculated = false;
#endif

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set_y( FLOAT  y_val )
{
#if 0
	if ( ! xy_calculated )
	{
		calculate_xy();
	}
#endif

	y_value = y_val;

#if 0
	pole_calculated = false;
#endif

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set_r( FLOAT  r_val )
{
#if 0
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	r_value = r_val;

	xy_calculated = false;
#else
	FLOAT	current_r_val;
	FLOAT	current_theta_val;

	this -> get_pole_coordinate( &current_r_val , &current_theta_val );
	this -> set_xy_from_pole( r_val , current_theta_val );
#endif

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set_theta( const Angle &  theta_val )
{
#if 0
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	theta_value = theta_val.radian();

	xy_calculated = false;
#else
	FLOAT	current_r_val;
	FLOAT	current_theta_val;

	this -> get_pole_coordinate( &current_r_val , &current_theta_val );
	this -> set_xy_from_pole( current_r_val , theta_val.radian() );
#endif

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::operator = ( const D2_Vector &  vec )
{
	return( this -> set( vec ) );
}


D2_VECTOR_INLINE
FLOAT  D2_Vector::x() const
{
#if 0
	if ( ! xy_calculated )
	{
		calculate_xy();
	}
#endif

	return( x_value );
}

D2_VECTOR_INLINE
FLOAT  D2_Vector::y() const
{
#if 0
	if ( ! xy_calculated )
	{
		calculate_xy();
	}
#endif

	return( y_value );
}

D2_VECTOR_INLINE
FLOAT  D2_Vector::r() const
{
#if 0
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	return( r_value );
#else
	FLOAT	r_val;
	FLOAT	theta_val;

	this -> get_pole_coordinate( &r_val , &theta_val );

	return( r_val );
#endif

}

D2_VECTOR_INLINE
Angle  D2_Vector::theta() const
{
#if 0
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	return( Angle( Angle::Radian , theta_value ) );
#else
	FLOAT	r_val;
	FLOAT	theta_val;

	this -> get_pole_coordinate( &r_val , &theta_val );

	return( Radian( theta_val ) );
#endif
}


D2_VECTOR_INLINE
D2_Vector  D2_Vector::normalize() const
{
#if 0
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	if ( ! xy_calculated )
	{
		calculate_xy();
	}

	D2_Vector	ret = (*this);

	ret.r_value = 1.0;

	if ( ret.r_value <= eps() )
	{
		ret.xy_calculated = false;
	}
	else
	{
		ret.x_value /= this -> r_value;
		ret.y_value /= this -> r_value;
	}

	return( ret );
#else
	return( D2_Vector( Pole , 1.0 , this -> theta() ) );
#endif
}


D2_VECTOR_INLINE
D2_Vector  D2_Vector::normalize_theta() const
{
#if 0
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	D2_Vector	ret = (*this);

	// [-PI , +PI)
	if ( ret.theta_value >= (+ float_traits<FLOAT>::PI()) )
	{
		do
		{
			ret.theta_value -= 2.0 * float_traits<FLOAT>::PI();

		} while( ret.theta_value >= float_traits<FLOAT>::PI() );
	}
	else if ( ret.theta_value < (- float_traits<FLOAT>::PI()) )
	{
		do
		{
			ret.theta_value += 2.0 * float_traits<FLOAT>::PI();

		} while( ret.theta_value < (- float_traits<FLOAT>::PI()) );
	}

	if ( ret.theta_value >= (+ float_traits<FLOAT>::PI())
	  || ret.theta_value <  (- float_traits<FLOAT>::PI()) )
	{
		ret.theta_value = (- float_traits<FLOAT>::PI());
	}

#if 0
	assert( (- float_traits<FLOAT>::PI()) - eps() <= ret.theta_value
		&& ret.theta_value < (+ float_traits<FLOAT>::PI()) + eps() );
#endif

	return( ret );
#else
	return( *this );
#endif
}


D2_VECTOR_INLINE
D2_Vector  D2_Vector::rotate( const Angle &  d ) const
{
#if 0
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	D2_Vector	ret = (*this);

	ret.theta_value += d.radian();
	ret.xy_calculated = false;

	return( ret );
#else
	return( D2_Vector( Pole , this -> r() , this -> theta() + d ) );
#endif
}

D2_VECTOR_INLINE
D2_Vector  D2_Vector::reverse_x() const
{
#if 0
	if ( ! xy_calculated )
	{
		calculate_xy();
	}
#endif

	return( D2_Vector( D2_Vector::XY , - (this -> x()) , this -> y() ) );
}

D2_VECTOR_INLINE
D2_Vector  D2_Vector::reverse_y() const
{
#if 0
	if ( ! xy_calculated )
	{
		calculate_xy();
	}
#endif

	return( D2_Vector( D2_Vector::XY , this -> x() , - (this -> y()) ) );
}


#if 0
D2_VECTOR_INLINE
void  D2_Vector::calculate_xy() const
{
#if 0
	assert( pole_calculated );
#endif

	x_value = r_value * float_traits<FLOAT>::cos( theta_value );
	y_value = r_value * float_traits<FLOAT>::sin( theta_value );

	xy_calculated = true;
}

D2_VECTOR_INLINE
void  D2_Vector::calculate_pole() const
{
#if 0
	assert( xy_calculated );
#endif

	FLOAT	r2 = x_value * x_value + y_value * y_value;

	if ( r2 < 0.0 )
	{
		r2 = 0.0;
	}

	r_value = std::sqrt( r2 );

	if ( r_value == 0.0 )
	{
		theta_value = 0.0;
	}
	else
	{
		theta_value = float_traits<FLOAT>::atan2( y_value , x_value );
	}

	pole_calculated = true;
}
#endif


D2_VECTOR_INLINE
D2_Vector  D2_Vector::operator + () const
{
	return( *this );
}

D2_VECTOR_INLINE
D2_Vector  D2_Vector::operator - () const
{
	return( D2_Vector::origin() - (*this) );
}

D2_VECTOR_INLINE
D2_Vector  D2_Vector::operator + ( const D2_Vector &  vec ) const
{
#if 0
	if ( ! xy_calculated )
	{
		calculate_xy();
	}
#endif

	return( D2_Vector( D2_Vector::XY ,
			   this -> x_value + vec.x() ,
			   this -> y_value + vec.y() ) );
}

D2_VECTOR_INLINE
D2_Vector  D2_Vector::operator - ( const D2_Vector &  vec ) const
{
#if 0
	if ( ! xy_calculated )
	{
		calculate_xy();
	}
#endif

	return( D2_Vector( D2_Vector::XY ,
			   this -> x_value - vec.x() ,
			   this -> y_value - vec.y() ) );
}

D2_VECTOR_INLINE
D2_Vector  D2_Vector::operator * ( FLOAT  d ) const
{
#if 0
	if ( pole_calculated )
	{
		return( D2_Vector( D2_Vector::Pole ,
				   this -> r_value * d ,
				   this -> theta_value ) );
	}
	else
	{
#if 0
		assert( xy_calculated );
#endif

		return( D2_Vector( D2_Vector::XY ,
				   this -> x_value * d ,
				   this -> y_value * d ) );
	}
#else
	return( D2_Vector( D2_Vector::XY ,
			   this -> x_value * d ,
			   this -> y_value * d ) );
#endif
}

D2_VECTOR_INLINE
D2_Vector  D2_Vector::operator / ( FLOAT  d ) const
{
	if ( d <= eps() )
	{
		this -> divided_by_zero_notify();

		return( D2_Vector( D2_Vector::XY , 0.0 , 0.0 ) );
	}

#if 0
	if ( pole_calculated )
	{
		return( D2_Vector( D2_Vector::Pole ,
				   this -> r_value / d ,
				   this -> theta_value ) );
	}
	else
	{
#if 0
		assert( xy_calculated );
#endif

		return( D2_Vector( D2_Vector::XY ,
				   this -> x_value / d ,
				   this -> y_value / d ) );
	}
#else
	return( D2_Vector( D2_Vector::XY ,
			   this -> x_value / d ,
			   this -> y_value / d ) );
#endif
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::operator += ( const D2_Vector &  vec )
{
	// checked: OK. if (this == &vec)

#if 0
	if ( ! xy_calculated )
	{
		calculate_xy();
	}
#endif

	x_value += vec.x();
	y_value += vec.y();

#if 0
	pole_calculated = false;
#endif

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::operator -= ( const D2_Vector &  vec )
{
	// checked: OK. if (this == &vec)

#if 0
	if ( ! xy_calculated )
	{
		calculate_xy();
	}
#endif

	x_value -= vec.x();
	y_value -= vec.y();

#if 0
	pole_calculated = false;
#endif

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::operator *= ( FLOAT  d )
{
#if 0
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	r_value *= d;

	xy_calculated = false;
#else
	x_value *= d;
	y_value *= d;
#endif

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::operator /= ( FLOAT  d )
{
#if 0
	if ( ! pole_calculated )
	{
		calculate_pole();
	}
#endif

	if ( d <= eps() )
	{
		this -> divided_by_zero_notify();

		return( *this );
	}

#if 0
	r_value /= d;

	xy_calculated = false;
#else
	x_value /= d;
	y_value /= d;
#endif

	return( *this );
}


D2_VECTOR_INLINE
D2_Vector  operator * ( FLOAT  d ,  const D2_Vector &  vec )
{
	return( vec * d );
}


D2_VECTOR_INLINE
bool   D2_Vector::equal( const D2_Vector &  vec , FLOAT  epsilon ) const
{
	return( (vec - *(this)).r() <= epsilon );
}


D2_VECTOR_INLINE
bool   D2_Vector::operator == ( const D2_Vector &  vec ) const
{
	return( (vec - *(this)).r() == 0.0 );
}

D2_VECTOR_INLINE
bool   D2_Vector::operator != ( const D2_Vector &  vec ) const
{
	return( (vec - *(this)).r() != 0.0 );
}


D2_VECTOR_INLINE
FLOAT  D2_Vector::x_abs() const
{
	return( float_traits<FLOAT>::fabs( this -> x() ) );
}

D2_VECTOR_INLINE
FLOAT  D2_Vector::y_abs() const
{
	return( float_traits<FLOAT>::fabs( this -> y() ) );
}

D2_VECTOR_INLINE
int    D2_Vector::x_sign( int  zero_value ) const
{
	if ( this -> x() > 0.0 )
	{
		return( + 1 );
	}
	else if ( this -> x() < 0.0 )
	{
		return( - 1 );
	}
	else
	{
		return( zero_value );
	}
}

D2_VECTOR_INLINE
int    D2_Vector::y_sign( int  zero_value ) const
{
	if ( this -> y() > 0.0 )
	{
		return( + 1 );
	}
	else if ( this -> y() < 0.0 )
	{
		return( - 1 );
	}
	else
	{
		return( zero_value );
	}
}


D2_VECTOR_INLINE
FLOAT  D2_Vector::r_square() const
{
	FLOAT	r_sq = x_value * x_value + y_value * y_value;

	if ( r_sq < 0.0 )
	{
		r_sq = 0.0;
	}

	return( r_sq );
}


D2_VECTOR_INLINE
FLOAT  D2_Vector::inner_product( const D2_Vector &  p ) const
{
	return( this -> x() * p.x() + this -> y() * p.y() );
}


D2_VECTOR_INLINE
FLOAT  D2_Vector::eps()
{
	return( EPSILON );
}


D2_VECTOR_INLINE
D2_Vector  D2_Vector::origin()
{
	return( D2_Vector( D2_Vector::XY , 0.0 , 0.0 ) );
}


#if 1
D2_VECTOR_INLINE
void   D2_Vector::set_xy_from_pole( FLOAT   r_val ,  FLOAT   theta_val ) const
{
	this -> x_value = r_val * float_traits<FLOAT>::cos( theta_val );
	this -> y_value = r_val * float_traits<FLOAT>::sin( theta_val );
}


D2_VECTOR_INLINE
void   D2_Vector::get_pole_coordinate( FLOAT *  r_value ,
				       FLOAT *  theta_value ) const
{
	FLOAT	r2 = x_value * x_value + y_value * y_value;

	if ( r2 < 0.0 )
	{
		r2 = 0.0;
	}

	*r_value = std::sqrt( r2 );

	if ( *r_value == 0.0 )
	{
		*theta_value = 0.0;
	}
	else
	{
		*theta_value = float_traits<FLOAT>::atan2( y_value , x_value );
	}
}
#endif
