
/* This is an SDL implementation of a BMP to ICO translator

   (Information taken from "Icons in Win32",
	http://www.microsoft.com/win32dev/ui/icons.htm)
 */

#include <stdio.h>

#include <SDL/SDL.h>
#include <SDL/SDL_endian.h>
#include "bmp2ico.h"

/*
 * Calculate an 8-bit (3 red, 3 green, 2 blue) dithered palette of colors
 */
static void DitherColors(SDL_Color *colors, int bpp)
{
	Uint8 r, g, b, i;

	/* Calculate the dither values for each index */
	switch (bpp) {
		case 8:
			for ( r=0; r<(1<<3); ++r ) {
				for ( g=0; g<(1<<3); ++g ) {
					for ( b=0; b<(1<<2); ++b ) {
						i = ((r<<(3+2))|(g<<2)|b);
						colors[i].r = r<<(8-3);
						colors[i].g = g<<(8-3);
						colors[i].b = b<<(8-2);
					}
				}
			}
			break;
		default:
			/* ?? */
			break;
	}
}

#define SETMASK(base, row, col, val) \
((Uint8 *)base)[sizeof(*base)+icon_plen+icon_mlen-(row+1)*mask_pitch+col/8] \
			|= (val<<(7-col%8));

static int SDL_SurfaceToICO(SDL_data *stored, SDL_Surface *icon,
							int transparent)
{
	SDL_Palette *pal_256;
	SDL_Surface *icon_256;
	Uint8 *pdata, *pwin32;
	Uint8 clear;
	int icon_len;
	int icon_plen;
	int icon_mlen;
	int icon_pitch;
	int mask_pitch;
	SDL_Rect bounds;
	int i, skip;
	int row, col;
	Win32ICO *icon_win32;

	/* Allocate the win32 bmp icon and set everything to zero */
	icon_pitch = ((icon->w+3)&~3);
	mask_pitch = ((icon->w+7)/8);
	icon_plen = icon->h*icon_pitch;
	icon_mlen = icon->h*mask_pitch;
	icon_len = sizeof(*icon_win32)+icon_plen+icon_mlen;
	icon_win32 = (Win32ICO *)malloc(icon_len);
	if ( icon_win32 == NULL ) {
		return;
	}
	memset(icon_win32, 0, icon_len);

	/* Set the basic BMP parameters */
	icon_win32->biSize = sizeof(*icon_win32)-sizeof(icon_win32->biColors);
	icon_win32->biWidth = icon->w;
	icon_win32->biHeight = icon->h*2;
	icon_win32->biPlanes = 1;
	icon_win32->biBitCount = 8;
	icon_win32->biSizeImage = icon_plen+icon_mlen;

	/* Allocate a standard 256 color icon surface */
	icon_256 = SDL_AllocSurface(SDL_SWSURFACE, icon->w, icon->h,
					 icon_win32->biBitCount, 0, 0, 0, 0);
	if ( icon_256 == NULL ) {
		return;
	}
	pal_256 = icon_256->format->palette;
	if (icon->format->palette && 
		(icon->format->BitsPerPixel == icon_256->format->BitsPerPixel)){
		Uint8 black;
		memcpy(pal_256->colors, icon->format->palette->colors,
					pal_256->ncolors*sizeof(SDL_Color));
		/* Make sure that 0 is black! */
		black = SDL_MapRGB(icon->format, 0x00, 0x00, 0x00);
		pal_256->colors[black] = pal_256->colors[0];
		pal_256->colors[0].r = 0x00;
		pal_256->colors[0].g = 0x00;
		pal_256->colors[0].b = 0x00;
	} else {
		DitherColors(pal_256->colors, icon_256->format->BitsPerPixel);
	}

	/* Now copy color data to the icon BMP */
	for ( i=0; i<(1<<icon_win32->biBitCount); ++i ) {
		icon_win32->biColors[i].rgbRed = pal_256->colors[i].r;
		icon_win32->biColors[i].rgbGreen = pal_256->colors[i].g;
		icon_win32->biColors[i].rgbBlue = pal_256->colors[i].b;
	}

	/* Convert icon to a standard surface format.  This may not always
	   be necessary, as Windows supports a variety of BMP formats, but
	   it greatly simplifies our code.
	*/ 
        bounds.x = 0;
        bounds.y = 0;
        bounds.w = icon->w;
        bounds.h = icon->h;
        if ( SDL_LowerBlit(icon, &bounds, icon_256, &bounds) < 0 ) {
		SDL_FreeSurface(icon_256);
                return;
	}
	if ( transparent ) {
		clear = *(Uint8 *)icon_256->pixels;
	}

	/* Copy pixels upside-down to icon BMP, masked with the icon mask */
	if ( SDL_MUSTLOCK(icon_256) || (icon_256->pitch != icon_pitch) ) {
		SDL_FreeSurface(icon_256);
		SDL_SetError("Warning: Unexpected icon_256 characteristics");
		return;
	}
	pdata = (Uint8 *)icon_256->pixels;
	pwin32 = (Uint8 *)icon_win32+sizeof(*icon_win32)+icon_plen-icon_pitch;
	skip = icon_pitch - icon->w;
	for ( row=0; row<icon->h; ++row ) {
		for ( col=0; col<icon->w; ++col ) {
			if ( transparent ) {
				if ( *pdata == clear ) {
					SETMASK(icon_win32, row, col, 1);
				} else {
					*pwin32 = *pdata;
				}
			} else {
				*pwin32 = *pdata;
			}
			++pdata;
			++pwin32;
		}
		pdata  += skip;
		pwin32 += skip;
		pwin32 -= 2*icon_pitch;
	}
	SDL_FreeSurface(icon_256);

	/* We're done! */
	stored->data = (void *)icon_win32;
	stored->dlen = icon_len;
	return(0);
}

static int ConvertBMPtoICO(int index, SDL_Surface *bmp, int transparent,
								SDL_RWops *ico)
{
	ICONDIRENTRY ico_entry;
	Win32ICO *icon_win32;
	int icon_len;
	SDL_data stored;

	/* Convert the surface to an icon */
	if ( SDL_SurfaceToICO(&stored, bmp, transparent) < 0 ) {
		return(-1);
	}
	icon_win32 = (Win32ICO *)stored.data;
	icon_len = stored.dlen;

	/* Fill in the icon entry structure */
	if ( (icon_win32->biWidth != 32) || (icon_win32->biHeight/2 != 32) ) {
		fprintf(stderr,
		"Warning: Icon size (%dx%d) not 32x32 -- not supported\n",
				icon_win32->biWidth, icon_win32->biHeight/2);
	}
	ico_entry.bWidth = icon_win32->biWidth;
	ico_entry.bHeight = icon_win32->biHeight/2;
	if ( icon_win32->biBitCount < 8 ) {
		ico_entry.bColorCount = 1 << icon_win32->biBitCount;
	} else {
		ico_entry.bColorCount = 0;
	}
	ico_entry.bReserved = 0;
	ico_entry.wPlanes = icon_win32->biPlanes;
	ico_entry.wBitCount = icon_win32->biBitCount;
	ico_entry.dwBytesInRes = icon_len;
	ico_entry.dwImageOffset = SDL_RWtell(ico);

	/* Write out the icon entry structure */
	SDL_RWseek(ico, 6+(index*sizeof(ICONDIRENTRY)), SEEK_SET);
	SDL_RWwrite(ico, &ico_entry.bWidth, 1, 1);
	SDL_RWwrite(ico, &ico_entry.bHeight, 1, 1);
	SDL_RWwrite(ico, &ico_entry.bColorCount, 1, 1);
	SDL_RWwrite(ico, &ico_entry.bReserved, 1, 1);
	SDL_WriteLE16(ico, ico_entry.wPlanes);
	SDL_WriteLE16(ico, ico_entry.wBitCount);
	SDL_WriteLE32(ico, ico_entry.dwBytesInRes);
	SDL_WriteLE32(ico, ico_entry.dwImageOffset);

	/* Write out image data */
	SDL_RWseek(ico, ico_entry.dwImageOffset, SEEK_SET);
	SDL_WriteLE32(ico, icon_win32->biSize);
	SDL_WriteLE32(ico, icon_win32->biWidth);
	SDL_WriteLE32(ico, icon_win32->biHeight);
	SDL_WriteLE16(ico, icon_win32->biPlanes);
	SDL_WriteLE16(ico, icon_win32->biBitCount);
	SDL_WriteLE32(ico, icon_win32->biCompression);
	SDL_WriteLE32(ico, icon_win32->biSizeImage);
	SDL_WriteLE32(ico, icon_win32->biXPelsPerMeter);
	SDL_WriteLE32(ico, icon_win32->biYPelsPerMeter);
	SDL_WriteLE32(ico, icon_win32->biClrUsed);
	SDL_WriteLE32(ico, icon_win32->biClrImportant);
	if ( ! SDL_RWwrite(ico, (Uint8 *)icon_win32+40, icon_len-40, 1) ) {
		return(-1);
	}
	free(icon_win32);
	return(0);
}

static void PrintUsage(char *argv0)
{
	fprintf(stderr,
"Usage: %s [-transparent] file.ico file.bmp [file.bmp] ...\n", argv0);
	exit(1);
}

int main(int argc, char *argv[])
{
	char *argv0;
	SDL_RWops *ico;
	int numicons;
	int transparent;
	long i;
	SDL_Surface *bmp;

	/* Initialize SDL */
	if ( SDL_Init(0) < 0 ) {
		fprintf(stderr, "Couldn't initialize SDL: %s\n",SDL_GetError());
		exit(-1);
	}
	atexit(SDL_Quit);

	/* Check command line */
	transparent = 0;
	argv0 = argv[0];
	if ( argv[1] && (strcmp(argv[1], "-transparent") == 0) ) {
		transparent = 1;
		++argv; --argc;
	}
	if ( argc <= 2 ) {
		PrintUsage(argv0);
	}

	/* Open the ICO file, and write the header */
	if ( strcasecmp(argv[1]+strlen(argv[1])-4, ".ico") != 0 ) {
		PrintUsage(argv0);
	}
	ico = SDL_RWFromFile(argv[1], "wb");
	if ( ico == NULL ) {
		fprintf(stderr, "Couldn't write %s\n", argv[1]);
		exit(2);
	}
	numicons = argc-2;
	SDL_WriteLE16(ico, 0);
	SDL_WriteLE16(ico, 1);
	SDL_WriteLE16(ico, numicons);
	SDL_RWseek(ico, 6+numicons*sizeof(ICONDIRENTRY), SEEK_SET);

	/* Write each icon */
	for ( i=2; argv[i]; ++i ) {
		fprintf(stderr, "Converting BMP file: %s\n", argv[i]);
		bmp = SDL_LoadBMP(argv[i]);
		if ( bmp == NULL ) {
			fprintf(stderr, "Couldn't read %s\n", argv[i]);
			exit(3);
		}
		if ( ConvertBMPtoICO(i-2, bmp, transparent, ico) < 0 ) {
			fprintf(stderr, "Couldn't convert BMP: %s\n",
							SDL_GetError());
			exit(3);
		}
		SDL_FreeSurface(bmp);
	}
	SDL_RWclose(ico);
	exit(0);
}
