/*
  Daemonshogi -- a GTK+ based, simple shogi (Japanese chess) program.

  Copyright (C) Masahiko Tokita    2002-2005,2009 
  Copyright (C) Hisashi Horikawa   2008-2010

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


// #define GTK_DISABLE_DEPRECATED 1
// #define GDK_DISABLE_DEPRECATED 1
// #define G_DISABLE_DEPRECATED 1

#include <config.h>
#include <gtk/gtk.h>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cassert>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include <utility>

G_BEGIN_DECLS
#include "interface.h"
#include "support.h"
G_END_DECLS
#include "canvas.h"
#include "history_window.h"
#include "misc.h"
#include "si/si.h"
#include "si/si-think.h"

#include "back.xpm"
#include "koma.xpm"

using namespace std;


static void daemon_canvas_draw_back_update( Canvas* canvas, const GdkRectangle* rect );

static void daemon_canvas_draw_back_update_from_tmp_pixmap( Canvas* canvas,
						     const GdkRectangle* rect );

static void daemon_canvas_change_size_sprite( Canvas* canvas );


static int KOMA_SPRITE_XY[][2] = {
    { -1,  -1}, /* なし */
    {  0,   0}, /* 先手歩   0x01 */
    { 40,   0}, /* 先手香車 0x02 */
    { 80,   0}, /* 先手桂馬 0x03 */
    {120,   0}, /* 先手銀   0x04 */
    {160,   0}, /* 先手金   0x05 */
    {200,   0}, /* 先手角   0x06 */
    {240,   0}, /* 先手飛車 0x07 */
    {280,   0}, /* 先手王   0x08 */
    {  0,  40}, /* 先手と   0x09 */
    { 40,  40}, /* 先手成香 0x0A */
    { 80,  40}, /* 先手成桂 0x0B */
    {120,  40}, /* 先手成銀 0x0C */
    {280,  40}, /* 先手玉   0x0D */
    {200,  40}, /* 先手馬   0x0E */
    {240,  40}, /* 先手竜   0x0F */
    { -1,  -1}, /* 先手竜   0x10 */
    {  0,  80}, /* 後手歩   0x11 */
    { 40,  80}, /* 後手香車 0x12 */
    { 80,  80}, /* 後手桂馬 0x13 */
    {120,  80}, /* 後手銀   0x14 */
    {160,  80}, /* 後手金   0x15 */
    {200,  80}, /* 後手角   0x16 */
    {240,  80}, /* 後手飛車 0x17 */
    {280,  80}, /* 後手王   0x18 */
    {  0, 120}, /* 後手と   0x19 */
    { 40, 120}, /* 後手成香 0x1A */
    { 80, 120}, /* 後手成桂 0x1B */
    {120, 120}, /* 後手成銀 0x1C */
    {280, 120}, /* 後手玉   0x1D */
    {200, 120}, /* 後手馬   0x1E */
    {240, 120}, /* 後手竜   0x1F */
};


/** 駒スプライトを生成する */
static void daemon_canvas_init_sprite(Canvas* canvas)
{
  // GdkPixbuf *im2;
  Sprite* sprite;
  int i;

  for (i = 0; i < SPRITEMAX; i++)
    canvas->sprite[i] = NULL;

  GdkPixbuf* koma_src = gdk_pixbuf_new_from_xpm_data(koma_xpm);
  if ( !koma_src ) {
    si_abort("No enough memory in daemon_canvas_init(). koma_src is NULL\n");
  }

  // スプライト用の駒 pixmap を作る
  canvas->koma_pix = load_pixmap_from_imlib_image_scaled( CAIRO_FORMAT_ARGB32, 
					 koma_src, 320, 160 );
  assert( canvas->koma_pix );

  for (i = 0; i < SPRITEMAX; i++) {
    sprite = daemon_sprite_new( canvas->koma_pix, 0, 0, 40, 40 ); // 座標は仮
    canvas->sprite[i] = sprite;
    // daemon_sprite_set_type(sprite, KOMA_SPRITE);
    // daemon_sprite_set_src_x(sprite, 0);
    // daemon_sprite_set_src_y(sprite, 0);
    // daemon_sprite_set_width(sprite, 40);
    // daemon_sprite_set_height(sprite, 40);
    daemon_sprite_set_visible( sprite, FALSE );
    daemon_sprite_set_movable( sprite, TRUE );
  }
}


/** 駒種類をセットする */
static void daemon_sprite_set_koma( Sprite* sprite, intptr_t koma ) 
{
  g_assert( sprite );
  g_assert(1 <= koma && koma <= 0x1F);

  // 表示するときにスケールする
  daemon_sprite_set_src_xy( sprite, 
			    KOMA_SPRITE_XY[koma][0], 
			    KOMA_SPRITE_XY[koma][1] );
  sprite->userdata = (void*) koma;
}


/** 駒種類を返す */
static intptr_t daemon_sprite_get_koma( Sprite* sprite ) 
{
  g_assert( sprite );
  return (intptr_t) sprite->userdata;
}


/** 持ち駒の種類の数を数える */
static int count_piece(const DBoard* board, TEBAN next) 
{
  gint count;
  gint i;

  count = 0;
  for (i=1; i <= 7; i++) {
    if (0 < daemon_dboard_get_piece(board, next, i)) {
      count++;
    }
  }

  return count;
}


static Sprite* daemon_canvas_get_sprite(Canvas* canvas, gint no) 
{
  g_assert(canvas != NULL);
  g_assert(0 <= no);
  g_assert(no < SPRITEMAX);

  return canvas->sprite[no];
}


static double get_scale_factor( const Canvas* canvas )
{
  if ( canvas->size == D_CANVAS_SIZE_SMALL )
    return 1.0;
  else if ( canvas->size == D_CANVAS_SIZE_MIDDLE )
    return ((double) D_CANVAS_SIZE_MIDDLE_X) / D_CANVAS_SIZE_SMALL_X;
  else if ( canvas->size == D_CANVAS_SIZE_BIG )
    return ((double) D_CANVAS_SIZE_BIG_X) / D_CANVAS_SIZE_SMALL_X;
  else {
    abort();
  }
}


/** スプライトを配置する */
static void daemon_canvas_dispose_sprite( Canvas* canvas ) 
{
  DBoard* board;
  Sprite* sprite;
  gint x, y, num;
  gint width, height;
  int base_x, base_y;
  double scale_factor;
  gint p;
  gint i, j;
  
  board = &(canvas->board);

  scale_factor = get_scale_factor( canvas );

  base_x = 140 * scale_factor;
  base_y = 10 * scale_factor;

  for (i = 0; i < SPRITEMAX; i++) {
    sprite = daemon_canvas_get_sprite(canvas, i);
    if ( !sprite )
      continue;
    // if (daemon_sprite_get_type(sprite) != KOMA_SPRITE) {
    // continue;
    // }
    daemon_sprite_set_visible(sprite, FALSE);
  }

  /* 盤に駒を配置 */
  num = 0;
  for (y = 1; y <= 9; y++) {
    for (x = 1; x <= 9; x++) {
      p = daemon_dboard_get_board(board, x, y);
      if (!p)
	continue;

      g_assert(1 <=p && p <= 0x1F);
      sprite = daemon_canvas_get_sprite(canvas, num++);
      g_assert(sprite != NULL);
      // while (sprite->type != KOMA_SPRITE) {
      // sprite = daemon_canvas_get_sprite(canvas, num++);
      // g_assert(sprite != NULL);
      // }
      width  = daemon_sprite_get_width(sprite) * scale_factor;
      height = daemon_sprite_get_height(sprite) * scale_factor;
      if (daemon_canvas_get_front(canvas) == SENTE) {
	daemon_sprite_set_xy(sprite, (10 - x) * width - width + base_x,
			     y * height - height + base_y);
	daemon_sprite_set_koma(sprite, p);
      }
      else {
	daemon_sprite_set_xy(sprite, (10 - (10-x)) * width - width + base_x,
			     (10-y) * height - height + base_y);
	daemon_sprite_set_koma(sprite, p ^ 0x10);
      }
      daemon_sprite_set_visible(sprite, TRUE);
    }
  }

  /* 持ち駒を配置する */
  {
    static const int base_y_value[9] = {
      0, 65, 65, 45, 45, 25, 5, 5, 5,
      // {0, 65, 65, 85, 85, 105, 125, 125, 125},
    };
    gint count, flg;
    gint x, y, skip_x;
    int komadai;

    // komadai = 0 で画面右手
    for (komadai = 0; komadai < 2; komadai++) {
      TEBAN teban;
      if (daemon_canvas_get_front(canvas) == SENTE)
	teban = komadai == 0 ? SENTE : GOTE;
      else
	teban = komadai == 0 ? GOTE : SENTE;

      count = count_piece(&canvas->board, teban); // 持ち駒の種類の数
      if (komadai == 0) {
	base_x = 511; // (510,210) - (640,380)
	base_y = 210 + base_y_value[count];
      }
      else {
	base_x = 129 - 40; // (0,0) - (130,170)
	base_y = 170 - base_y_value[count] - 40;
      }
      flg = 0; // 手前のとき、=0で左側、=1で右側

      // 大駒から順に表示
      for (i = 7; i >= 1; i--) {
	int p = daemon_dboard_get_piece(board, teban, i);
	if (!p)
	  continue;

	if (i == 1) {
	  skip_x = 80;
	  if (flg == 1) {
	    base_y += komadai == 0 ? +40 : -40;
	    flg = 0;
	  }
	} 
	else
	  skip_x = 20;

	// 駒を重ねて表示
	for (j = 0; j < p; j++) {
	  x = 5 + (skip_x - skip_x / p * (j + 1)) + (flg ? 60 : 0);

	  sprite = daemon_canvas_get_sprite(canvas, num++);
	  g_assert(sprite != NULL);
	  // while (sprite->type != KOMA_SPRITE) {
	  //   sprite = daemon_canvas_get_sprite(canvas, num++);
	  //   g_assert(sprite != NULL);
	  // }

	  daemon_sprite_set_xy(sprite, 
	       daemon_canvas_rate_xy(canvas, base_x + (komadai == 0 ? +x : -x)),
	       daemon_canvas_rate_xy(canvas, base_y));
	  daemon_sprite_set_koma(sprite, i + komadai * 0x10);
	  daemon_sprite_set_visible(sprite, TRUE);
	}

	if (flg == 1) {
	  base_y += komadai == 0 ? +40 : -40;
	  flg = 0;
	} 
	else
	  flg = 1;
      }
    }

    /* 駒箱 */
    if (daemon_canvas_get_mode(canvas) == D_CANVAS_MODE_EDIT) {
    count = count_pbox(&(canvas->board));
    base_x = 1;
    base_y = 210;
    flg = 0;
    skip_x = 20;

    for (i=8; 1 <= i; i--) {
      p = daemon_dboard_get_pbox(board, i);
      if (p == 0) {
	continue;
      }
      if (0 && i == 1) {
	skip_x = 80;
	flg = 0;
      } else {
	skip_x = 20;
      }
      for (j=0; j<p; j++) {
	x = 5 + (skip_x - skip_x / p * (j + 1)) + (flg ? 60 : 0);
	y = base_y_value[count];

	sprite = daemon_canvas_get_sprite(canvas, num++);
	g_assert(sprite != NULL);
	//	while (sprite->type != KOMA_SPRITE) {
	//sprite = daemon_canvas_get_sprite(canvas, num++);
	// g_assert(sprite != NULL);
	// }

	daemon_sprite_set_xy(sprite, daemon_canvas_rate_xy(canvas, base_x + x),
			     daemon_canvas_rate_xy(canvas, base_y + y));
	daemon_sprite_set_koma(sprite, i);
	daemon_sprite_set_visible(sprite, TRUE);
      }

      if (flg == 1) {
	base_y += 40;
	flg = 0;
      } else {
	flg = 1;
      }
    }
    }
  }
}


/** 初期化 */
static void daemon_canvas_init( Canvas* canvas )
{
  assert( canvas != NULL );
  assert( canvas->window );

  GtkWidget* window = canvas->window;

  daemon_canvas_set_size(canvas, D_CANVAS_SIZE_SMALL);
  canvas->pixmap[0] = cairo_image_surface_create( CAIRO_FORMAT_RGB24,
						  D_CANVAS_SIZE_SMALL_X,
						  D_CANVAS_SIZE_SMALL_Y );
  assert( canvas->pixmap[0] );
  canvas->pixmap[1] = cairo_image_surface_create( CAIRO_FORMAT_RGB24,
						  D_CANVAS_SIZE_MIDDLE_X,
						  D_CANVAS_SIZE_MIDDLE_Y );
  assert( canvas->pixmap[1] );
  canvas->pixmap[2] = cairo_image_surface_create( CAIRO_FORMAT_RGB24,
						  D_CANVAS_SIZE_BIG_X,
						  D_CANVAS_SIZE_BIG_Y );
  assert( canvas->pixmap[2] );

  // canvas->gc          = gdk_gc_new(window->window);
  canvas->drawingarea = lookup_widget(GTK_WIDGET(window), "drawingarea");
  canvas->drag        = FALSE;
  canvas->mode        = D_CANVAS_MODE_EDIT;
  canvas->back_src    = gdk_pixbuf_new_from_xpm_data(back_xpm);
  if ( !canvas->back_src ) {
    si_abort("No enough memory in daemon_canvas_init(). back_src is NULL\n");
  }
  canvas->back[0] = NULL;
  canvas->back[1] = NULL;
  canvas->back[2] = NULL;
  daemon_canvas_create_back_pixmap(canvas);

  bo_reset_to_hirate(&canvas->board);
  canvas->record = NULL;

  canvas->history = new History(canvas);
  canvas->move_property_dialog = NULL;

  canvas->tmp_pixmap = NULL;

  /* スプライト初期化 */
  daemon_canvas_init_sprite(canvas);

  /* スプライト配置 */
  daemon_canvas_dispose_sprite(canvas);

  canvas->front = SENTE;
  daemon_canvas_set_sensitive(canvas);
  canvas->flg_thinking = FALSE;
}


static void daemon_canvas_invalidate_rect( Canvas* canvas, 
					   const GdkRectangle* rect ) 
{
  gdk_window_invalidate_rect( gtk_widget_get_window(canvas->drawingarea),
			      rect, FALSE);
}


static void daemon_canvas_invalidate( Canvas* canvas ) 
{
  GdkRectangle rect;

  rect.x = 0;
  rect.y = 0;
  rect.width = canvas->width;
  rect.height = canvas->height;

  daemon_canvas_invalidate_rect(canvas, &rect);
}


void Canvas::update_window_title()
{
  if ( record && record->filename != "" ) {
    vector<string> path;
    str_split( &path, record->filename, '/' );
    gtk_window_set_title( GTK_WINDOW(window), 
			  ((record->changed ? string("*") : string("")) +
                           path.back() + " - " + _("Daemonshogi")).c_str() );
  }
  else {
    gtk_window_set_title( GTK_WINDOW(window), _("Daemonshogi") );
  }
}


static void daemon_canvas_draw_sprite_all( Canvas* canvas )
{
  Sprite* sprite;
  gint i;

  g_assert(canvas != NULL);

  daemon_sprite_draw_all( canvas->sprite, 
			  daemon_canvas_get_pixmap(canvas) );

  // 最後の手をハイライト (枠で囲む)
  // TODO: 新しいレイヤを作って, そこへ描画する
  if ( canvas->history && canvas->history->focus_line.size() > 0 &&
       canvas->history->cur_pos > 0 ) {
    DTe te = canvas->history->focus_line[canvas->history->cur_pos - 1].first;
    int x = te.to & 0xf;
    int y = (te.to >> 4);

    GdkRectangle rect;
    if (daemon_canvas_get_front(canvas) == SENTE) {
      rect.x = daemon_canvas_rate_xy(canvas, 
				     140 + (9 - x) * D_SPRITE_KOMA_WIDTH);
      rect.y = daemon_canvas_rate_xy(canvas, 
				     10 + (y - 1) * D_SPRITE_KOMA_HEIGHT);
    }
    else {
      rect.x = daemon_canvas_rate_xy(canvas, 
				     140 + (x - 1) * D_SPRITE_KOMA_WIDTH);
      rect.y = daemon_canvas_rate_xy(canvas, 
				     10 + (10 - y - 1) * D_SPRITE_KOMA_HEIGHT);
    }
    rect.width = daemon_canvas_rate_xy(canvas, D_SPRITE_KOMA_WIDTH);
    rect.height = daemon_canvas_rate_xy(canvas, D_SPRITE_KOMA_HEIGHT);

    cairo_t* cr = cairo_create( daemon_canvas_get_pixmap(g_canvas) );
    gdk_cairo_rectangle( cr, &rect );
    cairo_destroy( cr );

    daemon_canvas_invalidate_rect(g_canvas, &rect);
  }
}


/** gtk+オブジェクトを構築する */
void canvas_build(Canvas* canvas)
{
  printf( "%s: enter.\n", __func__ ); // DEBUG
  daemon_canvas_init(canvas);

  /* 背景描画 */
  daemon_canvas_draw_back(canvas);

  /* 手番、時刻更新 */
  daemon_canvas_draw_time(canvas);

  /* スプライト描画 */
  daemon_canvas_draw_sprite_all(canvas);

  /* 更新 */
  daemon_canvas_invalidate(canvas);
  
  /* ステータスバー更新 */
  daemon_canvas_update_statusbar(canvas);
}


/** デストラクタ */
void daemon_canvas_free(Canvas* canvas) 
{
  assert( canvas );
  int i;

  for (i = 0; i < 3; i++) {
    cairo_surface_destroy( canvas->pixmap[i] );
    canvas->pixmap[i] = NULL;
  }

  for (i = 0; i < 3; i++) {
    if (canvas->back[i]) {
      cairo_surface_destroy( canvas->back[i] );
      canvas->back[i] = NULL;
    }
  }
	
  if (canvas != NULL)
    delete canvas;
}


/*************************************************************/
/* set_* , get_* */
/*************************************************************/

GtkWidget* daemon_canvas_get_window(Canvas *canvas) {
  return canvas->window;
}


/*
GdkGC* daemon_canvas_get_gc(Canvas *canvas) {
  return canvas->gc;
}
*/


cairo_surface_t* daemon_canvas_get_pixmap( Canvas* canvas ) 
{
  g_assert(canvas != NULL);

  if (canvas->size == D_CANVAS_SIZE_SMALL) {
    return canvas->pixmap[0];
  } 
  else if (canvas->size == D_CANVAS_SIZE_MIDDLE) {
    return canvas->pixmap[1];
  } 
  else if (canvas->size == D_CANVAS_SIZE_BIG) {
    return canvas->pixmap[2];
  } 
  else {
    abort(); /* 仕様上ありえない分岐 */
  }

  return NULL;
}


cairo_surface_t* daemon_canvas_get_back( Canvas* canvas )
{
  g_assert(canvas != NULL);

  if (canvas->size == D_CANVAS_SIZE_SMALL) {
    return canvas->back[0];
  } 
  else if (canvas->size == D_CANVAS_SIZE_MIDDLE) {
    return canvas->back[1];
  } 
  else if (canvas->size == D_CANVAS_SIZE_BIG) {
    return canvas->back[2];
  } 
  else {
    abort(); /* 仕様上ありえない分岐 */
  }

  return NULL;
}


GtkWidget* daemon_canvas_get_drawinarea(Canvas *canvas) 
{
  g_assert(canvas != NULL);
  return canvas->drawingarea;
}


static gboolean daemon_canvas_isdrag(const Canvas* canvas) 
{
  g_assert(canvas != NULL);
  return canvas->drag;
}


void daemon_canvas_set_drag(Canvas *canvas, gboolean drag) 
{
  g_assert(canvas != NULL);
  g_assert(drag == TRUE || drag == FALSE);
  // printf("%s: drag start.\n", __func__); // DEBUG

  canvas->drag = drag;
}


gint daemon_canvas_get_dragno(Canvas *canvas) 
{
  g_assert(canvas != NULL);
  return canvas->drag_no;
}


void daemon_canvas_set_dragno(Canvas *canvas, gint no) 
{
  g_assert(canvas != NULL);
  g_assert(0 <= no);
  canvas->drag_no = no;
}


GdkPoint daemon_canvas_get_diff(Canvas* canvas) 
{
  g_assert(canvas != NULL);
  return canvas->diff;
}


void daemon_canvas_set_diff(Canvas *canvas, gint x, gint y) {
  g_assert(canvas != NULL);
  g_assert(0 <= x);
  g_assert(0 <= y);
  canvas->diff.x = x;
  canvas->diff.y = y;
}

D_CANVAS_MODE daemon_canvas_get_mode(const Canvas* canvas)
{
  return canvas->mode;
}


/** canvas のサイズをセットする */
void daemon_canvas_set_size( Canvas* canvas, D_CANVAS_SIZE size )
{
  g_assert(canvas != NULL);
  g_assert(size == D_CANVAS_SIZE_SMALL ||
	   size == D_CANVAS_SIZE_MIDDLE ||
	   size == D_CANVAS_SIZE_BIG);

  canvas->size = size;
  if (size == D_CANVAS_SIZE_SMALL) {
    canvas->width  = D_CANVAS_SIZE_SMALL_X;
    canvas->height = D_CANVAS_SIZE_SMALL_Y;
  } 
  else if (size == D_CANVAS_SIZE_MIDDLE) {
    canvas->width  = D_CANVAS_SIZE_MIDDLE_X;
    canvas->height = D_CANVAS_SIZE_MIDDLE_Y;
  } 
  else if (size == D_CANVAS_SIZE_BIG) {
    canvas->width  = D_CANVAS_SIZE_BIG_X;
    canvas->height = D_CANVAS_SIZE_BIG_Y;
  } 
}


void daemon_canvas_set_front(Canvas* canvas, TEBAN player) 
{
  assert( player == SENTE || player == GOTE );
  canvas->front = player;
}


TEBAN daemon_canvas_get_front(const Canvas* canvas) 
{
  return canvas->front;
}


/*************************************************************/
/* event */
/*************************************************************/

/** マウスを動かしたとき. マウスドラッグ */
void daemon_canvas_motion_notify(Canvas* canvas, GdkEventMotion* event)
{
  GdkModifierType mask;
  gint x, y;
  gint no;
  Sprite* sprite;
  GdkPoint diff;
  GdkRectangle rect;

  g_assert(canvas != NULL);
  g_assert(event != NULL);

  if (event->is_hint) {
    gdk_window_get_pointer (event->window, &x, &y, &mask);
  } else {
    x = event->x;
    y = event->y;
  }
  
  // ドラッグ中でなければ戻る
  if ( !daemon_canvas_isdrag(canvas) )
    return;

  // ドラッグ開始時点のスナップショット
  if ( !canvas->tmp_pixmap )
    daemon_canvas_create_tmp_pixmap(canvas);

  /* ダブルクリック判定用時刻を初期化 */
  canvas->tv.tv_sec  = 0;
  canvas->tv.tv_usec = 0;

  double scale_factor = get_scale_factor(canvas);

  no     = daemon_canvas_get_dragno(canvas);
  sprite = daemon_canvas_get_sprite(canvas, no);
  diff   = daemon_canvas_get_diff(canvas);

  rect.x      = daemon_sprite_get_x(sprite);
  rect.y      = daemon_sprite_get_y(sprite);
  rect.width  = daemon_sprite_get_width(sprite) * scale_factor;
  rect.height = daemon_sprite_get_height(sprite) * scale_factor;

  daemon_sprite_set_xy(sprite, event->x - diff.x, event->y - diff.y);

  /* 背景とドラッグ中の駒を描く */
  daemon_canvas_draw_back_update_from_tmp_pixmap(canvas, &rect);
  daemon_sprite_draw( daemon_canvas_get_sprite(canvas, no),
		      daemon_canvas_get_pixmap(canvas) );
  daemon_canvas_invalidate_rect(canvas, &rect);
}


/** マウスボタンが押された */
void daemon_canvas_button_press( Canvas* canvas, GdkEventButton* event)
{
  gint     no;
  gboolean doubleclick = FALSE;
  Sprite* sprite;
  gint sprite_x, sprite_y;
  gint dest_x, dest_y;
  D_CANVAS_AREA area;
  // SPRITE_TYPE type;
  D_CANVAS_MODE mode;
  gint p;

  g_assert(canvas != NULL);
  g_assert(event != NULL);

  if (event->type != GDK_BUTTON_PRESS)
    return;

  if (event->button != 1) {
    /* 1 番のボタン以外が押された */
    return;
  }

  mode = daemon_canvas_get_mode(canvas);
  if (mode == D_CANVAS_MODE_PRE_GAME)
    return;

  if (daemon_canvas_isdrag(canvas) == TRUE) {
    /* ドラッグ中にクリックされた */
    daemon_canvas_set_drag(canvas, FALSE);
    return;
  }

  no = daemon_sprite_point_on_sprite( canvas->sprite, event->x, event->y );
  if (no == -1) {
    /* クリックされたスプライトなし */
    return;
  }

  sprite = daemon_canvas_get_sprite(canvas, no);
  doubleclick = daemon_canvas_is_doubleclick(canvas);

  sprite_x = daemon_sprite_get_x(sprite);
  sprite_y = daemon_sprite_get_y(sprite);
  area = daemon_canvas_on_drag_area(canvas, sprite_x, sprite_y, &dest_x, &dest_y);
  // type = daemon_sprite_get_type(sprite);

  if ( mode == D_CANVAS_MODE_GAME || mode == D_CANVAS_MODE_BOOK ) {
    // if (type != KOMA_SPRITE)
    // return;

    if ( mode == D_CANVAS_MODE_GAME &&
         canvas->playertype[canvas->board.next] != SI_GUI_HUMAN ) {
      // printf("expected %d, but %d\n",
      //       SI_HUMAN, canvas->playertype[canvas->board.next]);  // DEBUG
      return;
    }

    p = daemon_sprite_get_koma(sprite);

    if (daemon_canvas_get_front(canvas) == SENTE) {
      if (canvas->board.next == SENTE && 0x10 < p) {
	return;
      } else if (canvas->board.next == GOTE && p < 0x10) {
	return;
      }
    } 
    else {
      if (canvas->board.next == SENTE && p < 0x10) {
	return;
      } else if (canvas->board.next == GOTE && 0x10 < p) {
	return;
      }
    }
  }
  else if ( mode == D_CANVAS_MODE_EDIT ) {
    if (doubleclick && area == D_CANVAS_AREA_BOARD /*&& type == KOMA_SPRITE*/) {
      /* ダブルクリックされた */
      gint koma;
      GdkRectangle rect;

      koma = daemon_sprite_get_koma(sprite);
      if ((koma & 0x0F) == 0x05 || (koma & 0x0F) == 0x08) {
	if (koma <= 0x10) {
	  koma += 0x10;
	} else {
	  koma -= 0x10;
	}
      } else if (koma <= 0x10) {
	if (koma & 0x08) {
	  koma += 0x10;
	}
	koma ^= 0x08;
      } else {
	if (koma & 0x08) {
	  koma -= 0x10;
	}
	koma ^= 0x08;
      }
      daemon_sprite_set_koma(sprite, koma);
      if ( daemon_canvas_get_front(canvas) == SENTE ) 
	daemon_dboard_set_board(&(canvas->board), dest_x, dest_y, koma);
      else {
	daemon_dboard_set_board(&canvas->board, 10 - dest_x, 10 - dest_y, 
			      koma ^ 0x10);
      }
    
      rect.x      = daemon_sprite_get_x(sprite);
      rect.y      = daemon_sprite_get_y(sprite);
      rect.width  = daemon_sprite_get_width(sprite);
      rect.height = daemon_sprite_get_height(sprite);
      /* 背景を描く */
      daemon_canvas_draw_back_update(canvas, &rect);
      /* 駒を描く */
      // daemon_canvas_draw_sprite(canvas, no);
      // daemon_canvas_draw_sprite_update_all(canvas, &rect);
      daemon_canvas_draw_sprite_all(canvas);
    
      daemon_canvas_invalidate(canvas);
    
      return;
    }
  }

  // スプライト上で押された -> ドラッグ開始 
  gint x, y;

  if ( daemon_sprite_ismovable(sprite) ) {
    x = daemon_sprite_get_x(sprite);
    y = daemon_sprite_get_y(sprite);

    daemon_sprite_set_drag_from( sprite, x, y );

    daemon_canvas_set_dragno(canvas, no);
    daemon_canvas_set_diff(canvas, event->x - x, event->y - y);
    daemon_canvas_set_drag(canvas, TRUE);
  }
}


/**
 * (x, y) の座標が盤面のどのエリアにあるか調べる。<br>
 * 返り値が<UL>
 * <LI> 0 : 該当エリアなし
 * <LI> 1 : 左上駒台
 * <LI> 2 : 右下駒台
 * <LI> 3 : 左下駒箱
 * <LI> 4 : 盤の上
 * </UL>
 * 返り値が 4 の場合は、その盤上の座標をdest_x dest_y にセットする
 *  (1, 1) 〜 (9, 9) の値。
 */
D_CANVAS_AREA daemon_canvas_on_drag_area(Canvas *canvas, gint x, gint y,
			       gint *dest_x, gint *dest_y) {
  GdkPoint point;
  GdkRectangle left_koma, right_koma, komabako, board;

  point.x = x / get_scale_factor(canvas) + D_SPRITE_KOMA_WIDTH / 2;
  point.y = y / get_scale_factor(canvas) + D_SPRITE_KOMA_HEIGHT / 2;

  left_koma.x = 1;
  left_koma.y = 1;
  left_koma.width  = 129;
  left_koma.height = 169;
  if (daemon_on_rectangle(&left_koma, &point) == TRUE) {
    return D_CANVAS_AREA_LEFT_KOMADAI;
  }

  right_koma.x = 511;
  right_koma.y = 210;
  right_koma.width  = 129;
  right_koma.height = 169;
  if (daemon_on_rectangle(&right_koma, &point) == TRUE) {
    return D_CANVAS_AREA_RIGHT_KOMADAI;
  }

  komabako.x = 1;
  komabako.y = 210;
  komabako.width  = 129;
  komabako.height = 169;
  if (daemon_on_rectangle(&komabako, &point) == TRUE) {
    return D_CANVAS_AREA_KOMABAKO;
  }

  board.x = 140;
  board.y = 11;
  board.width  = 360;
  board.height = 360;
  if (daemon_on_rectangle(&board, &point) == TRUE) {
    /* 10 - は駒の座標の右向き、左向きを反転させるため */
    *dest_x = 10 - ((point.x - board.x) / D_SPRITE_KOMA_WIDTH + 1);
    *dest_y = (point.y - board.y) / D_SPRITE_KOMA_HEIGHT + 1;
    return D_CANVAS_AREA_BOARD;
  }

  return D_CANVAS_AREA_NONE;
}


// extern int THINK_DEEP_MAX;

static Xy rotate_xy(Xy xy)
{
  int x = xy & 0xf;
  int y = xy >> 4;
  return ((10 - y) << 4) + (10 - x);
}


static void button_release_on_edit_mode(Canvas* canvas, GdkEventButton* event)
{
  gint to_x, to_y;
  gint from_x, from_y;
  gint sprite_x, sprite_y;
  gint sprite_from_x, sprite_from_y;
  gint no;
  D_CANVAS_AREA to_ret, from_ret;
  Sprite* sprite;
  PieceKind p, p2;
  // SPRITE_TYPE type;

  no            = daemon_canvas_get_dragno(canvas);
  sprite        = daemon_canvas_get_sprite(canvas, no);
  sprite_x      = daemon_sprite_get_x(sprite);
  sprite_y      = daemon_sprite_get_y(sprite);
  sprite_from_x = daemon_sprite_get_from_x(sprite);
  sprite_from_y = daemon_sprite_get_from_y(sprite);

  // type          = daemon_sprite_get_type(sprite);
  // assert( type == KOMA_SPRITE );

  to_x = to_y = 0;
  to_ret = daemon_canvas_on_drag_area(canvas, sprite_x, sprite_y,
					&to_x, &to_y);

  from_ret = daemon_canvas_on_drag_area(canvas, sprite_from_x, sprite_from_y,
					  &from_x, &from_y);

  p = daemon_sprite_get_koma(sprite);

  if (from_ret == D_CANVAS_AREA_NONE ||
	to_ret == D_CANVAS_AREA_NONE ||
	((p & 0xF) == 0x8 &&
	 (to_ret == D_CANVAS_AREA_LEFT_KOMADAI ||
	  to_ret == D_CANVAS_AREA_RIGHT_KOMADAI))) {
      /* 駒台、盤の外にドラッグしようとした */

    from_x = daemon_sprite_get_from_x(sprite);
    from_y = daemon_sprite_get_from_y(sprite);
    daemon_sprite_set_xy(sprite, from_x, from_y);

    canvas_redraw(canvas);
    return;
  } 

  // 駒を取り除く
  if (from_ret == D_CANVAS_AREA_BOARD) {
    if ( daemon_canvas_get_front(canvas) == SENTE )
      daemon_dboard_set_board(&(canvas->board), from_x, from_y, 0);
    else
      boSetPiece(&canvas->board, rotate_xy((from_y << 4) + from_x), 0);
  } 
  else if (from_ret == D_CANVAS_AREA_LEFT_KOMADAI) {
    daemon_dboard_dec_piece(&(canvas->board),
				daemon_canvas_get_front(canvas) ^ 1,
				p & 7);
  } 
  else if (from_ret == D_CANVAS_AREA_RIGHT_KOMADAI) {
    daemon_dboard_dec_piece(&(canvas->board),
				daemon_canvas_get_front(canvas),
				p & 7);
  } 
  else {
    p &= 0xF;
    p = (p != 8) ? p & 7 : p;
    daemon_dboard_dec_pbox(&(canvas->board), p);
  }

  // 駒を置く
  if (to_ret == D_CANVAS_AREA_BOARD) {
    Xy to_xy = daemon_canvas_get_front(canvas) == SENTE ? 
                        (to_y <<4) + to_x : rotate_xy((to_y <<4) + to_x);
    p2 = boGetPiece(&canvas->board, to_xy);
    if ( p2 != 0 ) {
      p2 &= 0xF;
      p2 = (p2 != 8) ? p2 & 7 : p2;
      daemon_dboard_add_pbox(&(canvas->board), p2);
    }
    boSetPiece( &canvas->board, to_xy, 
		daemon_canvas_get_front(canvas) == SENTE ? p : p ^ 0x10 );
  } 
  else if (to_ret == D_CANVAS_AREA_LEFT_KOMADAI) {
    daemon_dboard_add_piece(&(canvas->board),
				daemon_canvas_get_front(canvas) ^ 1,
				p & 7);
  } 
  else if (to_ret == D_CANVAS_AREA_RIGHT_KOMADAI) {
    daemon_dboard_add_piece(&(canvas->board),
				daemon_canvas_get_front(canvas),
				p & 7);
  } 
  else {
    p &= 0xF;
    p = (p != 8) ? p & 7 : p;
    daemon_dboard_add_pbox(&(canvas->board), p);
  }

  canvas_redraw(canvas);
}


static void append_sequence(Canvas* canvas, const TE& te)
{
  // 現在の局面ノードを得る
  KyokumenNode* node = g_canvas->record->mi;
  for (int i = 0; i < g_canvas->history->cur_pos; i++)
    node = node->lookup_child( g_canvas->history->focus_line[i].first );

  DTe dte( DTe::NORMAL_MOVE, te, canvas->board.board[te_to(te)] );

  KyokumenNode::Moves::const_iterator it;
  for (it = node->moves.begin(); it != node->moves.end(); it++) {
    if ( it->te == dte ) {
      // すでにある手順
      if ( canvas->history->focus_line[canvas->history->cur_pos].first != 
	   dte ) {
	// 別の枝へ
	canvas->history->sync(*canvas->record,
                              canvas->history->cur_pos, dte);
      }

      // 棋譜をなぞる
      g_signal_emit_by_name(
        G_OBJECT(lookup_widget(g_canvas->window, "next")),
        "activate",
        NULL);
      
      return;
    }
  }

  if ( canvas->history->focus_line.size() <= canvas->history->cur_pos ) {
    // 棋譜を伸ばす
    node->add_child(dte, 0, "");
    canvas->history->append_move(canvas->board, dte, 0);

    // 手を指す
    g_signal_emit_by_name(
      G_OBJECT(lookup_widget(g_canvas->window, "next")),
      "activate",
      NULL);
  }
  else {
    // 新しい分岐
    int r = daemon_canvas_yes_no_dialog(canvas, _("Do you add alternative?"));
    if (r == GTK_RESPONSE_YES) {
      node->add_child(dte, 0, "");
      canvas->history->sync(*canvas->record, canvas->history->cur_pos, dte);

      g_signal_emit_by_name(
        G_OBJECT(lookup_widget(g_canvas->window, "next")),
        "activate",
        NULL);
    }
    else {
      canvas_redraw(canvas);
    }
  }
}


static void button_release_on_game_mode(Canvas* canvas, GdkEventButton* event)
{
  gint to_x, to_y;
  gint from_x, from_y;
  gint sprite_x, sprite_y;
  gint sprite_from_x, sprite_from_y;
  gint no;
  D_CANVAS_AREA to_ret, from_ret;
  PieceKind p;
  Sprite *sprite;
  // SPRITE_TYPE type;

  no            = daemon_canvas_get_dragno(canvas);
  sprite        = daemon_canvas_get_sprite(canvas, no);
  sprite_x      = daemon_sprite_get_x(sprite);
  sprite_y      = daemon_sprite_get_y(sprite);
  sprite_from_x = daemon_sprite_get_from_x(sprite);
  sprite_from_y = daemon_sprite_get_from_y(sprite);

  // type          = daemon_sprite_get_type(sprite);
  // assert( type == KOMA_SPRITE );

  to_x = to_y = 0;
  to_ret = daemon_canvas_on_drag_area(canvas, sprite_x, sprite_y,
					&to_x, &to_y);

  from_ret = daemon_canvas_on_drag_area(canvas, sprite_from_x, sprite_from_y,
					  &from_x, &from_y);

  p = daemon_sprite_get_koma(sprite);

  if (from_ret == D_CANVAS_AREA_NONE || to_ret != D_CANVAS_AREA_BOARD ) {
    /* 駒台、盤の外にドラッグしようとした */

    from_x = daemon_sprite_get_from_x(sprite);
    from_y = daemon_sprite_get_from_y(sprite);
    daemon_sprite_set_xy(sprite, from_x, from_y);

    canvas_redraw(canvas);
    return;
  }
  
  TE te;
  gboolean just_te;

  if ( from_ret == D_CANVAS_AREA_LEFT_KOMADAI ||
       from_ret == D_CANVAS_AREA_RIGHT_KOMADAI ) {
    te = te_make_put( (to_y << 4) + to_x, p & 7 );
  } 
  else {
    te = te_make_move( (from_y << 4) + from_x,
		       (to_y << 4) + to_x,
		       false );
  }

  if (daemon_canvas_get_front(canvas) == GOTE) {
    if (te_from(te) ) {
      te = te_make_move( rotate_xy(te_from(te)), rotate_xy(te_to(te)),
			 te_promote(te) );
    }
    else
      te = te_make_put( rotate_xy(te_to(te)), te_pie(te) );
  }

  // 合法な手？
  just_te = FALSE;
  if ( te_is_put(te) )
    just_te = te_is_valid( &canvas->board, te);
  else {
    just_te = te_is_valid( &canvas->board, te);
    
    TE te2 = te;
    te_set_promote(&te2, true);
    bool just_nari = te_is_valid( &canvas->board, te2);
    
    if (just_nari) {
      if (just_te) {
        if ( daemon_canvas_yes_no_dialog(canvas, _("Promote ?")) == 
             GTK_RESPONSE_YES )
          te_set_promote(&te, true);
      }
      else {
	just_te = true;
        te_set_promote(&te, true);
      }
    }
  }

  if (just_te) {
    // 千日手？
    bool forbidden = false; // 連続王手
    if ( te_is_sennichite(&canvas->board, te, &forbidden)) {
      if (forbidden)
	just_te = false; // 連続王手ではない千日手は指してよい
    }
  }

  if (just_te) {
    if (canvas->mode == D_CANVAS_MODE_GAME) {
      canvas->gui_te = 
                DTe( DTe::NORMAL_MOVE, te, canvas->board.board[te_to(te)] );
      daemon_canvas_invalidate(canvas);
    }
    else if (canvas->mode == D_CANVAS_MODE_BOOK) {
      append_sequence(canvas, te);
    }
    else {
      abort();
    }
  } 
  else {
    from_x = daemon_sprite_get_from_x(sprite);
    from_y = daemon_sprite_get_from_y(sprite);
    daemon_sprite_set_xy(sprite, from_x, from_y);

    canvas_redraw(canvas);
  }
}


/** マウスボタンを放した -> 駒をドロップ */
void daemon_canvas_button_release(Canvas* canvas, GdkEventButton* event) 
{
  g_assert(canvas != NULL);
  g_assert(event != NULL);

  // printf("%s: enter.\n", __func__); // DEBUG

  if (event->type != GDK_BUTTON_RELEASE || event->button != 1) 
    return;

  if (daemon_canvas_isdrag(canvas) == FALSE)
    return;

  switch (daemon_canvas_get_mode(canvas)) {
  case D_CANVAS_MODE_EDIT:
    button_release_on_edit_mode(canvas, event);
    break;
  case D_CANVAS_MODE_GAME:
  case D_CANVAS_MODE_BOOK:
    button_release_on_game_mode(canvas, event);
    break;
  case D_CANVAS_MODE_PRE_GAME:
    return; // 動かせない
  default:
    abort();
  }

  daemon_canvas_set_drag(canvas, FALSE);
  cairo_surface_destroy( canvas->tmp_pixmap );
  canvas->tmp_pixmap = NULL;
}


/** 盤の大きさを変更する */
void daemon_canvas_change_size( Canvas* canvas, D_CANVAS_SIZE size ) 
{
  g_assert(canvas);
  g_assert(size == D_CANVAS_SIZE_SMALL ||
	   size == D_CANVAS_SIZE_MIDDLE ||
	   size == D_CANVAS_SIZE_BIG);

  daemon_canvas_set_size(canvas, size);

  gtk_widget_set_size_request(canvas->drawingarea, 
			      canvas->width, canvas->height);
  gtk_window_resize( GTK_WINDOW(canvas->window),
		                             canvas->width, canvas->height );

  daemon_canvas_change_size_sprite( canvas );
  
  // 再描画
  canvas_redraw(canvas);

  gtk_widget_show(canvas->drawingarea);
}


bool bookwindow_item_is_actived(Canvas* canvas) 
{
  GtkWidget* bookwindow_item = lookup_widget(canvas->window, "bookwindow");
  assert(bookwindow_item);
  return gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(bookwindow_item));
}


/** メニュー/棋譜ウィンドウのチェックをON/OFF */
void history_window_menuitem_set_active(Canvas* canvas, bool a)
{
  GtkWidget* bookwindow_item = lookup_widget(canvas->window, "bookwindow");
  assert(bookwindow_item);
  gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(bookwindow_item), a );
}


/** モード変更 */
void daemon_canvas_change_mode(Canvas *canvas, D_CANVAS_MODE mode) 
{
  printf("mode to %d\n", mode);

  D_CANVAS_MODE before_mode;

  before_mode = daemon_canvas_get_mode(canvas);

  if (mode == D_CANVAS_MODE_EDIT) {
    daemon_dboard_set_for_edit(&(canvas->board));
    if (bookwindow_item_is_actived(canvas)) {
      canvas->history->hide_window();
      history_window_menuitem_set_active(canvas, false);
    }

    // 棋譜をクリア
    if (canvas->record)
      daemon_record_free(canvas->record);
    canvas->record = daemon_record_new();
    canvas->history->clear();
    alt_moves_dialog_hide();
  }

  canvas->mode = mode;

  /* 背景の画像を編集モード用にする */
  daemon_canvas_create_back_pixmap(canvas);

  // 再描画
  canvas_redraw(canvas);

  /* ボタン、メニューの使用可／使用不可を設定する */
  daemon_canvas_set_sensitive(canvas);
  
  if (mode == D_CANVAS_MODE_BOOK) {
    if (!bookwindow_item_is_actived(canvas))
      canvas->history->show_window();
  }

  daemon_canvas_update_statusbar(canvas);
}


/** 画面の大きさ変更に合わせてスプライトの大きさを変更する */
static void daemon_canvas_change_size_sprite( Canvas* canvas )
{
  Sprite *sprite;
  // GdkBitmap *mask;
  // gint width, height;
  gint i;

  double scale_factor = get_scale_factor(canvas);

  for (i = 0; i < SPRITEMAX; i++) {
    sprite = daemon_canvas_get_sprite(canvas, i);
    if ( !sprite )
      continue;

    daemon_sprite_set_scale( sprite, scale_factor, scale_factor );
  }
}


static void daemon_canvas_draw_time_up(Canvas* canvas, int teban)
{
  if (daemon_canvas_get_mode(canvas) != D_CANVAS_MODE_GAME)
    return;

  int x, y, width, height, step;
  char buf[100];
  PangoLayout* layout;
 
  if ((teban == 0 && daemon_canvas_get_front(canvas) == SENTE) ||
      (teban == 1 && daemon_canvas_get_front(canvas) == GOTE)) {
    /* 画面右手 */
    x      = daemon_canvas_rate_xy(canvas, 516 - 1);
    y      = daemon_canvas_rate_xy(canvas, 158 + 10 + 4);
    width  = daemon_canvas_rate_xy(canvas, 129);
    height = daemon_canvas_rate_xy(canvas,  38 - 20);
    step   = daemon_canvas_rate_xy(canvas,  15);
  }
  else {
    /* 画面左手 */
    x      = daemon_canvas_rate_xy(canvas,   6 - 1);
    y      = daemon_canvas_rate_xy(canvas, 158 + 10 + 4);
    width  = daemon_canvas_rate_xy(canvas, 129);
    height = daemon_canvas_rate_xy(canvas,  38 - 20);
    step   = daemon_canvas_rate_xy(canvas,  15);
  }

  cairo_surface_t* pixmap = daemon_canvas_get_pixmap(canvas);
  cairo_surface_t* back = daemon_canvas_get_back(canvas);

  cairo_t* cr = cairo_create( pixmap ); // dest

  GdkRectangle rect;
  rect.x = x; rect.y = y; rect.width = width; rect.height = height;

  cairo_set_source_surface( cr, back, 0, 0 );
  gdk_cairo_rectangle( cr, &rect );
  cairo_fill( cr );

  // 秒読み
  if ( teban == canvas->board.next )
    time2string(buf, canvas->elapsed);
  else
    strcpy(buf, "        ");
  buf[8] = '/';
  time2string(buf + 9, 
	      g_game.total_time[teban] - g_game.total_elapsed_time[teban] );

  layout = gtk_widget_create_pango_layout(canvas->window, buf);
  cairo_move_to(cr, x, y);
  cairo_set_source_rgb(cr, 1.0, 1.0, 1.0);
  pango_cairo_show_layout(cr, layout);
  cairo_destroy(cr);

  g_object_unref(layout);

  daemon_canvas_invalidate_rect(canvas, &rect);
}


static gboolean on_timeout( gpointer data )
{
  g_canvas->elapsed = time(NULL) - g_game.think_start_time;
  daemon_canvas_draw_time_up( g_canvas, g_canvas->board.next );
  return TRUE;
}


/** 棋譜ファイルを読み込む */
void daemon_kiffile_load_ok_button_pressed(const char* filename)
{
  Record *record;
  D_LOADSTAT stat;

  // エラーのときも NULL ではない
  record = daemon_record_load( filename );
  stat = daemon_record_get_loadstat(record);
  if (stat != D_LOAD_SUCCESS) {
    // ファイル読み込み失敗
#ifndef NDEBUG
    KyokumenNode* node = record->mi;
    while (node) {
      if (node->moves.size() > 0) {
	KyokumenNode::Moves::const_iterator it;
	for (it = node->moves.begin(); it != node->moves.end(); it++)
	  daemon_dte_output(&it->te, stdout);
	printf("\n");
	node = node->moves.begin()->node;
      }
      else
	break;
    }
#endif // !NDEBUG
    daemon_messagebox( g_canvas, 
        (string(_("Failed to load file.")) + "\n" + record->strerror).c_str(), 
        GTK_MESSAGE_ERROR );

    daemon_record_free(record);
    return;
  }

  daemon_record_free(g_canvas->record);
  g_canvas->record = record;

  daemon_dboard_copy(&(record->first_board), &(g_canvas->board));

  // 注目する手筋を更新
  g_canvas->history->sync( *record, 0, record->mi->moves.begin()->te );
  alt_moves_dialog_update();

  daemon_canvas_change_mode(g_canvas, D_CANVAS_MODE_BOOK);
  g_canvas->update_window_title();
}


/** 棋譜ファイルを保存する */
void daemon_kiffile_save_ok_button_pressed( const string& filename, 
                                            FileWriter* writer,
                                            Record::FileType filetype )
{
  int ret;

  if (g_canvas->mode == D_CANVAS_MODE_EDIT) {
    if ( !g_canvas->record )
      g_canvas->record = daemon_record_new();
    daemon_dboard_copy(&(g_canvas->board), &(g_canvas->record->first_board));
  }

  ret = -1;
  switch (filetype)
  {
  case Record::D_FORMAT_CSA:
    ret = daemon_record_output_csa(g_canvas->record, writer );
    break;
  case Record::D_FORMAT_KIF:
    ret = daemon_record_output_kif(g_canvas->record, writer );
    break;
  default:
    abort();
  }

  if (ret != 0) {
    /* ファイル書き込み失敗 */
    daemon_messagebox(g_canvas, _("Failed to write the file."),
		      GTK_MESSAGE_ERROR);
  }
  else {
    g_canvas->record->filename = filename;
    g_canvas->record->filetype = filetype;
    g_canvas->record->changed = false;
    g_canvas->update_window_title();
  }
}


/**
 * マウスカーソルがcanvasから出たときに呼ばれる。
 * ドラッグ中ならばスプライトを戻す。
 */
void daemon_canvas_leave_notify(Canvas *canvas,
				GtkWidget *widget,
				GdkEventCrossing *event,
				gpointer user_data) {
  gint no;
  Sprite *sprite;
  gint x, y;
  GdkPoint diff;
  GdkRectangle rect;

  if ( !daemon_canvas_isdrag(canvas) )
    return;

  // ドラッグ中だったとき
  daemon_canvas_set_drag(canvas, FALSE);
  no     = daemon_canvas_get_dragno(canvas);
  sprite = daemon_canvas_get_sprite(canvas, no);
  diff   = daemon_canvas_get_diff(canvas);
  x      = daemon_sprite_get_from_x(sprite);
  y      = daemon_sprite_get_from_y(sprite);

  rect.x      = daemon_sprite_get_x(sprite);
  rect.y      = daemon_sprite_get_y(sprite);
  rect.width  = daemon_sprite_get_width(sprite);
  rect.height = daemon_sprite_get_height(sprite);

  daemon_sprite_set_xy(sprite, x, y);

  /* 再描画 */
  daemon_canvas_draw_back_update_from_tmp_pixmap(canvas, &rect);
  daemon_sprite_draw( daemon_canvas_get_sprite(canvas, no),
		         	     daemon_canvas_get_pixmap(canvas) );
  daemon_canvas_invalidate(canvas);
}


/** 主ウィンドウをすべて再描画 */
void canvas_redraw(Canvas* canvas)
{
  /* スプライト再配置 */
  daemon_canvas_dispose_sprite(canvas);

  /* 画面再描画 */
  daemon_canvas_draw_back(canvas);
  daemon_canvas_draw_sprite_all(canvas);
  daemon_canvas_draw_time(canvas);

  daemon_canvas_invalidate(canvas);
}


/**
 * メニュー項目のうち進む/戻る.
 * 棋譜モードで、進む/戻るボタンを押したときに呼び出される
 */
void daemon_canvas_set_kif_sensitive(Canvas* canvas) 
{
  GtkWidget *widget;
  int i;
  bool back_enable, forward_enable;
  DTe::Special s;

  g_assert(canvas);

  if ( canvas->mode == D_CANVAS_MODE_BOOK ) {
    back_enable = canvas->history->cur_pos != 0;
    if ( canvas->history->focus_line.size() > 0 && 
	 (s = canvas->history->focus_line.back().first.special, 
	           !(s == DTe::NORMAL_MOVE || s == DTe::SENNICHITE)) ) {
      forward_enable = canvas->history->cur_pos < 
	               canvas->history->focus_line.size() - 1;
    }
    else {
      forward_enable = canvas->history->cur_pos < 
                     canvas->history->focus_line.size();
    }
  }
  else {
    back_enable = false;
    forward_enable = false;
  }

  static const char* back_menuitems[] = {
    "first", "back", "button_first", "button_back", NULL};
  static const char* forward_menuitems[] = {
    "next", "last", "button_next", "button_last", NULL};
  
  for (i = 0; back_menuitems[i]; i++) {
    widget = lookup_widget(canvas->window, back_menuitems[i]);
    gtk_widget_set_sensitive(widget, back_enable );
  }

  for (i = 0; forward_menuitems[i]; i++) {
    widget = lookup_widget(canvas->window, forward_menuitems[i]);
    gtk_widget_set_sensitive(widget, forward_enable );
  }
}


/**
 * ゲームモードでのメニュー項目の有効・無効
 */
void daemon_canvas_set_game_sensitive(Canvas* canvas)
{
  GtkWidget* widget;

  // 投了できるか
  bool can_resign = canvas->mode == D_CANVAS_MODE_GAME &&
                    canvas->playertype[canvas->board.next] == SI_GUI_HUMAN;
  
  // 待ったできるか
  bool can_retract = can_resign && canvas->history->focus_line.size() != 0 &&
    canvas->playertype[1 - canvas->board.next] != SI_NETWORK_1;

  bool can_stop = canvas->mode == D_CANVAS_MODE_GAME; // &&
                  // canvas->playertype[0] != SI_NETWORK_1 && 
                  // canvas->playertype[1] != SI_NETWORK_1;
  widget = lookup_widget(canvas->window, "button_stop");
  gtk_widget_set_sensitive(widget, can_stop );
  widget = lookup_widget(canvas->window, "stop");
  gtk_widget_set_sensitive(widget, can_stop );

  widget = lookup_widget(canvas->window, "button_give_up");
  gtk_widget_set_sensitive(widget, can_resign);
  widget = lookup_widget(canvas->window, "give_up");
  gtk_widget_set_sensitive(widget, can_resign);

  // 待った
  widget = lookup_widget(canvas->window, "retract");
  gtk_widget_set_sensitive(widget, can_retract);
}


/** メニュー項目の有効・無効. */
void daemon_canvas_set_sensitive(Canvas *canvas)
{
  GtkWidget *widget;
  int i;

  // ゲーム中、詰め探索中は無効
  static const char* disable_on_game[] = {
    "fileopen", "file_save", "filesave", "button_new_game", "playgame", 
    "check_mate", "solve_hisshi",
    "connect_server", NULL
  };

  for (i = 0; disable_on_game[i]; i++) {
    widget = lookup_widget( canvas->window, disable_on_game[i] );
    gtk_widget_set_sensitive( widget,
                       daemon_canvas_get_mode(canvas) != D_CANVAS_MODE_GAME );
  }

  // 編集開始は棋譜モードからのみ
  widget = lookup_widget(canvas->window, "startedit");
  gtk_widget_set_sensitive( widget,
                            canvas->mode != D_CANVAS_MODE_GAME &&
                            canvas->mode != D_CANVAS_MODE_EDIT );
  widget = lookup_widget(canvas->window, "file_game_properties");
  gtk_widget_set_sensitive( widget,
                            canvas->mode != D_CANVAS_MODE_GAME &&
                            canvas->mode != D_CANVAS_MODE_EDIT );

  // 編集モードでのみ有効
  static const char* enable_only_edit[] = {
    "set_hirate", "set_mate", "all_koma_to_pbox", "rl_reversal", 
    "order_reversal", "edit_flip", NULL
  };

  for (i = 0; enable_only_edit[i]; i++) {
    widget = lookup_widget( canvas->window, enable_only_edit[i] );
    gtk_widget_set_sensitive( widget, canvas->mode == D_CANVAS_MODE_EDIT );
  }

  // 進む/戻る
  daemon_canvas_set_kif_sensitive( canvas );

  // 投了など
  daemon_canvas_set_game_sensitive( canvas );
}


/**
 * 詰め将棋の準備: ユーザの用意した局面が妥当かどうか
 */
bool daemon_game_mate_setup()
{
  Canvas* canvas = g_canvas;

  canvas->board.next = SENTE;

  printBOARD( &canvas->board ); // DEBUG

  boSetTsumeShogiPiece(&canvas->board);
  boPlayInit(&canvas->board);

  printBOARD( &canvas->board ); // DEBUG

  // 再描画 (持ち駒など)
  canvas_redraw(canvas);

  if ( !boCheckJustMate(&canvas->board) || !canvas->board.king_xy[1] ) 
    return 0;

  return 1;
}


/** 詰め将棋のmain loop */
void daemon_game_mate_main_loop(int tesuki, int max_depth, int node_count)
{
  Canvas* canvas;
  GtkWidget* dialog;
  int ret;

  canvas = g_canvas;

  dialog = gtk_message_dialog_new(GTK_WINDOW(canvas->window),
				  GTK_DIALOG_DESTROY_WITH_PARENT,
				  GTK_MESSAGE_INFO,
				  GTK_BUTTONS_CANCEL,
				  _("Now thinking..."));
  g_signal_connect_swapped(dialog, "response", 
			   G_CALLBACK(daemon_canvas_mate_think_cancel), NULL);
  gtk_widget_show(dialog);

  gmSetGameStatus(&g_game, SI_NORMAL);

  /* ハッシュテーブル初期化 */
  newHASH();

  TE te_ret;
  int node_limit = node_count;

  si_start_alarm();
  ret = bfs_hisshi( &canvas->board, tesuki, max_depth, &te_ret, 
		    &node_limit ); 
  si_end_alarm();

  gtk_widget_destroy(dialog);

  if ( ret == CHECK_MATE ) {
    DTe te;

    daemon_messagebox(canvas, _("Check mate."), GTK_MESSAGE_INFO);

    if (canvas->record != NULL)
      daemon_record_free(canvas->record);
    canvas->record = daemon_record_new();
    canvas->history->clear();

    daemon_dboard_copy(&(canvas->board), &(canvas->record->first_board));

    // 詰め手順を再現する
    BOARD* bo_tmp = bo_dup(&canvas->board);
    int32_t phi, delta;

    while ( hsGetBOARD( tesuki * 2 + canvas->board.next, 
			bo_tmp, &phi, &delta, &te_ret, 0) ) {
      if (!te_to(te_ret) ) {
        te.special = DTe::RESIGN;
        te.fm = 0; te.to = 0; te.nari = 0; te.uti = 0;
        te.tori = 0;
      }
      else {
        te.special = DTe::NORMAL_MOVE;
        te.fm = te_from(te_ret); 
	te.to = te_to(te_ret); 
	te.nari = te_promote(te_ret);
        te.uti = te_pie(te_ret);
        te.tori = bo_tmp->board[te.to];
      }

      daemon_dmoveinfo_add(canvas->record->mi, &te, 0);
      canvas->history->append_move(*bo_tmp, te, 0);

      if (!te_to(te_ret) )
        break;

      boMove_mate(bo_tmp, te_ret);
    }
    freeBOARD(bo_tmp);

    daemon_canvas_change_mode(canvas, D_CANVAS_MODE_BOOK);
  } 
  else {
    daemon_messagebox(canvas, _("No check mate."), GTK_MESSAGE_INFO);
  }
  
  // ハッシュテーブル解放
  freeHASH();
}


/** 詰め将棋でのキャンセルボタン */
void daemon_canvas_mate_think_cancel()
{
  gmSetGameStatus(&g_game, SI_CHUDAN_LOCAL);
}


PLAYERTYPE get_player_type(int combo_index)
{
  PLAYERTYPE playertype;

  switch (combo_index) {
  case 0:
    playertype = SI_GUI_HUMAN;
    break;
  case 1:
    playertype = SI_COMPUTER_2;
    break;
  case 2:
    playertype = SI_COMPUTER_3;
    break;
  case 3:
    playertype = SI_COMPUTER_4;
    break;
  case 4:
    playertype = SI_CHILD_PROCESS;
    break;
  default:
    si_abort("internal error: unknown player type = '%d'", combo_index);
  }

  return playertype;
}


/** 
 * メニュー -> ゲーム -> OK
 * @return エラー発生のとき 0
 */
int daemon_game_new_game_ok(GtkWidget* dialog_game) 
{
  PLAYERTYPE playertype[2];
  D_TEAI teai;
  const gchar* playername[2];
  const gchar* tmp;
  int i;

  GtkWidget *menu_sente;
  GtkWidget *menu_gote;
  GtkWidget *menu_teai;
  GtkWidget* entry_playername;

  menu_sente = lookup_widget(dialog_game, "optionmenu_sente");
  menu_gote  = lookup_widget(dialog_game, "optionmenu_gote");
  menu_teai  = lookup_widget(dialog_game, "optionmenu_teai");

  playertype[0] = get_player_type(
                     gtk_combo_box_get_active(GTK_COMBO_BOX(menu_sente)));
  playertype[1] = get_player_type(
                     gtk_combo_box_get_active(GTK_COMBO_BOX(menu_gote)));

  int t = gtk_combo_box_get_active(GTK_COMBO_BOX(menu_teai));
  switch (t) {
  case 0:
    teai = D_HIRATE;
    bo_reset_to_hirate( &g_canvas->board );
    break;
  case 1:
    teai = HANDICAP_ETC;
    boSetToBOARD( &g_canvas->board );
    g_canvas->board.mseq.count = 0;
    if (boCheckJustGame(&g_canvas->board) == 0) {
      daemon_messagebox(g_canvas, _("No good board."), GTK_MESSAGE_ERROR);
      return 0;
    }
    break;
  default:
    si_abort("unknown teai = '%d'", t);
  }

  for (i = 0; i < 2; i++) {
    entry_playername = lookup_widget(dialog_game, 
				     i == 0 ? "entry_sente" : "entry_gote");
    tmp = gtk_entry_get_text(GTK_ENTRY(entry_playername));
    if (tmp[0] == '\0') {
      if (playertype[i] == SI_GUI_HUMAN) {
	uid_t uid;
	struct passwd *curr_passwd;
	uid = getuid();
	curr_passwd = getpwuid(uid);
	playername[i] = curr_passwd->pw_name;
      }
      else 
	playername[i] = "COM";
    } 
    else 
      playername[i] = tmp;
  }

#if 0   // TODO: impl.
  static const char* exefile = "/opt/src/shogi/bonanza-4.0.4/src/bonanza/bonanza"; 
  static const char* dir = "/opt/src/shogi/bonanza-4.0.4/winbin";

  // bonanzaの起動
  for (i = 0; i < 2; i++) {
    g_canvas->child_process[i].clear();
    if (playertype[i] == SI_CHILD_PROCESS) {
      if (!spawn_async_with_pipe(&g_canvas->child_process[i])) {
	daemon_messagebox(g_canvas, _("failed: run bonanza"), 
			  GTK_MESSAGE_ERROR);
	if (i == 1)
	  g_canvas->child_process[0].kill();
	return 0;
      }
    }
  }
#endif 

  daemon_game_setup(playertype[0], playername[0],
		    playertype[1], playername[1], 1500 );

  return 1;
}


/** recordのクリアなどの準備 */
void daemon_game_setup(PLAYERTYPE playertype1, const char* playername1,
		       PLAYERTYPE playertype2, const char* playername2,
		       int total_time )
{
  if (g_canvas->record)
    daemon_record_free(g_canvas->record);
  g_canvas->record = daemon_record_new();
  g_canvas->update_window_title();

  daemon_record_set_player(g_canvas->record, SENTE, playername1);
  daemon_record_set_player(g_canvas->record, GOTE, playername2);

  daemon_dboard_copy(&g_canvas->board, &g_canvas->record->first_board);
  assert( g_canvas->record->first_board.mseq.count == 0 );
  g_canvas->history->clear();

  g_canvas->playertype[0] = playertype1;
  g_canvas->playertype[1] = playertype2;

  for (int i = 0; i < 2; i++) {
    g_game.total_time[i] = total_time;
    g_game.total_elapsed_time[i] = 0;
  }

  alt_moves_dialog_hide();

  daemon_canvas_change_mode(g_canvas, D_CANVAS_MODE_GAME);
}



/*************************************************************/
/* draw */
/*************************************************************/


/** キャンバス全体を背景画像でクリア */
void daemon_canvas_draw_back(Canvas* canvas) 
{
  GdkRectangle rect;
  
  rect.x = 0;
  rect.y = 0;
  rect.width  = canvas->width;
  rect.height = canvas->height;

  daemon_canvas_draw_back_update(canvas, &rect);
}


/** 指定した領域を背景画像でクリアする。 */
static void daemon_canvas_draw_back_update( Canvas* canvas, const GdkRectangle* rect ) 
{
  cairo_surface_t* pixmap = daemon_canvas_get_pixmap(canvas);
  cairo_surface_t* back   = daemon_canvas_get_back(canvas);
  
  cairo_t* cr = cairo_create( pixmap ); // dest
  // set_color(gc, 0, 0, 0);
  // gdk_gc_set_clip_rectangle(gc, rect);
  cairo_set_source_surface( cr, back, 0, 0 );
  gdk_cairo_rectangle( cr, rect );
  cairo_fill( cr );
  cairo_destroy( cr );
}


static void daemon_canvas_draw_back_update_from_tmp_pixmap(Canvas* canvas, 
						    const GdkRectangle* rect) 
{
  cairo_surface_t* pixmap     = daemon_canvas_get_pixmap( canvas );
  cairo_surface_t* tmp_pixmap = canvas->tmp_pixmap;
  // cairo_surface_t* back       = daemon_canvas_get_back(canvas);
  
  // gdk_gc_set_clip_rectangle(gc, rect);
  cairo_t* cr = cairo_create( pixmap ); // dest
  cairo_set_source_surface( cr, tmp_pixmap, 0, 0 );
  gdk_cairo_rectangle( cr, rect );
  cairo_fill( cr );
  cairo_destroy( cr );
}


/** 消費時間を描く */
void daemon_canvas_draw_time(Canvas* canvas) 
{
  // GdkFont *font;
  GdkRectangle rect;
  gint x, y, width, height, step;
  gchar buf[BUFSIZ];
  PangoLayout* layout;
  int teban;

  g_assert(canvas != NULL); 
  
  cairo_surface_t* pixmap      = daemon_canvas_get_pixmap(canvas);
  cairo_surface_t* back        = daemon_canvas_get_back(canvas);
/*
  rect.x      = 0;
  rect.y      = 0;
  rect.width  = canvas->width;
  rect.height = canvas->height;
*/
  x = y = width = height = step = 0;

  for (teban = 0; teban < 2; teban++) {
    if ((teban == 0 && daemon_canvas_get_front(canvas) == SENTE) ||
	(teban == 1 && daemon_canvas_get_front(canvas) == GOTE)) {
      /* 画面右手 */
      x      = daemon_canvas_rate_xy(canvas, 516);
      y      = daemon_canvas_rate_xy(canvas, 158);
      width  = daemon_canvas_rate_xy(canvas, 129);
      height = daemon_canvas_rate_xy(canvas,  38);
      step   = daemon_canvas_rate_xy(canvas,  15);
    }
    else {
      /* 画面左手 */
      x      = daemon_canvas_rate_xy(canvas,   6);
      y      = daemon_canvas_rate_xy(canvas, 158);
      width  = daemon_canvas_rate_xy(canvas, 129);
      height = daemon_canvas_rate_xy(canvas,  38);
      step   = daemon_canvas_rate_xy(canvas,  15);
    }

    // font = gtk_style_get_font(canvas->window->style);
    cairo_t* cr = cairo_create( pixmap ); // dest
    //  set_color_rgb( cr, 0xFFFF, 0xFFFF, 0XFFFF );

    // gdk_gc_set_clip_rectangle(gc, &rect);
    // gdk_gc_set_clip_origin(gc, 0, 0);

    cairo_set_source_surface( cr, back, 0, 0);
    cairo_rectangle( cr, x, y, width, height );
    cairo_fill( cr );

    daemon_canvas_draw_time_up(canvas, teban);

    if (canvas->record != NULL) {
      sprintf(buf, "%-20s", canvas->record->player_name[teban].c_str() );

      layout = gtk_widget_create_pango_layout(canvas->window, buf);

      cairo_move_to( cr, x, y + step * 2 + 3 );
      pango_cairo_show_layout( cr, layout );

      g_object_unref(layout);
    }
    cairo_destroy( cr );
  }

  /* 手番、前の手を表示 */
  x      = daemon_canvas_rate_xy(canvas, 516);
  y      = daemon_canvas_rate_xy(canvas, 131 - 14);
  width  = daemon_canvas_rate_xy(canvas, 129);
  height = daemon_canvas_rate_xy(canvas,  38);
  step   = daemon_canvas_rate_xy(canvas,  16);

  // font = gtk_style_get_font(canvas->window->style);
  // set_color(gc, 0xFFFF, 0xFFFF, 0XFFFF);
  cairo_t* cr = cairo_create( pixmap ); // dest
  
  // gdk_gc_set_clip_rectangle(gc, &rect);
  // gdk_gc_set_clip_origin(gc, 0, 0);
  cairo_set_source_surface(cr, back, 0, 0);
  cairo_rectangle( cr, x, y, width, height );
  cairo_fill(cr);

  // 手番を表示
  gint count;
  sprintf(buf, "%s", canvas->board.next == SENTE ? 
          _("next SENTE (BLACK)") : _("next GOTE (WHITE)"));
  y += step;
  // gdk_draw_string(pixmap, font, gc, x, y, buf);
  layout = gtk_widget_create_pango_layout(canvas->window, buf);

  cairo_set_source_rgb(cr, 1.0, 1.0, 1.0);
  cairo_move_to(cr, x, y);
  pango_cairo_show_layout(cr, layout);

  g_object_unref(layout);

  if (daemon_canvas_get_mode(canvas) == D_CANVAS_MODE_GAME) {
    y += step;
    count = canvas->history->focus_line.size();
    if (count > 0) {
      // 最後の手を表示
      DTe te = canvas->history->focus_line[count - 1].first;
      create_te_string2(&(canvas->board), count, &te, buf);
      layout = gtk_widget_create_pango_layout(canvas->window, buf);

      cairo_move_to(cr, x, y);
      pango_cairo_show_layout(cr, layout);

      g_object_unref(layout);
    }
  }
  cairo_destroy(cr);

  daemon_canvas_invalidate_rect(canvas, &rect);
}


/** ステータスバーを更新 */
void daemon_canvas_update_statusbar(Canvas* canvas) 
{
  GtkWidget* mode_label = lookup_widget( canvas->window, "mode_label" );
  assert( mode_label );
  GtkWidget* flip_label = lookup_widget( canvas->window, "flip_label" );
  assert( flip_label );

  switch ( daemon_canvas_get_mode(canvas) ) {
  case D_CANVAS_MODE_BOOK:
    gtk_label_set_text( GTK_LABEL(mode_label), _("PLAYBACK MODE") );
    break;
  case D_CANVAS_MODE_GAME:
    gtk_label_set_text( GTK_LABEL(mode_label), _("GAME MODE") );
    break;
  case D_CANVAS_MODE_EDIT:
    gtk_label_set_text( GTK_LABEL(mode_label), _("EDIT MODE") );
    break;
  case D_CANVAS_MODE_PRE_GAME:
    break;
  default:
    abort();
  }

  if ( canvas->front == SENTE )
    gtk_label_set_text( GTK_LABEL(flip_label), "" );
  else
    gtk_label_set_text( GTK_LABEL(flip_label), _("Flip") );
}


/*************************************************************/
/* etc */
/*************************************************************/

/**
 * canvas->back の内容を初期化する。モードによって, 駒台を塗りつぶす
 */
void daemon_canvas_create_back_pixmap(Canvas* canvas) 
{
  GdkRectangle rect;

  /* 駒台の座標 */
  static const gint x1 = 1, y1 = 210, x2 = 129, y2 = 378;

  g_assert(canvas != NULL);

  if ( canvas->back[0] )
    cairo_surface_destroy( canvas->back[0] );
  if ( canvas->back[1] )
    cairo_surface_destroy( canvas->back[1] );
  if ( canvas->back[2] )
    cairo_surface_destroy( canvas->back[2] );

  canvas->back[0] = load_pixmap_from_imlib_image_scaled( CAIRO_FORMAT_ARGB32,
			                        canvas->back_src,
						D_CANVAS_SIZE_SMALL_X,
						D_CANVAS_SIZE_SMALL_Y );
  assert( canvas->back[0] );
  canvas->back[1] = load_pixmap_from_imlib_image_scaled( CAIRO_FORMAT_ARGB32,
						canvas->back_src,
						D_CANVAS_SIZE_MIDDLE_X,
						D_CANVAS_SIZE_MIDDLE_Y );
  assert( canvas->back[1] );
  canvas->back[2] = load_pixmap_from_imlib_image_scaled( CAIRO_FORMAT_ARGB32,
						canvas->back_src,
						D_CANVAS_SIZE_BIG_X,
						D_CANVAS_SIZE_BIG_Y );
  assert( canvas->back[2] );

  if (canvas->mode == D_CANVAS_MODE_EDIT)
    return;

  // 使用しない駒エリアを黒で塗りつぶす
/*
  rect.x      = 0;
  rect.y      = 0;
  rect.width  = canvas->width;
  rect.height = canvas->height;
*/
  // gdk_gc_set_clip_rectangle(canvas->gc, &rect);
  // gdk_gc_set_clip_origin(canvas->gc, 0, 0);

  // set_color(canvas->gc, 0, 0, 0);

  cairo_t* cr = cairo_create( canvas->back[0] );
  cairo_set_source_rgb( cr, 0, 0, 0 );
  cairo_rectangle( cr, x1, y1, x2, y2 );
  cairo_fill( cr );
  cairo_destroy( cr );

  cr = cairo_create( canvas->back[1] );
  cairo_set_source_rgb( cr, 0, 0, 0 );
  cairo_rectangle( cr, 
            x1 * D_CANVAS_SIZE_MIDDLE_X / D_CANVAS_SIZE_SMALL_X,
		     y1 * D_CANVAS_SIZE_MIDDLE_X / D_CANVAS_SIZE_SMALL_X,
		     x2 * D_CANVAS_SIZE_MIDDLE_X / D_CANVAS_SIZE_SMALL_X + 1,
		     y2 * D_CANVAS_SIZE_MIDDLE_X / D_CANVAS_SIZE_SMALL_X );
  cairo_fill( cr );
  cairo_destroy( cr );

  cr = cairo_create( canvas->back[2] );
  cairo_set_source_rgb( cr, 0, 0, 0 );
  cairo_rectangle( cr, 
            x1 * D_CANVAS_SIZE_BIG_X / D_CANVAS_SIZE_SMALL_X,
		     y1 * D_CANVAS_SIZE_BIG_X / D_CANVAS_SIZE_SMALL_X,
		     x2 * D_CANVAS_SIZE_BIG_X / D_CANVAS_SIZE_SMALL_X + 1,
		     y2 * D_CANVAS_SIZE_BIG_X / D_CANVAS_SIZE_SMALL_X );
  cairo_fill( cr );
  cairo_destroy( cr );
}


/**
 * ダブルクリックチェック。
 * @param canvas 対象のDragTestCanvas
 * @return ダブルクリックならば TRUEを返す
 */
gboolean daemon_canvas_is_doubleclick(Canvas* canvas) 
{
  struct timeval  tv;
  // struct timezone tz;

  gettimeofday(&tv, NULL );

  if (check_timeval(&(canvas->tv), &tv) == TRUE) {
    canvas->tv.tv_sec  = 0;
    canvas->tv.tv_usec = 0;
    return TRUE;
  }

  // gettimeofday(&(canvas->tv), &tz);
  canvas->tv.tv_sec  = tv.tv_sec;
  canvas->tv.tv_usec = tv.tv_usec;

  return FALSE;
}


/** ドラッグ中に使う作業用の pixmap を作成する */
void daemon_canvas_create_tmp_pixmap(Canvas* canvas) 
{
  Sprite *sprite;
  GdkRectangle rect;
  gint no;

  g_assert(canvas != NULL);
  g_assert(daemon_canvas_isdrag(canvas) == 1);

  cairo_surface_t* pixmap = daemon_canvas_get_pixmap(canvas);

  canvas->tmp_pixmap = cairo_surface_create_similar( pixmap,
				     CAIRO_CONTENT_COLOR_ALPHA,
				     canvas->width,
				     canvas->height );
  if ( !canvas->tmp_pixmap ) {
    si_abort("In daemon_canvas_create_tmp_pixmap(). tmp_pixmap is NULL.\n");
  }
  
  no = daemon_canvas_get_dragno(canvas);
  sprite = daemon_canvas_get_sprite(canvas, no);
  rect.x      = daemon_sprite_get_x(sprite);
  rect.y      = daemon_sprite_get_y(sprite);
  rect.width  = daemon_sprite_get_width(sprite);
  rect.height = daemon_sprite_get_height(sprite);

  daemon_sprite_set_visible(sprite, FALSE);
  /* 背景を描く */
  daemon_canvas_draw_back_update(canvas, &rect);
  /* 駒を描く */
  daemon_sprite_draw_overlap( canvas->sprite, &rect, 
			      daemon_canvas_get_pixmap(canvas) );
  daemon_sprite_set_visible(sprite, TRUE);

  rect.x = 0;
  rect.y = 0;
  rect.width  = canvas->width;
  rect.height = canvas->height;

  // gdk_gc_set_clip_rectangle(gc, &rect);
  // gdk_draw_drawable(canvas->tmp_pixmap, gc, pixmap, 0, 0, 0, 0,
  //	  canvas->width, canvas->height);
  cairo_t* cr = cairo_create( canvas->tmp_pixmap );
  cairo_set_source_surface( cr, pixmap, 0, 0 );
  gdk_cairo_rectangle(cr, &rect);
  cairo_fill(cr);
  cairo_destroy(cr);

  daemon_sprite_draw( daemon_canvas_get_sprite(canvas, no), 
		      daemon_canvas_get_pixmap(canvas) );
}


/** 拡大表示にあわせて、実際の座標を返す */
int daemon_canvas_rate_xy( const Canvas* canvas, int value ) 
{
  return value * get_scale_factor(canvas);
}


/**
 * プログラムの一手を te に格納する。
 * @param bo_ 対象のBOARD
 * @param best_move 指し手 (出力)
 *
 * @return GAMESTAT
 */
INPUTSTATUS si_input_next_computer( const BOARD* bo_, 
				    DTe* best_move, int* /*dmy */ )
{
  INPUTSTATUS ret;
  TE te_ret;
#ifndef NDEBUG
  int side = bo_->next;
  uint64_t hash = bo_->hash_all;
#endif

  BOARD bo = BOARD(*bo_);

  if ( jouseki(&bo, &te_ret) ) {
    // 定跡があった
    *best_move = DTe( DTe::NORMAL_MOVE, te_ret, bo.board[te_to(te_ret)] );
    return SI_NORMAL;
  }

  si_start_alarm();
  // 通常探索 (内部で詰将棋も読む)
  PhiDelta score;
  ret = bfs_minmax(&bo, 25, &score, &te_ret);
  si_end_alarm();

  if ( ret == SI_NORMAL ) {
    if ( te_to(te_ret) )
      *best_move = DTe( DTe::NORMAL_MOVE, te_ret, bo.board[te_to(te_ret)] );
    else {
/*
      hsGetBOARDMinMax(&bo, &score, &tt_move, 0);
*/
      printf("score = (%d,%d)\n", score.phi, score.delta ); // DEBUG
      if (score.phi >= +VAL_INF || score.delta <= -VAL_INF) {
	// 本当は勝ってる
	TE tt_move; 
	if ( hsGetBOARD(bo.next, &bo, &score.phi, &score.delta, &tt_move, 0)
	     && (score.phi == 0 && score.delta >= +VAL_MAY_MATCH) ) {
	  printf("tsumi = %s\n", te_str(tt_move, bo).c_str() );
	  *best_move = DTe( DTe::NORMAL_MOVE, 
			    tt_move, bo.board[te_to(tt_move)] );
	  return ret;
	}
	
	if ( hsGetBOARD(bo.next + 2, &bo, &score.phi, &score.delta, &tt_move, 0)
	     && (score.phi == 0 && score.delta >= +VAL_MAY_MATCH) ) {
	  printf("hisshi = %s\n", te_str(tt_move, bo).c_str() );
	  *best_move = DTe( DTe::NORMAL_MOVE, 
			    tt_move, bo.board[te_to(tt_move)] );
	  return ret;
	}

	printf("Why win???\n"); // DEBUG
      }

      // DEBUGここから
      printf("%s: TORYO!!\n", __func__);
#if 0
      MOVEINFO mi;
      make_moves(&bo, &mi);
      for (int i = 0; i < mi.count; i++) {
	TE tt_move;
	boMove( &bo, mi.te[i] );
	bool r = hsGetBOARDMinMax(&bo, &score, &tt_move, 0);
	boBack( &bo );
	if (r) {
	  printf("%s->(%d,%d) ", te_str(mi.te[i], bo).c_str(), 
	                                    	 score.phi, score.delta);
	}
    else
	  printf("%s->unk ", te_str(mi.te[i], bo).c_str() );
      }
      printf("\n");
#endif // DEBUGここまで

      *best_move = DTe( DTe::RESIGN, 0, 0 );
    }
  }

  assert(side == bo.next);
  assert(hash == bo.hash_all);
  return ret;
}


/**
 * 人間のGUIからの一手を te に格納する。
 * @param bo 対象のBOARD
 * @param te TE
 * @return GAMESTAT
 */
INPUTSTATUS daemon_input_next_gui_human_impl( const BOARD* bo, 
					      DTe* te, int* /*dmy*/ ) 
{
  DTe tmp_te;
  DTe* gui_te = &g_canvas->gui_te;

  gui_te->clear();
  
  while ( tmp_te == *gui_te ) {
    if ( gmGetGameStatus(&g_game) != 0 ) {
      // GUI操作での中断, あるいはサーバからの時間切れが入った。
      return gmGetGameStatus(&g_game);
    }
    processing_pending( true );
  }
  
  *te = *gui_te;

  return SI_NORMAL;
}


/** 
 * 子プロセスからの入力を受け取る.
 * \todo impl.
 */
INPUTSTATUS daemon_input_next_child_process_impl( const BOARD* bo, 
						  DTe* te, int* /*dmy*/ )
{
  assert(0);

  return SI_NORMAL;
}


/**
 * 対局のメインルーチン.
 * この関数を呼び出す前に daemon_game_setup() で準備すること。
 */
void daemon_game_main_loop()
{
  g_assert(daemon_canvas_get_mode(g_canvas) == D_CANVAS_MODE_GAME);

  BOARD *board;
  GAME *game;
  INPUTSTATUS minmax_return;
  DTe dte;
  int elapsed_time = -1;

  Canvas* canvas = g_canvas;
  board = &canvas->board;
  game = &g_game;

  int i;
  for (i = 0; i < 2; i++) {
    switch (canvas->playertype[i]) {
    case SI_GUI_HUMAN:
      gmSetInputFunc(game, i, daemon_input_next_gui_human_impl);
      break;
    case SI_COMPUTER_2:
    case SI_COMPUTER_3:
    case SI_COMPUTER_4:
      // TODO: レベル分け
      gmSetInputFunc(game, i, si_input_next_computer );
      break;
    case SI_NETWORK_1: // CSA server
      gmSetInputFunc(game, i, daemon_input_next_network_impl);
      break;
    case SI_CHILD_PROCESS: // bonanza
      gmSetInputFunc(game, i, daemon_input_next_child_process_impl);
      break;
    default:
      abort();
    }
  }

  boPlayInit(board);
  gmSetGameStatus(game, SI_NORMAL);

  newHASH();

  // 消費時間の表示を更新
  canvas->timeout_id = g_timeout_add_seconds( 1, on_timeout, NULL );

  // main loop
  while ( true ) {
    canvas->elapsed = 0;
    gm_set_think_start_time(&g_game);
    elapsed_time = -1;

    daemon_canvas_set_game_sensitive(canvas);
    processing_pending( false );

#ifndef NDEBUG
    int book_count;
    BookEnt* book_ent = book_get( board->hash_all, &book_count );

    printf("book: ");
    if (book_ent) {
      for ( int i = 0; i < book_count; i++ )
	printf( "%s ", te_str(book_ent[i].book_move, *board).c_str() );
      printf("\n");
      free( book_ent );
    }
    else
      printf("not found.\n");
#endif // !NDEBUG

    // 手の入力
    int elapsed_time__ = 0;
    minmax_return = (*game->si_input_next[board->next])( board, 
						 &dte, &elapsed_time__ );
    printf("side %d minmax_return = %d\n", board->next, minmax_return); // DEBUG
    if (canvas->playertype[board->next] == SI_NETWORK_1) 
      elapsed_time = elapsed_time__;
    else {
      // 手をサーバに送信 (GUI, コンピュータ)
      if (canvas->playertype[board->next == 0] == SI_NETWORK_1) {
	// すでに時間切れ?
	if ( net_queue_is_pending() )
	  minmax_return = csa_wait_for_result(board, &dte, &elapsed_time);
	if ( gmGetGameStatus(&g_game) == SI_TIMEOUT ) {
	  printf("time up!!\n"); // DEBUG
	  minmax_return = SI_TIMEOUT;
	}

	switch (minmax_return) {
        case SI_NORMAL:
	  // 通常の指し手, 投了
	  conn_send_move(net_get_server_fd(), &canvas->board, dte);
	  // 時間切れのことがある. 千日手のことがある
	  minmax_return = csa_wait_for_result(board, &dte, &elapsed_time);
	  break;
	case SI_TIMEOUT:
	case SI_CHUDAN_REMOTE:
	  // 送信不要
	  break;
	case SI_CHUDAN_LOCAL:
	  csa_send_chudan_request();
	  minmax_return = csa_wait_for_result(board, &dte, &elapsed_time);
	  break;
	default:
	  abort(); // ない
	}
      }
      else {
	// サーバを介さない場合
	elapsed_time = time(NULL) - g_game.think_start_time;
	
	if (minmax_return == SI_NORMAL && dte.special == DTe::NORMAL_MOVE) {
	  bool forbidden = false;
	  if ( te_is_sennichite(&canvas->board, dte.pack(), &forbidden) )
	    dte.special = forbidden ? DTe::ILLEGAL : DTe::SENNICHITE;
	}
      }
    }

    // 指し手を記録する
    if ( minmax_return != SI_CHUDAN_LOCAL && 
         minmax_return != SI_CHUDAN_REMOTE ) {
      if (minmax_return == SI_TIMEOUT)
	dte.special = DTe::TIME_UP;
      daemon_dmoveinfo_add(canvas->record->mi, &dte, elapsed_time);
      canvas->history->append_move(canvas->board, dte, elapsed_time);
      gm_add_elapsed_time( game, board->next, elapsed_time );
    }

    if ( minmax_return == SI_NORMAL && 
	 (dte.special == DTe::NORMAL_MOVE || dte.special == DTe::SENNICHITE) ) {
      boMove( board, dte.pack() );
      canvas->history->select_nth(board->mseq.count);

      // 千日手チェック用
      bo_inc_repetition(board);
    }

    if ( !(minmax_return == SI_NORMAL && dte.special == DTe::NORMAL_MOVE) )
      break;

    // 再描画
    canvas_redraw(canvas);
  }

  // サーバと切断
  if ( canvas->playertype[0] == SI_NETWORK_1 || 
       canvas->playertype[1] == SI_NETWORK_1 ) {
    csa_send_logout();
    disconnect_server();
  }

  g_source_remove(canvas->timeout_id);
  canvas->timeout_id = 0;

  // ゲーム結果を表示
  int winner = -1;
  switch (minmax_return) {
  case SI_NORMAL:
    switch (dte.special) 
    {
    case DTe::RESIGN:
    case DTe::ILLEGAL:
      winner = 1 - board->next;
      break;
    case DTe::KACHI:
      winner = board->next;
      break;
    case DTe::JISHOGI:
    case DTe::SENNICHITE:
      winner = 2; // draw
      break;
    default:
      abort(); 
    }
    break;
  case SI_TIMEOUT:
    winner = 1 - board->next;
    break;
  case SI_CHUDAN_LOCAL:
  case SI_CHUDAN_REMOTE:
    winner = -1;
    break;
  default:
    abort(); // ない
  }

  if (winner < 0) 
    daemon_messagebox(canvas, _("Stoped."), GTK_MESSAGE_INFO);
  else if (winner == 0)
    daemon_messagebox(canvas, _("Sente(black) win."), GTK_MESSAGE_INFO);
  else if (winner == 1)
    daemon_messagebox(canvas, _("Gote(white) win."), GTK_MESSAGE_INFO);
  else
    daemon_messagebox( canvas, _("Game was drawn."), GTK_MESSAGE_INFO );

  /* ハッシュテーブル解放 */
  freeHASH();

  // 局面は投了時のまま
  daemon_canvas_change_mode(canvas, D_CANVAS_MODE_BOOK);
}


/** 駒箱内の駒の種類の数を数える */
gint count_pbox(DBoard *board) {
  gint count;
  gint i;

  count = 0;
  for (i=1; i <= 8; i++) {
    if (0 < daemon_dboard_get_pbox(board, i)) {
      count++;
    }
  }

  return count;
}


/**
 * 棋譜ウィンドウ/ステータスバーに表示する手の文字列を生成する。
 * @param board 局面
 * @param no    プレイヤー
 * @param te    指し手. NULLでもよい
 * @param buf   出力先バッファ
 */
void create_te_string(const DBoard* board, int no, const DTe* te, char* buf) 
{
  const char* x[] = {
    "",     _("1_"), _("2_"), _("3_"), _("4_"),
    _("5_"),_("6_"), _("7_"), _("8_"), _("9_"),
  };
  const char* y[] = {
    "",     _("1 "), _("2 "), _("3 "), _("4 "),
    _("5 "),_("6 "), _("7 "), _("8 "), _("9 "),
  };
  const char* piece[] = {
    "",           _("FU"), _("KYO"),     _("KEI"),
    _("GIN"),     _("KIN"),_("KAKU"),    _("HISYA"),
    _("OH"),      _("TO"), _("NARIKYO"), _("NARIKEI"),
    _("NARIGIN"), "",      _("UMA"),     _("RYU"),
  };
  const char* nari = _(" NARI");
  const char* uti = _(" UTI");
  int p;

  if (!te) {
    sprintf(buf, "%s", _("game start") );
    return;
  }

  if (te->special != DTe::NORMAL_MOVE) {
    switch (te->special) {
    case DTe::RESIGN:
      sprintf(buf, "%s", _("Resign"));
      return;
    case DTe::KACHI:
      sprintf(buf, "%s", _("King in enemies side, WIN"));
      return;
    case DTe::ILLEGAL:
      sprintf(buf, "%s", _("Illegal move"));
      return;
    case DTe::TIME_UP:
      sprintf(buf, "%s", _("Time up"));
      return;
    case DTe::JISHOGI:
      sprintf(buf, "%s", _("Impasse"));
      return;
    case DTe::SENNICHITE:
      // 特別
      break;
    default:
      si_abort( "internal error: special = %d\n", te->special);
    }
  }

  // 通常の指し手
  if ( te->fm ) {
    p = daemon_dboard_get_board(board, te->fm & 0xF, te->fm >> 4);
    p &= 0xF;
    sprintf(buf, "%s%s%s%s%s(%d%d)",
	    (no % 2) ? "▲" : "▽",
	    x[te->to & 0xF],
	    y[te->to >> 4],
	    piece[p],
	    te->nari ? nari : "",
	    te->fm & 0xF,
	    te->fm >> 4
	    );
  } 
  else {
    sprintf(buf, "%s%s%s%s%s",
	    (no % 2) ? "▲" : "▽",
	    x[te->to & 0xF],
	    y[te->to >> 4],
	    piece[te->uti],
	    uti);
  }

  // 千日手
  if ( te->special == DTe::SENNICHITE ) 
    sprintf( buf + strlen(buf), ",%s", _("Repetition"));
}


void create_te_string2(const DBoard* board, int no, const DTe* te, char* buf) 
{
  const char* x[10] = {
    "",     _("1_"), _("2_"), _("3_"), _("4_"),
    _("5_"),_("6_"), _("7_"), _("8_"), _("9_"),
  };
  const char* y[10] = {
    "",     _("1 "), _("2 "), _("3 "), _("4 "),
    _("5 "),_("6 "), _("7 "), _("8 "), _("9 "),
  };
  const char* piece[16] = {
    "",           _("FU"), _("KYO"),     _("KEI"),
    _("GIN"),     _("KIN"),_("KAKU"),    _("HISYA"),
    _("OH"),      _("TO"), _("NARIKYO"), _("NARIKEI"),
    _("NARIGIN"), "",      _("UMA"),     _("RYU"),
  };
  const char* nari = _(" NARI");
  const char* uti = _(" UTI");
  const char* game_start = _("game start");
  const char* teme = _("teme");
  int p;
  // DBoard *tmp_board;

  if (no == 0) {
    sprintf(buf, "%d %s %s",
	    no,
	    teme,
	    game_start);
  } 
  else if ( te->fm ) {
    BOARD* tmp_board = bo_dup(board);
    // daemon_dboard_copy(board, tmp_board);
    // daemon_dboard_back(tmp_board, te);
    boBack_mate(tmp_board);

    p = daemon_dboard_get_board(tmp_board, te->fm & 0xF, te->fm >> 4);
    p &= 0xF;
    sprintf(buf, "%d %s %s%s%s%s(%d,%d)",
	    no,
	    teme,
	    x[te->to & 0xF],
	    y[te->to >> 4],
	    piece[p],
	    te->nari ? nari : "",
	    te->fm & 0xF,
	    te->fm >> 4
	    );
    // daemon_dboard_free(tmp_board);
    freeBOARD(tmp_board);
  } 
  else {
    sprintf(buf, "%d %s %s%s%s%s",
	    no,
	    teme, 
	    x[te->to & 0xF],
	    y[te->to >> 4],
	    piece[te->uti],
	    uti
	    );
  }
}

