/*
 * Copyright(C) 2010 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.lunchorder;

import java.util.ArrayList;

import org.apache.cayenne.access.DataContext;

import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;

import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.common.ALPermissionException;

import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.DatabaseOrmService;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 注文表のフォームデータを管理するクラスです。 <BR>
 * 
 */
public class LunchOrderMapReturnInputFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
      .getLogger(LunchOrderMapReturnInputFormData.class.getName());

  /** 子テーブル（商品入力フォーム）一覧 */
  private ArrayList<Object> itemFieldList;

  /** 弁当屋名 */
  private ALStringField storeName;

  /** コメント */
  private ALStringField comment;

  /** 締め切り時刻（時） */
  private ALNumberField deadLineTime_hour;

  /** 締め切り時刻（分） */
  private ALNumberField deadLineTime_minute;

  /** 最低条件有無フラグ */
  private ALStringField minStatus;

  /** 最低条件（金額） */
  private ALNumberField minPrice;

  /** 最低条件（数） */
  private ALNumberField minNum;

  /** 入力最低条件値 */
  private ALNumberField minProperty;

  /** 登録商品価格 */
  private ALNumberField[] item_id = new ALNumberField[100];

  /** 登録商品名 */
  private ALStringField[] item_name = new ALStringField[100];

  /** 登録商品価格 */
  private ALNumberField[] item_price = new ALNumberField[100];

  /** 登録商品コメント */
  private ALStringField[] item_comment = new ALStringField[100];

  /** 商品入力フィールドの数 */
  private int itemFieldlength = 0;

  /** 更新商品入力フィールドの数 */
  private int itemUpdataFieldSize = 0;

  /** プレビュー画面からの遷移かどうかを判断するためのフラグを入れておく変数 */
  private String returnMode;

  /** 現在のモード */
  private String mode;


  DataContext dataContext = DatabaseOrmService.getInstance().getDataContext();

  /**
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

  }

  /**
   * 親テーブル用フィールドを初期化します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {
    // 弁当屋名
    storeName = new ALStringField();
    storeName.setFieldName("弁当屋名");
    storeName.setTrim(true);
    // コメント
    comment = new ALStringField();
    comment.setFieldName("コメント");
    comment.setTrim(true);
    // 締め切り時刻(時)
    deadLineTime_hour = new ALNumberField();
    // 締め切り時刻（分）
    deadLineTime_minute = new ALNumberField();

    // 最低条件有無フラグ
    minStatus = new ALStringField();
    // 最低注文金額
    minPrice = new ALNumberField();
    // 最低注文数
    minNum = new ALNumberField();

    // 最低条件値
    minProperty = new ALNumberField();
    minProperty.setFieldName("条件");
  }

  /**
   * 子テーブル用フィールドを初期化します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initItemField(RunData rundata) {
    itemFieldlength = getItemFieldSize(rundata);// フィールドの数を取得

    for (int i = 1; i < itemFieldlength; i++) {// フィールドの数分初期化

      item_id[i] = new ALNumberField();// hiddenで送るフィールド初期化

      item_name[i] = new ALStringField();// 登録商品を入力するフィールド初期化
      item_name[i].setNotNull(true);// NotNull制約を付け加える
      item_name[i].setFieldName("商品名:" + (i) + "行目");// フィールド名（エラーメッセージ表示時）

      item_price[i] = new ALNumberField();// 価格を入力するフィールド初期化
      item_price[i].setNotNull(true);// NotNull制約を付け加える
      item_price[i].setFieldName("価格:" + (i) + "行目");// フィールド名（エラーメッセージ表示時）

      item_comment[i] = new ALStringField();// コメントを入力するフィールド初期化
      item_comment[i].setNotNull(true);// NotNull制約を付け加える
      item_comment[i].setFieldName("商品コメント:" + (i) + "行目");// フィールド名（エラーメッセージ表示時）

      

    }
  }

  /**
   * プレビュー画面から遷移時にセッションに格納している情報を取得 <BR>
   *      
   */
  public void loadPreViewReturnList(RunData rundata, Context context) {
    try {
      itemFieldList = new ArrayList<Object>();

      // 表示する各項目情報をセッションから取り出す
      //弁当屋名
      String sessionStoreName = new String(ALEipUtils.getTemp(rundata, context,
          "com.aimluck.eip.lunchorder.storeNameListKey"));
      storeName.setValue(sessionStoreName);
      //弁当屋コメント
      String sessionStoreComment = new String(ALEipUtils.getTemp(rundata,
          context, "com.aimluck.eip.lunchorder.storeCommentListKey"));
      comment.setValue(sessionStoreComment);
      //締め切り時刻（時）
      String sessionDeadLineHour = new String(ALEipUtils.getTemp(rundata,
          context, "com.aimluck.eip.lunchorder.deadLineHourListKey"));
      deadLineTime_hour.setValue(sessionDeadLineHour);
      //締め切り時刻（分）
      String sessionDeadLineMinute = new String(ALEipUtils.getTemp(rundata,
          context, "com.aimluck.eip.lunchorder.deadLineMinuteListKey"));
      deadLineTime_minute.setValue(sessionDeadLineMinute);
      //最低条件種類
      String sessionMinStatus = new String(ALEipUtils.getTemp(rundata, context,
          "com.aimluck.eip.lunchorder.minStatusListKey"));
      minStatus.setValue(sessionMinStatus);
      //最低条件値
      String sessionMinProperty = new String(ALEipUtils.getTemp(rundata,
          context, "com.aimluck.eip.lunchorder.minPropertyListKey"));
      minProperty.setValue(sessionMinProperty);
      //商品ID
      String sessionItemId = new String(ALEipUtils.getTemp(rundata, context,
          "com.aimluck.eip.lunchorder.itemIdListKey"));
      String[] previewItemId = sessionItemId.split(",",-1);
      //previewItemId = sessionItemId;
      //商品名
      String sessionItemName = new String(ALEipUtils.getTemp(rundata, context,
          "com.aimluck.eip.lunchorder.itemNameListKey"));
      String[] previewItemName = sessionItemName.split(",",-1);
    
      //商品コメント
      String sessionItemComment = new String(ALEipUtils.getTemp(rundata,
          context, "com.aimluck.eip.lunchorder.itemCommentListKey"));
      String[] previewItemComment = sessionItemComment.split(",",-1);
    
      //商品価格
      String sessionItemPrice = new String(ALEipUtils.getTemp(rundata, context,
          "com.aimluck.eip.lunchorder.itemPriceListKey"));
      String[] previewItemPrice = sessionItemPrice.split(",",-1);
      
      //商品最低条件
      String sessionItemMinOrderFlag = new String(ALEipUtils.getTemp(rundata,
          context, "com.aimluck.eip.lunchorder.itemMinOrderFlagListKey"));
      String[] previewItemMinOrderFlag = sessionItemMinOrderFlag.split(",",-1);
      
      int listSize = previewItemName.length;
      // セッション情報をvelocityで使うpreViewListに乗せかえる処理
      for (int i = 0; i < listSize; i++) {
        String returnPreviewItemId = previewItemId[i].trim();
        String returnPreviewItemName = previewItemName[i].trim();
        String returnPreviewItemComment = previewItemComment[i].trim();
        String returnPreviewItemPrice = previewItemPrice[i].trim();
        if (returnPreviewItemPrice.equals("")) {
          break;//プレビュー時に表示させる商品がなっかた場合、プレビュー画面からの商品項目のセッションを返さない
        }
        String returnPreviewItemMinOrderFlag = previewItemMinOrderFlag[i].trim();
        LunchOrderPreViewResultData rd = new LunchOrderPreViewResultData();

        rd.initField();
        rd.setItemId(returnPreviewItemId);
        rd.setItemName(returnPreviewItemName);
        rd.setItemComment(returnPreviewItemComment);
        rd.setItemPrice(returnPreviewItemPrice);
        rd.setItemMinOrderFlag(returnPreviewItemMinOrderFlag);
        itemFieldList.add(rd);
      }

    } catch (Exception ex) {
      logger.error("Exception", ex);
    }

  }

  /**
   * 注文表にある商品入力フィールドの数を取得するメソッド
   * 
   * @param rundata
   * @return
   */
  public int getItemFieldSize(RunData rundata) {
    String itemFieldSizes[] = (rundata.getParameters()
        .getStrings("item_name[]"));
    int itemFieldSize = itemFieldSizes.length;
    return itemFieldSize;
  }

  /**
   * 注文表にある更新商品入力フィールドの数を取得するメソッド（hiddenパラメータがある商品）
   * 
   * @param rundata
   * @return
   */
  public int getUpdataItemFieldSize(RunData rundata) {
    try {
      String itemUpdataFieldSizes[] = (rundata.getParameters()
          .getStrings("item_id[]"));

      for (int i = 0; i < itemUpdataFieldSizes.length; i++) {
        String itemId = itemUpdataFieldSizes[i];
        if (!(itemId.equals(""))) {
          itemUpdataFieldSize = i;
        }
      }
    } catch (Exception ex) {
      // itemUpdataFieldSizes[]がＮＵＬＬだったら０を返す
      logger.error("Exception", ex);
      return 0;
    }
    return itemUpdataFieldSize;
  }

  /**
   * プレビュー画面からの遷移かどうかの判断フラグ取得 <BR>
   * 
   * @return
   */
  public String getReturnMode() {
    return returnMode;
  }

  /**
   * 弁当屋名を取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getStoreName() {
    return storeName;
  }

  /**
   * コメントを取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getComment() {
    return comment;
  }

  /**
   * 締め切り時刻（hour）を取得します。 <BR>
   * 
   * @return
   */
  public ALNumberField getDeadLineTimeHour() {
    return deadLineTime_hour;
  }

  /**
   * 締め切り時刻（minute）を取得します。 <BR>
   * 
   * @return
   */
  public ALNumberField getDeadLineTimeMinute() {
    return deadLineTime_minute;
  }

  /**
   * 最低条件を取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getMinStatus() {
    return minStatus;
  }

  /**
   * 最低金額を取得します。 <BR>
   * 
   * @return
   */
  public ALNumberField getMinPrice() {
    return minPrice;
  }

  /**
   * 最低注文数を取得します。 <BR>
   * 
   * @return
   */
  public ALNumberField getMinNum() {
    return minNum;
  }

  /**
   * 最低条件値を取得します。 <BR>
   * 
   * @return
   */
  public ALNumberField getMinProperty() {
    return minProperty;
  }

  /**
   * 子テーブル（商品入力フィールド）の数を取得します。引数にrundataを使わないﾊﾟﾀｰﾝ <BR>
   * 
   * @return
   */
  public int getItemFieldlength() {
    return itemFieldlength;
  }

  /**
   * 子テーブル（商品入力フィールド）一覧を取得します。 <BR>
   * 
   * @return
   */
  public ArrayList<Object> getItemFieldList() {
    return itemFieldList;
  }

  /**
   * 新規追加モード(insert)でのプレビューか編集モード(edit)でのプレビューかの
   * modeを取得するために呼び出す <BR>
   * 
   * @return
   */
  @Override
  public boolean doViewForm(ALAction action, RunData rundata, Context context) {
    try {
      init(action, rundata, context);
      boolean isedit = (ALEipUtils.getTemp(rundata, context,
          ALEipConstants.ENTITY_ID) != null);

      int aclType = ALAccessControlConstants.VALUE_ACL_INSERT;
      if (isedit) {
        aclType = ALAccessControlConstants.VALUE_ACL_UPDATE;
      }
      doCheckAclPermission(rundata, context, aclType);

      action.setMode(isedit ? ALEipConstants.MODE_EDIT_FORM
          : ALEipConstants.MODE_NEW_FORM);
      mode = action.getMode();

      ArrayList msgList = new ArrayList();
      boolean res = (isedit) ? loadFormData(rundata, context, msgList)
          : setFormData(rundata, context, msgList);
      action.setResultData(this);
      action.addErrorMessages(msgList);
      action.putData(rundata, context);
      return res;
    } catch (ALPermissionException e) {
      ALEipUtils.redirectPermissionError(rundata);
      return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }

  }

  protected void setValidator() {
  }

  protected boolean validate(ArrayList msgList) {

    return (msgList.size() == 0);
  }

  protected boolean loadFormData(RunData rundata, Context context,
      ArrayList msgList) {

    return true;
  }

  protected boolean deleteFormData(RunData rundata, Context context,
      ArrayList msgList) {

    return true;
  }

  protected boolean insertFormData(RunData rundata, Context context,
      ArrayList msgList) {

    return true;
  }

  protected boolean updateFormData(RunData rundata, Context context,
      ArrayList msgList) {

    return true;
  }

}
